/*******************************************************************************

************
** Function: 
************

  Compares wealth changes in the 2007-2009 SCF panel to the HRS in 2006-2008 and 2008-2010.

************
** Inputs  : 
************

  - Survey of consumer finances 2007-2009 panel
	- RAND HRS Longitudinal File

************
** Outputs : 
************

  - FIGH2b.eps
	
*******************************************************************************/

clear *
set more off
macro drop _all

set scheme hack, permanently

********************************************************************************
** Director names
********************************************************************************

do GetDirNames.do

********************************************************************************
** Set age cutoff for the figures
********************************************************************************

global AGE_CUTOFF 65

********************************************************************************
** Load SCF data
********************************************************************************

/*
Summary Extract Data: 
Extract data set of summary variables used in the FEDS article appendix tables – 
all dollar variables inflation-adjusted to 2013 dollars

For more about the panel, see:
https://www.federalreserve.gov/econres/scf_2009p.htm

For the codebook, see:
https://www.federalreserve.gov/econres/files/codebk2009p.txt

The variables in the codebook are the INTERNAL variable names (e.g., P12345).

To actually understand the variables in the EXTRACT, however, one must consult
the SAS program that transforms internal variables into extract variables.

That is here:
https://www.federalreserve.gov/econres/files/bulletin.macro.txt

Also available are: Replicate weights.  

Details:
For many purposes, one must consider the sampling error of SCF estimates.  
However, because detailed information on the sample design cannot be released, 
and because of the complexity of the SCF design, users cannot apply some of the 
standard procedures for variance estimation.  A set of sample replicates has 
been created with bootstrap techniques and analysis weights have been computed 
independently for each replicate.  Analysts may use these weights to make 
approximate estimates of sampling variance.  Replicate weights corresponding 
to P42001 are available.  See the codebook for more details. 

We are currently using the weights in the file itself (wgt09).

*/

use ${scf_panel}, clear

renvars, l

********************************************************************************
** Variables
********************************************************************************

desc *networth*
/*
networth07      double  %10.0g                
networth09      long    %12.0g                
networthdif     double  %10.0g                
networthpct     double  %10.0g 
*/

/*
from bulletin macro:

*   total net worth;
    NETWORTH=ASSET-DEBT;
    IF (NETWORTH<=.Z) THEN PUT Y1= &PID= FIN= NFIN= DEBT= LIQ= CDS= NMMF=
      STOCKS=  BOND=  RETQLIQ=  SAVBND=  CASHLI=  OTHMA=  OTHFIN= 
      VEHIC= HOUSES= ORESRE= NNRESRE= BUS= OTHNFIN=
      MRTHEL= RESDBT= OTHLOC= CCBAL= INSTALL= ODEBT=;
*/

descr *house*
/*
houses07        double  %10.0g                
hhouses07       byte    %8.0g                 
housecl07       byte    %8.0g                 
houses09        double  %10.0g                
hhouses09       byte    %8.0g                 
housecl09       byte    %8.0g                 
housesdif       double  %10.0g                
housespct       double  %10.0g 
*/

desc *wgt*
/*
wgt09           double  %10.0g 
*/

/*
from the bulletin macro:
*   divide weight by 5 so totals estimated on the 5 implicates
    jointly are correct;
    WGT=X42001/5;
*/


desc age*
/*
age07           byte    %8.0g                 
agecl07         byte    %8.0g                 
ageft07         byte    %8.0g                 
ageall07        byte    %8.0g                 
age09           byte    %8.0g                 
agecl09         byte    %8.0g                 
ageft09         byte    %8.0g                 
ageall09        byte    %8.0g 
*/

/*
from the bulletin macro:
*   age of the reference person and categorical variable:
    1:<35, 2:35-44, 3:45-54, 4:55-64, 5:65-74, 6:>=75;
    AGE=X14;
    AGECL=1+(AGE GE 35)+(AGE GE 45)+(AGE GE 55)+(AGE GE 65)+(AGE GE 75);
*/

/*
The household’s characteristics are in part based on the household head. In the 
SCF if a couple is economically dominant in the PEU, then the head is the male 
in an opposite sex couple or the older person in a same-sex couple. If a single 
individual is economically dominant, that person is the household head.
*/

* age categories
lab def AGECL 1 "1.<35" 2 "2.35-44" 3 "3.45-54" 4 "4.55-64" 5 "5.65-74" 6 "6.>=75"
lab val agecl09 AGECL

gen agecl09_5yr = irecode(age09,34,39,44,49,54,59,64,69,74)
lab def AGECL5YR 0 "<35" 1 "35-39" 2 "40-44" 3 "45-49" 4 "50-54" 5 "55-59" ///
                 6 "60-64" 7 "65-69" 8 "70-74" 9 "75+"
lab val agecl09_5yr AGECL5YR	

tab agecl09
/*
    agecl09 |      Freq.     Percent        Cum.
------------+-----------------------------------
      1.<35 |      2,385       12.37       12.37
    2.35-44 |      3,325       17.24       29.61
    3.45-54 |      4,395       22.79       52.40
    4.55-64 |      4,400       22.82       75.21
    5.65-74 |      2,825       14.65       89.86
     6.>=75 |      1,955       10.14      100.00
------------+-----------------------------------
      Total |     19,285      100.00
*/

tab agecl09_5yr
/*

agecl09_5yr |      Freq.     Percent        Cum.
------------+-----------------------------------
        <35 |      2,385       12.37       12.37
      35-39 |      1,460        7.57       19.94
      40-44 |      1,865        9.67       29.61
      45-49 |      1,955       10.14       39.75
      50-54 |      2,440       12.65       52.40
      55-59 |      2,290       11.87       64.27
      60-64 |      2,110       10.94       75.21
      65-69 |      1,680        8.71       83.93
      70-74 |      1,145        5.94       89.86
        75+ |      1,955       10.14      100.00
------------+-----------------------------------
      Total |     19,285      100.00
*/

count
// 19,285

* homeownership
/*
from bulletin macro:
*   homeownership class: 1=owns ranch/farm/mobile home/house/condo/
    coop/etc., 2=otherwise;
*/
lab def HOUSECL 1 "1. owns home" 2 "2. otherwise"
lab val housecl?? HOUSECL
tab housecl07 housecl09,m
tab housecl07 housecl09,m cell nofreq
/*
             |       housecl09
   housecl07 | 1. owns h  2. otherw |     Total
-------------+----------------------+----------
1. owns home |    14,200        390 |    14,590 
2. otherwise |       645      4,050 |     4,695 
-------------+----------------------+----------
       Total |    14,845      4,440 |    19,285 

             |       housecl09
   housecl07 | 1. owns h  2. otherw |     Total
-------------+----------------------+----------
1. owns home |     73.63       2.02 |     75.65 
2. otherwise |      3.34      21.00 |     24.35 
-------------+----------------------+----------
       Total |     76.98      23.02 |    100.00 
*/

********************************************************************************
** Deflator
********************************************************************************

// CPIAUCNS	Consumer Price Index for All Urban Consumers: All Items in U.S. 
//          City Average, Index 1982-1984=100, Monthly, Not Seasonally Adjusted

// Dec 2010 --> 219.179
// Dec 2013 --> 233.049

gen deflator10 = (233.049) / (219.179)

* to convert from 2013 to 2010 values, *divide* by deflator10.

********************************************************************************
** Build the SCF panel
********************************************************************************

gen group = .
replace group = 1 if (housecl07==1 & housecl09==1) // keepers
replace group = 2 if (housecl07==1 & housecl09==2) // liquidators
replace group = 3 if (housecl07==2 & housecl09==2) // continuous renters
replace group = 4 if (housecl07==2 & housecl09==1) // new owners

lab def G ///
1 "1. keepers" ///
2 "2. liquidators" ///
3 "3. cont renters" ///
4 "4. new owners"

lab val group G

gen nw2007 = (networth07 / 1e3) / deflator10 // 2007 networth in 1000s of 2010 dollars
gen nw2009 = (networth09 / 1e3) / deflator10 // 2009 networth in 1000s of 2010 dollars

//gen nwdiff = (networthdif / 1e3) / deflator10 // 2009-2007 change in nw in 1000s of 2010 dollars

drop if (age09<${AGE_CUTOFF} | mi(age09)) //  <-- NOTE: AGE RESTRICTION!

keep y1 group nw2007 nw2009 wgt09

reshape long nw@, i(y1) j(year)
// years = 2007, 2009

collapse (p25) p25=nw ///
         (p50) p50=nw ///
				 (p75) p75=nw ///
				 (p90) p90=nw ///
				 (count) N=nw ///
				 [aw=wgt09], ///
				 by(group year)
				 
list, noobs sepby(group)				 
				 
gen str13 sample = "SCF 2007-2009"	

* save		
save ${save}/tmp_scf.dta, replace

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

clear

********************************************************************************
** HRS
********************************************************************************

local V1 hhidpn ///
		     r*wtcrnh ///
		     h*pickhh ///	
				 h*afhous ///
         h*atotb /// 
		     r*agey_b s*agey_b

local V2 ///
         r@wtcrnh ///
		     h@pickhh ///
		     h@afhous ///
         h@atotb ///
		     r@agey_b s@agey_b
                 
use `V1' using $randhrs, clear
qui: reshape long `V2', i(hhidpn) j(w)

// merge in deflator:
merge m:1 w using ${save}/CPIdeflator, nogen keep(1 3) keepusing(deflator10)

// convert nominal values to constant 2010 dollars
gen atotb10 = (1e-3)*hatotb/deflator10 // net value of total wealth

// Find age of eldest in household
gen ageEldest = .
replace ageEldest = ragey_b if (ragey_b >= sagey_b & ragey_b <.) | (!mi(ragey_b) & mi(sagey_b))
replace ageEldest = sagey_b if (ragey_b  < sagey_b & sagey_b <.) | (!mi(sagey_b) & mi(ragey_b))
// -> pick oldest age in the household

// Home ownership indicator
recode hafhous (1/5=1) (6=0) (7/9=.m), gen(rownhm)
// -> indicator = 1 if own home, = 0 otherwise

// Year
gen year = 1992 + (w-1)*2

// Selected waves
keep if inrange(w,8,10)

// Sort
sort  hhidpn w
order hhidpn w
xtset hhidpn w

// Ages
by hhidpn: egen age08 = max(cond(w== 9,ageEldest,.))
by hhidpn: egen age10 = max(cond(w==10,ageEldest,.))

// Which households
by hhidpn: egen pick08 = max(cond(w== 9,hpickhh,.))
by hhidpn: egen pick10 = max(cond(w==10,hpickhh,.))

// Indicators
by hhidpn: egen own06 = max(cond(w== 8,rownhm,.))
by hhidpn: egen own08 = max(cond(w== 9,rownhm,.))
by hhidpn: egen own10 = max(cond(w==10,rownhm,.))

lab def G ///
1 "1. keepers" ///
2 "2. liquidators" ///
3 "3. cont renters" ///
4 "4. new owners"

*************
** 2006-2008
*************

preserve

keep if inrange(w,8,9) // 2006-2008

gen ownprev = own06
gen owncurr = own08

gen age  = age08
gen pick = pick08

gen group = .
replace group = 1 if (ownprev==1 & owncurr==1) // keepers
replace group = 2 if (ownprev==1 & owncurr==0) // liquidators
replace group = 3 if (ownprev==0 & owncurr==0) // continuous renters
replace group = 4 if (ownprev==0 & owncurr==1) // new owners
lab val group G

// one-wave wealth changes
sort hhidpn w
gen dwlth = S1.atotb10

tabstat dwlth if (ageEldest>=${AGE_CUTOFF} & ageEldest<. & hpickhh==1) [aw=rwtcrnh], ///
  s(n p25 p50 p75) f(%9.0fc) by(group) m
	
keep if (age>=${AGE_CUTOFF} & !mi(age))  // <-- NOTE: AGE RESTRICTION!
keep if (pick==1)

tabstat dwlth [aw=rwtcrnh], ///
  s(n p25 p50 p75) f(%9.0fc) by(group) m
	
collapse (p25) p25=atotb10 ///
         (p50) p50=atotb10 ///
				 (p75) p75=atotb10 ///
				 (p90) p90=atotb10 ///
				 [aw=rwtcrnh], ///
				 by(group year)
				 
list, noobs clean				 
				 	
gen str13 sample = "HRS 2006-2008"	
	
save ${save}/tmp_hrs_2006_2008.dta, replace

restore

*************
** 2008-2010
*************

preserve

keep if inrange(w,9,10) // 2008-2010

gen ownprev = own08
gen owncurr = own10

gen age  = age10
gen pick = pick10

gen group = .
replace group = 1 if (ownprev==1 & owncurr==1) // keepers
replace group = 2 if (ownprev==1 & owncurr==0) // liquidators
replace group = 3 if (ownprev==0 & owncurr==0) // continuous renters
replace group = 4 if (ownprev==0 & owncurr==1) // new owners
lab val group G

// one-wave wealth changes
sort hhidpn w
gen dwlth = S1.atotb10

tabstat dwlth if (ageEldest>=${AGE_CUTOFF} & ageEldest<. & hpickhh==1) [aw=rwtcrnh], ///
  s(n p25 p50 p75) f(%9.0fc) by(group) m
	
keep if (age>=${AGE_CUTOFF} & !mi(age)) // <-- NOTE: AGE RESTRICTION!
keep if (pick==1)

tabstat dwlth [aw=rwtcrnh], ///
  s(n p25 p50 p75) f(%9.0fc) by(group) m

collapse (p25) p25=atotb10 ///
         (p50) p50=atotb10 ///
				 (p75) p75=atotb10 ///
				 (p90) p90=atotb10 ///
				 [aw=rwtcrnh], ///
				 by(group year)
				 
list				 
	
gen str13 sample = "HRS 2008-2010"	
	
save ${save}/tmp_hrs_2008_2010.dta, replace

restore

********************************************************************************
** Append files together
********************************************************************************

use ${save}/tmp_scf.dta, clear
append using ///
  ${save}/tmp_hrs_2006_2008.dta ///
	${save}/tmp_hrs_2008_2010.dta

sort sample group year	
	
list sample group year p50, sepby(sample)	noobs
	
********************************************************************************
** Figure
********************************************************************************	
		
graph tw ///
  (sc p50 year if group==1 & sample=="SCF 2007-2009", c(d) mc(blue ) lc(blue ) msym(Oh) lp(dash ) lw(*2.5)) ///
  (sc p50 year if group==2 & sample=="SCF 2007-2009", c(d) mc(red  ) lc(red  ) msym(T ) lp(solid) lw(*2.5)) ///
	(sc p50 year if group==3 & sample=="SCF 2007-2009", c(d) mc(black) lc(black) msym(x ) lp(dash ) lw(*2.5)) ///
	///
  (sc p50 year if group==1 & sample=="HRS 2006-2008", c(d) mc(blue ) lc(blue ) msym(Oh) lp(dash )) ///
  (sc p50 year if group==2 & sample=="HRS 2006-2008", c(d) mc(red  ) lc(red  ) msym(T ) lp(solid)) ///
	(sc p50 year if group==3 & sample=="HRS 2006-2008", c(d) mc(black) lc(black) msym(x ) lp(dash )) ///
	///
  (sc p50 year if group==1 & sample=="HRS 2008-2010", c(d) mc(blue ) lc(blue ) msym(Oh) lp(dash )) ///
  (sc p50 year if group==2 & sample=="HRS 2008-2010", c(d) mc(red  ) lc(red  ) msym(T ) lp(solid)) ///
	(sc p50 year if group==3 & sample=="HRS 2008-2010", c(d) mc(black) lc(black) msym(x ) lp(dash )) ///	
	///
	, xlabel(2005(1)2011) ///
	  ylabel(,angle(0)) /// <-- Settings for age-65 cutoff
		text(360 2006 "HRS 2006-2008" 320 2010 "HRS 2008-2010" 280 2008 "SCF 2007-2009") ///
		///ylabel(0(100)400,angle(0)) /// <-- Settings for age-50 cutoff
		///text(340 2006 "HRS 2006-2008" 310 2010 "HRS 2008-2010" 250 2008 "SCF 2007-2009") ///
	  ytitle("Median net worth (1000s of 2010 $)") ///
	  legend(order(1 "Keepers" 2 "Liquidators" 3 "Renters") row(1))	
		
graph2tex, epsfile(${figs}/FIGH2b)		
	
********************************************************************************
** Clean up
********************************************************************************

erase ${save}/tmp_scf.dta
erase ${save}/tmp_hrs_2006_2008.dta
erase ${save}/tmp_hrs_2008_2010.dta
