/*******************************************************************************

************
** Function: 
************

  Creates data extracts from model data with end-of-life wealth trajectories.
	These are merged in with the decedent sample in Decedents_Analysis_Trajectories.do
	to construct the data-versus-model comparisons in the article.

************
** Inputs  :
************
	
	- res-merged.dta    (see Model_Get_Baseline_Sample.do)
	
************
** Outputs :
************
	
	- model_traj_data_parents.dta
	- model_traj_data_own_rent.dta
  - model_traj_data_nhr_cr.dta

*******************************************************************************/

clear *
macro drop _all
set more off 

********************************************************************************
** Directory names
********************************************************************************

do GetDirNames.do

********************************************************************************
** Data
********************************************************************************

use ${save}/baseline-merged.dta, clear

********************************************************************************
** Trajectories
********************************************************************************

keep if (nCln==1)
// drop clones

sort idCln t

* drop those who survive until age 95
by idCln: egen tmp = max(Surv95)
drop if (tmp==1)
drop tmp

gen xnriw=.
replace xnriw=0 if (   dead==1)
replace xnriw=1 if (F1.dead==1)
replace xnriw=2 if (F2.dead==1)
replace xnriw=3 if (F3.dead==1)
replace xnriw=4 if (F4.dead==1)

tab xnriw if t>0

* net worth
gen nwp = .
replace nwp = Nwp65  if (t==0) // initial wealth at age 65
replace nwp = ap + h if (t >0) // later ages
* estate
by idCln: egen tmp = max(Beq) // assign Beq to all observations for each (id nCln)
replace nwp = tmp if (dead==1)  // previously assigned only to t=0
drop tmp

* housing
gen own = .
replace own = (h>0) if ~mi(h) & (dead==0)     // core interviews
* get housing bequest, check if >0
by idCln: egen tmp = max(HBeq)
replace own = (tmp>0) if ~mi(tmp) & (dead==1) // exit interview
drop tmp

keep if inrange(xnriw,0,4)
// keep only interviews relevant for trajectories

keep if (t>0)
// don't count age 65 as an interview

******************************
** All (Parents)
******************************

preserve

format nwp %9.0fc

collapse (p10) p10=nwp ///
         (p25) p25=nwp ///
				 (p50) p50=nwp ///
				 (p75) p75=nwp ///
				 (p90) p90=nwp ///
				 , by(xnriw)

list, clean noobs sep(0)			 
				 
save ${save}/model_traj_data_parents.dta, replace
// save copy				 
				 				 
restore

/*

QUANTILES:

MATLAB:
     t=0        t=1      t=2       t=3       t=4
         0         0         0         0         0    p10
   34.2827   51.7319   67.4317   76.3486   81.5133    p25
  139.8675  142.4917  147.4231  150.7249  153.1950    p50
  261.1148  263.8451  273.1466  282.1720  292.3856    p75
  508.9348  516.4706  547.7520  582.1896  616.9839    p90
	
STATA:
    xnriw      p10       p25        p50        p75        p90  
        0   0.0000   34.2834   139.8680   261.1150   508.9400  
        1   0.0000   51.7337   142.4920   263.8460   516.5010  
        2   0.0000   67.4317   147.4230   273.1480   547.7870  
        3   0.0000   76.3487   150.7250   282.1720   582.1950  
        4   0.0000   81.5138   153.1954   292.3870   616.9940  

OBSERVATIONS:
				
MATLAB:

      224,917  t=0
      219,898  t=1
      213,508  t=2
      205,743  t=3
      196,363  t=4

STATA:			
. tab xnriw if t>0

      xnriw |      Freq.     Percent        Cum.
------------+-----------------------------------
          0 |    224,917       21.21       21.21
          1 |    219,898       20.74       41.95
          2 |    213,508       20.13       62.08
          3 |    205,743       19.40       81.48
          4 |    196,363       18.52      100.00
------------+-----------------------------------
      Total |  1,060,429      100.00

// -> Exact match.
			
*/

******************************
** Wealth trajectories by homeownership (own at T-1)
******************************

preserve

by idCln: egen own_1x = max(cond(xnriw==1,own,.))

format nwp %9.0fc

collapse (p10) p10=nwp ///
         (p25) p25=nwp ///
				 (p50) p50=nwp ///
				 (p75) p75=nwp ///
				 (p90) p90=nwp ///
				 , by(xnriw own_1x)

list, clean noobs sep(0)

save ${save}/model_traj_data_own_rent.dta, replace
// save copy
			 
restore			 

******************************
** Wealth trajectories by nursing home stats (NH at T-1)
******************************

preserve

by idCln: egen nhr_1x = max(cond(xnriw==1,NHval,.))

format nwp %9.0fc

collapse (p10) p10=nwp ///
         (p25) p25=nwp ///
				 (p50) p50=nwp ///
				 (p75) p75=nwp ///
				 (p90) p90=nwp ///
				 , by(xnriw nhr_1x)

list, clean noobs sep(0)

save ${save}/model_traj_data_nhr_cr.dta, replace
// save copy
			 
restore			 
			 
********************************************************************************
