/*******************************************************************************

************
** Function: 
************

  Creates a data extract using the text file output from quantitative model 
	In the MATLAB code, this output is stored in 'res' struct. To export the
	data from this struct into text files readable by Stata, see WriteResToTxt.m.

************
** Inputs  :
************
	
	- baseline-cross-section-data.txt  // cross-sectional data output from the model
	- baseline-panel-data.txt          // panel data output from the model
	
************
** Outputs :
************

	- baseline-merged.dta              // file with merged cross-sectional and panel data

************
** Notes:
************
	
** -------------------	
** Cross section data:
** -------------------

%% Individual-level variables:

% res.pan.once._____
%
%  EpInd65: nPan x 1. parent prod. at 65
%  Nwp65:   nPan x 1. net worth at 65
%  h65:     nPan x 1. housing at 65
%  NwpLtc:  nPan x 1 x nCln. net worth upon entering LTC
%  OwnLtc:  nPan x 1. if owner when LTC hits.
%  KeepLtc: nPan x 1. if house is kept when LTC hits.
%  Beq:     nPan x 1 x nCln. bequest
%  HBeq:    nPan x 1 x nCln. housing bequest
%  FBeq:    nPan x 1 x nCln. financial bequest
%  Surv95   nPan x 1. if survived until 95.
            
% res.pan.age._____
%  Liq:      nPan x 1 x nCln. House liquidation.
%  Died:     nPan x 1. Age at death
%  EnterLtc: nPan x 1. Age at LTC

** -----------
** Panel data:
** -----------

%% Individual-interview-level panel variables:

% res.pan.binary._____._____
%
% first blank:
%  IC, FHC, NH, MA, Htm
% 
% second blank:
%  ever: nPan x 1 x nCln. ever in state.
%  val:  nPan x TPan x nCln. state at interview.
%  time: nPan x TPan x nCln. time in state during interview period.
%  ccl:  nPan x TPan. contemporaneous clone. state at interview.

% res.pan.stocks._____
%  ak, ap, h, ind: nPan x TPan x nCln. child assets, parent assets, house
%                  value, closest linear index

% res.pan.flows._____.______
%
% first blank: 
%  Qp, Qk, QPaltrComp, QKaltrComp, gp, gk, gpUnc, gkUnc, ep
%
% second blank:
%  val:    nPan x TPan x nCln. Cumulative flow during interview period.
%  polval: nPan x TPan x nCln. Policy function at interview.
%  ccl:    nPan x TPan. Contemporaneous clone. Cumulative flow.
%  polccl: nPan x TPan. Contemporaneous clone. Policy at interview.
%
% NOTE: gpUnc, gkUnc have only fields val, polval.

*******************************************************************************/

clear *
macro drop _all
set more off 

********************************************************************************
** Directory names
********************************************************************************

do GetDirNames

********************************************************************************
** Set filenames (global macros)
********************************************************************************

global CrossSectionInputFile baseline-cross-section-data.txt
// file with cross-sectional data

global CrossSectionOutputFile baseline-cross-section-data.dta
// where to output cross-sectional data, once imported

global PanelInputFile baseline-panel-data.txt
// file with panel data

global PanelOutputFile baseline-panel-data.dta
// where to output panel data, once imported

global OutputFile baseline-merged.dta
// final output file with merged cross-section and panel results

********************************************************************************
** Import cross-section data from text file
********************************************************************************

import delimited using ${ModelOutput}/${CrossSectionInputFile}, case(preserve) varnames(1)

/*
import delimited [using] filename [, import_delimited_options]

    delimiters("chars"[, collapse|asstring])  use chars as delimiters
		
        By default,
        import delimited will check if the file is delimited by tabs or
        commas based on the first line of data.  Specify delimiters("\t") to
        use a tab character

    varnames(#|nonames)                       treat row # of data as variable
                                                names or the data do not have
                                                variable names
    case(preserve|lower|upper)                preserve the case or read
                                                variable names as lowercase
                                                (the default) or uppercase
*/

renvars Died Liq EnterLtc, prefix(age)
// prefix these variables with age

local VarsWithNaNs ///
  ageEnterLtc ageLiq NwpLtc
	
foreach var of local VarsWithNaNs	{
  destring `var', replace force
  // 'force' b/c includes "NaN" values
}

describe
list in 1

qui compress
save ${save}/${CrossSectionOutputFile}, replace

********************************************************************************
** Import data from text file
********************************************************************************

clear

import delimited using ${ModelOutput}/${PanelInputFile}, case(preserve) varnames(1)

describe
list in 1

qui compress
save ${save}/${PanelOutputFile}, replace

********************************************************************************
** Merge files
********************************************************************************

* assign all cross-section data to t=0
use ${save}/${CrossSectionOutputFile}, clear

gen t = 0
// time 0, equivalent to age 65

merge 1:1 id nCln t using ${save}/${PanelOutputFile}, nogen
// should be zero matches

sort id nCln t

egen idCln = group(id nCln)
xtset idCln t
sort  idCln t

// check: should be no gaps! (OK if unbalanced)

tab dead nCln
// -> should be 250,000 cases where dead=1

********************************************************************************
** Save
********************************************************************************

qui compress

save ${save}/${OutputFile}, replace

********************************************************************************
** Erase temporary files
********************************************************************************

erase ${save}/${CrossSectionOutputFile}
erase ${save}/${PanelOutputFile}

********************************************************************************
