/*******************************************************************************

************
** Function: 
************

  Creates age-at-which-dollars-transferred histogram figure from the model data.

************
** Inputs  :
************
	
	- baseline-merged.dta	(see Model_Get_Baseline_Sample.do)
	
************
** Outputs :
************
	
	- FIG12b.eps

*******************************************************************************/

clear *
macro drop _all
set more off 

********************************************************************************
** Directory names
********************************************************************************

do GetDirNames.do

********************************************************************************
** Data
********************************************************************************

use ${save}/baseline-merged.dta, clear

********************************************************************************
** Dollars transferred by age, histograms
********************************************************************************

keep if (nCln==1)
// drop clones

* net worth
gen nwp = .
replace nwp = Nwp65  if (t==0) // initial wealth at age 65
replace nwp = ap + h if (t >0) // later ages
* add bequest to net worth
by idCln: egen tmp = max(Beq)  // assign Beq to all observations for each (id nCln)
replace nwp = tmp if (dead==1) // previously assigned only to t=0
drop tmp                       // want to assign to nwp to apply wealth threshold below

* define bequest data for analysis (assign bequest data to exit interview)
gen beq = nwp if (dead==1)

* age
gen age = 65 + 2*t
// age in years
by idCln: egen tmp = max(ageDied)
replace age = tmp if (dead==1)
// use exact age at death for exit interview
drop tmp

drop if (t==0)
// don't include age 65
// drop these after re-assigning bequest/age died data (which is assigned to t=0 for now)
// but drop before calculating wealth threshold

* find exclusion threshold
* use only core interviews after age 65
_pctile nwp if (dead!=1), p(95)
return list
local exclThr = `=r(r1)'
di "Dropping observations with wealth above: " %9.4fc `exclThr'
// Dropping observations with wealth above: 1202.4410

count if (nwp < `exclThr') & (dead!=1) // note: core interviews only
//  2,138,164

keep if (nwp < `exclThr')
// note: applies to core and exit interviews!

gen ivp = gpval + Qpval
// parent transfer
// gpval: altruistic gift
// Qpval: exchange-motivated gift

* define ivt data for analysis
gen ivt = ivp if (dead!=1)
// ivt: core interviews only, age 67+

* define components of ivt data for more detailed analysis
gen ivtg = gpval if (dead!=1)  // altruistic
gen ivtQ = Qpval if (dead!=1)  // exchange-motivated
// ivt: core interviews only, age 67+

* define total transfer
egen xfr = rowtotal(ivt beq),m

/*
gen ivk = gkval + Qkval
// child transfer

gen ivnet = ivp - ivk
// net transfer (parent minus child)
*/

//tabstat ivp, by(t) s(n mean)

* Age at which average IVT given
* (take only core interviews)
tabstat age [aw=ivt], s(mean) f(%3.1f)
/*
    variable |      mean
-------------+----------
         age |      75.0
------------------------
*/
// matches

/*
* Age at which average IVT given
* Equivalent formulation
gen ageXivt = age*ivt
qui summ ageXivt
local num = `=r(sum)'
qui summ ivt
local den = `=r(sum)'
local AverageAge = `num'/`den'
di "At at which average IVP given: " %3.1f `AverageAge'
// At at which average IVP given: 75.0
// matches
*/

* Age at which altruistic gift given
tabstat age [aw=gpval] if (dead!=1), s(mean) f(%3.1f)
/*
    variable |      mean
-------------+----------
         age |      73.3
------------------------
*/


* Age at which average BEQ given
tabstat age [aw=beq], s(mean) f(%3.1f)
//tabstat age [aw=nwp] if (dead==1), s(mean) f(%3.1f) // -> equivalent
/*
    variable |      mean
-------------+----------
         age |      83.6
------------------------
*/

* IVT/Bequest Ratio
qui summ ivt if (dead!=1)
local ivtsum = `=r(sum)'
qui summ beq
local beqsum = `=r(sum)'
local ratio = `ivtsum'/`beqsum'
di "IVT/Bequest ratio is: " %4.3f `ratio'
/*
IVT/Bequest ratio is: 0.177
*/

// ********************************
// All of the above match exactly.
// ********************************

* IVT share of all transfers (compare with output below)
local share = `ivtsum' / (`ivtsum' + `beqsum')
di "IVT as share of all transfers is: " %4.3f `share'
/*
IVT as share of all transfers is: 0.151
*/

gen ageCat = irecode(age,69.999,74.999,79.999,84.999,89.999)
lab def AGECAT 0 "65-69" 1 "70-74" 2 "75-79" 3 "80-84" 4 "85-89" 5 "90+"
lab val ageCat AGECAT	
tabstat age, by(ageCat) s(min max)
/*
ageCat |       min       max
-------+--------------------
 65-69 |        65   69.9565
 70-74 |        70   74.9565
 75-79 |        75   79.9565
 80-84 |        80   84.9565
 85-89 |        85   89.9565
   90+ |        90   94.9565
-------+--------------------
 Total |        65   94.9565
----------------------------
*/

** Among all dollars given as IVT, what is distribution across ages
tab ageCat [aw=ivt]

** Among all dollars given as bequests, what is distribution across ages                                                     
tab ageCat [aw=beq]

** Among all dollars given, what is distribution across ages                                                     
tab ageCat [aw=xfr]

**************************
** Graph that decomposes IVT into altruistic and exchange-motivated
**************************

** Among all dollars given as *altruistic* IVT, what is distribution across ages
tab ageCat [aw=ivtg]
** Among all dollars given as *exchange-motivated* IVT, what is distribution across ages
tab ageCat [aw=ivtQ]

preserve

collapse (sum) ivtg ivtQ beq xfr, by(ageCat)									 
							
egen ivtgTot = total(ivtg)
gen ivtgHist = ivtg/ivtgTot     // -> compare to tab above
egen ivtQTot = total(ivtQ)
gen ivtQHist = ivtQ/ivtQTot     // -> compare to tab above
egen beqTot = total(beq)
gen beqHist = beq/beqTot        // -> compare to tab above
egen xfrTot = total(xfr)
gen xfrHist = xfr/xfrTot        // -> compare to tab above
list ageCat *Hist, sep(0)

* Shares
gen ivtgShr = ivtg/xfrTot  // -> altruistic ivt / total transfers
gen ivtQShr = ivtQ/xfrTot  // -> exchange ivt / total transfers
gen beqShr  = beq/xfrTot   // -> beq / total transfers
gen xfrShr  = xfr/xfrTot

gen xfrChkShr = ivtgShr + ivtQShr + beqShr // check that shares add to total in each row

list ageCat *Shr, sep(0)

* Check math
egen ivtgShrSum = total(ivtgShr)
egen ivtQShrSum = total(ivtQShr)
egen beqShrSum = total(beqShr)
gen checkSum = ivtgShrSum + ivtQShrSum + beqShrSum
egen xfrShrSum = total(xfrShr)
list *Sum in 1, sep(0) clean noobs
/*
    ivtgSh~m   ivtQSh~m   beqShr~m   checkSum   xfrShr~m  
    .1202378   .0302956   .8494666          1          1  
*/

* IVT/Bequest ratio:
gen ivtBeqRat = (ivtgShrSum + ivtQShrSum) / beqShrSum
list ivtBeqRat in 1, sep(0) clean noobs
/*
    ivtBeq~t  
    .1772094 
*/

* Bars
gen bar1 = xfrShr             // total
gen bar2 = ivtQShr + ivtgShr  // ivt
gen bar3 = ivtQShr            // exchange ivt

** Stacked bar 
*  (actually graph three bars, each successive bar in front of the others)
cap graph drop fig2
gr tw ///
(bar bar1  ageCat, barwidth(0.5) ) ///
(bar bar2 ageCat, barwidth(0.5) ) ///
(bar bar3 ageCat, barwidth(0.5) ) ///
, legend(order(1 "Bequests" 2 "Altruistic IVT" 3 "Exchange IVT") rows(1)) ///
  ytitle("Fraction of all dollars transferred" "by parent households ages 65+") ///
	xtitle("Age (5-year bins)") ///
	xlab(0 "65-69" 1 "70-74" 2 "75-79" 3 "80-84" 4 "85-89" 5 "90+") ///
	ylab(0(0.05).30, angle(0) gmax) name(fig2)

graph2tex, epsfile(${figs}/FIG12b) 
	
restore

********************************************************************************
