/*******************************************************************************

************
** Function: 
************

  Generates respondent-level data extract with helper counts, 
	core interviews 1998-2012, exit interviews 2002-2012.

************
** Inputs  : 
************ 

  Functional limitations and helpers, respondent-level file (E_R, G_R)

************
** Outputs : 
************

  - helper_counts.dta
	
************
** Notes   : 
************
	
	Notes on R and HP Files:
	-----------------------
	
  Spouse helpers are not recorded in HP files prior to 2000 (w=5). There is no
	entry for a spouse/partner helper. Their hours are not recorded.
	
	Employee of institution helpers are not recorded in HP files prior to 2002
	(w=6). Their hours are not recorded in the HP files of ANY interview wave.
	
	Spouse and employee of institution helpers can be identified in R files though
	these files do not record their hours. The variables are 'who helps' and 
	'helper type' or 'helper relationship' for ADLs, IADLs, and MNY help. 
	
	Unfortunately, the 'who' and the 'type' variables don't always agree on 
	whether R has an NH helper. They match for 2002 and 2006+ but there are a 
	number of disagreements in 2004 (in these cases, 'type' matches with the HP
	file while 'who' does not ... 'type' seems more reliable in this case) and
	in 1998 and 2000 most NH helpers are recorded in the 'who' (opn) question and
	not in the 'type' question (except for a very small number of cases). 
	Suggested solution: For 2002+, use 'type' and for 1998-2000 use both 'who' and 
	'type' (but primarily 'who'). 
	
	Given that the 'type' questions appear more reliable for measuring NH helpers
	in the 2002+ wave, it seems likely that we will undercount NH helpers using
	the 'who' variables in 1998-2000 even using the R files (the HP files do not
	include NH helpers in these waves except by accident).
	
	Note also the R files for 1998-2000 do not record helper 'type' for children 
	(or most other categories of persons with opn's). Children must be identified
	from the HP files or by linking to a dataset of child opn's (such as the RAND
	family files). We use the information in the HP files.
	
	Bottom line: for helper information, data are most reliable in 2002+. For 
	calculating care hours from all sources except spouse/partner and NH, we can 
	use 1998+. 

*******************************************************************************/

clear *
set more off
macro drop _all

********************************************************************************
** Directory names
********************************************************************************

do GetDirNames.do

********************************************************************************
** Core 2002-2012
********************************************************************************
/*
=====================
Nursing home resident
=====================

*A028 R IN NURSING HOME

==========
Checkpoint
==========

NG013 CHKPNT: COUNT OF LG001 THROUGH LG012

NOTE: ADLs asked if *G013>0

====
ADLs
====

NG014 DIFFICULTY- DRESSING
NG015 HELP W/DRESS

NG016 DIFFICULTY WALKING
NG020 ADL WALK HELP

NG021 DIFFICULTY BATHING
NG022 ADL BATHE HELP

NG023 DIFFICULTY EATING
NG024 ADL EAT HELP

NG025 DIFFICULTY GET IN/OUT BED
NG029 ADL BED HELP

NG030 DIFFICULTY USING TOILET
NG031 ADL TOILET HELP

===========
ADL helpers
===========

NG032_* WHO HELPS MOST- ADL- *

  096.  EMPLOYEE OF INSTITUTION

NG033_* ADL HELPER RELATIONSHIP TO R- *

   2.  SPOUSE/PARTNER
	22.  EMPLOYEE OF INSTITUTION
	
=====
IADLs
=====

NG041 IADL MEAL PREPARATION DIFFICULTY
NG042 WHY- MEAL PREPARATION DIFFICULTY
NG043 IADL MEAL PREPARATION HELP

NG044 IADL GROC SHOP DIFFICULTY
NG045 WHY- GROC SHOP DIFFICULTY
NG046 IADL GROC SHOP HELP

NG047 IADL MAKING PHONE CALLS DIFFICULTY
NG048 WHY- MAKING PHONE CALLS DIFFICULTY
NG049 IADL MAKING PHONE CALLS HELP

NG050 IADL TAKING MEDICATION DIFFICULTY
NG051 IADL TAKING MEDS IF NEEDED DIFFICULTY
NG052 WHY- TAKING MEDICATIONS DIFFICULTY
NG053 IADL TAKING MEDICATIONS HELP	

============
IADL helpers
============

NG054_1 IADLS- WHO HELPS -*
NG055_1 IADL HELPER RELATIONSHIP TO R-*

===
MNY
===

NG059 IADL MANAGING MONEY DIFFICULTY
NG060 WHY- MANAGING MONEY DIFFICULTY
NG061 IADL MANAGING MONEY HELP	

===========
MNY helpers
===========

NG062_* WHO HELPS MANAGE MONEY-*
NG063_* MONEY HELPER RELATIONSHIP TO R- *
	
*/

** 2002-2012
foreach file in h12 h10 h08 h06 h04 h02 {
	use HHID PN ///
      ?G032_* ?G033_*	///                                                                - ADL helpers
      ?G054_* ?G055_* ///                                                                - IADL helpers
      ?G062_* ?G063_*	///                                                                - MNY helpers
	using "${hrscore}/`file'g_r.dta", clear
	merge 1:1 HHID PN using "${hrscore}/`file'a_r.dta", keepusing(?A028) nogen
	renvars,l
	renvars ?g*, predrop(1)	
	renvars ?a*, predrop(1)
	descr
	
	ren a028 nhmliv
	
	* Notes:
	*  2. spouse/partner
	* 22. employee of institution
	
	* Identify spouse helpers
	egen sppADLhlpr  = anymatch(g033_*), values(2)
	egen sppIADLhlpr = anymatch(g055_*), values(2)
	egen sppMNYhlpr  = anymatch(g063_*), values(2)
	egen spphlpr = rowmax(spp*hlpr)
	
	* Identify nursing home helpers
	egen nhADLhlpr  = anymatch(g033_*), values(22)
	egen nhIADLhlpr = anymatch(g055_*), values(22)
	egen nhMNYhlpr  = anymatch(g063_*), values(22)
	egen nhhlpr = rowmax(nh*hlpr)
	
	* Any helper
	quietly destring g032_* g054_* g062_*, replace
	egen anyADLhlpr  = rownonmiss(g032_*)
	recode anyADLhlpr (1/max=1)
	egen anyIADLhlpr = rownonmiss(g054_*)
	recode anyIADLhlpr (1/max=1)
	egen anyMNYhlpr  = rownonmiss(g062_*)
	recode anyMNYhlpr (1/max=1)
	egen anyhlpr = rowmax(anyADLhlpr anyIADLhlpr anyMNYhlpr)
	
  drop g032_* g054_* g062_* ///
       g033_* g055_* g063_* ///
			 spp*???hlpr nh*???hlpr any*???hlpr
	
	describe
	
	tempfile `file'
	save ``file''
}

********************************************************************************
** Core 2000
********************************************************************************
/*
=====================
Nursing home resident
=====================

G558 R IN NURSING HOME

====
ADLs
====

G2722_* E70Y2.CKPT FOR SKIP/ASK ADL SERIES-*****

G2723 E73Y1.DRESS DIFF
G2724 E73F.ADL DRESS HELP

G2725 E72.WALK DIFF
G2729 E72Y1.ADL WALK HELP

G2742 E74.BATHING DIFF
G2745 E74F.ADL BATHE HELP

G2752 E75.EAT DIFF
G2755 E75F.ADL EAT HELP

G2762 E76.BED DIFF
G2768 E76F.ADL BED HELP

G2775 E77.TOILET DIFF
G2778 E77F.ADL TOILET HELP

===========
ADL helpers
===========

G2800 E83.WHO HELP-11

  036. SPOUSE
	100. EMPLOYEE OF FACILITY

G2806 E83A.TYPE HELPER-1

   3. EMPLOYEE OF "INSTITUTION"

G2814 E84.WHO HELP-2
G2815 E84A.TYPE HELPER-2
G2823 E85.WHO HELP-3
G2824 E85A.TYPE HELPER-3
G2827 E86.WHO HELP-4
G2828 E86A.TYPE HELPER-4
G2831 E87.WHO HELP-5
G2832 E87A.TYPE HELPER-5
G2835 E88.WHO HELP-6
G2836 E88A.TYPE HELPER-6
G2839 E89.WHO HELP-7
G2840 E89A.TYPE HELPER-7	
	
=====
IADLs
=====

G2860 E95.IADL MEALS DIFF
G2862 E95B.MEALS -WHY DONT
G2863 E95C.IADL MEALS HELP

G2865 E96.IADL GROC DIFF
G2867 E96B.GROC -WHY DONT
G2868 E96C.IADL GROC HELP

G2870 E97.IADLS PHONE DIFF
G2872 E97B.PHONE -WHY DONT
G2873 E97C.IADL PHONE HELP

G2875 E98.IADLS MEDICATION DIFF
G2876 E98A.IADLS MED IF NEEDED DIFF
G2877 E98B.MED-WHY DONT
G2878 E98C.IADL MED HELP

============
IADL helpers
============

G2880 E99.IADLS-WHO HELP-1
G2881 E99A.TYPE IADL HELPER-1
G2889 E100.IADLS-WHO HELP-2
G2890 E100A.TYPE IADL HELPER-2
G2894 E101.IADLS-WHO HELP-3
G2895 E101A.TYPE IADL HELPER-3
G2900 E102.IADLS-WHO HELP-4
G2901 E102A.TYPE IADL HELPER-4
G2906 E103.IADLS-WHO HELP-5
G2907 E103A.TYPE IADL HELPER-5
G2912 E104.IADLS-WHO HELP-6
G2913 E104A.TYPE IADL HELPER-6

===
MNY
===

G2916 E106.IADL MONEY DIFF
G2917 E106A.MONEY HEALTH PROBLEM
G2918 E106C.IADL MONEY WHO HELP-1   // Does anyone ever help you manage your money?

===========
MNY helpers
===========

G2919 E107. MONEY HELP-1
G2920 E107A.TYPE MONEY HELPER-1
G2923 E108. MONEY HELPER-2
G2924 E108A.TYPE MONEY HELPER-2
	
*/

use HHID PN ///
    G2800 G2806 G2814 G2815 G2823 G2824 G2827 G2828 G2831 G2832 G2835 G2836 G2839 G2840 /// - ADL helpers
    G2880 G2881 G2889 G2890 G2894 G2895 G2900 G2901 G2906 G2907 G2912 G2913             /// - IADL helpers
    G2919 G2920 G2923 G2924                                                             /// - MNY helpers 
using "${hrscore}/h00e_r.dta", clear

merge 1:1 HHID PN using "${hrscore}/h00cs_r.dta", keepusing(G558) nogen
// -> get nursing home data

renvars,l

ren g558 nhmliv

** Identify types of helpers

* Notes:
* ------
* Who helps?
*  "036". spouse/partner
*  "100". employee of facility
*
* Type of helper?
*      3. employee of "institution" (very, very few of these)

* Convert 'who helps' from string to numeric
destring g2800 g2814 g2823 g2827 g2831 g2835 g2839 , replace
destring g2880 g2889 g2894 g2900 g2906 g2912, replace
destring g2919 g2923, replace

* Identify spouse helpers using 'who helps'
egen sppADLhlpr  = anymatch(g2800 g2814 g2823 g2827 g2831 g2835 g2839), values(36)
egen sppIADLhlpr = anymatch(g2880 g2889 g2894 g2900 g2906 g2912), values(36)
egen sppMNYhlpr  = anymatch(g2919 g2923), values(36)
egen spphlpr = rowmax(spp*hlpr)

* Identify nursing home helpers using 'who helps' ...
egen nhADLhlpr_1  = anymatch(g2800 g2814 g2823 g2827 g2831 g2835 g2839), values(100)
egen nhIADLhlpr_1 = anymatch(g2880 g2889 g2894 g2900 g2906 g2912), values(100)
egen nhMNYhlpr_1  = anymatch(g2919 g2923), values(100)

* ... and 'helper type'
egen nhADLhlpr_2  = anymatch(g2806 g2815 g2824 g2828 g2832 g2836 g2840), values(3)
egen nhIADLhlpr_2 = anymatch(g2881 g2890 g2895 g2901 g2907 g2913), values(3)
egen nhMNYhlpr_2  = anymatch(g2920 g2924), values(3)

tab1 nh???*hlpr_?
// -> nearly all helpers are identified using 'who helps' rather than 'helper type'

egen nhADLhlpr  = rowmax(nhADLhlpr_?)
egen nhIADLhlpr = rowmax(nhIADLhlpr_?)
egen nhMNYhlpr  = rowmax(nhMNYhlpr_?)
// -> combine each instance of two measures of nh helpers into one

egen nhhlpr = rowmax(nh*hlpr)

* Any helper
egen anyADLhlpr  = rownonmiss(g2800 g2814 g2823 g2827 g2831 g2835 g2839)
recode anyADLhlpr (1/max=1)
egen anyIADLhlpr = rownonmiss(g2880 g2889 g2894 g2900 g2906 g2912)
recode anyIADLhlpr (1/max=1)
egen anyMNYhlpr  = rownonmiss(g2919 g2923)
recode anyMNYhlpr (1/max=1)
egen anyhlpr = rowmax(anyADLhlpr anyIADLhlpr anyMNYhlpr)

drop nh???*hlpr_? /// 
     g2800 g2814 g2823 g2827 g2831 g2835 g2839 ///
     g2880 g2889 g2894 g2900 g2906 g2912 ///
		 g2919 g2923 ///
		 g2806 g2815 g2824 g2828 g2832 g2836 g2840 ///
		 g2881 g2890 g2895 g2901 g2907 g2913 ///
		 g2920 g2924 ///
		 spp*???hlpr nh*???hlpr any*???hlpr

descr
tempfile h00
save `h00'

********************************************************************************
** Core 1998
********************************************************************************

/*
=========
CORE 1998
=========

=====================
Nursing home resident
=====================

F517 R IN NURSING HOME

====
ADLs
====

F2424_* ADLCK.CKPT FOR SKIPPING ADL SERIES-*****

F2425 E73F.DRESS DIFF
F2426 E73F.ADL DRESS HELP
F2427 E72.WALK DIFF
F2431 E72.ADL WALK HELP
F2444 E74.BATHING DIFF
F2447 E74F.ADL BATHE HELP
F2454 E75.EAT DIFF
F2457 E75F.ADL EAT HELP
F2464 E76.BED DIFF
F2470 E76F.ADL BED HELP
F2477 E77.TOILET DIFF
F2480 E77F.ADL TOILET HELP

===========
ADL helpers
===========

F2502 E83.WHO HELP-11
F2508 E83A.TYPE HELPER-1
F2516 E84.WHO HELP-2
F2517 E84A.TYPE HELPER-2
F2525 E85.WHO HELP-3
F2526 E85A.TYPE HELPER-3
F2529 E86.WHO HELP-4
F2530 E86A.TYPE HELPER-4
F2533 E87.WHO HELP-5
F2534 E87A.TYPE HELPER-5
F2537 E88.WHO HELP-6
F2538 E88A.TYPE HELPER-6
F2541 E89.WHO HELP-7
F2542 E89A.TYPE HELPER-7
	
'WHO'

036. SPOUSE
100. EMPLOYEE OF FACILITY

'TYPE'

  3. EMPLOYEES OF "INSTITUTION"
	
=====
IADLs
=====

F2562 E95.IADL MEALS DIFF
F2564 E95B.MEALS -WHY DONT
F2565 E95C.IADL MEALS HELP
F2567 E96.IADL GROC DIFF
F2569 E96B.GROC -WHY DONT
F2570 E96C.IADL GROC HELP
F2572 E97.IADLS PHONE DIFF
F2574 E97B.PHONE -WHY DONT
F2575 E97C.IADL PHONE HELP
F2577 E98.IADLS MEDICATION DIFF
F2578 E98A                       // -> 'if' question is unlabeled in codebook
F2579 E98B.MED-WHY DONT
F2580 E98C.IADL MED HELP

============
IADL helpers
============

F2582 E99.IADLS-WHO HELP,1
F2583 E99A.TYPE IADL HELPER-1
F2591 E100.IADLS-WHO HELP,2
F2592 E100A.TYPE IADL HELPER-2
F2596 E101.IADLS-WHO HELP,3
F2597 E101A.TYPE IADL HELPER-3
F2602 E102.IADLS-WHO HELP,4
F2603 E102A.TYPE IADL HELPER-4
F2608 E103.IADLS-WHO HELP,5
F2609 E103A.TYPE IADL HELPER-5
F2614 E104.IADLS-WHO HELP,6
F2615 E104A.TYPE IADL HELPER-6

===
MNY
===

F2618 E106.IADL MONEY DIFF
F2619 E106A.MONEY HEALTH PROBLEM
F2620 E106C.IADL MONEY WHO HELP,1

===========
MNY helpers
===========

F2621 E107. MONEY HELP-1
F2622 E107A.TYPE MONEY HELPER-1
F2625 E108. MONEY HELPER-2
F2626 E108A.TYPE MONEY HELPER-2
	
*/

use HHID PN ///
    F2502 F2516 F2525 F2529 F2533 F2537 F2541                                     /// - ADL helpers - who
    F2508 F2517 F2526 F2530 F2534 F2538 F2542                                     /// - ADL helpers - type
		F2582 F2591 F2596 F2602 F2608 F2614                                           /// - IADL helpers - who
    F2583 F2592 F2597 F2603 F2609 F2615                                           /// - IADL helpers - type
		F2621 F2625                                                                   /// - MNY helpers - who
    F2622 F2626                                                                   /// - MNY helpers - type
using "${hrscore}/h98e_r.dta", clear

merge 1:1 HHID PN using "${hrscore}/h98cs_r.dta", keepusing(F517) nogen
// -> get nursing home data

renvars,l

ren f517 nhmliv

** Identify types of helpers

* Notes:
* ------
* Who helps?
*  "036". spouse/partner
*  "100". employee of facility
*
* Type of helper?
*      3. employee of "institution" (very, very few of these)

* Convert 'who helps' from string to numeric
destring f2502 f2516 f2525 f2529 f2533 f2537 f2541, replace
destring f2582 f2591 f2596 f2602 f2608 f2614, replace
destring f2621 f2625, replace

* Identify spouse helpers using 'who helps'
egen sppADLhlpr  = anymatch(f2502 f2516 f2525 f2529 f2533 f2537 f2541), values(36)
egen sppIADLhlpr = anymatch(f2582 f2591 f2596 f2602 f2608 f2614), values(36)
egen sppMNYhlpr  = anymatch(f2621 f2625), values(36)
egen spphlpr = rowmax(spp*hlpr)

* Identify nursing home helpers using 'who helps' ...
egen nhADLhlpr_1  = anymatch(f2502 f2516 f2525 f2529 f2533 f2537 f2541), values(100)
egen nhIADLhlpr_1 = anymatch(f2582 f2591 f2596 f2602 f2608 f2614), values(100)
egen nhMNYhlpr_1  = anymatch(f2621 f2625), values(100)

* ... and 'helper type'
egen nhADLhlpr_2  = anymatch(f2508 f2517 f2526 f2530 f2534 f2538 f2542), values(3)
egen nhIADLhlpr_2 = anymatch(f2583 f2592 f2597 f2603 f2609 f2615), values(3)
egen nhMNYhlpr_2  = anymatch(f2622 f2626), values(3)

tab1 nh???*hlpr_?
// -> nearly all helpers are identified using 'who helps' rather than 'helper type'

egen nhADLhlpr  = rowmax(nhADLhlpr_?)
egen nhIADLhlpr = rowmax(nhIADLhlpr_?)
egen nhMNYhlpr  = rowmax(nhMNYhlpr_?)
// -> combine each instance of two measures of nh helpers into one

egen nhhlpr = rowmax(nh*hlpr)

* Any helper
egen anyADLhlpr  = rownonmiss(f2502 f2516 f2525 f2529 f2533 f2537 f2541)
recode anyADLhlpr (1/max=1)
egen anyIADLhlpr = rownonmiss(f2582 f2591 f2596 f2602 f2608 f2614)
recode anyIADLhlpr (1/max=1)
egen anyMNYhlpr  = rownonmiss(f2621 f2625)
recode anyMNYhlpr (1/max=1)
egen anyhlpr = rowmax(anyADLhlpr anyIADLhlpr anyMNYhlpr)

drop nh???*hlpr_? /// 
		 f2502 f2516 f2525 f2529 f2533 f2537 f2541 ///
		 f2582 f2591 f2596 f2602 f2608 f2614 ///
		 f2621 f2625 ///
		 f2508 f2517 f2526 f2530 f2534 f2538 f2542 ///
		 f2583 f2592 f2597 f2603 f2609 f2615 ///
		 f2622 f2626 ///
		 spp*???hlpr nh*???hlpr any*???hlpr
		 
descr
tempfile h98
save `h98'

********************************************************************************
** Append files
********************************************************************************

use `h12', clear
append using `h10' `h08' `h06' `h04' `h02' `h00' `h98', gen(w)
replace w = 11 - w

order hhid pn w
sort hhid pn w

********************************************************************************
** Check helper variables across waves
********************************************************************************

* any helper
tab w anyhlpr, row nof m

* any nh helper
tab w nhhlpr, row nof m

* any sp/p helper
tab w spphlpr, row nof m

// -> actually looks good across waves, even for 1998-2000

* living in a nursing home
tab w nhmliv, row nof m

// -> also looks comparable across waves

********************************************************************************
** Examine NH resident cases
********************************************************************************

tab w anyhlpr if (nhmliv==1), m row nof
// -> about 15% of NH residents have no reported helper...

tab nhmliv nhhlpr, m row nof
// -> about 80% of those living in NH have an NH helper...
//    and about 20% do not...

* Examine cases where R is NH resident but has no NH helpers
tab w if (nhmliv==1 & nhhlpr==0)
// -> cases appear across waves, fairly evenly

tab w anyhlpr if (nhmliv==1 & nhhlpr==0), m row nof
// -> about 2/3 of NH residents who don't have an NH helper have no reported helper ...

********************************************************************************
** Save core interview data
********************************************************************************

* check observation count
unique hhid pn
tab w,m

/*
Number of unique values of hhid pn is  33320
Number of records is  157534

          w |      Freq.     Percent        Cum.
------------+-----------------------------------
          4 |     21,384       13.57       13.57
          5 |     19,580       12.43       26.00
          6 |     18,167       11.53       37.54
          7 |     20,129       12.78       50.31
          8 |     18,469       11.72       62.04
          9 |     17,217       10.93       72.97
         10 |     22,034       13.99       86.95
         11 |     20,554       13.05      100.00
------------+-----------------------------------
      Total |    157,534      100.00
*/

save ${save}/helper_counts, replace

********************************************************************************

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

********************************************************************************
** Exit 2004-2012
********************************************************************************

** 2002-2012
foreach file in x12 x10 x08 x06 x04 x02 {
	use HHID PN ///
      ?G032_* ?G033_*	///                                                                - ADL helpers
      ?G054_* ?G055_* ///                                                                - IADL helpers
      ?G062_* ?G063_*	///                                                                - MNY helpers
	using "${hrsexit}/`file'g_r.dta", clear
	noisily capture {
	merge 1:1 HHID PN using "${hrsexit}/`file'a_r.dta", keepusing(?A028) nogen
	}
	noisily capture {
	merge 1:1 HHID PN using "${hrsexit}/`file'a_r.dta", keepusing(?A167) nogen
	}
	renvars,l
	renvars ?g*, predrop(1)	
	renvars ?a*, predrop(1)
	descr
	
	noisily capture {
	ren a028 nhmliv
	}

	noisily capture {
	ren a167 nhmliv
	}	
	
	* Notes:
	*  2. spouse/partner
	* 22. employee of institution
	
	* Identify spouse helpers
	egen sppADLhlpr  = anymatch(g033_*), values(2)
	egen sppIADLhlpr = anymatch(g055_*), values(2)
	egen sppMNYhlpr  = anymatch(g063_*), values(2)
	egen spphlpr = rowmax(spp*hlpr)
	
	* Identify nursing home helpers
	egen nhADLhlpr  = anymatch(g033_*), values(22)
	egen nhIADLhlpr = anymatch(g055_*), values(22)
	egen nhMNYhlpr  = anymatch(g063_*), values(22)
	egen nhhlpr = rowmax(nh*hlpr)
	
	* Any helper
	quietly destring g032_* g054_* g062_*, replace
	egen anyADLhlpr  = rownonmiss(g032_*)
	recode anyADLhlpr (1/max=1)
	egen anyIADLhlpr = rownonmiss(g054_*)
	recode anyIADLhlpr (1/max=1)
	egen anyMNYhlpr  = rownonmiss(g062_*)
	recode anyMNYhlpr (1/max=1)
	egen anyhlpr = rowmax(anyADLhlpr anyIADLhlpr anyMNYhlpr)
	
  drop g032_* g054_* g062_* ///
       g033_* g055_* g063_* ///
			 spp*???hlpr nh*???hlpr any*???hlpr
	
	describe
	
	tempfile `file'
	save ``file''
}

********************************************************************************
** Append files
********************************************************************************

use `x12', clear
append using `x10' `x08' `x06' `x04' `x02', gen(w)
replace w = 11 - w

order hhid pn w
sort hhid pn w

* check observation count
unique hhid pn
tab w,m

/*
          w |      Freq.     Percent        Cum.
------------+-----------------------------------
          6 |      1,501       18.76       18.76
          7 |      1,227       15.34       34.10
          8 |      1,310       16.37       50.47
          9 |      1,330       16.62       67.09
         10 |      1,446       18.07       85.16
         11 |      1,187       14.84      100.00
------------+-----------------------------------
      Total |      8,001      100.00
*/

********************************************************************************
** Merge to core interview data
********************************************************************************

merge 1:1 hhid pn w using ${save}/helper_counts
// should be no matches:
// master is exit
// using is core

* indicator for exit interviews
recode _merge (1=1) (2=0), gen(xiw)
drop _merge
lab drop _merge

********************************************************************************
** Save
********************************************************************************

qui compress

save ${save}/helper_counts, replace

********************************************************************************
