/*******************************************************************************

************
** Function: 
************

  Creates data extract with I/ADL information from exit interviews 2002-2012.
	Includes also age at death.

************
** Inputs  : 
************ 

	- Section G_R
	- Section A_R

************
** Outputs : 
************

	- exit_adls.dta
		
*******************************************************************************/

clear *
set more off
macro drop _all

********************************************************************************
** Directory names
********************************************************************************

do GetDirNames.do

********************************************************************************
** X: FUNCTIONAL LIMITATIONS AND HELPERS (G, RESPONDENT) 
********************************************************************************
  
** 2002-2012
foreach file in X12 X10 X08 X06 x04 x02 {
  use HHID PN ?G015 ?G020 ?G022 ?G024 ?G029 ?G031 /// ?G033_* ///
              ?G043 ?G046 ?G049 ?G053             /// ?G055_1 ///
              ?G061 ?G129  ///
	    using "${hrsexit}/`file'G_R.dta", clear  
			
	* nursing home residency, age		
	* note: nursing home variable name varies across waves. 
	capture noisily {
	merge 1:1 HHID PN using "${hrsexit}/`file'A_R.dta", keepusing(?A028 ?A019) nogen
	}
	capture noisily {
	merge 1:1 HHID PN using "${hrsexit}/`file'A_R.dta", keepusing(?A167 ?A019) nogen
	}
	
	renvars,l
	renvars ?g*, predrop(1)	
	renvars ?a*, predrop(1)
	
	descr
	
	ren a019 xage
	
	capture noisily {
	ren a028 xnhmliv
	}

	capture noisily {
	ren a167 xnhmliv
	}	
	
  tempfile `file'
	save ``file''
}
  
use `X12', clear
append using `X10' `X08' `X06' `X04' `X02', gen(w)
replace w = 11 - w

order hhid pn w
sort hhid pn w
						 

* NOTE:
* The I/ADL questions in the exit interview correspond to 'help' with activities
* rather than 'difficulty' with activities, which are the variables we use from the core IWs. 
* For example, the wording for the walk question is: 'Because of a health or 
* memory problem did anyone help [her/him ] get across a room (in the last three 
* months of [her/his] life)?' in the exit interviews. The corresponding core variable
* that we use asks 'Because of a health or memory problem do you have any difficulty 
* with walking across a room?'

* Rename (I)ADL names
rename (g015    g020    g022   g024   g029   g031    g043    g046   g049    g053   g061) ///
       (xdress  xwalkr  xbath  xeat   xbed   xtoilt  xmeals  xshop  xphone  xmeds  xmoney)  

* Need to recode ADLs and IADLs into binary variables
* ADLs                                        Functional limitation:
recode xdress (5 = 0) ( 6 7 = 1 ) (8 9 = .)    // dress
recode xwalkr (5 = 0) ( 6 7 = 1 ) (8 9 = .)    // walk
recode xbath  (5 = 0) ( 6 7 = 1 ) (8 9 = .)    // bathe
recode xeat   (5 = 0) ( 6 7 = 1 ) (8 9 = .)    // eat
recode xbed   (5 = 0) ( 6 7 = 1 ) (8 9 = .)    // bed
recode xtoilt (5 = 0) ( 6 7 = 1 ) (8 9 = .)    // toilt
* IADLs
recode xmeals (5 = 0) ( 6 7 = 1 ) (8 9 = .)    // meal
recode xshop  (5 = 0) ( 6 7 = 1 ) (8 9 = .)    // shop
recode xphone (5 = 0) ( 6 7 = 1 ) (8 9 = .)    // phone
recode xmeds  (5 = 0) ( 6 7 = 1 ) (8 9 = .)    // medicine
recode xmoney (5 = 0) ( 6 7 = 1 ) (8 9 = .)    // money
// -> question regarding use of map not available in exit IW
// 1.  YES
// 5.  NO
// 6.  COULD NOT DO
// 7.  DID NOT DO

tab xmeals if xnhmliv<5,m  
tab xshop  if xnhmliv<5,m
tab xmeds  if xnhmliv<5,m
//    xnhmliv shows NH status:  5=no, 1=yes, 2=hospice
// -> NHRs are NOT asked about help with "meal", "shop", and "medicine"  
tab xphone if xnhmliv<5,m
tab xmoney if xnhmliv<5,m
// -> NHRs are asked about "phone" and "money"
// Suggestion: set "meal", "shop", "medicine" equal to 1 for NHR 
replace xmeals=1 if xnhmliv<5  
replace xshop=1  if xnhmliv<5
replace xmeds=1  if xnhmliv<5
// -> This becomes important when we impute NH hours based on the number of
//    functional limitations as otherwise we would underestimate the required hours

* G129: days in bed over last three months before death
* Coding varies: 
* 2004-2010 0-100, 993 everyday, 998 dk, 999 rf
* 2002      0-90 , 93  everyday, 98  dk  99  rf
* NOTE: the survey uses G129 < 86 as an ask criterion for ADLs!

ren g129 xnrdaysBed
tab w xnrdaysBed if xnrdaysBed>=86,m

recode xnrdaysBed (86/100 = 90) (993=90) (998=.d) (999=.d) if (w >6)
recode xnrdaysBed (86/90  = 90) ( 93=90) ( 98=.d) ( 99=.d) if (w==6)
// xnrdaysBed: number of days R stayed in bed more than half the day because of
//            illness or injury during last three months before death.
//            90 means that R stayed in bed every day
//            .d means that proxy reporter "does not know"/"refuses" to answer
foreach ADL in xdress xwalkr xbath xeat xbed xtoilt {
  tab `ADL' if xnrdaysBed==90,m
}
// -> If R stayed in bed for more than 85 days because of illness or injury
//    questions regarding ADLs are skipped. 
//    But, the HRS still asks about who most often helps R with 
//    dressing, bathing, eating, getting in and out of bed, using the toilet, 
//    and getting across a room.
foreach ADL in xdress xwalkr xbath xeat xbed xtoilt {
  tab `ADL' if xnrdaysBed==.d,m
}
// -> Those who answered .d are still asked.
// Suggestion: set ADLs equal to 1 for those with xnrdaysBed=90
foreach ADL in xdress xwalkr xbath xeat xbed xtoilt {
  replace `ADL'= 1 if xnrdaysBed==90 
}

foreach IADL in xmeals  xshop  xphone  xmeds  xmoney {
  tab `IADL' if xnrdaysBed==90,m
}
// ->  But, the HRS does ask about IADLs even for those who stayed in bed for 
//     more than 86 days. 


* Construction of disability index: totadl = adla + iadlza, where
* adla   = sum(BATHA,DRESSA,EATA,BEDA,WALKRA)
* iadlza = sum(PHONEA,MONEYA,MEDSA,SHOPA,MEALA)
egen    xadla  = rowtotal(xdress xwalkr xbath xeat xbed), missing
egen    xiadlza= rowtotal(xmeals xshop xphone xmeds xmoney), missing
// -> rowtotal assigns a zero if all values are missing; if "missing"
//    is specified it assigns a missing value ONLY if ALL values are missing. 

********************************************************************************
** Drop unneeded variables
********************************************************************************

drop xdress xwalkr xbath xeat xbed xtoilt xmeals xshop xphone xmeds xmoney
drop xnrdaysBed xnhmliv

********************************************************************************
** save
********************************************************************************

qui compress

save ${save}/exit_adls, replace

********************************************************************************
