/*******************************************************************************

************
** Function: 
************

  Combines helper information from several .do files into a single .dta file.
	Defines care arrangements, hours fractions, helper indicators, etc.

************
** Inputs  : 
************ 

  - helpers_core.dta
  - helpers_exit.dta
	- helpers_core_add_waves.dta
	- helpers_counts.dta
						
************
** Outputs : 
************

  - helpers_combined.dta
	
************
** Notes   : 
************

  The combined data set includes the following waves:

  Core 1998-2010
	Exit 2004-2012
	
*******************************************************************************/

clear *
set more off
macro drop _all

********************************************************************************
** Directory names
********************************************************************************

do GetDirNames.do

********************************************************************************
** Combine helper files, save
********************************************************************************

use ${save}/helpers_core, clear

merge 1:1 hhid pn w using ${save}/helpers_core_add_waves, nogen
// should be no matches:
// first file is 2002-2012
// second file is 1998-2000

merge 1:1 hhid pn w using ${save}/helpers_exit, nogen
// should be no matches:
// first file is core
// second file is exit (2004-2012)

* save
save ${save}/helpers_combined.dta, replace

********************************************************************************
** Merge to helper counts (from functional limitations R file)
********************************************************************************

use ${save}/helper_counts, clear

merge 1:1 hhid pn w using ${save}/helpers_combined.dta

recode _merge (1  =0) /// master only
              (2 3=1) /// using only or matched
		          , gen(inhpfile)

drop _merge	
lab drop _merge

// this measure is not perfect because: (1) some individuals who live in nursing homes
// do not appear in the hp files in 2002-2012, (2) individuals with NH helpers may not
// appear in hp file prior to 2002 unless they have other helpers, (3) same for individuals
// with sp/p helpers before 2000.

// better to use 'anyhlpr' which is defined below

********************************************************************************
** Drop waves not used in paper
********************************************************************************

drop if (xiw==1 & w<7)
drop if (xiw==0 & w>10)

* use only: core 1998-2010, exit 2004-2012

********************************************************************************
** Correct indicator for nursing home helper / residency
********************************************************************************

renvars anyhlpr nhhlpr spphlpr, postfix(GR)
// GR for section G-R

gen nhanyhlpr = 0
replace nhanyhlpr = 1 if (nhnhlpr>0 & nhnhlpr<.) // count from g-hp file
replace nhanyhlpr = 1 if (nhhlprGR==1)           // indicator from functional limitations g-r file
replace nhanyhlpr = 1 if (nhmliv==1)             // indicator for nh residency from a-r file

gen nhanyhlprInfo = 0
replace nhanyhlprInfo = nhanyhlprInfo + 1   if (nhnhlpr>0 & nhnhlpr<.)
replace nhanyhlprInfo = nhanyhlprInfo + 10  if (nhhlprGR==1)
replace nhanyhlprInfo = nhanyhlprInfo + 100 if (nhmliv==1)

lab def NHINFO ///
  0 "no nh care" ///
	1 "nh hlpr in g-hp file only" ///
 11 "nh hlpr in both g-r/g-hp files" ///
100 "nh res only, no nh hlpr in g-r/g-hp files" ///
110 "nh res, hlpr in g-r file only" ///
111 "nh res, nh hlpr in both g-r/g-hp files"

lab val nhanyhlprInfo NHINFO

tab nhanyhlprInfo, m
  
/*
                          nhanyhlprInfo |      Freq.     Percent        Cum.
----------------------------------------+-----------------------------------
                             no nh care |    137,794       96.04       96.04
              nh hlpr in g-hp file only |          1        0.00       96.04
         nh hlpr in both g-r/g-hp files |        125        0.09       96.12
nh res only, no nh hlpr in g-r/g-hp fil |      1,131        0.79       96.91
          nh res, hlpr in g-r file only |        691        0.48       97.39
 nh res, nh hlpr in both g-r/g-hp files |      3,739        2.61      100.00
----------------------------------------+-----------------------------------
                                  Total |    143,481      100.00
*/
		
********************************************************************************
** Correct indicator for spouse/partner helper
********************************************************************************

gen spanyhlpr = 0
replace spanyhlpr = 1 if (spnhlpr>0 & spnhlpr<.) // count from g-hp file
replace spanyhlpr = 1 if (spphlprGR==1)          // indicator from functional limitations g-r file

********************************************************************************
** Correct indicator for any helper (add new cases with NH helpers)
********************************************************************************

gen anyhlpr = anyhlprGR
replace anyhlpr = 1 if (nhanyhlpr==1)
replace anyhlpr = 1 if (spanyhlpr==1)

********************************************************************************
** Finalize indicators for other categories
********************************************************************************

foreach pref in kd yg ic hc fc {
  gen `pref'anyhlpr = 0
	replace `pref'anyhlpr = 1 if (`pref'nhlpr>0 & `pref'nhlpr<.)
}

********************************************************************************
** Correction for informal care (add new cases with spouse helpers)
********************************************************************************

replace icanyhlpr = 1 if (spanyhlpr==1)

********************************************************************************
** Correction for formal care (add new cases with NH helpers)
********************************************************************************

replace fcanyhlpr = 1 if (nhanyhlpr==1)

********************************************************************************
** Fill in zeros
********************************************************************************			

replace hrswkly = 0 if (anyhlpr==0)

foreach pref in kd sp yg ic nh hc fc {
	replace `pref'hrswkly = 0 if (anyhlpr==0)       // no helper of any type
	replace `pref'hrswkly = 0 if (`pref'anyhlpr==0) // no helper of this type
}

summ *hrswkly, sep(0)

********************************************************************************
** New total weekly hours variable, set to missing for those with NH helper
********************************************************************************

clonevar tohrswkly = hrswkly

** enforce that all individuals with nursing home helpers missing total hours
replace tohrswkly = . if (nhanyhlpr == 1)
// -> (4,888 real changes made, 4,888 to missing)

********************************************************************************
** Hour shares by helper type
********************************************************************************

** NOTE: does not appropriately take into account nursing homes.
**       results relevant only for non-nh users with no missing hours (hrswklyf==0).

gen kdhrsfrac = kdhrswkly / hrswkly

gen yghrsfrac = yghrswkly / hrswkly

gen ichrsfrac = ichrswkly / hrswkly
gen nhhrsfrac = nhhrswkly / hrswkly
gen hchrsfrac = hchrswkly / hrswkly
gen fchrsfrac = fchrswkly / hrswkly

fsum kdhrsfrac if (hrswklyf==0), s(n mean p25 p50 p75 p90 max)
fsum yghrsfrac if (hrswklyf==0), s(n mean p25 p50 p75 p90 max)

fsum ichrsfrac fchrsfrac nhhrsfrac hchrsfrac if (hrswklyf==0), s(n mean p25 p50 p75 p90 max)

********************************************************************************
** Top-code hours from each source at 168 hours per week
********************************************************************************

* NOTE: do after calculating hours shares (above).

foreach pref in to kd sp yg ic nh hc fc {
  di "`pref'hrswkly:"
  replace `pref'hrswkly = 168 if (`pref'hrswkly>168 & `pref'hrswkly<.)
}

/*
tohrswkly:
(544 real changes made)
kdhrswkly:
(73 real changes made)
sphrswkly:
(0 real changes made)
yghrswkly:
(183 real changes made)
ichrswkly:
(351 real changes made)
nhhrswkly:
(0 real changes made)
hchrswkly:
(57 real changes made)
fchrswkly:
(57 real changes made)
*/

********************************************************************************
** care arrangements
********************************************************************************

***************************************
* detailed measure
***************************************

gen caretyped = 0
replace caretyped = 1 if (icanyhlpr==1 & hcanyhlpr==0 & nhanyhlpr==0) // -> only ic helper
replace caretyped = 2 if (icanyhlpr==0 & hcanyhlpr==1 & nhanyhlpr==0) // -> only fhc helper
replace caretyped = 3 if (icanyhlpr==0 & hcanyhlpr==0 & nhanyhlpr==1) // -> only nh helper
replace caretyped = 4 if (icanyhlpr==1 & hcanyhlpr==1 & nhanyhlpr==0) // -> only ic+fhc helper
replace caretyped = 5 if (icanyhlpr==1 & hcanyhlpr==0 & nhanyhlpr==1) // -> only ic+nh helper
replace caretyped = 6 if (icanyhlpr==0 & hcanyhlpr==1 & nhanyhlpr==1) // -> only fhc+nh helper
replace caretyped = 7 if (icanyhlpr==1 & hcanyhlpr==1 & nhanyhlpr==1) // -> all three types of helpers
replace caretyped = .m if (icanyhlpr==0 & hcanyhlpr==0 & nhanyhlpr==0 & anyhlpr==1) 
  // .m cases: has helper according to G-R file, but does not appear in helper file,
	//           type of helper in G-R file is neither spouse/partner nor NH helper

lab def CTYPED ///
  0 "0. no care" ///
	1 "1. ihc only" ///
	2 "2. fhc only" ///
	3 "3. nh only" ///
	4 "4. ihc + fhc" ///
	5 "5. ihc + nh" ///
	6 "6. fhc + nh" ///
	7 "7. ihc + fhc + nh"
lab val caretype CTYPED

tab caretyped
/*
        caretyped |      Freq.     Percent        Cum.
------------------+-----------------------------------
       0. no care |    116,759       80.78       80.78
      1. ihc only |     17,064       11.81       92.58
      2. fhc only |        958        0.66       93.25
       3. nh only |      1,832        1.27       94.51
     4. ihc + fhc |      3,563        2.47       96.98
      5. ihc + nh |      4,006        2.77       99.75
      6. fhc + nh |        104        0.07       99.82
7. ihc + fhc + nh |        256        0.18      100.00
------------------+-----------------------------------
            Total |    144,542      100.00
*/

tab caretyped if (caretyped>0)
/*
        caretyped |      Freq.     Percent        Cum.
------------------+-----------------------------------
      1. ihc only |     17,064       61.42       61.42
      2. fhc only |        958        3.45       64.87
       3. nh only |      1,832        6.59       71.46
     4. ihc + fhc |      3,563       12.82       84.29
      5. ihc + nh |      4,006       14.42       98.70
      6. fhc + nh |        104        0.37       99.08
7. ihc + fhc + nh |        256        0.92      100.00
------------------+-----------------------------------
            Total |     27,783      100.00
*/

tab w if (caretyped==.m)
/*
          w |      Freq.     Percent        Cum.
------------+-----------------------------------
          4 |          4        0.91        0.91
          5 |          1        0.23        1.14
          6 |        420       95.45       96.59
          7 |          4        0.91       97.50
          8 |          2        0.45       97.95
          9 |          1        0.23       98.18
         10 |          6        1.36       99.55
         11 |          2        0.45      100.00
------------+-----------------------------------
      Total |        440      100.00
*/
// -> cases in which G-R file indicates a helper, but individual does not appear in HP file
//    and the helper is neither a spouse nor NH (we tally these two types)
// -> almost all are in wave 6 (2002) (core)
//    appears to be an HRS mistake

* core vs exit
tab caretyped xiw, col nof
/*
                  |   RECODE of _merge
        caretyped |         0          1 |     Total
------------------+----------------------+----------
       0. no care |     84.48      13.91 |     80.78 
      1. ihc only |     10.83      29.45 |     11.81 
      2. fhc only |      0.61       1.66 |      0.66 
       3. nh only |      0.84       8.95 |      1.27 
     4. ihc + fhc |      1.62      17.73 |      2.47 
      5. ihc + nh |      1.50      25.73 |      2.77 
      6. fhc + nh |      0.05       0.42 |      0.07 
7. ihc + fhc + nh |      0.07       2.15 |      0.18 
------------------+----------------------+----------
            Total |    100.00     100.00 |    100.00  
*/

* core vs exit, among care recipients
tab caretyped xiw if (caretyped>0), col nof
/*
                  |   RECODE of _merge
        caretyped |         0          1 |     Total
------------------+----------------------+----------
      1. ihc only |     69.77      34.21 |     61.42 
      2. fhc only |      3.91       1.93 |      3.45 
       3. nh only |      5.42      10.40 |      6.59 
     4. ihc + fhc |     10.44      20.59 |     12.82 
      5. ihc + nh |      9.68      29.88 |     14.42 
      6. fhc + nh |      0.34       0.49 |      0.37 
7. ihc + fhc + nh |      0.44       2.50 |      0.92 
------------------+----------------------+----------
            Total |    100.00     100.00 |    100.00 
*/

***************************************
* condensed measure
***************************************

recode caretyped ///
  (0       = 0) /// no care
  (1       = 1) /// ic only
	(4       = 2) /// ic + fhc only
	(2       = 3) /// fhc only
	(3 5 6 7 = 4) /// any nh care (nh only, ihc+nh, fhc+nh, ihc+fhc+nhc)
	, gen(caretype)

lab def CTYPE 0 "0. no care" 1 "1. ihc only" 2 "2. ihc+fhc" 3 "3. fhc only" 4 "4. any nh care"
lab val caretype CTYPE

tab caretype if (caretype>0)
/*
     RECODE of |
     caretyped |      Freq.     Percent        Cum.
---------------+-----------------------------------
   1. ihc only |     17,064       61.42       61.42
    2. ihc+fhc |      3,563       12.82       74.24
   3. fhc only |        958        3.45       77.69
4. any nh care |      6,198       22.31      100.00
---------------+-----------------------------------
         Total |     27,783      100.00
*/

* core vs exit, among care recipients
tab caretype xiw if (caretype>0), col nof
/*
     RECODE of |   RECODE of _merge
     caretyped |         0          1 |     Total
---------------+----------------------+----------
   1. ihc only |     69.77      34.21 |     61.42 
    2. ihc+fhc |     10.44      20.59 |     12.82 
   3. fhc only |      3.91       1.93 |      3.45 
4. any nh care |     15.87      43.27 |     22.31 
---------------+----------------------+----------
         Total |    100.00     100.00 |    100.00 
*/

***************************************
* indicator for receiving only informal care
***************************************

recode caretype (1=1) (0 2 3 4=0), gen(iconly)

***************************************
* care arrangements, defining ic as >50% informal care
***************************************

* NOTE: may be incorrect in cases where some non-nh helpers have missing hours.
*       may therefore be preferable to use variable defined without using hours fractions.

* NOTE: missing spouse hours in 1998. use pre-2000 data with caution.

clonevar caretype50 = caretype
replace caretype50 = 1  if (caretype==2 & ichrsfrac> .50 & ichrsfrac<.) // ihc+fhc, >50% ihc, reassign to ihc
replace caretype50 = 3  if (caretype==2 & ichrsfrac<=.50)               // ihc+fhc, <=50% ihc, reassign to fhc
replace caretype50 = .m if (caretype==2 & mi(ichrsfrac))  // assign missing if fractions missing, cant reassign
//replace caretype50 = . if (w<5)

lab def CTYPE50 0 "0. no care" 1 "1. ihc (ihc>50%)" 3 "3. fhc (fhc>=50%)" 4 "4. any nh care"
lab val caretype50 CTYPE50

* comparisons
tab caretype if (w>=5)
tab caretype50 if (w>=5)

* comparisons, for those with no missing hours for a non-nh helpers
tab caretype if (w>=5 & hrswklyf<2)
tab caretype50 if (w>=5 & hrswklyf<2)

***************************************
* indicator for receiving >50% informal care and no nursing home care
***************************************

* NOTE: may be incorrect in cases where some non-nh helpers have missing hours.
*       may therefore be preferable to use variable defined without using hours fractions.

recode caretype50 (1=1) (0 2 3 4=0), gen(iconly50)

* comparisons
tab iconly if (w>=5)
tab iconly50 if (w>=5)

* comparisons, for those with no missing hours for a non-nh helpers
tab iconly if (w>=5 & hrswklyf<2)
tab iconly50 if (w>=5 & hrswklyf<2)

***************************************
* measure that ignores nursing home care
***************************************

gen caretypen = 0
replace caretypen = 1 if (icanyhlpr==1 & hcanyhlpr==0) // -> only ic helper
replace caretypen = 2 if (icanyhlpr==0 & hcanyhlpr==1) // -> only fhc helper
replace caretypen = 3 if (icanyhlpr==1 & hcanyhlpr==1) // -> both ic + fhc helper
replace caretypen = 4 if (icanyhlpr==0 & hcanyhlpr==0 & anyhlpr==1) 
  // '4' cases: has a helper that is neither IC nor FHC.
	//            may be NH or may be cases where R has helper in G-R file but not in G-HP files.

lab def CTYPEN ///
  0 "0. no care" ///
	1 "1. ihc only" ///
	2 "2. fhc only" ///
	3 "3. ihc + fhc" ///
	4 "4. other helper case"
lab val caretypen CTYPEN

tab caretypen
/*
           caretypen |      Freq.     Percent        Cum.
---------------------+-----------------------------------
          0. no care |    116,530       81.22       81.22
         1. ihc only |     20,629       14.38       95.59
         2. fhc only |      1,062        0.74       96.33
        3. ihc + fhc |      3,819        2.66       99.00
4. other helper case |      1,441        1.00      100.00
---------------------+-----------------------------------
               Total |    143,481      100.00
*/

tab caretypen if caretypen>0
/*
           caretypen |      Freq.     Percent        Cum.
---------------------+-----------------------------------
         1. ihc only |     20,629       76.54       76.54
         2. fhc only |      1,062        3.94       80.48
        3. ihc + fhc |      3,819       14.17       94.65
4. other helper case |      1,441        5.35      100.00
---------------------+-----------------------------------
               Total |     26,951      100.00
*/

** comparing results to another measure:
tab caretypen if (nhanyhlpr==0) & (caretypen>0),m
/*
           caretypen |      Freq.     Percent        Cum.
---------------------+-----------------------------------
         1. ihc only |     16,723       78.64       78.64
         2. fhc only |        958        4.51       83.15
        3. ihc + fhc |      3,563       16.76       99.91
4. other helper case |         20        0.09      100.00
---------------------+-----------------------------------
               Total |     21,264      100.00
*/

tab caretyped if (nhanyhlpr==0) & (caretyped>0),m
/*

        caretyped |      Freq.     Percent        Cum.
------------------+-----------------------------------
      1. ihc only |     16,723       78.64       78.64
      2. fhc only |        958        4.51       83.15
     4. ihc + fhc |      3,563       16.76       99.91
               .m |         20        0.09      100.00
------------------+-----------------------------------
            Total |     21,264      100.00
*/
// should be identical

lab var caretypen "care arrangement, ignoring NH use"

********************************************************************************
** home care - formal / informal mix
********************************************************************************

recode caretype ///
 (0 = .) /// no care
 (1 = 1) /// ihc only
 (2 = 2) /// ihc + fhc
 (3 = 3) /// fhc only
 (4 = .) /// any nh care
 , gen(hcmix)

lab def HCMIX 1 "1. ihc only" 2 "2. ihc+fhc" 3 "3. fhc only"
lab val hcmix HCMIX
lab var hcmix "home care formal/informal mix"

* core vs exit
tab hcmix xiw, col nof
/*
  RECODE of |
   caretype |
 (RECODE of |   RECODE of _merge
 caretyped) |         0          1 |     Total
------------+----------------------+----------
1. ihc only |     82.94      60.30 |     79.05 
 2. ihc+fhc |     12.41      36.30 |     16.51 
3. fhc only |      4.65       3.41 |      4.44 
------------+----------------------+----------
      Total |    100.00     100.00 |    100.00  
*/

tabstat ichrsfrac, by(hcmix) s(p5 p10 p25 p50 p75 p90 p95)
tabstat hchrsfrac, by(hcmix) s(p5 p10 p25 p50 p75 p90 p95)

** refined measure that defines someone as a non-mixer if 90%+ of their hours
** come from one type of source
** restrict to 2000+ (wave 5+) b/c missing spouse hours in 1998

clonevar hcmix90 = hcmix
replace hcmix90 = 1 if (hcmix90==2 & ichrsfrac>=.90 & ichrsfrac<.)
replace hcmix90 = 3 if (hcmix90==2 & hchrsfrac>=.90 & hchrsfrac<.)
replace hcmix90 = . if (w<5)

** refined measure that defines someone as a non-mixer if 80%+ of their hours
** come from one type of source
** restrict to 2000+ (wave 5+) b/c missing spouse hours in 1998

clonevar hcmix80 = hcmix
replace hcmix80 = 1 if (hcmix80==2 & ichrsfrac>=.80 & ichrsfrac<.)
replace hcmix80 = 3 if (hcmix80==2 & hchrsfrac>=.80 & hchrsfrac<.)
replace hcmix80 = . if (w<5)

** compare
tab hcmix if w>=5
tab hcmix90
tab hcmix80

** core vs. exit
tab hcmix xiw if w>=5, col nof
tab hcmix90 xiw, col nof
tab hcmix80 xiw, col nof

/*
Original:

  RECODE of |
   caretype |
 (RECODE of |   RECODE of _merge
 caretyped) |         0          1 |     Total
------------+----------------------+----------
1. ihc only |     83.09      60.30 |     78.63 
 2. ihc+fhc |     12.33      36.30 |     17.02 
3. fhc only |      4.58       3.41 |      4.35 
------------+----------------------+----------
      Total |    100.00     100.00 |    100.00 

90% cutoff:

  RECODE of |
   caretype |
 (RECODE of |   RECODE of _merge
 caretyped) |         0          1 |     Total
------------+----------------------+----------
1. ihc only |     85.39      70.57 |     82.49 
 2. ihc+fhc |      7.85      20.27 |     10.28 
3. fhc only |      6.77       9.16 |      7.24 
------------+----------------------+----------
      Total |    100.00     100.00 |    100.00 

80% cutoff:

  RECODE of |
   caretype |
 (RECODE of |   RECODE of _merge
 caretyped) |         0          1 |     Total
------------+----------------------+----------
1. ihc only |     86.73      74.51 |     84.34 
 2. ihc+fhc |      5.29      13.62 |      6.92 
3. fhc only |      7.98      11.86 |      8.74 
------------+----------------------+----------
      Total |    100.00     100.00 |    100.00 
*/

********************************************************************************
** indicators for cases where young-gen. supplies > X% of care
********************************************************************************

* > 50%

recode caretype ///
 (0 = 0) /// no care
 (1 = 1) /// ihc only
 (2 = 1) /// ihc + fhc
 (3 = 0) /// fhc only
 (4 = 0) /// any nh care
 , gen(ygonly50)

replace ygonly50 = 0  if (inlist(caretype,1,2) & yghrsfrac<=.50)
replace ygonly50 = .m if (inlist(caretype,1,2) & mi(yghrsfrac))

lab var ygonly "yg hlprs > 50% care hours, no nh hlpr"

tab ygonly50
/*
   yg > 50% |
care hours, |
 no nh hlpr |      Freq.     Percent        Cum.
------------+-----------------------------------
          0 |    133,818       94.78       94.78
          1 |      7,377        5.22      100.00
------------+-----------------------------------
      Total |    141,195      100.00
*/
tab ygonly50 if caretype>0
/*
   yg > 50% |
care hours, |
 no nh hlpr |      Freq.     Percent        Cum.
------------+-----------------------------------
          0 |     17,288       70.09       70.09
          1 |      7,377       29.91      100.00
------------+-----------------------------------
      Total |     24,665      100.00
*/

* 100% 

recode caretype ///
 (0 = 0) /// no care
 (1 = 1) /// ihc only
 (2 = 0) /// ihc + fhc
 (3 = 0) /// fhc only
 (4 = 0) /// any nh care
 , gen(ygonly)

replace ygonly = 0 if (ygonly==1 & spanyhlpr==1)
replace ygonly = 0 if (ygnhlpr < nhlpr)

lab var ygonly "yg hlprs provide all care"

tab ygonly
/*
   yg hlprs |
provide all |
       care |      Freq.     Percent        Cum.
------------+-----------------------------------
          0 |    137,796       96.05       96.05
          1 |      5,665        3.95      100.00
------------+-----------------------------------
      Total |    143,461      100.00
*/

tab ygonly if caretype>0
/*
   yg hlprs |
provide all |
       care |      Freq.     Percent        Cum.
------------+-----------------------------------
          0 |     21,266       78.96       78.96
          1 |      5,665       21.04      100.00
------------+-----------------------------------
      Total |     26,931      100.00
*/

********************************************************************************
** drop unneeded variables
********************************************************************************

drop anyhlprGR nhhlprGR spphlprGR
// Section G-R helper counts

********************************************************************************
** save
********************************************************************************

qui compress

save ${save}/helpers_combined.dta, replace

********************************************************************************
