/*******************************************************************************

************
** Function: 
************

  Creates end-of-life wealth trajectories figures.

************
** Inputs  : 
************

  - decedent_sample_single_final.dta

************
** Outputs : 
************

  Output for the text:
	
	- FIG1a.eps
  - FIG1b.eps
	- FIG2a.eps
	- FIG2b.eps
  - FIG4b.eps
	- FIG4a.eps
	- FIG7_DATA.txt
	- FIG8a.eps
	- FIG8b.eps
	- FIG10a.eps
	- FIG10b.eps	
	- FIG11a.eps
	- FIG11b.eps	
	
	Output for Appendix A:
	
	- TABA2b_SAMPLE_COUNTS.txt
	
	Output for Appendix H:
	
	- FIGH1a.eps
	- FIGH1b.eps 
	- FIGH1c.eps
	- APPENDIX_H_STATISTICS.txt
	
	Output for Appendix K:
	
	- FIGK1a.eps
	- FIGK1b.eps
	- FIGK1c.eps
	- FIGK1d.eps
	- FIGK1e.eps
	- FIGK1f.eps
	
*******************************************************************************/

clear *
set more off
macro drop _all

********************************************************************************
** Director names
********************************************************************************

do GetDirNames.do

********************************************************************************
** Settings
********************************************************************************

set scheme economist

** Line thickness:

global lw_p10  *0.166
global lw_p25  *0.333
global lw_p50  *0.500
global lw_p75  *0.666
global lw_p90  *0.833

********************************************************************************
** DATA
********************************************************************************	

use ${save}/decedent_sample_single_final, clear

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

graph drop _all

********************************************************************************
** Sample counts
********************************************************************************
/*
The following sample counts appear in Appendix Table A2, Panel B.
*/

* turn on log
cap log close
log using ${logs}/TABA2b_SAMPLE_COUNTS.txt, text replace	

tab xnriw               // -> unbalanced
/*
      xnriw |      Freq.     Percent        Cum.
------------+-----------------------------------
          0 |      3,227       17.95       17.95
          1 |      3,227       17.95       35.91
          2 |      3,151       17.53       53.44
          3 |      3,010       16.75       70.18
          4 |      2,316       12.89       83.07
          5 |      1,668        9.28       92.35
          6 |      1,011        5.62       97.97
          7 |        364        2.03      100.00
------------+-----------------------------------
      Total |     17,974      100.00
*/

tab xnriw if totnriw>=4 // -> balanced? (no: actually, not everyone has wealth data at end point)

/*
      xnriw |      Freq.     Percent        Cum.
------------+-----------------------------------
          0 |      2,316       15.84       15.84
          1 |      2,316       15.84       31.68
          2 |      2,316       15.84       47.51
          3 |      2,316       15.84       63.35
          4 |      2,316       15.84       79.19
          5 |      1,668       11.41       90.60
          6 |      1,011        6.91       97.51
          7 |        364        2.49      100.00
------------+-----------------------------------
      Total |     14,623      100.00
*/

tab xnriw if fullWlthHist4==1 & (riwstat==1 | xIW==1) // -> balanced

/*
      xnriw |      Freq.     Percent        Cum.
------------+-----------------------------------
          0 |      2,290       15.82       15.82
          1 |      2,290       15.82       31.64
          2 |      2,290       15.82       47.46
          3 |      2,290       15.82       63.29
          4 |      2,290       15.82       79.11
          5 |      1,654       11.43       90.53
          6 |      1,007        6.96       97.49
          7 |        363        2.51      100.00
------------+-----------------------------------
      Total |     14,474      100.00
*/

* count of all decedents with 4+ core interviews:
unique hhidpn if (fullWlthHist4==1) & (riwstat==1 | xIW==1)
/*
Number of unique values of hhidpn is  2290
Number of records is  14474
*/
// -> 2,290 decedents
// -> 14,474 interviews (core + exit)

* parents
unique hhidpn if (fullWlthHist4==1) & (riwstat==1 | xIW==1) & (nochild==0)
/*
Number of unique values of hhidpn is  2060
Number of records is  13019
*/

* childless
unique hhidpn if (fullWlthHist4==1) & (riwstat==1 | xIW==1) & (nochild==1)
/*
Number of unique values of hhidpn is  230
Number of records is  1455
*/

* T-1 owners
unique hhidpn if (fullWlthHist4==1) & (riwstat==1 | xIW==1) & (own_1x==1)
/*
Number of unique values of hhidpn is  1000
Number of records is  6310
*/

* T-1 renters
unique hhidpn if (fullWlthHist4==1) & (riwstat==1 | xIW==1) & (own_1x==0)
/*
Number of unique values of hhidpn is  1266
Number of records is  8013
*/

* T-1 NHRs
unique hhidpn if (fullWlthHist4==1) & (riwstat==1 | xIW==1) & (nhr_1x==1)
/*
Number of unique values of hhidpn is  555
Number of records is  3518
*/

* T-1 CRs
unique hhidpn if (fullWlthHist4==1) & (riwstat==1 | xIW==1) & (nhr_1x==0)
/*
Number of unique values of hhidpn is  1735
Number of records is  10956
*/

* close log
cap log close

********************************************************************************
** Wealth trajectories
********************************************************************************

******************************
* Unconditional wealth trajectories
******************************

/*
FIGURE 4 PANEL (B)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel

tab xnriw if (xnriw<=4)

keep xnriw wlthhist  crwtall 
collapse  (p10) p10=wlthhist ///
          (p25) p25=wlthhist (p50) p50=wlthhist (p75) p75=wlthhist ///
          (p90) p90=wlthhist [aw=crwtall],  by(xnriw)
		  
gen nxnriw = -xnriw

gr tw ///
(line p10 nxnriw  if xnriw<=4, lc(gs1) lp(solid)    lw(${lw_p10}))  ///
(line p25 nxnriw  if xnriw<=4, lc(gs1) lp(dash)     lw(${lw_p25}))  ///
(line p50 nxnriw  if xnriw<=4, lc(gs1) lp(solid)    lw(${lw_p50}))  ///
(line p75 nxnriw  if xnriw<=4, lc(gs1) lp(dash)     lw(${lw_p75}))  ///
(line p90 nxnriw  if xnriw<=4, lc(gs1) lp(dash_dot) lw(${lw_p90})),  ///
ylabel(0(100)700) ///
xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
yscale(range(0 700)) ///
///legend( order(1 "p10, p25, p50, p75, p90") cols(1) ) ///
///legend(order(1 "p25" 2 "p50" 3 "p75" 4 "p90") rows(1) ) ///
legend(order(1 "p10" 2 "p25" 3 "p50" 4 "p75" 5 "p90") rows(1) ) ///
xti("# waves until exit interview (0 = exit interview)") ///
yti("net worth")
restore

graph2tex, epsfile(${figs}/FIG4b) 

******************************
* Data for the unconditional wealth trajectories
******************************

* turn on log
cap log close
log using ${logs}/FIG7_DATA.txt, text replace

tabstat wlthhist [aw=crwtall] if xnriw<=4 & totnriw>=4 & nochild==0, by(xnriw) ///
	s(p10 p25 p50 p75 p90) f(%9.0f) nototal

/*
   xnriw |       p10       p25       p50       p75       p90
---------+--------------------------------------------------
       0 |         0         0        25       200       521
       1 |         0         1        52       222       577
       2 |         0         2        79       259       617
       3 |         0         7        90       277       651
       4 |         0        16        98       285       635
------------------------------------------------------------
*/

* close log
cap log close

******************************
** Wealth trajectories by child status
******************************

/*
FIGURE 4 PANEL (A)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel

tab xnriw nochild if (xnriw<=4)

keep xnriw wlthhist nochild  crwtall 
collapse  (p10) p10=wlthhist (p25) p25=wlthhist (p50) p50=wlthhist ///
          (p75) p75=wlthhist (p90) p90=wlthhist [aw=crwtall],  by(xnriw nochild)

gen nxnriw = -xnriw

gr tw ///
(line p10 nxnriw  if xnriw<=4 & nochild==0, lc(gs1) lp(solid) lw(${lw_p10}))  ///
(line p10 nxnriw  if xnriw<=4 & nochild==1, lc(gs1) lp(dash)  lw(${lw_p10})) ///
(line p25 nxnriw  if xnriw<=4 & nochild==0, lc(gs1) lp(solid) lw(${lw_p25}))  ///
(line p25 nxnriw  if xnriw<=4 & nochild==1, lc(gs1) lp(dash)  lw(${lw_p25})) ///
(line p50 nxnriw  if xnriw<=4 & nochild==0, lc(gs1) lp(solid) lw(${lw_p50}))  ///
(line p50 nxnriw  if xnriw<=4 & nochild==1, lc(gs1) lp(dash)  lw(${lw_p50})) ///
(line p75 nxnriw  if xnriw<=4 & nochild==0, lc(gs1) lp(solid) lw(${lw_p75}))  ///
(line p75 nxnriw  if xnriw<=4 & nochild==1, lc(gs1) lp(dash)  lw(${lw_p75})) ///
(line p90 nxnriw  if xnriw<=4 & nochild==0, lc(gs1) lp(solid) lw(${lw_p90}))  ///
(line p90 nxnriw  if xnriw<=4 & nochild==1, lc(gs1) lp(dash)  lw(${lw_p90})), ///
ylabel(0(100)800) ///
xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
yscale(range(0 800)) ///
legend( order(1 "p10, p25, p50, p75, p90: Parents" 2 "p10, p25, p50, p75, p90: Childless") cols(1) ) ///
xti("# waves until exit interview (0 = exit interview)") ///
yti("net worth")
restore

graph2tex, epsfile(${figs}/FIG4a) 

******************************
** Wealth trajectories by homeownership (own at T-1)
******************************

/*
FIGURE 1 PANEL (A)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel

tab xnriw own_1x if (xnriw<=4)

keep xnriw wlthhist own_1x crwtall 
collapse  (p10) p10=wlthhist (p25) p25=wlthhist (p50) p50=wlthhist ///
          (p75) p75=wlthhist (p90) p90=wlthhist [aw=crwtall],  by(xnriw own_1x)

gen nxnriw = -xnriw

gr tw ///
(line p10 nxnriw  if xnriw<=4 & own_1x==1, lc(gs4) lp(solid) lw(${lw_p10}))  ///
(line p10 nxnriw  if xnriw<=4 & own_1x==0, lc(gs4) lp(dash)  lw(${lw_p10})) ///
(line p25 nxnriw  if xnriw<=4 & own_1x==1, lc(gs4) lp(solid) lw(${lw_p25}))  ///
(line p25 nxnriw  if xnriw<=4 & own_1x==0, lc(gs4) lp(dash)  lw(${lw_p25})) ///
(line p50 nxnriw  if xnriw<=4 & own_1x==1, lc(gs4) lp(solid) lw(${lw_p50}))  ///
(line p50 nxnriw  if xnriw<=4 & own_1x==0, lc(gs4) lp(dash)  lw(${lw_p50})) ///
(line p75 nxnriw  if xnriw<=4 & own_1x==1, lc(gs4) lp(solid) lw(${lw_p75}))  ///
(line p75 nxnriw  if xnriw<=4 & own_1x==0, lc(gs4) lp(dash)  lw(${lw_p75})) ///
(line p90 nxnriw  if xnriw<=4 & own_1x==1, lc(gs4) lp(solid) lw(${lw_p90}))  ///
(line p90 nxnriw  if xnriw<=4 & own_1x==0, lc(gs4) lp(dash)  lw(${lw_p90})), ///
legend( order(1 "p10, p25, p50, p75, p90: Owners" 2 "p10, p25, p50, p75, p90: Renters") cols(1) ) ///
ylabel(0(100)900) ///
xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
yscale(range(0 900)) ///
xti("# waves until exit interview (0 = exit interview)") ///
yti("net worth")
restore

graph2tex, epsfile(${figs}/FIG1a) 

******************************
** Wealth trajectories by nursing home status (NHR at T-1)
******************************

/*
FIGURE 2 PANEL (A)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel

tab xnriw nhr_1x if (xnriw<=4)

keep xnriw wlthhist nhr_1x crwtall 
collapse  (p10) p10=wlthhist (p25) p25=wlthhist (p50) p50=wlthhist ///
          (p75) p75=wlthhist (p90) p90=wlthhist [aw=crwtall],  by(xnriw nhr_1x)

gen nxnriw = -xnriw

gr tw ///
(line p10 nxnriw  if xnriw<=4 & nhr_1x==1, lc(gs4) lp(solid) lw(${lw_p10}))  ///
(line p10 nxnriw  if xnriw<=4 & nhr_1x==0, lc(gs4) lp(dash)  lw(${lw_p10})) ///
(line p25 nxnriw  if xnriw<=4 & nhr_1x==1, lc(gs4) lp(solid) lw(${lw_p25}))  ///
(line p25 nxnriw  if xnriw<=4 & nhr_1x==0, lc(gs4) lp(dash)  lw(${lw_p25})) ///
(line p50 nxnriw  if xnriw<=4 & nhr_1x==1, lc(gs4) lp(solid) lw(${lw_p50}))  ///
(line p50 nxnriw  if xnriw<=4 & nhr_1x==0, lc(gs4) lp(dash)  lw(${lw_p50})) ///
(line p75 nxnriw  if xnriw<=4 & nhr_1x==1, lc(gs4) lp(solid) lw(${lw_p75}))  ///
(line p75 nxnriw  if xnriw<=4 & nhr_1x==0, lc(gs4) lp(dash)  lw(${lw_p75})) ///
(line p90 nxnriw  if xnriw<=4 & nhr_1x==1, lc(gs4) lp(solid) lw(${lw_p90}))  ///
(line p90 nxnriw  if xnriw<=4 & nhr_1x==0, lc(gs4) lp(dash)  lw(${lw_p90})), ///
legend( order(1 "p10, p25, p50, p75, p90: NHR" 2 "p10, p25, p50, p75, p90: CR") cols(1) ) ///
ylabel(0(100)700) ///
xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
yscale(range(0 700)) ///
xti("# waves until exit interview (0 = exit interview)") ///
yti("net worth")
restore

graph2tex, epsfile(${figs}/FIG2a)

********************************************************************************
** Owners/Keepers // Owners/Liquidators // Renters // Median trajectories
********************************************************************************

/*
FIGURE 1 PANEL (B)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)

by hhidpn: egen own_0 = max(cond(xnriw==0,ownHist,.))
by hhidpn: egen own_1 = max(cond(xnriw==1,ownHist,.))
by hhidpn: egen own_2 = max(cond(xnriw==2,ownHist,.))
by hhidpn: egen own_3 = max(cond(xnriw==3,ownHist,.))
by hhidpn: egen own_4 = max(cond(xnriw==4,ownHist,.))

gen keep_43 = (own_4==1 & own_3==1)
gen sell_43 = (own_4==1 & own_3==0)
gen rent_43 = (own_4==0 & own_3==0)

gen keep_32 = (own_3==1 & own_2==1)
gen sell_32 = (own_3==1 & own_2==0)
gen rent_32 = (own_3==0 & own_2==0)

gen keep_21 = (own_2==1 & own_1==1)
gen sell_21 = (own_2==1 & own_1==0)
gen rent_21 = (own_2==0 & own_1==0)

gen keep_10 = (own_1==1 & own_0==1)
gen sell_10 = (own_1==1 & own_0==0)
gen rent_10 = (own_1==0 & own_0==0)

expand 2 if (inlist(xnriw,3,2,1)), gen(dup) // -> each of the middle obs need to be duplicated
tab xnriw dup // -> check

local d = 0.25 // -> x-axis increment

replace xnriw = xnriw + `d' if (dup==1)
tab xnriw dup

gen group = .

* Keepers group
replace group = 1 if (xnriw==   4  & keep_43==1) // keepers 4->3 at wave 4
replace group = 1 if (xnriw==3+`d' & keep_43==1) // keepers 4->3 at wave 3
replace group = 1 if (xnriw==   3  & keep_32==1) // keepers 3->2 at wave 3
replace group = 1 if (xnriw==2+`d' & keep_32==1) // keepers 3->2 at wave 2
replace group = 1 if (xnriw==   2  & keep_21==1) // keepers 2->1 at wave 2
replace group = 1 if (xnriw==1+`d' & keep_21==1) // keepers 2->1 at wave 1
replace group = 1 if (xnriw==   1  & keep_10==1) // keepers 1->0 at wave 1
replace group = 1 if (xnriw==   0  & keep_10==1) // keepers 1->0 at wave 0

* Sellers group
replace group = 2 if (xnriw==   4  & sell_43==1) // sellers 4->3 at wave 4
replace group = 2 if (xnriw==3+`d' & sell_43==1) // sellers 4->3 at wave 3
replace group = 2 if (xnriw==   3  & sell_32==1) // sellers 3->2 at wave 3
replace group = 2 if (xnriw==2+`d' & sell_32==1) // sellers 3->2 at wave 2
replace group = 2 if (xnriw==   2  & sell_21==1) // sellers 2->1 at wave 2
replace group = 2 if (xnriw==1+`d' & sell_21==1) // sellers 2->1 at wave 1
replace group = 2 if (xnriw==   1  & sell_10==1) // sellers 1->0 at wave 1
replace group = 2 if (xnriw==   0  & sell_10==1) // sellers 1->0 at wave 0

* Renters group
replace group = 3 if (xnriw==   4  & rent_43==1) // renters 4->3 at wave 4
replace group = 3 if (xnriw==3+`d' & rent_43==1) // renters 4->3 at wave 3
replace group = 3 if (xnriw==   3  & rent_32==1) // renters 3->2 at wave 3
replace group = 3 if (xnriw==2+`d' & rent_32==1) // renters 3->2 at wave 2
replace group = 3 if (xnriw==   2  & rent_21==1) // renters 2->1 at wave 2
replace group = 3 if (xnriw==1+`d' & rent_21==1) // renters 2->1 at wave 1
replace group = 3 if (xnriw==   1  & rent_10==1) // renters 1->0 at wave 1
replace group = 3 if (xnriw==   0  & rent_10==1) // renters 1->0 at wave 0

drop if missing(group)

collapse (p50) p50=wlthhist [aw=crwtall], by(group xnriw)

order group xnriw
sort group xnriw
list, sepby(group)

gen segment = .
replace segment = 1 if (xnriw==4 | xnriw==3+`d')
replace segment = 2 if (xnriw==3 | xnriw==2+`d')
replace segment = 3 if (xnriw==2 | xnriw==1+`d')
replace segment = 4 if (xnriw==1 | xnriw==0    )

gen nxnriw = -round(xnriw)

gr tw ///
(sc p50 nxnriw if (group==1 & segment==1), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==2), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==3), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==4), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==1), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==2), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==3), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==4), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==1), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==2), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==3), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==4), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
, legend(order(1 "p50: Keepers" 5 "p50: Liquidators" 9 "p50: Renters") rows(2)) ///
  ylabel(0(50)250) ///
  xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
  yscale(range(0 250)) ///
  xti("# waves until exit interview (0 = exit interview)") ///
  yti("net worth")

restore

graph2tex, epsfile(${figs}/FIG1b)

********************************************************************************
** Community residents // NH entrants // NH stayers // Median trajectories
********************************************************************************

/*
FIGURE 2 PANEL (B)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)

by hhidpn: egen nhr_0 = max(cond(xnriw==0,rnhmliv,.))
by hhidpn: egen nhr_1 = max(cond(xnriw==1,rnhmliv,.))
by hhidpn: egen nhr_2 = max(cond(xnriw==2,rnhmliv,.))
by hhidpn: egen nhr_3 = max(cond(xnriw==3,rnhmliv,.))
by hhidpn: egen nhr_4 = max(cond(xnriw==4,rnhmliv,.))

gen stay_43 = (nhr_4==1 & nhr_3==1) // stay: nh resident both periods
gen entr_43 = (nhr_4==0 & nhr_3==1) // entr: nh resident in 2nd period only
gen cmty_43 = (nhr_4==0 & nhr_3==0) // cmty: community resident in both periods

gen stay_32 = (nhr_3==1 & nhr_2==1)
gen entr_32 = (nhr_3==0 & nhr_2==1)
gen cmty_32 = (nhr_3==0 & nhr_2==0)

gen stay_21 = (nhr_2==1 & nhr_1==1)
gen entr_21 = (nhr_2==0 & nhr_1==1)
gen cmty_21 = (nhr_2==0 & nhr_1==0)

gen stay_10 = (nhr_1==1 & nhr_0==1)
gen entr_10 = (nhr_1==0 & nhr_0==1)
gen cmty_10 = (nhr_1==0 & nhr_0==0)

expand 2 if (inlist(xnriw,3,2,1)), gen(dup) // -> each of the middle obs need to be duplicated
tab xnriw dup // -> check

local d = 0.25 // -> x-axis increment

replace xnriw = xnriw + `d' if (dup==1)
tab xnriw dup

gen group = .

* Community residents group
replace group = 1 if (xnriw==   4  & cmty_43==1) // community residents 4->3 at wave 4
replace group = 1 if (xnriw==3+`d' & cmty_43==1) // community residents 4->3 at wave 3
replace group = 1 if (xnriw==   3  & cmty_32==1) // community residents 3->2 at wave 3
replace group = 1 if (xnriw==2+`d' & cmty_32==1) // community residents 3->2 at wave 2
replace group = 1 if (xnriw==   2  & cmty_21==1) // community residents 2->1 at wave 2
replace group = 1 if (xnriw==1+`d' & cmty_21==1) // community residents 2->1 at wave 1
replace group = 1 if (xnriw==   1  & cmty_10==1) // community residents 1->0 at wave 1
replace group = 1 if (xnriw==   0  & cmty_10==1) // community residents 1->0 at wave 0

* Entrants group
replace group = 2 if (xnriw==   4  & entr_43==1) // entrants 4->3 at wave 4
replace group = 2 if (xnriw==3+`d' & entr_43==1) // entrants 4->3 at wave 3
replace group = 2 if (xnriw==   3  & entr_32==1) // entrants 3->2 at wave 3
replace group = 2 if (xnriw==2+`d' & entr_32==1) // entrants 3->2 at wave 2
replace group = 2 if (xnriw==   2  & entr_21==1) // entrants 2->1 at wave 2
replace group = 2 if (xnriw==1+`d' & entr_21==1) // entrants 2->1 at wave 1
replace group = 2 if (xnriw==   1  & entr_10==1) // entrants 1->0 at wave 1
replace group = 2 if (xnriw==   0  & entr_10==1) // entrants 1->0 at wave 0

* Stayers group
replace group = 3 if (xnriw==   4  & stay_43==1) // stayers 4->3 at wave 4
replace group = 3 if (xnriw==3+`d' & stay_43==1) // stayers 4->3 at wave 3
replace group = 3 if (xnriw==   3  & stay_32==1) // stayers 3->2 at wave 3
replace group = 3 if (xnriw==2+`d' & stay_32==1) // stayers 3->2 at wave 2
replace group = 3 if (xnriw==   2  & stay_21==1) // stayers 2->1 at wave 2
replace group = 3 if (xnriw==1+`d' & stay_21==1) // stayers 2->1 at wave 1
replace group = 3 if (xnriw==   1  & stay_10==1) // stayers 1->0 at wave 1
replace group = 3 if (xnriw==   0  & stay_10==1) // stayers 1->0 at wave 0

drop if missing(group)

tab group xnriw

collapse (p50) p50=wlthhist [aw=crwtall], by(group xnriw)

order group xnriw
sort group xnriw
list, sepby(group)

gen segment = .
replace segment = 1 if (xnriw==4 | xnriw==3+`d')
replace segment = 2 if (xnriw==3 | xnriw==2+`d')
replace segment = 3 if (xnriw==2 | xnriw==1+`d')
replace segment = 4 if (xnriw==1 | xnriw==0    )

gen nxnriw = -round(xnriw)

gr tw ///
(sc p50 nxnriw if (group==1 & segment==1), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==2), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==3), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==4), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==1), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==2), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==3), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==4), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==1), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==2), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==3), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==4), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
, legend(order(1 "p50: Remain CR" 5 "p50: Enter NH" 9 "p50: Remain NHR") rows(2)) ///
  ylabel(0(25)125) ///
  xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
  yscale(range(0 125)) ///
  xti("# waves until exit interview (0 = exit interview)") ///
  yti("net worth")

restore

graph2tex, epsfile(${figs}/FIG2b)

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
** Owners/Keepers // Owners/Liquidators+NH Entrant // Owners/Liquidators+Community Stayer
********************************************************************************

/*
APPENDIX FIGURE H.1 PANEL (A)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)

by hhidpn: egen own_0 = max(cond(xnriw==0,ownHist,.))
by hhidpn: egen own_1 = max(cond(xnriw==1,ownHist,.))
by hhidpn: egen own_2 = max(cond(xnriw==2,ownHist,.))
by hhidpn: egen own_3 = max(cond(xnriw==3,ownHist,.))
by hhidpn: egen own_4 = max(cond(xnriw==4,ownHist,.))

by hhidpn: egen nhr_0 = max(cond(xnriw==0,rnhmliv,.))
by hhidpn: egen nhr_1 = max(cond(xnriw==1,rnhmliv,.))
by hhidpn: egen nhr_2 = max(cond(xnriw==2,rnhmliv,.))
by hhidpn: egen nhr_3 = max(cond(xnriw==3,rnhmliv,.))
by hhidpn: egen nhr_4 = max(cond(xnriw==4,rnhmliv,.))

gen keep_43 = (own_4==1 & own_3==1)                       // own @t-1, own  @t
gen lqnh_43 = (own_4==1 & own_3==0 & nhr_4==0 & nhr_3==1) // own @t-1, rent @t, CR @t-1, NHR @t
gen lqcr_43 = (own_4==1 & own_3==0 & nhr_4==0 & nhr_3==0) // own @t-1, rent @t, CR @t  , CR  @t

gen keep_32 = (own_3==1 & own_2==1)
gen lqnh_32 = (own_3==1 & own_2==0 & nhr_3==0 & nhr_2==1)
gen lqcr_32 = (own_3==1 & own_2==0 & nhr_3==0 & nhr_2==0) 

gen keep_21 = (own_2==1 & own_1==1)
gen lqnh_21 = (own_2==1 & own_1==0 & nhr_2==0 & nhr_1==1)
gen lqcr_21 = (own_2==1 & own_1==0 & nhr_2==0 & nhr_1==0) 

gen keep_10 = (own_1==1 & own_0==1)
gen lqnh_10 = (own_1==1 & own_0==0 & nhr_1==0 & nhr_0==1)
gen lqcr_10 = (own_1==1 & own_0==0 & nhr_1==0 & nhr_0==0) 

expand 2 if (inlist(xnriw,3,2,1)), gen(dup) // -> each of the middle obs need to be duplicated
tab xnriw dup // -> check

local d = 0.25 // -> x-axis increment

replace xnriw = xnriw + `d' if (dup==1)
tab xnriw dup

gen group = .

* Keepers group
replace group = 1 if (xnriw==   4  & keep_43==1) // keepers 4->3 at wave 4
replace group = 1 if (xnriw==3+`d' & keep_43==1) // keepers 4->3 at wave 3
replace group = 1 if (xnriw==   3  & keep_32==1) // keepers 3->2 at wave 3
replace group = 1 if (xnriw==2+`d' & keep_32==1) // keepers 3->2 at wave 2
replace group = 1 if (xnriw==   2  & keep_21==1) // keepers 2->1 at wave 2
replace group = 1 if (xnriw==1+`d' & keep_21==1) // keepers 2->1 at wave 1
replace group = 1 if (xnriw==   1  & keep_10==1) // keepers 1->0 at wave 1
replace group = 1 if (xnriw==   0  & keep_10==1) // keepers 1->0 at wave 0

* Liquidators NHR group
replace group = 2 if (xnriw==   4  & lqnh_43==1) // liq+NH 4->3 at wave 4
replace group = 2 if (xnriw==3+`d' & lqnh_43==1) // liq+NH 4->3 at wave 3
replace group = 2 if (xnriw==   3  & lqnh_32==1) // liq+NH 3->2 at wave 3
replace group = 2 if (xnriw==2+`d' & lqnh_32==1) // liq+NH 3->2 at wave 2
replace group = 2 if (xnriw==   2  & lqnh_21==1) // liq+NH 2->1 at wave 2
replace group = 2 if (xnriw==1+`d' & lqnh_21==1) // liq+NH 2->1 at wave 1
replace group = 2 if (xnriw==   1  & lqnh_10==1) // liq+NH 1->0 at wave 1
replace group = 2 if (xnriw==   0  & lqnh_10==1) // liq+NH 1->0 at wave 0

* Liquidators CR group
replace group = 3 if (xnriw==   4  & lqcr_43==1) // liq+CR 4->3 at wave 4
replace group = 3 if (xnriw==3+`d' & lqcr_43==1) // liq+CR 4->3 at wave 3
replace group = 3 if (xnriw==   3  & lqcr_32==1) // liq+CR 3->2 at wave 3
replace group = 3 if (xnriw==2+`d' & lqcr_32==1) // liq+CR 3->2 at wave 2
replace group = 3 if (xnriw==   2  & lqcr_21==1) // liq+CR 2->1 at wave 2
replace group = 3 if (xnriw==1+`d' & lqcr_21==1) // liq+CR 2->1 at wave 1
replace group = 3 if (xnriw==   1  & lqcr_10==1) // liq+CR 1->0 at wave 1
replace group = 3 if (xnriw==   0  & lqcr_10==1) // liq+CR 1->0 at wave 0

drop if missing(group)

collapse (p50) p50=wlthhist [aw=crwtall], by(group xnriw)

order group xnriw
sort group xnriw
list, sepby(group)

gen segment = .
replace segment = 1 if (xnriw==4 | xnriw==3+`d')
replace segment = 2 if (xnriw==3 | xnriw==2+`d')
replace segment = 3 if (xnriw==2 | xnriw==1+`d')
replace segment = 4 if (xnriw==1 | xnriw==0    )

gen nxnriw = -round(xnriw)

gr tw ///
(sc p50 nxnriw if (group==1 & segment==1), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==2), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==3), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==4), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==1), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==2), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==3), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==4), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==1), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==2), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==3), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==4), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
, legend(order(1 "p50: Keepers" 5 "p50: Liq + Enter NH" 9 "p50: Liq + Stay CR") rows(2)) ///
  ylabel(0(50)250) ///
  xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
  yscale(range(0 250)) ///
  xti("# waves until exit interview (0 = exit interview)") ///
  yti("net worth")

restore

graph2tex, epsfile(${figs}/FIGH1a)

********************************************************************************
** Owners/Keepers // Owners/Liquidators // With OOP Medical Expenditures Added Back in
********************************************************************************

/*
APPENDIX FIGURE H.1 PANEL (B)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)

by hhidpn: egen own_0 = max(cond(xnriw==0,ownHist,.))
by hhidpn: egen own_1 = max(cond(xnriw==1,ownHist,.))
by hhidpn: egen own_2 = max(cond(xnriw==2,ownHist,.))
by hhidpn: egen own_3 = max(cond(xnriw==3,ownHist,.))
by hhidpn: egen own_4 = max(cond(xnriw==4,ownHist,.))

gen keep_43 = (own_4==1 & own_3==1)
gen sell_43 = (own_4==1 & own_3==0)

gen keep_32 = (own_3==1 & own_2==1)
gen sell_32 = (own_3==1 & own_2==0)

gen keep_21 = (own_2==1 & own_1==1)
gen sell_21 = (own_2==1 & own_1==0)

gen keep_10 = (own_1==1 & own_0==1)
gen sell_10 = (own_1==1 & own_0==0)

expand 2 if (inlist(xnriw,3,2,1)), gen(dup) // -> each of the middle obs need to be duplicated
tab xnriw dup // -> check

local d = 0.25 // -> x-axis increment

replace xnriw = xnriw + `d' if (dup==1)
tab xnriw dup

gen group = .

* Keepers group
replace group = 1 if (xnriw==   4  & keep_43==1) // keepers 4->3 at wave 4
replace group = 1 if (xnriw==3+`d' & keep_43==1) // keepers 4->3 at wave 3
replace group = 1 if (xnriw==   3  & keep_32==1) // keepers 3->2 at wave 3
replace group = 1 if (xnriw==2+`d' & keep_32==1) // keepers 3->2 at wave 2
replace group = 1 if (xnriw==   2  & keep_21==1) // keepers 2->1 at wave 2
replace group = 1 if (xnriw==1+`d' & keep_21==1) // keepers 2->1 at wave 1
replace group = 1 if (xnriw==   1  & keep_10==1) // keepers 1->0 at wave 1
replace group = 1 if (xnriw==   0  & keep_10==1) // keepers 1->0 at wave 0

* Sellers group
replace group = 2 if (xnriw==   4  & sell_43==1) // sellers 4->3 at wave 4
replace group = 2 if (xnriw==3+`d' & sell_43==1) // sellers 4->3 at wave 3
replace group = 2 if (xnriw==   3  & sell_32==1) // sellers 3->2 at wave 3
replace group = 2 if (xnriw==2+`d' & sell_32==1) // sellers 3->2 at wave 2
replace group = 2 if (xnriw==   2  & sell_21==1) // sellers 2->1 at wave 2
replace group = 2 if (xnriw==1+`d' & sell_21==1) // sellers 2->1 at wave 1
replace group = 2 if (xnriw==   1  & sell_10==1) // sellers 1->0 at wave 1
replace group = 2 if (xnriw==   0  & sell_10==1) // sellers 1->0 at wave 0

drop if missing(group)

/*
* compute wealth trajectory with cumulative oop medical expenditure added back 
* in starting after first wave in the trajectory

gen oop = total_oop / 1e3
// convert to 1000s of 2010 dollars
replace oop = 0 if (xnriw==4)
// set oop to zero at first wave in trajectory panel
sort hhid pn w
by hhid pn: gen cumoop = sum(oop) 
// cumulative sum of oop after first wave
gen wlthhist_plus_oop = wlthhist + cumoop
// wealth history plus cumulative oop med exp
*/

* add oop medical expenditures to wealth trajectory:
* (here we do not add cumulative oopme but instead, for each interview pair,
*  we add between-interview oopme back to wealth at the second interview in
*  the pair. we do not change wealth at the first interview in the pair.)

gen oop = total_oop_HH / 1e3
// convert to 1000s of 2010 dollars
// note: using total household oop, including respondent and spouse (both core and exit)
gen wlthhist_plus_oop = .
replace wlthhist_plus_oop = wlthhist if inlist(xnriw,4,3,2,1) 
// start of segment, add nothing
replace wlthhist_plus_oop = wlthhist + oop if inlist(xnriw,3+`d',2+`d',1+`d',0)
// end of segment, add oop med exp back in

collapse (p50) p50=wlthhist p50oop=wlthhist_plus_oop [aw=crwtall], by(group xnriw)

order group xnriw
sort group xnriw
list, sepby(group)

gen segment = .
replace segment = 1 if (xnriw==4 | xnriw==3+`d')
replace segment = 2 if (xnriw==3 | xnriw==2+`d')
replace segment = 3 if (xnriw==2 | xnriw==1+`d')
replace segment = 4 if (xnriw==1 | xnriw==0    )

gen nxnriw = -round(xnriw)

gr tw ///
(sc p50 nxnriw if (group==1 & segment==1), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==2), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==3), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==4), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==1), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==2), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==3), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==4), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50oop nxnriw if (group==1 & segment==1), c(d) lc(gs1) mc(gs1) lp(dot) ms(D) lw(medthick) msiz(1)) ///
(sc p50oop nxnriw if (group==1 & segment==2), c(d) lc(gs1) mc(gs1) lp(dot) ms(D) lw(medthick) msiz(1)) ///
(sc p50oop nxnriw if (group==1 & segment==3), c(d) lc(gs1) mc(gs1) lp(dot) ms(D) lw(medthick) msiz(1)) ///
(sc p50oop nxnriw if (group==1 & segment==4), c(d) lc(gs1) mc(gs1) lp(dot) ms(D) lw(medthick) msiz(1)) ///
(sc p50oop nxnriw if (group==2 & segment==1), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50oop nxnriw if (group==2 & segment==2), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50oop nxnriw if (group==2 & segment==3), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50oop nxnriw if (group==2 & segment==4), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
, legend(order(1 "p50: Keepers" 5 "p50: Liquidators" 9 "p50+OOPME: Keep." 13 "p50+OOPME: Liq.") rows(2)) ///
  ylabel(0(50)250) ///
  xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
  yscale(range(0 250)) ///
  xti("# waves until exit interview (0 = exit interview)") ///
  yti("net worth")

restore

graph2tex, epsfile(${figs}/FIGH1b)

********************************************************************************
** Liquidators // Synthetic Liquidators as-if-owners // Synthetic Liquidators as-if-renters
********************************************************************************

/*
APPENDIX FIGURE H.1 PANEL (C)
*/

preserve

* change in net worth
gen wlthhistDiff = (wlthhist - cL1wlthhist) 
// note: not annualized
// note: difference uses carried forward, lagged wealth history

keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)

sort hhidpn w
set seed 1234

by hhidpn: egen own_0 = max(cond(xnriw==0,ownHist,.))
by hhidpn: egen own_1 = max(cond(xnriw==1,ownHist,.))
by hhidpn: egen own_2 = max(cond(xnriw==2,ownHist,.))
by hhidpn: egen own_3 = max(cond(xnriw==3,ownHist,.))
by hhidpn: egen own_4 = max(cond(xnriw==4,ownHist,.))

gen keep_43 = (own_4==1 & own_3==1)
gen sell_43 = (own_4==1 & own_3==0)
gen rent_43 = (own_4==0 & own_3==0)

gen keep_32 = (own_3==1 & own_2==1)
gen sell_32 = (own_3==1 & own_2==0)
gen rent_32 = (own_3==0 & own_2==0)

gen keep_21 = (own_2==1 & own_1==1)
gen sell_21 = (own_2==1 & own_1==0)
gen rent_21 = (own_2==0 & own_1==0)

gen keep_10 = (own_1==1 & own_0==1)
gen sell_10 = (own_1==1 & own_0==0)
gen rent_10 = (own_1==0 & own_0==0)

expand 2 if (inlist(xnriw,3,2,1)), gen(dup) // -> each of the middle obs need to be duplicated
tab xnriw dup // -> check

local d = 0.25 // -> x-axis increment

replace xnriw = xnriw + `d' if (dup==1)
tab xnriw dup

gen group = .

* Keepers group
replace group = 1 if (xnriw==   4  & keep_43==1) // keepers 4->3 at wave 4
replace group = 1 if (xnriw==3+`d' & keep_43==1) // keepers 4->3 at wave 3
replace group = 1 if (xnriw==   3  & keep_32==1) // keepers 3->2 at wave 3
replace group = 1 if (xnriw==2+`d' & keep_32==1) // keepers 3->2 at wave 2
replace group = 1 if (xnriw==   2  & keep_21==1) // keepers 2->1 at wave 2
replace group = 1 if (xnriw==1+`d' & keep_21==1) // keepers 2->1 at wave 1
replace group = 1 if (xnriw==   1  & keep_10==1) // keepers 1->0 at wave 1
replace group = 1 if (xnriw==   0  & keep_10==1) // keepers 1->0 at wave 0

* Sellers group
replace group = 2 if (xnriw==   4  & sell_43==1) // sellers 4->3 at wave 4
replace group = 2 if (xnriw==3+`d' & sell_43==1) // sellers 4->3 at wave 3
replace group = 2 if (xnriw==   3  & sell_32==1) // sellers 3->2 at wave 3
replace group = 2 if (xnriw==2+`d' & sell_32==1) // sellers 3->2 at wave 2
replace group = 2 if (xnriw==   2  & sell_21==1) // sellers 2->1 at wave 2
replace group = 2 if (xnriw==1+`d' & sell_21==1) // sellers 2->1 at wave 1
replace group = 2 if (xnriw==   1  & sell_10==1) // sellers 1->0 at wave 1
replace group = 2 if (xnriw==   0  & sell_10==1) // sellers 1->0 at wave 0

* Renters group
replace group = 3 if (xnriw==   4  & rent_43==1) // renters 4->3 at wave 4
replace group = 3 if (xnriw==3+`d' & rent_43==1) // renters 4->3 at wave 3
replace group = 3 if (xnriw==   3  & rent_32==1) // renters 3->2 at wave 3
replace group = 3 if (xnriw==2+`d' & rent_32==1) // renters 3->2 at wave 2
replace group = 3 if (xnriw==   2  & rent_21==1) // renters 2->1 at wave 2
replace group = 3 if (xnriw==1+`d' & rent_21==1) // renters 2->1 at wave 1
replace group = 3 if (xnriw==   1  & rent_10==1) // renters 1->0 at wave 1
replace group = 3 if (xnriw==   0  & rent_10==1) // renters 1->0 at wave 0

drop if missing(group)

* assign counterfactual changes in net worth from renters to liquidators
* find nearest neighbors using carried forward, lagged wealth history
gen to_impute = (group==2 | group==3)
// want only groups 2 (liquidators) and 3 (renters)
gen z = wlthhistDiff if (group==3)
// only use data from group 3 (renters)
mi set wide
mi register imputed z
* impute one wave at a time (end-points only)
mi impute pmm z cL1wlthhist if (to_impute==1 & xnriw==3+`d'), knn(1) add(1) noisily
mi impute pmm z cL1wlthhist if (to_impute==1 & xnriw==2+`d'), knn(1) add(1) noisily
mi impute pmm z cL1wlthhist if (to_impute==1 & xnriw==1+`d'), knn(1) add(1) noisily
mi impute pmm z cL1wlthhist if (to_impute==1 & xnriw==0    ), knn(1) add(1) noisily
desc _*_z
egen zi = rowmax(_*_z)
egen N  = rownonmiss(_*_z)
tab N if (group==2) & inlist(xnriw,3+`d',2+`d',1+`d',0) // -> expect 1s only
tab N if (group==3) & inlist(xnriw,3+`d',2+`d',1+`d',0) // -> expect 4s only
mi extract 0, clear // clear the imputation, delete imputed data

gen ctfl = cL1wlthhist + zi if (group==2)
// carried forward lagged wealth history + counterfactually assigned change
// note: z = wlthhistDiff = wlthhist - cL1wlthhist

gen wlthHistCtfl = .
replace wlthHistCtfl = wlthhist if (group==2) & inlist(xnriw,4,3,2,1) 
// start of the segment, use original wealth history
replace wlthHistCtfl = ctfl     if (group==2) & inlist(xnriw,3+`d',2+`d',1+`d',0)
// end of segment, use counterfactual wealth history

collapse (p50) p50=wlthhist p50c=wlthHistCtfl [aw=crwtall], by(group xnriw)

order group xnriw
sort group xnriw
list, sepby(group)

gen segment = .
replace segment = 1 if (xnriw==4 | xnriw==3+`d')
replace segment = 2 if (xnriw==3 | xnriw==2+`d')
replace segment = 3 if (xnriw==2 | xnriw==1+`d')
replace segment = 4 if (xnriw==1 | xnriw==0    )

gen nxnriw = -round(xnriw)

gr tw ///
(sc p50 nxnriw if (group==1 & segment==1), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==2), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==3), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==1 & segment==4), c(d) lc(gs1) mc(gs1) lp(solid) ms(O) lw(medthick) msiz(1)) ///
///
(sc p50 nxnriw if (group==2 & segment==1), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==2), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==3), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==2 & segment==4), c(d) lc(gs4) mc(gs4) lp(dash) ms(S) lw(medthick) msiz(1)) ///
///
(sc p50 nxnriw if (group==3 & segment==1), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==2), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==3), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
(sc p50 nxnriw if (group==3 & segment==4), c(d) lc(gs8) mc(gs4) lp(dash_dot) ms(T) lw(medthick) msiz(1)) ///
///
(sc p50c nxnriw if (group==2 & segment==1), c(d) lc(red) mc(red) lp(dash) ms(Dh) lw(medthick) msiz(1)) ///
(sc p50c nxnriw if (group==2 & segment==2), c(d) lc(red) mc(red) lp(dash) ms(Dh) lw(medthick) msiz(1)) ///
(sc p50c nxnriw if (group==2 & segment==3), c(d) lc(red) mc(red) lp(dash) ms(Dh) lw(medthick) msiz(1)) ///
(sc p50c nxnriw if (group==2 & segment==4), c(d) lc(red) mc(red) lp(dash) ms(Dh) lw(medthick) msiz(1)) ///
///
, legend(order(1 "p50: Keepers" 5 "p50: Liquidators" 9 "p50: Renters" 13 "p50: Liq as-if-renters") rows(2)) ///
  ylabel(0(50)250) ///
  xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
  yscale(range(0 250)) ///
  xti("# waves until exit interview (0 = exit interview)") ///
  yti("net worth")

restore

graph2tex, epsfile(${figs}/FIGH1c)

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
** Keepers/Liquidators // Statistics for Appendix H
********************************************************************************

* turn on log
cap log close
log using ${logs}/APPENDIX_H_STATISTICS.txt, text replace

preserve

keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)

by hhidpn: egen own_0 = max(cond(xnriw==0,ownHist,.))
by hhidpn: egen own_1 = max(cond(xnriw==1,ownHist,.))
by hhidpn: egen own_2 = max(cond(xnriw==2,ownHist,.))
by hhidpn: egen own_3 = max(cond(xnriw==3,ownHist,.))
by hhidpn: egen own_4 = max(cond(xnriw==4,ownHist,.))

gen keep_43 = (own_4==1 & own_3==1)
gen sell_43 = (own_4==1 & own_3==0)
gen rent_43 = (own_4==0 & own_3==0)

gen keep_32 = (own_3==1 & own_2==1)
gen sell_32 = (own_3==1 & own_2==0)
gen rent_32 = (own_3==0 & own_2==0)

gen keep_21 = (own_2==1 & own_1==1)
gen sell_21 = (own_2==1 & own_1==0)
gen rent_21 = (own_2==0 & own_1==0)

gen keep_10 = (own_1==1 & own_0==1)
gen sell_10 = (own_1==1 & own_0==0)
gen rent_10 = (own_1==0 & own_0==0)

gen post_sell = .
replace post_sell = 1 if (xnriw==3 & sell_43==1)
replace post_sell = 1 if (xnriw==2 & sell_32==1)
replace post_sell = 1 if (xnriw==1 & sell_21==1)
replace post_sell = 1 if (xnriw==0 & sell_10==1)
// identify wave after selling

gen pre_sell = .
replace pre_sell = 1 if (xnriw==4 & sell_43==1)
replace pre_sell = 1 if (xnriw==3 & sell_32==1)
replace pre_sell = 1 if (xnriw==2 & sell_21==1)
replace pre_sell = 1 if (xnriw==1 & sell_10==1)
// identify wave after selling

tab pre_sell
tab post_sell
// same numbers of obs

tabstat wlthhist [aw=crwtall] if pre_sell ==1, c(s) s(n p50) f(%5.0fc)
tabstat wlthhist [aw=crwtall] if post_sell==1, c(s) s(n p50) f(%5.0fc)
// weights lead #s of obs to be different b/c some have zero weight

// Median wealth pre  liquidation in waves T-4 to T-1 is 139K.
// Median wealth post liquidation in waves T-3 to T-0 is  23K.

// Change in medians is 139-23 = 116K.

// This is the change we wish to explain.

// NOTE: This is not the same thing as the median change in wealth.

* change in net worth
gen wlthhistDiff = (wlthhist - cL1wlthhist) 
// note: not annualized
// note: difference uses carried forward, lagged wealth history

tabstat wlthhistDiff [aw=crwtall] if post_sell==1, c(s) s(n p50) f(%5.0fc)

// Median change in wealth when selling is -67K.

**********
** Comparing those who entered an NH versus remained in the community
**********

by hhidpn: egen nhr_0 = max(cond(xnriw==0,rnhmliv,.))
by hhidpn: egen nhr_1 = max(cond(xnriw==1,rnhmliv,.))
by hhidpn: egen nhr_2 = max(cond(xnriw==2,rnhmliv,.))
by hhidpn: egen nhr_3 = max(cond(xnriw==3,rnhmliv,.))
by hhidpn: egen nhr_4 = max(cond(xnriw==4,rnhmliv,.))

** Stay NH, enter NH, remain in the community

gen stay_43 = (nhr_4==1 & nhr_3==1) // stay: nh resident both periods
gen entr_43 = (nhr_4==0 & nhr_3==1) // entr: nh resident in 2nd period only
gen cmty_43 = (nhr_4==0 & nhr_3==0) // cmty: community resident in both periods

gen stay_32 = (nhr_3==1 & nhr_2==1)
gen entr_32 = (nhr_3==0 & nhr_2==1)
gen cmty_32 = (nhr_3==0 & nhr_2==0)

gen stay_21 = (nhr_2==1 & nhr_1==1)
gen entr_21 = (nhr_2==0 & nhr_1==1)
gen cmty_21 = (nhr_2==0 & nhr_1==0)

gen stay_10 = (nhr_1==1 & nhr_0==1)
gen entr_10 = (nhr_1==0 & nhr_0==1)
gen cmty_10 = (nhr_1==0 & nhr_0==0)

// Liquidators who are NH residents pre- and post-liquidation
gen stay_post_sell = .
replace stay_post_sell = 1 if (xnriw==3 & sell_43==1 & stay_43==1)
replace stay_post_sell = 1 if (xnriw==2 & sell_32==1 & stay_32==1)
replace stay_post_sell = 1 if (xnriw==1 & sell_21==1 & stay_21==1)
replace stay_post_sell = 1 if (xnriw==0 & sell_10==1 & stay_10==1)
// identify wave before selling

gen stay_pre_sell = .
replace stay_pre_sell = 1 if (xnriw==4 & sell_43==1 & stay_43==1)
replace stay_pre_sell = 1 if (xnriw==3 & sell_32==1 & stay_32==1)
replace stay_pre_sell = 1 if (xnriw==2 & sell_21==1 & stay_21==1)
replace stay_pre_sell = 1 if (xnriw==1 & sell_10==1 & stay_10==1)
// identify wave after selling

// Liquidators who are enter NH residents post-liquidation
gen entr_post_sell = .
replace entr_post_sell = 1 if (xnriw==3 & sell_43==1 & entr_43==1)
replace entr_post_sell = 1 if (xnriw==2 & sell_32==1 & entr_32==1)
replace entr_post_sell = 1 if (xnriw==1 & sell_21==1 & entr_21==1)
replace entr_post_sell = 1 if (xnriw==0 & sell_10==1 & entr_10==1)
// identify wave before selling

gen entr_pre_sell = .
replace entr_pre_sell = 1 if (xnriw==4 & sell_43==1 & entr_43==1)
replace entr_pre_sell = 1 if (xnriw==3 & sell_32==1 & entr_32==1)
replace entr_pre_sell = 1 if (xnriw==2 & sell_21==1 & entr_21==1)
replace entr_pre_sell = 1 if (xnriw==1 & sell_10==1 & entr_10==1)
// identify wave after selling

// Liquidators who are community residents before and after liquidation
gen cmty_post_sell = .
replace cmty_post_sell = 1 if (xnriw==3 & sell_43==1 & cmty_43==1)
replace cmty_post_sell = 1 if (xnriw==2 & sell_32==1 & cmty_32==1)
replace cmty_post_sell = 1 if (xnriw==1 & sell_21==1 & cmty_21==1)
replace cmty_post_sell = 1 if (xnriw==0 & sell_10==1 & cmty_10==1)
// identify wave before selling

gen cmty_pre_sell = .
replace cmty_pre_sell = 1 if (xnriw==4 & sell_43==1 & cmty_43==1)
replace cmty_pre_sell = 1 if (xnriw==3 & sell_32==1 & cmty_32==1)
replace cmty_pre_sell = 1 if (xnriw==2 & sell_21==1 & cmty_21==1)
replace cmty_pre_sell = 1 if (xnriw==1 & sell_10==1 & cmty_10==1)
// identify wave after selling

tab stay_pre_sell
tab stay_post_sell
// same numbers of obs

tab entr_pre_sell
tab entr_post_sell
// same numbers of obs

tab cmty_pre_sell
tab cmty_post_sell
// same numbers of obs

// stay NH
tabstat wlthhist [aw=crwtall] if stay_pre_sell ==1, c(s) s(n p50) f(%5.0fc)
tabstat wlthhist [aw=crwtall] if stay_post_sell==1, c(s) s(n p50) f(%5.0fc)
// weights lead #s of obs to be different b/c some have zero weight

// enter NH
tabstat wlthhist [aw=crwtall] if entr_pre_sell ==1, c(s) s(n p50) f(%5.0fc)
tabstat wlthhist [aw=crwtall] if entr_post_sell==1, c(s) s(n p50) f(%5.0fc)
// weights lead #s of obs to be different b/c some have zero weight

// remain CR
tabstat wlthhist [aw=crwtall] if cmty_pre_sell ==1, c(s) s(n p50) f(%5.0fc)
tabstat wlthhist [aw=crwtall] if cmty_post_sell==1, c(s) s(n p50) f(%5.0fc)
// weights lead #s of obs to be different b/c some have zero weight

/*

# liquidators total: 741

# liquidators who are NHRs pre- and post-liquidation: 64 (8.6%)
# liquidators who are CRs pre-liquidation and NHRs post-liquidation: 182 (24.6%)
# liquidators who were CRs pre- and post-liquidation: 485. (65.5%)
(sum = 64+182+485 = 731.  Implies that 10 (1.3%) liquidators are NHRs pre and CRs post.)

NH stayers:
Pre-liquidation NW = 126
Post-liquidation NW = 21
Change = -105 (83.3%)

NH entrants:
Pre-liquidation NW = 138
Post-liquidation NW = 15
Change = -123 (89.1%)

CR stayers:
Pre-liquidation NW = 138
Post-liquidation NW = 27
Change = -111 (80.4%)

*/

**********
** Assess the role of out-of-pocket medical spending.
**********

gen oop = total_oop_HH / 1e3
// convert to 1000s of 2010 dollars
// note: using total household oop, including respondent and spouse (both core and exit)

gen wlthhist_plus_oop = wlthhist + oop
// net worth plus medical spending accruing between current/previous interview

tabstat wlthhist          [aw=crwtall] if pre_sell ==1, c(s) s(n p50) f(%5.0fc)
tabstat wlthhist_plus_oop [aw=crwtall] if post_sell==1, c(s) s(n p50) f(%5.0fc)
// weights lead #s of obs to be different b/c some have zero weight

// Median wealth pre  liquidation in waves T-4 to T-1 is                       139K.
// Median wealth post liquidation in waves T-3 to T-0 without replacing OOP is  23K.
// Median wealth post liquidation in waves T-3 to T-0 after   replacing OOP is  55K.

// Change in medians is 139-23 = (139-55) + (55-23) = 116K.
//                             =   84     +   22    = 116K.

// Median expenditures account for 55 - 23 = 22K of the 116K decrease in.
// That is, 22/116 = 19%. 

**********
** Counterfactuals: liquidators as if renters.
**********

sort hhidpn w
set seed 1234

gen post_rent = .
replace post_rent = 1 if (xnriw==3 & rent_43==1)
replace post_rent = 1 if (xnriw==2 & rent_32==1)
replace post_rent = 1 if (xnriw==1 & rent_21==1)
replace post_rent = 1 if (xnriw==0 & rent_10==1)
// identify wave after selling

* assign counterfactual changes in net worth from renters to liquidators
* find nearest neighbors using carried forward, lagged wealth history
gen to_impute = (post_sell==1 | post_rent==1)
// want only (liquidators) and (renters)
gen z = wlthhistDiff if (post_rent==1)
// only use data from (renters)
mi set wide
mi register imputed z
* impute one wave at a time (end-points only)
mi impute pmm z cL1wlthhist if (to_impute==1 & xnriw==3), knn(1) add(1) noisily
mi impute pmm z cL1wlthhist if (to_impute==1 & xnriw==2), knn(1) add(1) noisily
mi impute pmm z cL1wlthhist if (to_impute==1 & xnriw==1), knn(1) add(1) noisily
mi impute pmm z cL1wlthhist if (to_impute==1 & xnriw==0), knn(1) add(1) noisily
desc _*_z
egen zi = rowmax(_*_z)
egen N  = rownonmiss(_*_z)
tab N if (post_sell==1) & inlist(xnriw,3,2,1,0) // -> expect 1s only
tab N if (post_rent==1) & inlist(xnriw,3,2,1,0) // -> expect 4s only
mi extract 0, clear // clear the imputation, delete imputed data

gen ctfl = cL1wlthhist + zi if (post_sell==1)
// carried forward lagged wealth history + counterfactually assigned change
// note: z = wlthhistDiff = wlthhist - cL1wlthhist

gen wlthHistCtfl = .
replace wlthHistCtfl = wlthhist if (pre_sell ==1) & inlist(xnriw,4,3,2,1) 
// start of the segment, use original wealth history
replace wlthHistCtfl = ctfl     if (post_sell==1) & inlist(xnriw,3,2,1,0)
// end of segment, use counterfactual wealth history

tabstat wlthHistCtfl [aw=crwtall] if pre_sell ==1, c(s) s(n p50) f(%5.0fc)
tabstat wlthHistCtfl [aw=crwtall] if post_sell==1, c(s) s(n p50) f(%5.0fc)

// Median wealth pre  liquidation in waves T-4 to T-1 is                           139K.
// Median wealth post liquidation in waves T-3 to T-0 for synthetic liquidators is  83K.
// (synthetic liquidators = liquidators as if renters)

// Change in medians for synthetic liquidators is 139-83 = 56K.

// (Change in medians) - (Change for synthetic liquidators) = 116 - 56 = 60K.

// Median expenditures account for 60K of the 116K decrease in.
// That is, 60/116 = 52%. 

restore

* close log
cap log close

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

graph drop _all

********************************************************************************
** Trajectories with confidence intervals
********************************************************************************

set scheme hack, permanent

******************************
* Subroutine to generate 95% confidence intervals
******************************

cap program drop CalculateCI
program define CalculateCI

* write time until death t=4,3,2,1,0
qui gen t = _n - 1 in 1/5

* loop over quantiles
foreach q in 10 25 50 75 90 {
	
	local qq = `q'/100
	
	qui gen p`q' = .
	qui gen ci95u`q' = .
	qui gen ci95l`q' = .
	
	* loop over x-axis values
	forvalues t = 0(1)4 {
	
	  qui: qreg wlthhist [pw=crwtall] if xnriw==`t', q(`qq') vce(robust)
		mat b = e(b)
		mat V = e(V)
		
		qui: replace p`q' = b[1,1] if (t==`t')
		qui: replace ci95u`q' = b[1,1] + 1.96 * sqrt(V[1,1]) if (t==`t')
		qui: replace ci95l`q' = b[1,1] - 1.96 * sqrt(V[1,1]) if (t==`t')
	
	}
}
end

******************************
** Subroutine to plot quantiles with confidence intervals
******************************

cap program drop PlotQuantilesWithCI
program define PlotQuantilesWithCI

args yIncrement yMaxValue FigName SubTitle

cap graph drop `FigName'

gen nxnriw = -t

local linestyleCI "lp(dash ) lw(*0.5)"
local linestyleQ  "lp(solid) lw(*1  )"

gr tw ///
(line ci95u10 nxnriw, `linestyleCI' lc(maroon)) ///
(line ci95l10 nxnriw, `linestyleCI' lc(maroon)) ///
(line ci95u25 nxnriw, `linestyleCI' lc(navy)) ///
(line ci95l25 nxnriw, `linestyleCI' lc(navy)) ///
(line ci95u50 nxnriw, `linestyleCI' lc(forest_green)) ///
(line ci95l50 nxnriw, `linestyleCI' lc(forest_green)) ///
(line ci95u75 nxnriw, `linestyleCI' lc(dkorange)) ///
(line ci95l75 nxnriw, `linestyleCI' lc(dkorange)) ///
(line ci95u90 nxnriw, `linestyleCI' lc(teal)) ///
(line ci95l90 nxnriw, `linestyleCI' lc(teal)) ///
(line p10 nxnriw, `linestyleQ' lc(maroon)) ///
(line p25 nxnriw, `linestyleQ' lc(navy)) ///
(line p50 nxnriw, `linestyleQ' lc(forest_green)) ///
(line p75 nxnriw, `linestyleQ' lc(dkorange)) ///
(line p90 nxnriw, `linestyleQ' lc(teal)) ///
, ///
ylabel(0(`=`yIncrement'')`=`yMaxValue'', angle(0) gmax) ///
xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
yscale(range(0 `yMaxValue')) ///
legend(order(11 "p10" 12 "p25" 13 "p50" 14 "p75" 15 "p90" 1 "CI") rows(1)) ///
xti("# waves until exit interview (0 = exit interview)") ///
yti("net worth") ///
subti(`SubTitle') ///
name(`FigName')

end

******************************
* Unconditional wealth trajectories with 95% confidence intervals
******************************

/*
APPENDIX FIGURE K.1 PANEL (F)
*/

preserve
keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)
keep xnriw wlthhist crwtall 
* Run subroutine to get confidence intervals
CalculateCI
* Make plots
PlotQuantilesWithCI 100 700 fig1
restore

graph2tex, epsfile(${figs}/FIGK1f)

******************************
** Wealth trajectories by child status with 95% confidence intervals
******************************

/*
APPENDIX FIGURE K.1 PANEL (E)
*/

** Kids
preserve
keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)
keep if (nochild==0)
keep xnriw wlthhist crwtall 
* Run subroutine to get confidence intervals
CalculateCI
* Make plots
PlotQuantilesWithCI 200 1200 fig1 "Parents"
restore

** No Kids
preserve
keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)
keep if (nochild==1)
keep xnriw wlthhist crwtall 
* Run subroutine to get confidence intervals
CalculateCI
* Make plots
PlotQuantilesWithCI 200 1200 fig2 "Childless"
restore

** Combine
grc1leg fig1 fig2

graph2tex, epsfile(${figs}/FIGK1e) 

******************************
** Wealth trajectories by homeownership (own at T-1) with 95% confidence intervals
******************************

/*
APPENDIX FIGURE K.1 PANEL (A)
*/

** Own at T-1
preserve
keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)
keep if (own_1x==1)
keep xnriw wlthhist crwtall 
* Run subroutine to get confidence intervals
CalculateCI
* Make plots
PlotQuantilesWithCI 150 1050 fig1 "Owners at last core interview"
restore

** Rent at T-1
preserve
keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)
keep if (own_1x==0)
keep xnriw wlthhist crwtall 
* Run subroutine to get confidence intervals
CalculateCI
* Make plots
PlotQuantilesWithCI 150 1050 fig2 "Renters at last core interview"
restore

** Combine
grc1leg fig1 fig2

graph2tex, epsfile(${figs}/FIGK1a)

******************************
** Wealth trajectories by NH status (at T-1) with 95% confidence intervals
******************************

/*
APPENDIX FIGURE K.1 PANEL (C)
*/

** NHR at T
preserve
keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)
keep if (nhr_1x==1)
keep xnriw wlthhist crwtall 
* Run subroutine to get confidence intervals
CalculateCI
* Make plots
PlotQuantilesWithCI 100 800 fig1 "NHR at last core interview"
restore

** CR at T
preserve
keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)
keep if (nhr_1x==0)
keep xnriw wlthhist crwtall 
* Run subroutine to get confidence intervals
CalculateCI
* Make plots
PlotQuantilesWithCI 100 800 fig2 "CR at at last core interview"
restore

** Combine
grc1leg fig1 fig2

graph2tex, epsfile(${figs}/FIGK1c)

********************************************************************************
** Owners/Keepers // Owners/Liquidators // Renters // Median trajectories 
** WITH CONFIDENCE INTERVALS
********************************************************************************

/*
APPENDIX FIGURE K.1 PANEL (B)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)

by hhidpn: egen own_0 = max(cond(xnriw==0,ownHist,.))
by hhidpn: egen own_1 = max(cond(xnriw==1,ownHist,.))
by hhidpn: egen own_2 = max(cond(xnriw==2,ownHist,.))
by hhidpn: egen own_3 = max(cond(xnriw==3,ownHist,.))
by hhidpn: egen own_4 = max(cond(xnriw==4,ownHist,.))

gen keep_43 = (own_4==1 & own_3==1)
gen sell_43 = (own_4==1 & own_3==0)
gen rent_43 = (own_4==0 & own_3==0)

gen keep_32 = (own_3==1 & own_2==1)
gen sell_32 = (own_3==1 & own_2==0)
gen rent_32 = (own_3==0 & own_2==0)

gen keep_21 = (own_2==1 & own_1==1)
gen sell_21 = (own_2==1 & own_1==0)
gen rent_21 = (own_2==0 & own_1==0)

gen keep_10 = (own_1==1 & own_0==1)
gen sell_10 = (own_1==1 & own_0==0)
gen rent_10 = (own_1==0 & own_0==0)

expand 2 if (inlist(xnriw,3,2,1)), gen(dup) // -> each of the middle obs need to be duplicated
tab xnriw dup // -> check

local d = 0.25 // -> x-axis increment

replace xnriw = xnriw + `d' if (dup==1)
tab xnriw dup

gen group = .

* Keepers group
replace group = 1 if (xnriw==   4  & keep_43==1) // keepers 4->3 at wave 4
replace group = 1 if (xnriw==3+`d' & keep_43==1) // keepers 4->3 at wave 3
replace group = 1 if (xnriw==   3  & keep_32==1) // keepers 3->2 at wave 3
replace group = 1 if (xnriw==2+`d' & keep_32==1) // keepers 3->2 at wave 2
replace group = 1 if (xnriw==   2  & keep_21==1) // keepers 2->1 at wave 2
replace group = 1 if (xnriw==1+`d' & keep_21==1) // keepers 2->1 at wave 1
replace group = 1 if (xnriw==   1  & keep_10==1) // keepers 1->0 at wave 1
replace group = 1 if (xnriw==   0  & keep_10==1) // keepers 1->0 at wave 0

* Sellers group
replace group = 2 if (xnriw==   4  & sell_43==1) // sellers 4->3 at wave 4
replace group = 2 if (xnriw==3+`d' & sell_43==1) // sellers 4->3 at wave 3
replace group = 2 if (xnriw==   3  & sell_32==1) // sellers 3->2 at wave 3
replace group = 2 if (xnriw==2+`d' & sell_32==1) // sellers 3->2 at wave 2
replace group = 2 if (xnriw==   2  & sell_21==1) // sellers 2->1 at wave 2
replace group = 2 if (xnriw==1+`d' & sell_21==1) // sellers 2->1 at wave 1
replace group = 2 if (xnriw==   1  & sell_10==1) // sellers 1->0 at wave 1
replace group = 2 if (xnriw==   0  & sell_10==1) // sellers 1->0 at wave 0

* Renters group
replace group = 3 if (xnriw==   4  & rent_43==1) // renters 4->3 at wave 4
replace group = 3 if (xnriw==3+`d' & rent_43==1) // renters 4->3 at wave 3
replace group = 3 if (xnriw==   3  & rent_32==1) // renters 3->2 at wave 3
replace group = 3 if (xnriw==2+`d' & rent_32==1) // renters 3->2 at wave 2
replace group = 3 if (xnriw==   2  & rent_21==1) // renters 2->1 at wave 2
replace group = 3 if (xnriw==1+`d' & rent_21==1) // renters 2->1 at wave 1
replace group = 3 if (xnriw==   1  & rent_10==1) // renters 1->0 at wave 1
replace group = 3 if (xnriw==   0  & rent_10==1) // renters 1->0 at wave 0

drop if missing(group)

* write groups
qui gen g = 1 + floor( (_n-1) / 8 ) in 1/24

* write time until death t=4,3,2,1,0
qui gen t = mod(_n-1,8) in 1/24
recode t (0=0) (1=1) (2=`=1+`d'') (3=2) (4=`=2+`d'') (5=3) (6=`=3+`d'') (7=4)

* check
list g t in 1/24, sepby(g)

* loop over quantiles
foreach q in 50 {
	
	local qq = `q'/100
	
	qui gen p`q' = .
	qui gen ci95u`q' = .
	qui gen ci95l`q' = .
	
  * loop over groups
  foreach g in 1 2 3 {	
	
	* loop over x-axis values
	foreach t in 0 1 `=1+`d'' 2 `=2+`d'' 3 `=3+`d'' 4 {
	
	  qui: qreg wlthhist [pw=crwtall] if xnriw==`t' & group==`g', q(`qq') vce(robust)
		mat b = e(b)
		mat V = e(V)
		
		qui: replace p`q' = b[1,1] if (t==`t' & g==`g')
		qui: replace ci95u`q' = b[1,1] + 1.96 * sqrt(V[1,1]) if (t==`t' & g==`g')
		qui: replace ci95l`q' = b[1,1] - 1.96 * sqrt(V[1,1]) if (t==`t' & g==`g')
	
	}
  }
}

keep g t p50 ci95u50 ci95l50

* list
list g t p50 ci95u50 ci95l50 in 1/24, sepby(g)

gen segment = .
replace segment = 1 if (t==4 | t==3+`d')
replace segment = 2 if (t==3 | t==2+`d')
replace segment = 3 if (t==2 | t==1+`d')
replace segment = 4 if (t==1 | t==0    )

gen nxnriw = -round(t)

local yMaxValue   250
local yIncrement   50
local linestyleCI "lp(dash ) lw(*0.5)"
local linestyleQ  "lp(solid) lw(*1  )"

cap graph drop _all

gr tw ///
/// keepers
(line ci95u50 nxnriw if (g==1 & segment==1), `linestyleCI' lc(maroon)) ///
(line ci95u50 nxnriw if (g==1 & segment==2), `linestyleCI' lc(maroon)) ///
(line ci95u50 nxnriw if (g==1 & segment==3), `linestyleCI' lc(maroon)) ///
(line ci95u50 nxnriw if (g==1 & segment==4), `linestyleCI' lc(maroon)) ///
(line ci95l50 nxnriw if (g==1 & segment==1), `linestyleCI' lc(maroon)) ///
(line ci95l50 nxnriw if (g==1 & segment==2), `linestyleCI' lc(maroon)) ///
(line ci95l50 nxnriw if (g==1 & segment==3), `linestyleCI' lc(maroon)) ///
(line ci95l50 nxnriw if (g==1 & segment==4), `linestyleCI' lc(maroon)) ///
(line p50 nxnriw if (g==1 & segment==1), `linestyleQ' lc(maroon)) ///
(line p50 nxnriw if (g==1 & segment==2), `linestyleQ' lc(maroon)) ///
(line p50 nxnriw if (g==1 & segment==3), `linestyleQ' lc(maroon)) ///
(line p50 nxnriw if (g==1 & segment==4), `linestyleQ' lc(maroon)) ///
/// liquidators
(line ci95u50 nxnriw if (g==2 & segment==1), `linestyleCI' lc(navy)) ///
(line ci95u50 nxnriw if (g==2 & segment==2), `linestyleCI' lc(navy)) ///
(line ci95u50 nxnriw if (g==2 & segment==3), `linestyleCI' lc(navy)) ///
(line ci95u50 nxnriw if (g==2 & segment==4), `linestyleCI' lc(navy)) ///
(line ci95l50 nxnriw if (g==2 & segment==1), `linestyleCI' lc(navy)) ///
(line ci95l50 nxnriw if (g==2 & segment==2), `linestyleCI' lc(navy)) ///
(line ci95l50 nxnriw if (g==2 & segment==3), `linestyleCI' lc(navy)) ///
(line ci95l50 nxnriw if (g==2 & segment==4), `linestyleCI' lc(navy)) ///
(line p50 nxnriw if (g==2 & segment==1), `linestyleQ' lc(navy)) ///
(line p50 nxnriw if (g==2 & segment==2), `linestyleQ' lc(navy)) ///
(line p50 nxnriw if (g==2 & segment==3), `linestyleQ' lc(navy)) ///
(line p50 nxnriw if (g==2 & segment==4), `linestyleQ' lc(navy)) ///
/// renters
(line ci95u50 nxnriw if (g==3 & segment==1), `linestyleCI' lc(dkorange)) ///
(line ci95u50 nxnriw if (g==3 & segment==2), `linestyleCI' lc(dkorange)) ///
(line ci95u50 nxnriw if (g==3 & segment==3), `linestyleCI' lc(dkorange)) ///
(line ci95u50 nxnriw if (g==3 & segment==4), `linestyleCI' lc(dkorange)) ///
(line ci95l50 nxnriw if (g==3 & segment==1), `linestyleCI' lc(dkorange)) ///
(line ci95l50 nxnriw if (g==3 & segment==2), `linestyleCI' lc(dkorange)) ///
(line ci95l50 nxnriw if (g==3 & segment==3), `linestyleCI' lc(dkorange)) ///
(line ci95l50 nxnriw if (g==3 & segment==4), `linestyleCI' lc(dkorange)) ///
(line p50 nxnriw if (g==3 & segment==1), `linestyleQ' lc(dkorange)) ///
(line p50 nxnriw if (g==3 & segment==2), `linestyleQ' lc(dkorange)) ///
(line p50 nxnriw if (g==3 & segment==3), `linestyleQ' lc(dkorange)) ///
(line p50 nxnriw if (g==3 & segment==4), `linestyleQ' lc(dkorange)) ///
, ///
ylabel(0(`=`yIncrement'')`=`yMaxValue'', angle(0) gmax) ///
xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
yscale(range(0 `yMaxValue')) ///
legend(order(9 "Keepers: p50"    21 "Liquidators: p50"    36 "Renters: p50"  ///
             1 "95% CI"          16 "95% CI"              31 "95% CI" ) rows(2)) ///
xti("# waves until exit interview (0 = exit interview)") ///
yti("net worth") ///
name(fig)

restore

graph2tex, epsfile(${figs}/FIGK1b)

********************************************************************************
** Community residents // NH entrants // NH stayers // Median trajectories 
** WITH CONFIDENCE INTERVALS
********************************************************************************

/*
APPENDIX FIGURE K.1 PANEL (D)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel
keep if (xnriw<=4)

by hhidpn: egen nhr_0 = max(cond(xnriw==0,rnhmliv,.))
by hhidpn: egen nhr_1 = max(cond(xnriw==1,rnhmliv,.))
by hhidpn: egen nhr_2 = max(cond(xnriw==2,rnhmliv,.))
by hhidpn: egen nhr_3 = max(cond(xnriw==3,rnhmliv,.))
by hhidpn: egen nhr_4 = max(cond(xnriw==4,rnhmliv,.))

gen stay_43 = (nhr_4==1 & nhr_3==1) // stay: nh resident both periods
gen entr_43 = (nhr_4==0 & nhr_3==1) // entr: nh resident in 2nd period only
gen cmty_43 = (nhr_4==0 & nhr_3==0) // cmty: community resident in both periods

gen stay_32 = (nhr_3==1 & nhr_2==1)
gen entr_32 = (nhr_3==0 & nhr_2==1)
gen cmty_32 = (nhr_3==0 & nhr_2==0)

gen stay_21 = (nhr_2==1 & nhr_1==1)
gen entr_21 = (nhr_2==0 & nhr_1==1)
gen cmty_21 = (nhr_2==0 & nhr_1==0)

gen stay_10 = (nhr_1==1 & nhr_0==1)
gen entr_10 = (nhr_1==0 & nhr_0==1)
gen cmty_10 = (nhr_1==0 & nhr_0==0)

expand 2 if (inlist(xnriw,3,2,1)), gen(dup) // -> each of the middle obs need to be duplicated
tab xnriw dup // -> check

local d = 0.25 // -> x-axis increment

replace xnriw = xnriw + `d' if (dup==1)
tab xnriw dup

gen group = .

* Community residents group
replace group = 1 if (xnriw==   4  & cmty_43==1) // community residents 4->3 at wave 4
replace group = 1 if (xnriw==3+`d' & cmty_43==1) // community residents 4->3 at wave 3
replace group = 1 if (xnriw==   3  & cmty_32==1) // community residents 3->2 at wave 3
replace group = 1 if (xnriw==2+`d' & cmty_32==1) // community residents 3->2 at wave 2
replace group = 1 if (xnriw==   2  & cmty_21==1) // community residents 2->1 at wave 2
replace group = 1 if (xnriw==1+`d' & cmty_21==1) // community residents 2->1 at wave 1
replace group = 1 if (xnriw==   1  & cmty_10==1) // community residents 1->0 at wave 1
replace group = 1 if (xnriw==   0  & cmty_10==1) // community residents 1->0 at wave 0

* Entrants group
replace group = 2 if (xnriw==   4  & entr_43==1) // entrants 4->3 at wave 4
replace group = 2 if (xnriw==3+`d' & entr_43==1) // entrants 4->3 at wave 3
replace group = 2 if (xnriw==   3  & entr_32==1) // entrants 3->2 at wave 3
replace group = 2 if (xnriw==2+`d' & entr_32==1) // entrants 3->2 at wave 2
replace group = 2 if (xnriw==   2  & entr_21==1) // entrants 2->1 at wave 2
replace group = 2 if (xnriw==1+`d' & entr_21==1) // entrants 2->1 at wave 1
replace group = 2 if (xnriw==   1  & entr_10==1) // entrants 1->0 at wave 1
replace group = 2 if (xnriw==   0  & entr_10==1) // entrants 1->0 at wave 0

* Stayers group
replace group = 3 if (xnriw==   4  & stay_43==1) // stayers 4->3 at wave 4
replace group = 3 if (xnriw==3+`d' & stay_43==1) // stayers 4->3 at wave 3
replace group = 3 if (xnriw==   3  & stay_32==1) // stayers 3->2 at wave 3
replace group = 3 if (xnriw==2+`d' & stay_32==1) // stayers 3->2 at wave 2
replace group = 3 if (xnriw==   2  & stay_21==1) // stayers 2->1 at wave 2
replace group = 3 if (xnriw==1+`d' & stay_21==1) // stayers 2->1 at wave 1
replace group = 3 if (xnriw==   1  & stay_10==1) // stayers 1->0 at wave 1
replace group = 3 if (xnriw==   0  & stay_10==1) // stayers 1->0 at wave 0

drop if missing(group)

* write groups
qui gen g = 1 + floor( (_n-1) / 8 ) in 1/24

* write time until death t=4,3,2,1,0
qui gen t = mod(_n-1,8) in 1/24
recode t (0=0) (1=1) (2=`=1+`d'') (3=2) (4=`=2+`d'') (5=3) (6=`=3+`d'') (7=4)

* check
list g t in 1/24, sepby(g)

* loop over quantiles
foreach q in 50 {
	
	local qq = `q'/100
	
	qui gen p`q' = .
	qui gen ci95u`q' = .
	qui gen ci95l`q' = .
	
  * loop over groups
  foreach g in 1 2 3 {	
	
	* loop over x-axis values
	foreach t in 0 1 `=1+`d'' 2 `=2+`d'' 3 `=3+`d'' 4 {
	
	  qui: qreg wlthhist [pw=crwtall] if xnriw==`t' & group==`g', q(`qq') vce(robust)
		mat b = e(b)
		mat V = e(V)
		
		qui: replace p`q' = b[1,1] if (t==`t' & g==`g')
		qui: replace ci95u`q' = b[1,1] + 1.96 * sqrt(V[1,1]) if (t==`t' & g==`g')
		qui: replace ci95l`q' = b[1,1] - 1.96 * sqrt(V[1,1]) if (t==`t' & g==`g')
	
	}
  }
}

keep g t p50 ci95u50 ci95l50

* list
list g t p50 ci95u50 ci95l50 in 1/24, sepby(g)

gen segment = .
replace segment = 1 if (t==4 | t==3+`d')
replace segment = 2 if (t==3 | t==2+`d')
replace segment = 3 if (t==2 | t==1+`d')
replace segment = 4 if (t==1 | t==0    )

gen nxnriw = -round(t)

local yMaxValue   125
local yIncrement   25
local linestyleCI "lp(dash ) lw(*0.5)"
local linestyleQ  "lp(solid) lw(*1  )"

cap graph drop _all

gr tw ///
/// keepers
(line ci95u50 nxnriw if (g==1 & segment==1), `linestyleCI' lc(maroon)) ///
(line ci95u50 nxnriw if (g==1 & segment==2), `linestyleCI' lc(maroon)) ///
(line ci95u50 nxnriw if (g==1 & segment==3), `linestyleCI' lc(maroon)) ///
(line ci95u50 nxnriw if (g==1 & segment==4), `linestyleCI' lc(maroon)) ///
(line ci95l50 nxnriw if (g==1 & segment==1), `linestyleCI' lc(maroon)) ///
(line ci95l50 nxnriw if (g==1 & segment==2), `linestyleCI' lc(maroon)) ///
(line ci95l50 nxnriw if (g==1 & segment==3), `linestyleCI' lc(maroon)) ///
(line ci95l50 nxnriw if (g==1 & segment==4), `linestyleCI' lc(maroon)) ///
(line p50 nxnriw if (g==1 & segment==1), `linestyleQ' lc(maroon)) ///
(line p50 nxnriw if (g==1 & segment==2), `linestyleQ' lc(maroon)) ///
(line p50 nxnriw if (g==1 & segment==3), `linestyleQ' lc(maroon)) ///
(line p50 nxnriw if (g==1 & segment==4), `linestyleQ' lc(maroon)) ///
/// liquidators
(line ci95u50 nxnriw if (g==2 & segment==1), `linestyleCI' lc(navy)) ///
(line ci95u50 nxnriw if (g==2 & segment==2), `linestyleCI' lc(navy)) ///
(line ci95u50 nxnriw if (g==2 & segment==3), `linestyleCI' lc(navy)) ///
(line ci95u50 nxnriw if (g==2 & segment==4), `linestyleCI' lc(navy)) ///
(line ci95l50 nxnriw if (g==2 & segment==1), `linestyleCI' lc(navy)) ///
(line ci95l50 nxnriw if (g==2 & segment==2), `linestyleCI' lc(navy)) ///
(line ci95l50 nxnriw if (g==2 & segment==3), `linestyleCI' lc(navy)) ///
(line ci95l50 nxnriw if (g==2 & segment==4), `linestyleCI' lc(navy)) ///
(line p50 nxnriw if (g==2 & segment==1), `linestyleQ' lc(navy)) ///
(line p50 nxnriw if (g==2 & segment==2), `linestyleQ' lc(navy)) ///
(line p50 nxnriw if (g==2 & segment==3), `linestyleQ' lc(navy)) ///
(line p50 nxnriw if (g==2 & segment==4), `linestyleQ' lc(navy)) ///
/// renters
(line ci95u50 nxnriw if (g==3 & segment==1), `linestyleCI' lc(dkorange)) ///
(line ci95u50 nxnriw if (g==3 & segment==2), `linestyleCI' lc(dkorange)) ///
(line ci95u50 nxnriw if (g==3 & segment==3), `linestyleCI' lc(dkorange)) ///
(line ci95u50 nxnriw if (g==3 & segment==4), `linestyleCI' lc(dkorange)) ///
(line ci95l50 nxnriw if (g==3 & segment==1), `linestyleCI' lc(dkorange)) ///
(line ci95l50 nxnriw if (g==3 & segment==2), `linestyleCI' lc(dkorange)) ///
(line ci95l50 nxnriw if (g==3 & segment==3), `linestyleCI' lc(dkorange)) ///
(line ci95l50 nxnriw if (g==3 & segment==4), `linestyleCI' lc(dkorange)) ///
(line p50 nxnriw if (g==3 & segment==1), `linestyleQ' lc(dkorange)) ///
(line p50 nxnriw if (g==3 & segment==2), `linestyleQ' lc(dkorange)) ///
(line p50 nxnriw if (g==3 & segment==3), `linestyleQ' lc(dkorange)) ///
(line p50 nxnriw if (g==3 & segment==4), `linestyleQ' lc(dkorange)) ///
, ///
ylabel(0(`=`yIncrement'')`=`yMaxValue'', angle(0) gmax) ///
xlabel(-4 "4" -3 "3" -2 "2" -1 "1" 0 "0") ///
yscale(range(0 `yMaxValue')) ///
legend(order(9 "Remain CR: p50"    21 "Enter NH: p50"    36 "Remain NHR: p50"  ///
             1 "95% CI"            16 "95% CI"           31 "95% CI" ) rows(2)) ///
xti("# waves until exit interview (0 = exit interview)") ///
yti("net worth") ///
name(fig)

restore

graph2tex, epsfile(${figs}/FIGK1d)

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

graph drop _all

********************************************************************************
** Data vs. Model Comparisons
********************************************************************************

set scheme hack, permanently

** Line thickness:

global lw_p10  *0.50
global lw_p25  *1.00
global lw_p50  *1.50
global lw_p75  *2.00
global lw_p90  *2.50

********************************************************************************
** Wealth trajectories by child status
********************************************************************************

******************************
** Wealth trajectories by child status -- Parents only
******************************

/*
FIGURE 11 PANEL (A)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel

tab xnriw nochild if (xnriw<=4)
keep if inrange(xnriw,0,4)

keep xnriw wlthhist nochild  crwtall 
collapse  (p10) p10=wlthhist (p25) p25=wlthhist (p50) p50=wlthhist ///
          (p75) p75=wlthhist (p90) p90=wlthhist [aw=crwtall],  by(xnriw nochild)

keep if (nochild==0)
// parents					
					
append using ${save}/model_traj_data_parents.dta, gen(model)
					
list, clean noobs					
			
gr tw (sc p10 p25 p50 p75 p90 xnriw if (model==1), ///
       lc(blue blue blue blue blue) ///
			 msymbol(x x x x x) ///
			 mc(blue blue blue blue blue) ///
       lp(solid solid solid solid solid) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) ///
			(sc p10 p25 p50 p75 p90 xnriw if (model==0), ///
       lc(red red red red red) ///
			 msymbol(Oh Oh Oh Oh Oh) ///
			 mc(red red red red red) ///
			 lp(dash dash dash dash dash) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) /// 
			 , legend(order(1 "Model" 6 "Data")) ///
			   ylabel(0(100)800, gmax angle(0)) ///
				 xscale(reverse) xti("Waves until exit interview") yti("Net worth")					
								
restore

graph2tex, epsfile(${figs}/FIG11a)

******************************
** Wealth trajectories by child status -- Childless only
******************************

/*
FIGURE 11 PANEL (B)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel

tab xnriw nochild if (xnriw<=4)
keep if inrange(xnriw,0,4)

keep xnriw wlthhist nochild  crwtall 
collapse  (p10) p10=wlthhist (p25) p25=wlthhist (p50) p50=wlthhist ///
          (p75) p75=wlthhist (p90) p90=wlthhist [aw=crwtall],  by(xnriw nochild)

keep if (nochild==1)
// childless					
					
append using ${save}/model_traj_data_childless.dta, gen(model)
					
list, clean noobs					
			
gr tw (sc p10 p25 p50 p75 p90 xnriw if (model==1), ///
       lc(blue blue blue blue blue) ///
			 msymbol(x x x x x) ///
			 mc(blue blue blue blue blue) ///
       lp(solid solid solid solid solid) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) ///
			(sc p10 p25 p50 p75 p90 xnriw if (model==0), ///
       lc(red red red red red) ///
			 msymbol(Oh Oh Oh Oh Oh) ///
			 mc(red red red red red) ///
			 lp(dash dash dash dash dash) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) /// 
			 , legend(order(1 "Model" 6 "Data")) ///
			   ylabel(0(100)800, gmax angle(0)) ///
				 xscale(reverse) xti("Waves until exit interview") yti("Net worth")					
								
restore

graph2tex, epsfile(${figs}/FIG11b)

********************************************************************************
** Wealth trajectories by homeownership
********************************************************************************

******************************
** Wealth trajectories by homeownership (own at T-1) -- Owners only
******************************

/*
FIGURE 8 PANEL (A)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel

tab xnriw own_1x if (xnriw<=4)
keep if inrange(xnriw,0,4)

keep xnriw wlthhist own_1x crwtall 
collapse  (p10) p10=wlthhist (p25) p25=wlthhist (p50) p50=wlthhist ///
          (p75) p75=wlthhist (p90) p90=wlthhist [aw=crwtall],  by(xnriw own_1x)
					
append using ${save}/model_traj_data_own_rent.dta, gen(model)

keep if (own_1x==1)
// owners at T-1
			
list, clean noobs			
			
gr tw (sc p10 p25 p50 p75 p90 xnriw if (model==1), ///
       lc(blue blue blue blue blue) ///
			 msymbol(x x x x x) ///
			 mc(blue blue blue blue blue) ///
       lp(solid solid solid solid solid) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) ///
			(sc p10 p25 p50 p75 p90 xnriw if (model==0), ///
       lc(red red red red red) ///
			 msymbol(Oh Oh Oh Oh Oh) ///
			 mc(red red red red red) ///
			 lp(dash dash dash dash dash) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) /// 
			 , legend(order(1 "Model" 6 "Data")) ///
			   ylabel(0(100)900, gmax angle(0)) ///
				 xscale(reverse) xti("Waves until exit interview") yti("Net worth")					
								
restore

graph2tex, epsfile(${figs}/FIG8a)

******************************
** Wealth trajectories by homeownership (own at T-1) -- Renters only
******************************

/*
FIGURE 8 PANEL (B)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel

tab xnriw own_1x if (xnriw<=4)
keep if inrange(xnriw,0,4)

keep xnriw wlthhist own_1x crwtall 
collapse  (p10) p10=wlthhist (p25) p25=wlthhist (p50) p50=wlthhist ///
          (p75) p75=wlthhist (p90) p90=wlthhist [aw=crwtall],  by(xnriw own_1x)
					
append using ${save}/model_traj_data_own_rent.dta, gen(model)

keep if (own_1x==0)
// renters at T-1
			
list, clean noobs		
			
gr tw (sc p10 p25 p50 p75 p90 xnriw if (model==1), ///
       lc(blue blue blue blue blue) ///
			 msymbol(x x x x x) ///
			 mc(blue blue blue blue blue) ///
       lp(solid solid solid solid solid) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) ///
			(sc p10 p25 p50 p75 p90 xnriw if (model==0), ///
       lc(red red red red red) ///
			 msymbol(Oh Oh Oh Oh Oh) ///
			 mc(red red red red red) ///
			 lp(dash dash dash dash dash) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) /// 
			 , legend(order(1 "Model" 6 "Data")) ///
			   ylabel(0(100)900, gmax angle(0)) ///
				 xscale(reverse) xti("Waves until exit interview") yti("Net worth")					
								
restore

graph2tex, epsfile(${figs}/FIG8b)

******************************
** Wealth trajectories by nursing home status (NH at T-1) -- NHR only
******************************

/*
FIGURE 10 PANEL (B)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel

tab xnriw nhr_1x if (xnriw<=4)
keep if inrange(xnriw,0,4)

keep xnriw wlthhist nhr_1x crwtall 
collapse  (p10) p10=wlthhist (p25) p25=wlthhist (p50) p50=wlthhist ///
          (p75) p75=wlthhist (p90) p90=wlthhist [aw=crwtall],  by(xnriw nhr_1x)
					
append using ${save}/model_traj_data_nhr_cr.dta, gen(model)

keep if (nhr_1x==1)
// NHR at T-1
			
list, clean noobs			
			
gr tw (sc p10 p25 p50 p75 p90 xnriw if (model==1), ///
       lc(blue blue blue blue blue) ///
			 msymbol(x x x x x) ///
			 mc(blue blue blue blue blue) ///
       lp(solid solid solid solid solid) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) ///
			(sc p10 p25 p50 p75 p90 xnriw if (model==0), ///
       lc(red red red red red) ///
			 msymbol(Oh Oh Oh Oh Oh) ///
			 mc(red red red red red) ///
			 lp(dash dash dash dash dash) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) /// 
			 , legend(order(1 "Model" 6 "Data")) ///
			   ylabel(0(100)800, gmax angle(0)) ///
				 xscale(reverse) xti("Waves until exit interview") yti("Net worth")					
								
restore

graph2tex, epsfile(${figs}/FIG10b)

******************************
** Wealth trajectories by nursing home status (NH at T-1) -- CR only
******************************

/*
FIGURE 10 PANEL (A)
*/

preserve

keep if (fullWlthHist4==1) // -> balanced panel

tab xnriw nhr_1x if (xnriw<=4)
keep if inrange(xnriw,0,4)

keep xnriw wlthhist nhr_1x crwtall 
collapse  (p10) p10=wlthhist (p25) p25=wlthhist (p50) p50=wlthhist ///
          (p75) p75=wlthhist (p90) p90=wlthhist [aw=crwtall],  by(xnriw nhr_1x)
					
append using ${save}/model_traj_data_nhr_cr.dta, gen(model)

keep if (nhr_1x==0)
// CR at T-1
			
list, clean noobs			
			
gr tw (sc p10 p25 p50 p75 p90 xnriw if (model==1), ///
       lc(blue blue blue blue blue) ///
			 msymbol(x x x x x) ///
			 mc(blue blue blue blue blue) ///
       lp(solid solid solid solid solid) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) ///
			(sc p10 p25 p50 p75 p90 xnriw if (model==0), ///
       lc(red red red red red) ///
			 msymbol(Oh Oh Oh Oh Oh) ///
			 mc(red red red red red) ///
			 lp(dash dash dash dash dash) ///
			 lw(${lw_p10} ${lw_p25} ${lw_p50} ${lw_p75} ${lw_p90}) ///
       c(d d d d d)) /// 
			 , legend(order(1 "Model" 6 "Data")) ///
			   ylabel(0(100)800, gmax angle(0)) ///
				 xscale(reverse) xti("Waves until exit interview") yti("Net worth")					
								
restore

graph2tex, epsfile(${figs}/FIG10a)

********************************************************************************

cap graph drop _all

********************************************************************************
