/*******************************************************************************

************
** Function: 
************

  Generates tables for the decedent sample.

************
** Inputs  : 
************

  - decedent_sample_single_final.dta    (Decedents_ProcessData.do)

************
** Outputs  : 
************

  - TAB2b.tex
	- TABA4.tex
	- TABA5.tex

	- TAB7_and_TAB8.txt
	
*******************************************************************************/

clear *
set more off
macro drop _all

********************************************************************************
** Director names
********************************************************************************

do GetDirNames.do

********************************************************************************
** DATA
********************************************************************************	

use ${save}/decedent_sample_single_final, clear

********************************************************************************
** Additional variables
********************************************************************************

gen estVal10_1000s = estVal10/1000
gen pi_1000s = pi/1000
gen hitot10_1000s = hitot10/1000
gen atotb10_1000s = atotb10/1000

qui tab raracem, gen(race)
lab var race1 "Race: white"
lab var race2 "Race: Black"
lab var race3 "Race: other"		

gen hchild_parents  = (hchild) if (hchild>0)
gen kid10mi_parents = (kid10mi) if (hchild>0)

* care measures conditional on receiving care
gen tohrswklyinnmnew_condgt0 = tohrswklyinnmnew if (tohrswklyinnmnew>0)
gen yghrswklynew_condgt0 = yghrswklynew if (yghrswklynew>0)
gen ygonly50new_ifany = ygonly50new if (anyhlprnew==1)

gen w4  = (w== 4)
gen w5  = (w== 5)
gen w6  = (w== 6)
gen w7  = (w== 7)
gen w8  = (w== 8)
gen w9  = (w== 9)
gen w10 = (w==10)
gen w11 = (w==11)

********************************************************************************
** Labels
********************************************************************************

lab var age      "Age"
lab var sex      "Female"

lab var race1    "Race: white"
lab var race2    "Race: Black"
lab var race3    "Race: other"
lab var rahispan "Hispanic"

lab var educ1    "Educ: less HS/GED"
lab var educ2    "Educ: high school or GED" 
lab var educ3    "Educ: some college"
lab var educ4    "Educ: college graduate"
lab var raedyrs  "Education (years)"

lab var hcpl     "Coupled" 
lab var everCpl  "Ever coupled"
         
lab var Ihchild  "Has children"   
lab var haschild "Has children"       
lab var hchild   "Number of children"
lab var hchild_parents "Number of children (parents)"

lab var kidres   "Any coresident child"
lab var kid10mi  "Any child within 10 miles"
lab var kid10mi_parents  "Any child within 10 miles (parents)"

lab var pi_1000s "Avg. household income (1000s)"
lab var hitot10  "Household income"
lab var hitot10_1000s "Household income (1000s)"
lab var atotb10_1000s "Household net worth (1000s)"

lab var anyBeq         "Non-zero estate"
lab var estVal10_1000s "Estate value (1000s)"
lab var homeBeq        "Home bequest"
lab var homeBeqOrIvtr  "Home bequest+ (includes IVT)"

lab var ownHist  "Own home"
lab var everOwn  "Ever own home"

lab var ivtrhome "Home inter-vivos transfer"	

lab var adla       "Number of ADL limitations"
lab var iadlza     "Number of IADL limitations" 
lab var rmemrye    "Ever had memory disease"

lab var tohrswklyinnmnew6ymean "Avg. total weekly LTC hours" 
lab var yghrswklynew6ymean     "Avg. weekly hours from children"
lab var rnhmliv6ymean          "Share of interviews in NH"

lab var anyhlprnew "Any LTC helper"
lab var tohrswklyinnmnew_condgt0 "Weekly LTC hours (if $>0$)"
lab var yghrswklynew_condgt0     "Weekly LTC hours from children (if $>0$)"
lab var ygonly50new_ifany      "$>50\%$ LTC from children (LTC recipients)"

lab var rnhmliv "Nursing home resident"
lab var everNH  "Ever live in NH"

lab var w4  "Interview 1998"
lab var w5  "Interview 2000"
lab var w6  "Interview 2002"
lab var w7  "Interview 2004"
lab var w8  "Interview 2006"
lab var w9  "Interview 2008"
lab var w10 "Interview 2010"
lab var w11 "Interview 2012"	

********************************************************************************
** TABLE 2 PANEL (B)
** TABLE 7 (DATA ROW, PERCENTILES)
********************************************************************************

**  For Table 2 Panel (A), see: CoreIWs_Analysis_Tables.do

preserve

replace estVal10 = estVal10 / 1000

est clear
eststo: estpost tabstat estVal10 [aw=crwtall], by(nochild) s(n mean p10 p25 p50 p75 p90 p95) ///
											   c(s)
local OPTS ///
         cells("count(lab(N) fmt(%9.0fc)) mean(lab(Mean) fmt(%9.0fc)) p10(fmt(%9.0fc)) p25(fmt(%9.0fc)) p50(fmt(%9.0fc)) p75(fmt(%9.0fc)) p90(fmt(%9.0fc)) p95(fmt(%9.0fc))") ///
		 coeflab(0 "Children" 1 "No Children" Total "\hline All") ///
         noobs nonum mlab(none) ///
		 collab(,lhs("(b) Estates")) ///
		 nofloat ///
		 substitute( \begin{tabular}{l*{1}{cccccccc}} ///
		             \begin{tabular*}{\textwidth}{@{\extracolsep{\fill}}lrrrrrrrr} ///
					 \end{tabular} \end{tabular*})		 
		 
esttab ., `OPTS'
esttab . using ${tabs}/TAB2b.tex, booktabs replace `OPTS'
cat ${tabs}/TAB2b.tex
	
restore	

/*
\begin{tabular*}{\textwidth}{@{\extracolsep{\fill}}lrrrrrrrr}
\toprule
(b) Estates &           N&        Mean&         p10&         p25&         p50&         p75&         p90&         p95\\
\midrule
Children    &       2,803&         230&           0&           0&          22&         198&         521&         806\\
No Children &         355&         205&           0&           0&          13&         198&         639&       1,043\\
\hline All  &       3,158&         226&           0&           0&          20&         198&         521&         834\\
\bottomrule
*/

********************************************************************************
** TABLE 7
** TABLE 8
********************************************************************************

preserve

keep if (nochild==0) // -> NOTE: parents

egen h = rowtotal(ahous10 ahoub10), m // value of primary PLUS secondary residences
replace h = h/1000                    // (NOT net of mortgages), 1000s

gen nw = (atotb10)/1000               // total wealth, 1000s
gen f  = nw - h         // total wealth excluding value of residences, 1000s
// -> note: housing now includes both primary and secondary residences
//          "f" is all residual wealth excluding "h". 
//          therefore, f + h = nw.

* define liquid and illiquid components of non-housing wealth
egen f_illiq = rowtotal( arles10 atran10 absns10 ) , m
replace f_illiq = f_illiq/1000
gen  f_liq = (f - f_illiq)

* check that categories all add up to total net worth
gen chknw = h + f_liq + f_illiq
gen  chkzero = nw - chknw
tabstat nw chknw chkzero, s(n mean min p25 p50 p75 p90 max) c(s)
drop chk*

* define liquid and illiquid components of net worth
gen nw_liq = f_liq
gen nw_illiq = nw - f_liq

* additional illiquid categories
gen tran = atran10/1000
gen bsns = absns10/1000
gen rles = arles10/1000

* check that illiquid categories add up to total:
gen chkilliq = h + tran + bsns + rles
gen chkzero = nw_illiq - chkilliq
tabstat nw_illiq chkilliq chkzero, s(n mean min p25 p50 p75 p90 max) c(s)
drop chk*

gen nonowner = 1-ownHist if !mi(ownHist)
gen e = (estVal10)/1000 // estate value, 1000s

* turn on log
cap log close
log using ${logs}/TAB7_AND_TAB8.txt, text replace

* Output for TABLE 8 (except non-negligible shares, see below)
est clear
qui: eststo: estpost summ h f f_liq f_illiq nw_liq nw_illiq [aw=rwtall] if (xnriw==1), det
esttab ., ///
	cells("p25(fmt(%4.0fc)) p50 p75 p90 p95") nonum mlab(none) noobs varwidth(24) compress ///
	coeflab(h  "Housing wealth (T-1)" ///
	        f  "Non-housing wealth (T-1)" ///
					f_liq " Of which: Liquid" ///
					f_illiq " Of which: Illiquid" ///
					nw_liq "Liquid wealth" ///
					nw_illiq "Illiquid wealth" ///
	        )

* TABLE 8:
/*
--------------------------------------------------------------------------
                               p25       p50       p75       p90       p95
--------------------------------------------------------------------------
Housing wealth (T-1)             0         0       104       230       365
Non-housing wealth (T-1)         0         5        87       352       643
 Of which: Liquid                0         2        58       239       493
 Of which: Illiquid              0         0         5        24       141
Liquid wealth                    0         2        58       239       493
Illiquid wealth                  0         9       136       300       521
--------------------------------------------------------------------------
*/					
					
					
* Output for TABLE 7 (except non-negligible shares, see below)		
est clear
qui: eststo: estpost summ e [aw=crwtall] if (xIW==1), det
esttab ., ///
	cells("p25(fmt(%4.0fc)) p50 p75 p90 p95") nonum mlab(none) noobs varwidth(24) compress ///
	coeflab(e "Estate value")					
	
* TABLE 7
/*
--------------------------------------------------------------------------
                               p25       p50       p75       p90       p95
--------------------------------------------------------------------------
Estate value                     0        22       198       521       806
--------------------------------------------------------------------------
*/			
	
	
* Non-negligible shares for TABLE 8:	

foreach var in h f f_liq f_illiq nw_liq nw_illiq {

  qui: gen tmp = (~(`var' <= 15)) if !mi(`var')
	qui: summ tmp [aw=rwtall] if (xnriw==1)

  di "Prob( `var' <= 15 ) = "  %3.2f r(mean)
	
	drop tmp
}	


/*
Prob( h <= 15 ) = 0.45            Table 8: Data, housing, non-negligible share
Prob( f <= 15 ) = 0.41            Table 8: Data, non-housing, non-negligible share
Prob( f_liq <= 15 ) = 0.37        Table 8: Data, liquid non-housing, non-negligible share
Prob( f_illiq <= 15 ) = 0.13      Table 8: Data, illiquid non-housing, non-negligible share
Prob( nw_liq <= 15 ) = 0.37       Table 8: Data, liquid net worth, non-negligible share
Prob( nw_illiq <= 15 ) = 0.48     Table 8: Data, illiquid net worth, non-negligible share
*/


* Non-negligible share for TABLE 7:

qui: gen tmp = (~(e <= 15)) if !mi(e)
qui: summ tmp [aw=crwtall] if (xIW==1)
	
di "Prob( estate <= 15 ) = " %3.2f r(mean)
	
drop tmp

/*
Prob( estate <= 15 ) = 0.51       TABLE 7: Data, bequest distribution, non-negligible share
*/

restore

* close log
cap log close

********************************************************************************
** APPENDIX TABLE A.4
********************************************************************************

preserve

** sample
keep if (xIW==1)

** variables
local X ///
  anyBeq estVal10_1000s homeBeq homeBeqOrIvtr ///
  tohrswklyinnmnew6ymean yghrswklynew6ymean ///
  rnhmliv6ymean ///
  age sex race1 race2 race3 rahispan raedyrs ///
  haschild hchild_parents ///
  pi_1000s ///
  everCpl everOwn ///
  w7 w8 w9 w10 w11
	
* note: religion and census division excluded for space
	
** label fixes
lab var age "Age at death"
	
** weight
local WEIGHT [aw=crwtall]	
	
** numbers of observations
qui: count
local N_ALL = `=r(N)'
qui: count if (nochild==0)
local N_PARENTS = `=r(N)'
qui: count if (fullWlthHist4==1)
local N_BALPAN = `=r(N)'

** stats
	
est clear
* All:
eststo: estpost tabstat `X' `WEIGHT', c(s) s(mean semean)
estadd scalar N = `N_ALL', replace	
* Parents:
eststo: estpost tabstat `X' `WEIGHT' if (nochild==0), c(s) s(mean semean)
estadd scalar N = `N_PARENTS', replace	
* Balanced panel:
eststo: estpost tabstat `X' `WEIGHT' if (fullWlthHist4==1), c(s) s(mean semean)
estadd scalar N = `N_BALPAN', replace	

** output:

local OPTS ///
  main(mean a2) aux(semean a1) order(`X') ///
	wide nogaps nostar label varwidth(24) compress unstack nonum noobs nonote nofloat ///
	collab("Mean" "SE", ///
	  lhs("& \multicolumn{2}{c}{All Decedents} & \multicolumn{2}{c}{Parents} & \multicolumn{2}{c}{Balanced Panel}  \\ \cline{2-3} \cline{4-5} \cline{6-7}"  ) ) ///
	mlab(none) ///		
	stats(N, label("Observations") fmt(%9.0fc)) ///
	substitute(\begin{tabular}{l*{3}{cc}} ///
	           \begin{tabular*}{\textwidth}{@{\extracolsep{\fill}}lcccccc} ///
						 \end{tabular} ///
						 \end{tabular*} )
	
local FILENAME ${tabs}/TABA4.tex	

esttab *, `OPTS'
esttab * using `FILENAME', `OPTS' replace booktabs
cat `FILENAME'

restore

********************************************************************************
** APPENDIX TABLE A.5
********************************************************************************

preserve

** sample
keep if (xIW==1 | riwstat==1)

** variables
local X ///
  age sex race1 race2 race3 rahispan raedyrs hcpl ///
  haschild hchild_parents kid10mi_parents ///
  atotb10_1000s hitot10_1000s ownHist ///
	adla iadlza rmemrye ///
	rnhmliv ///
	anyhlprnew ///
	tohrswklyinnmnew_condgt0 ///  
	yghrswklynew_condgt0 ///
	ygonly50new_ifany ///	
  w4 w5 w6 w7 w8 w9 w10 w11
	
* note: religion and census division excluded for space
	
* variables in exit interviews only
local X_EXIT_ONLY ///	
	
** variables in core interviews only
local X_CORE_ONLY	///
  atotb10_1000s hitot10_1000s kid10mi_parents
	
** refine variable lists
local X_CORE : list X - X_EXIT_ONLY
local X_EXIT : list X - X_CORE_ONLY

	
** weight
local WEIGHT [aw=crwtall]	
	
** numbers of observations
qui: count if (riwstat==1)  // all, core
local N_ALL_CORE = `=r(N)'
qui: count if (xIW==1)      // all, exit
local N_ALL_EXIT = `=r(N)'
qui: count if (nochild==0) & (riwstat==1 & hcpl==0) // parents, core, single
local N_PARENTS_CORE = `=r(N)'
qui: count if (nochild==0) & (xIW==1)               // parents, exit
local N_PARENTS_EXIT = `=r(N)'
qui: count if (fullWlthHist4==1) & (riwstat==1) // balanced panel, core
local N_BALPAN_CORE = `=r(N)'
qui: count if (fullWlthHist4==1) & (xIW==1)     // balanced panel, exit
local N_BALPAN_EXIT = `=r(N)'

** stats
	
est clear
* all, core
eststo: estpost tabstat `X_CORE' `WEIGHT' if (riwstat==1), c(s) s(mean semean)
estadd scalar N = `N_ALL_CORE', replace	
* all, exit
eststo: estpost tabstat `X_EXIT' `WEIGHT' if (xIW==1), c(s) s(mean semean)
estadd scalar N = `N_ALL_EXIT', replace	
* parents, core, single
eststo: estpost tabstat `X_CORE' `WEIGHT' if (nochild==0 & riwstat==1 & hcpl==0), c(s) s(mean semean)
estadd scalar N = `N_PARENTS_CORE', replace	
* parents, exit
eststo: estpost tabstat `X_EXIT' `WEIGHT' if (nochild==0 & xIW==1), c(s) s(mean semean)
estadd scalar N = `N_PARENTS_EXIT', replace	
* Balanced panel, core
eststo: estpost tabstat `X_CORE' `WEIGHT' if (fullWlthHist4==1 & riwstat==1), c(s) s(mean semean)
estadd scalar N = `N_BALPAN_CORE', replace	
* Balanced panel, exit
eststo: estpost tabstat `X_EXIT' `WEIGHT' if (fullWlthHist4==1 & xIW==1), c(s) s(mean semean)
estadd scalar N = `N_BALPAN_EXIT', replace	

** output:

local OPTS ///
  main(mean a2) aux(semean a1) order(`X') ///
	wide nogaps nostar label varwidth(24) compress unstack nonum noobs nonote nofloat ///
	collab("Mean" "SE", ///
	  lhs("& \multicolumn{4}{c}{All Decedents} & \multicolumn{4}{c}{Parents} & \multicolumn{4}{c}{Balanced Panel}  \\ \cline{2-5} \cline{6-9} \cline{10-13} & \multicolumn{2}{c}{Core IWs} & \multicolumn{2}{c}{Exit IWs} & \multicolumn{2}{c}{Core IWs} & \multicolumn{2}{c}{Exit IWs} & \multicolumn{2}{c}{Core IWs} & \multicolumn{2}{c}{Exit IWs} \\ \cline{2-3} \cline{4-5} \cline{6-7} \cline{8-9} \cline{10-11} \cline{12-13}"  ) ) ///
	mlab(none) ///		
	stats(N, label("Observations") fmt(%9.0fc)) ///
	substitute(\begin{tabular}{l*{6}{cc}} ///
	           \begin{tabular*}{\textwidth}{@{\extracolsep{\fill}}lcccccccccccc} ///
						 \end{tabular} ///
						 \end{tabular*} )
	
local FILENAME ${tabs}/TABA5.tex	

esttab *, `OPTS'
esttab * using `FILENAME', `OPTS' replace booktabs
cat `FILENAME'

restore

********************************************************************************
