/*******************************************************************************

************
** Function: 
************

  Makes tables from core data.

************
** Inputs  :
************
	
	- core_sample.dta	(see CoreIWs_GetSample.do)
	
************
** Outputs :
************
		
	- TAB1_and_TAB5a.tex
	- TAB2a.tex
	- TABA3.tex
  
	- TAB4c.txt
	- TAB4a.txt
	- FIG9_DATA_LEFT_PANEL.txt
	- FIG9_DATA_RIGHT_PANELS.txt

*******************************************************************************/

clear *
macro drop _all
set more off 

********************************************************************************
** Directory names
********************************************************************************

do GetDirnames.do

********************************************************************************
** DATA
********************************************************************************

use ${save}/core_sample, clear

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

********************************************************************************
** Additional variables
********************************************************************************

* variables for descriptive statistics tables

gen childless = (hchild==0) if !mi(hchild)

gen children = 1 - childless

gen female = (ragender==2) if !mi(ragender)

gen raracem2 = (raracem==2) if !mi(raracem)
gen raracem3 = (raracem==3) if !mi(raracem)

gen educ1 = (raeduc==1) if !mi(raeduc)
gen educ2 = (raeduc==2 | raeduc==3) if !mi(raeduc)
gen educ3 = (raeduc==4) if !mi(raeduc)
gen educ4 = (raeduc==5) if !mi(raeduc)
lab var educ1 "Education: less than high school" 
lab var educ2 "Education: high school or GED" 
lab var educ3 "Education: some college"
lab var educ4 "Education: college and above"

gen hchild_if_parent = hchild if (hchild>0)

gen child10mi = (hlv10mikn>0) if !mi(hlv10mikn)
replace child10mi = 1 if (hresdkn>0 & hresdkn<.)
replace child10mi = 0 if (children==0)

gen child10mi_if_parent = child10mi if (hchild>0)

gen cogfunction2 = (cogfunction==2) if !mi(cogfunction)
gen cogfunction3 = (cogfunction==3) if !mi(cogfunction)

gen hatotb10_1e3 = atotb10/1e3
gen hitot10_1e3 = hitot10/1e3

xtile hatotb10q5 = atotb10 [pw=rwtcrnh], nq(5)

gen rnrsnit_condgt0 = rnrsnit if (rnrsnit>0)

gen yganyhlprnew_ifany = yganyhlprnew if (anyhlprnew==1)
gen yghrswklynew_condgt0 = yghrswklynew if (yghrswklynew>0)

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

********************************************************************************
** TABLE 1 AND TABLE 5a
********************************************************************************

**  Net worth distribution, all and owners vs. renters

est clear
eststo: estpost tabstat atotb10_1000s [aw=rwtall] ///
	if (hpickhh==1 & inrange(ageEldest,65,69)) ///
	, by(rownhm) ///
	  s(n mean p10 p25 p50 p75 p90 p95) c(s)
	
	
local OPTS ///
         cells("count(lab(N) fmt(%9.0fc)) mean(lab(Mean) fmt(%9.0fc)) p10(fmt(%9.0fc)) p25(fmt(%9.0fc)) p50(fmt(%9.0fc)) p75(fmt(%9.0fc)) p90(fmt(%9.0fc)) p95(fmt(%9.0fc))") ///
		 coeflab(0 "Renters" 1 "Owners" Total "\hline All") ///
         noobs nonum mlab(none) ///
		 nofloat ///
		 substitute( \begin{tabular}{l*{1}{cccccccc}} ///
		             \begin{tabular*}{\textwidth}{@{\extracolsep{\fill}}lrrrrrrrr} ///
					 \end{tabular} \end{tabular*})		 
		 
esttab ., `OPTS' compress
esttab . using ${tabs}/TAB1_and_TAB5a.tex, booktabs replace `OPTS'
cat ${tabs}/TAB1_and_TAB5a.tex

/*
------------------------------------------------------------------------------------------
                   N      Mean       p10       p25       p50       p75       p90       p95
------------------------------------------------------------------------------------------
Renters        3,031        92        -1         0         3        25       156       314
Owners        11,678       663        47       113       291       684     1,394     2,183
\hline All    14,709       551         2        52       204       560     1,206     1,915
------------------------------------------------------------------------------------------
*/

********************************************************************************
** TABLE 2a
********************************************************************************

**  Net worth distribution, all and parents vs. childless

**  For PANEL B, see: Decedents_Analysis_Tables.do

est clear
eststo: estpost tabstat atotb10_1000s [aw=rwtall] ///
	if (hpickhh==1 & inrange(ageEldest,65,69)) ///
	, by(nochild) ///
	  s(n mean p10 p25 p50 p75 p90 p95) c(s)
	
local OPTS ///
         cells("count(lab(N) fmt(%9.0fc)) mean(lab(Mean) fmt(%9.0fc)) p10(fmt(%9.0fc)) p25(fmt(%9.0fc)) p50(fmt(%9.0fc)) p75(fmt(%9.0fc)) p90(fmt(%9.0fc)) p95(fmt(%9.0fc))") ///
		 coeflab(0 "Children" 1 "No Children" Total "\hline All") ///
         noobs nonum mlab(none) ///
		 collab(,lhs("(a) Net worth")) ///
		 nofloat ///
		 substitute( \begin{tabular}{l*{1}{cccccccc}} ///
		             \begin{tabular*}{\textwidth}{@{\extracolsep{\fill}}lrrrrrrrr} ///
					 \end{tabular} \end{tabular*})		 
		 
esttab ., `OPTS' compress
esttab . using ${tabs}/TAB2a.tex, booktabs replace `OPTS'
cat ${tabs}/TAB2a.tex

/*
------------------------------------------------------------------------------------------
                   N      Mean       p10       p25       p50       p75       p90       p95
------------------------------------------------------------------------------------------
Children      13,568       558         2        54       206       553     1,229     1,966
No Child~n     1,008       501         0        34       206       651     1,102     1,685
\hline All    14,576       553         2        53       206       560     1,212     1,919
------------------------------------------------------------------------------------------
*/

/*
The difference between these numbers and the calculations below that show a median of 203 is
the conditioning on nochild. There are 269 obs with missing nochild but non-missing wealth.
Excluding these, the median is 206 (as it is here). Including them, the median is 203 (as
it is below). 
*/

********************************************************************************
** TABLE 4a
********************************************************************************

** Distribution of housing / non-housing wealth at ages 65-69

preserve

keep if (children==1)                 // parents
keep if (inrange(ageEldest,65,69))    // ages 65-69
keep if (hpickhh==1)                  // one member per HH

gen a = (atotb10/1e3)                 // all wealth, 1000s
egen h = rowtotal(ahous10 ahoub10), m // value of primary PLUS secondary residences
replace h = h/1e3                     // (NOT net of mortgages), 1000s
gen f = a - h                         // value of wealth excluding value of housing, 1000s

est clear
eststo: estpost tabstat h f a [aw=rwtall], s(p10 p25 p50 p75 p90 p95) c(s)
	
local OPTS ///
     cells("p10(fmt(%9.0fc)) p25(fmt(%9.0fc)) p50(fmt(%9.0fc)) p75(fmt(%9.0fc)) p90(fmt(%9.0fc)) p95(fmt(%9.0fc))") ///
		 coeflab(h "Housing" f "Non-housing" a "\hline Net worth" ) ///
     noobs nonum mlab(none) ///
		 nofloat ///
		 substitute( \begin{tabular}{l*{1}{cccccc}} ///
		             \begin{tabular*}{\textwidth}{@{\extracolsep{\fill}}lrrrrrr} ///
					       \end{tabular} \end{tabular*})		 
		
local FILENAME ${tabs}/junk.tex
		
* turn on log
cap log close
log using ${logs}/TAB4a.txt, text replace		
		
esttab ., `OPTS' compress
esttab using `FILENAME', booktabs replace `OPTS'
cat `FILENAME'
rm `FILENAME'

/*
\begin{tabular*}{\textwidth}{@{\extracolsep{\fill}}lrrrrrr}
\toprule
            &         p10&         p25&         p50&         p75&         p90&         p95\\
\midrule
Housing     &           0&          46&         134&         261&         501&         727\\
Non-housing &         -40&           0&          43&         274&         808&       1,355\\
\hline Net worth&           2&          54&         206&         553&       1,229&       1,966\\
\bottomrule
\end{tabular*}
*/

* close log
cap log close

restore

********************************************************************************
** FIGURE 9 // LEFT PANEL
********************************************************************************

** Home ownership rates with age

* turn on log
cap log close
log using ${logs}/FIG9_DATA_LEFT_PANEL.txt, text replace

tabstat rownhm [aw=rwtall] if (hpickhh==1 & nochild==0), c(s) s(mean) format(%4.3f) by(ageEldestCat) nototal

/*

ageEldestCat |      mean
-------------+----------
       50-54 |     0.765
       55-59 |     0.799
       60-64 |     0.808
       65-69 |     0.813
       70-74 |     0.799
       75-79 |     0.782
       80-84 |     0.713
       85-89 |     0.622
       90-94 |     0.497
------------------------
*/

* close log
cap log close

********************************************************************************
** FIGURE 9 // RIGHT PANELS
********************************************************************************

** Home ownership rates ages 65+ by wealth quintiles X age groups

preserve

keep if (children==1)                 // parents
keep if (ageEldest>=65 & ageEldest<.) // ages 65+
keep if (hpickhh==1)                  // one member per HH

* define wealth quintiles for this sub-sample
* define separate quintiles for each age category
levelsof ageEldestCat, local(agebins)
gen q = .
foreach bin of local agebins {
  xtile tmp = atotb10 [aw=rwtall] if (ageEldestCat==`bin'), nq(5)
  replace q = tmp if (ageEldestCat==`bin')
	drop tmp
}

* check quintiles
tab ageEldestCat q
tab ageEldestCat q [aw=rwtall], row nof

* turn on log
cap log close
log using ${logs}/FIG9_DATA_RIGHT_PANELS.txt, text replace

* result:
table ageEldestCat q [aw=rwtall], contents(mean rownhm) format(%4.3f)

/*
---------------------------------------------
ageEldest |                 q                
Cat       |     1      2      3      4      5
----------+----------------------------------
    65-69 | 0.269  0.895  0.949  0.970  0.980
    70-74 | 0.240  0.875  0.949  0.960  0.969
    75-79 | 0.215  0.846  0.931  0.952  0.963
    80-84 | 0.112  0.725  0.873  0.921  0.930
    85-89 | 0.045  0.555  0.786  0.852  0.870
    90-94 | 0.013  0.282  0.682  0.775  0.724
---------------------------------------------
*/

* close log
cap log close

restore

********************************************************************************
** TABLE 4 // PANEL (c)
********************************************************************************

** Share of wealth in housing, among owners
	
* NOTE: housing is value of primary + secondary residences, (NOT net of mortgage).
	
preserve

keep if (ageEldestCat>=65 & ageEldestCat<.) // eldest hh member is age 65+
keep if (hchild>0 & hchild<.)               // parents
keep if (hpickhh==1)                        // one member per hh (no double counting)
keep if (rownhm==1)                         // owners

gen a = (atotb10/1e3)                 // all wealth, 1000s
egen h = rowtotal(ahous10 ahoub10), m // value of primary PLUS secondary residences
replace h = h/1e3                     // (NOT net of mortgages), 1000s
gen f = a - h                         // value of wealth excluding value of housing, 1000s

* define ratio of h/a
gen h_over_a = 100 * (h/a) if (a>0)
// note: restricted to a>0
		
* what share of cases are a<=0?
count if (ageEldestCat>=65 & ageEldestCat<.) & (hchild>0 & hchild<.) & (hpickhh==1) & (rownhm==1) & (a<.)
local d = `=r(N)'
count if (ageEldestCat>=65 & ageEldestCat<.) & (hchild>0 & hchild<.) & (hpickhh==1) & (rownhm==1) & (a<=0)
local n = `=r(N)'
di "Percentage with a<=0 among 65+ owner households = 100*(`n')/(`d') = `=100*`n'/`d''"

/*
Percentage with a<=0 among 65+ owner households = 100*(395)/(39174) = 1.008321846122428
-> approx 1 percentage has non-positive assets.
*/
	
* deal with a<=0 cases by replacing h/a with 0 if h=0 and with max value if h>0	
summ h_over_a
replace h_over_a = 0         if (mi(h_over_a) & rownhm==1 & h==0 & a<=0)
replace h_over_a = `=r(max)' if (mi(h_over_a) & rownhm==1 & h >0 & a<=0)

* top-code at 100%
replace h_over_a = 100 if (h_over_a > 100) & !mi(h_over_a)

* turn on log
cap log close
log using ${logs}/TAB4c.txt, text replace

tabstat h_over_a [aw=rwtcrnh] ///
  if (ageEldestCat>=65 & ageEldestCat<=69) & (hchild>0 & hchild<.) & (hpickhh==1) & (rownhm==1) ///
  , s(p10 p25 p50 p75 p90 p95) f(%9.0fc)

/*
    variable |       p10       p25       p50       p75       p90       p95
-------------+------------------------------------------------------------
    h_over_a |        21        38        65        99       100       100
--------------------------------------------------------------------------
*/

* close log
cap log close

restore			
			
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

********************************************************************************
** DESCRIPTIVE STATISTICS TABLES FOR APPENDICES:
********************************************************************************

********************************************************************************
** Labels:
********************************************************************************

lab var female "Female"
lab var raracem2 "Race: Black/African Amer."
lab var raracem3 "Race: Other"
lab var rahispan "Ethnicity: Hispanic"
lab var hcpl "Coupled"
lab var children "Children"
lab var hchild_if_parent "Number of children (parents)"
lab var child10mi_if_parent "Child w/in 10 miles (parents)"
lab var ragey_b "Age"
lab var radla "Number of ADL limitations"
lab var riadlza "Number of IADL limitations"
lab var cogfunction2 "Cognitive function: Impaired"
lab var cogfunction3 "Cognitive function: Demented"
lab var hitot10_1e3 "Household income (1000s)"
lab var hatotb10_1e3 "Net worth (1000s)"
lab var rownhm "Homeowner"
lab var rnhmliv "Nursing home resident"
lab var rnrshom "Nursing home stay last 2 years"
lab var rnrsnit_condgt0 "Nursing home nights (if $>0$)"
lab var disablednew "Disabled (21+ hours/week LTC)"
lab var anyhlprnew "Any LTC helper"
lab var yganyhlprnew_ifany "Any child LTC helper (LTC recipients)"
lab var yghrswklynew_condgt0 "Child weekly LTC hours (if $>0$)"

********************************************************************************
** Complete summary statistics: All | Parents | Single/NH-Eligible
********************************************************************************

local VARS ///
  female raracem2 raracem3 rahispan educ1 educ2 educ3 educ4 hcpl ///
	children hchild_if_parent child10mi_if_parent ///
	ragey_b radla riadlza cogfunction2 cogfunction3 ///
	hitot10_1e3 hatotb10_1e3 rownhm ///
	rnhmliv rnrshom rnrsnit_condgt0 ///
	disablednew anyhlprnew yganyhlprnew_ifany yghrswklynew_condgt0

global IF ///
  (ragey_b>=65 & ragey_b < .)
	
** numbers of unique individuals
unique hhidpn if (${IF})
local N_ALL = `=r(N)'
local U_ALL = `=r(unique)'
unique hhidpn if (${IF} & children==1)
local N_PARENTS = `=r(N)'
local U_PARENTS = `=r(unique)'
/*
unique hhidpn if (${IF} & children==0)
local N_CHILDLESS = `=r(N)'
local U_CHILDLESS = `=r(unique)'
*/
unique hhidpn if (${IF} & ((radla>=2 & radla<.) | inlist(cogfunction,2,3)) & (hcpl==0))
local N_SINGLES = `=r(N)'
local U_SINGLES = `=r(unique)'

** numbers of unique households
unique hhid if (${IF})
local H_ALL = `=r(unique)'
unique hhid if (${IF} & children==1)
local H_PARENTS = `=r(unique)'
/*
unique hhid if (${IF} & children==0)
local H_CHILDLESS = `=r(unique)'
*/
unique hhid if (${IF} & ((radla>=2 & radla<.) | inlist(cogfunction,2,3)) & (hcpl==0))
local H_SINGLES = `=r(unique)'

est clear
eststo: estpost tabstat `VARS' if (${IF}) [aw=rwtcrnh], c(s) s(mean semean)
estadd scalar N = `N_ALL', replace
estadd scalar U = `U_ALL', replace
estadd scalar H = `H_ALL', replace
eststo: estpost tabstat `VARS' if (${IF} & children==1) [aw=rwtcrnh], c(s) s(mean semean)
estadd scalar N = `N_PARENTS', replace
estadd scalar U = `U_PARENTS', replace
estadd scalar H = `H_PARENTS', replace
/*
eststo: estpost tabstat `VARS' if (${IF} & children==0) [aw=rwtcrnh], c(s) s(mean semean)
estadd scalar N = `N_CHILDLESS', replace	
estadd scalar U = `U_CHILDLESS', replace	
estadd scalar H = `H_CHILDLESS', replace	
*/
eststo: estpost tabstat `VARS' if (${IF} & ((radla>=2 & radla<.) | inlist(cogfunction,2,3)) & (hcpl==0)) [aw=rwtcrnh], c(s) s(mean semean)
estadd scalar N = `N_SINGLES', replace
estadd scalar U = `U_SINGLES', replace
estadd scalar H = `H_SINGLES', replace

** Add median income and wealth
qui summ hitot10 if (${IF}) [aw=rwtcrnh], det
local HITOT_P50_ALL = `=round(r(p50)/1e3)'
qui summ hitot10 if (${IF} & children==1) [aw=rwtcrnh], det
local HITOT_P50_PARENTS = `=round(r(p50)/1e3)'
/*
qui summ hitot10 if (${IF} & children==0) [aw=rwtcrnh], det
local HITOT_P50_CHILDLESS = `=round(r(p50)/1e3)'
*/
qui summ hitot10 if (${IF} & ((radla>=2 & radla<.) | inlist(cogfunction,2,3)) & (hcpl==0)) [aw=rwtcrnh], det
local HITOT_P50_SINGLES = `=round(r(p50)/1e3)'

qui summ atotb10 if (${IF}) [aw=rwtcrnh], det
local HATOTB_P50_ALL = `=round(r(p50)/1e3)'
qui summ atotb10 if (${IF} & children==1) [aw=rwtcrnh], det
local HATOTB_P50_PARENTS = `=round(r(p50)/1e3)'
/*
qui summ atotb10 if (${IF} & children==0) [aw=rwtcrnh], det
local HATOTB_P50_CHILDLESS = `=round(r(p50)/1e3)'
*/
qui summ atotb10 if (${IF} & ((radla>=2 & radla<.) | inlist(cogfunction,2,3)) & (hcpl==0)) [aw=rwtcrnh], det
local HATOTB_P50_SINGLES = `=round(r(p50)/1e3)'

** Output
local OPTS ///
  main(mean a2) aux(semean a2) order(`VARS') ///
	wide nogaps nostar label varwidth(24) compress nonum noobs nonote ///
	collab("Mean" "SE", ///
	  lhs("& \multicolumn{2}{c}{All} & \multicolumn{2}{c}{Parents} & \multicolumn{2}{c}{NH-eligible Singles} \\ \cline{2-3} \cline{4-5} \cline{6-7} "  ) ) ///
	mlab(none) ///	
	stats(N U H, label("Observations" "Unique individuals" "Unique households") ///
	       fmt(%9.0fc %9.0fc %9.0fc) ) ///
	nofloat ///
	refcat(hatotb10_1e3 "{[}Median{]} & {[}`HITOT_P50_ALL'{]} & & {[}`HITOT_P50_PARENTS'{]} & & {[}`HITOT_P50_SINGLES'{]} &  \\ % "	///
	       rownhm "{[}Median{]} & {[}`HATOTB_P50_ALL'{]} & & {[}`HATOTB_P50_PARENTS'{]} & & {[}`HATOTB_P50_SINGLES'{]}  & \\ % " ///
	       , nolabel) ///
	substitute(\begin{tabular}{l*{3}{cc}} ///
	           \begin{tabular*}{\textwidth}{@{\extracolsep{\fill}}lcccccc} ///
						 \end{tabular} ///
						 \end{tabular*} )
	
/*
    lhs("& \multicolumn{2}{c}{All} & \multicolumn{2}{c}{Parents} & \multicolumn{2}{c}{Childless} & \multicolumn{2}{c}{NH-eligible Singles} \\ \cline{2-3} \cline{4-5} \cline{6-7} \cline{8-9} "  ) ) ///

	refcat(hatotb10_1e3 "{[}Median{]} & {[}`HITOT_P50_ALL'{]} & & {[}`HITOT_P50_PARENTS'{]} & & {[}`HITOT_P50_CHILDLESS'{]} & & {[}`HITOT_P50_SINGLES'{]} &  \\ % "	///
	       rownhm "{[}Median{]} & {[}`HATOTB_P50_ALL'{]} & & {[}`HATOTB_P50_PARENTS'{]} & & {[}`HATOTB_P50_CHILDLESS'{]} & & {[}`HATOTB_P50_SINGLES'{]}  & \\ % " ///
	       , nolabel) ///		
*/	
	
local FILENAME ${tabs}/TABA3.tex	

esttab *, `OPTS'
esttab * using `FILENAME', `OPTS' replace booktabs
cat `FILENAME'

********************************************************************************
