/*******************************************************************************

************
** Function: 
************

  Produces figure on nursing home entry and housing asset liquidation as well as
	statistics cited in the text.

************
** Inputs  :
************
	
	- core_sample.dta	(see CoreIWs_GetSample.do)
	
************
** Outputs :
************
	
	- HOME_LIQUIDATION_STATISTICS.txt
  - FIG3.eps

*******************************************************************************/

clear *
macro drop _all
set more off 

********************************************************************************
** Directory names
********************************************************************************

do GetDirNames.do

********************************************************************************
** DATA
********************************************************************************

use ${save}/core_sample, clear

********************************************************************************
***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   **
   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***  
***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   **   
********************************************************************************

********************************************************************************
** Additional variables
********************************************************************************

* home liquidation
gen homeliq = 1 - rownhm if (L.rownhm==1)

** nursing home entry

sort hhidpn w
gen enterNH = (L1.rnhmliv==0 & rnhmliv==1) if !mi(L1.rnhmliv,rnhmliv)
// -> R enters nursing home

sort hhhid w hhidpn
by hhhid w: egen hhMemberEnterNH = max(enterNH)
// -> any HH member enters a nursing home

drop enterNH
// -> to avoid mistakes

sort hhidpn w

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

********************************************************************************
** Hazard of home liquidation
********************************************************************************
					
** = Pr( don't own at t | owned at t-1 )
					
********************************************************************************
** Statistics for the text
********************************************************************************

* turn on log
cap log close
log using ${logs}/HOME_LIQUIDATION_STATISTICS.txt, text replace		

* overall share of owning households who liquidate 
tab homeliq [aw=rwtcrnh] if (ageEldest>=65 & ageEldest<. & hpickhh==1)
/*
    homeliq |      Freq.     Percent        Cum.
------------+-----------------------------------
          0 | 34,727.775       94.15       94.15
          1 | 2,158.2248        5.85      100.00
------------+-----------------------------------
      Total |     36,886      100.00
*/
// -> between adjacent core interview waves, among households with a member age 65+,
//    average unconditional liquidation rate is 5.85%.

* share of liquidating households in which a member moves into a nursing home
tab hhMemberEnterNH [aw=rwtcrnh] if (ageEldest>=65 & !mi(ageEldest)) & hpickhh==1 & homeliq==1
/*
hhMemberEnt |
       erNH |      Freq.     Percent        Cum.
------------+-----------------------------------
          0 |  1,875.674       86.00       86.00
          1 | 305.325954       14.00      100.00
------------+-----------------------------------
      Total |      2,181      100.00
*/

// 14% of all liquidations observed are coincident with a member moving into a nursing home.


* probability of liquidation among owners who move into a nursing home
tabstat homeliq [aw=rwtcrnh] if (ragey_b>=65 & ragey_b<.) & (L.rnhmliv==0 & rnhmliv==1), c(s) s(n mean)
/*
    variable |         N      mean
-------------+--------------------
     homeliq |       720  .4321021
----------------------------------
*/
// hazard of liquidation at nursing home entry is 43%.

* close log
cap log close

********************************************************************************
** FIGURE 3
********************************************************************************

preserve

* find nh entry events for individuals 65+
gen enternh = (ragey_b>=65 & ragey_b<. & L.rnhmliv==0 & rnhmliv==1) if !mi(L.rnhmliv,rnhmliv)
tab enternh,m

* count for each individual
by hhidpn: egen numentries = total(enternh),m

* identify ever-owners (only these matter for home liquidations)
by hhidpn: egen everowner = max(rownhm)

* how many nh entry events do we have? how many individuals have multiple events?
tab numentries if w==4, m
tab numentries if w==4
tab numentries if everowner==1 & w==4
// -> 967 individuals.
//    955 have 1 move.
//     11 have 2 moves.
//      1 has 3 moves.

* keep ever-owners only
keep if (everowner==1)

* keep only those with nh entries
keep if (numentries>0 & numentries<.)
tab w
// -> 967 individuals.

* identify each nh entry with its number, in order
by hhidpn: gen whichentry = sum(enternh) if (enternh==1)
tab whichentry
// -> 980 moves. 
//    Of those: 967 are 1st moves, 12 are 2nd moves, 1 is a 3rd move.

* for those with multiple entries, replicate data, each entry is its own person effectively
expand numentries

* re-create unique identifier 
sort hhidpn w
by hhidpn w: gen dup = _n
tab dup
* sort, check whether unique
sort hhidpn dup w
isid hhidpn dup w
* new panel setup
egen newhhidpn = group(hhidpn dup)
sort newhhidpn w
xtset newhhidpn w

* count how many nh entries we have
tab w
// -> 980 nursing home moves (among those who we observe to be owners in sample)

* create entry identifier, where multiple entries select the correct copy of the original obs.
gen entry = enternh
replace entry = 0 if (enternh==1 & whichentry!=dup)
tab enternh
tab entry
// -> entry count should align with count of nh entries above

* identify timeline relative to entry
* (-7,-6,...-1,  0,   1,2,...,7   )
*   pre-entry  entry   post-entry             

gen timeline = .
replace timeline = 0 if (entry==1)
forvalues n = 1/7 {
  replace timeline = -`n' if (F`n'.entry==1)
	replace timeline =  `n' if (L`n'.entry==1)
}

tab timeline

* result:
* (singles and couples)
tabstat homeliq [aw=rwtcrnh], by(timeline) c(s) s(n mean semean)
/*
timeline |         N      mean  se(mean)
---------+------------------------------
      -6 |         0         .         .
      -5 |       170   .035765  .0142849
      -4 |       321  .0592735  .0132004
      -3 |       427  .0730189  .0126052
      -2 |       556  .1013942  .0128128
      -1 |       666  .1629118  .0143203
       0 |       720  .4321021  .0184741
       1 |       161  .3353208   .037323
       2 |        55  .3523352  .0650064
       3 |        18  .1838178  .0939427
       4 |         9  .1914659  .1391073
       5 |         2         0         0
---------+------------------------------
   Total |      3105  .1954592  .0071177
----------------------------------------
*/

collapse (mean) y = homeliq ///
        (count) n = homeliq ///
			(semean) se = homeliq [aw=rwtcrnh], by(timeline)
list, sep(0)
* use only cells with 25+ observations
drop if (n<25)

format y %3.2f

* graph with 95% CI's added (approx +/- 2 standard errors)
gen y_plus_2se = y + 2*se
gen y_mins_2se = y - 2*se		
		
graph twoway ///
  (sc y timeline, c(d) lp(dash) lc(blue) mc(blue) ms(Oh) mlabel(y) mlabpos(1)) ///
	(line y_plus_2se y_mins_2se timeline, lp(dash_dot dash_dot) lc(red red)) ///
	, xti("Interviews relative to move into nursing home (=0)") ///
	  yti("Housing liquidation hazard") ///
		legend(order(1 "Hazard" 2 "95% CI")) ///
		xlabel(-5(1)2) ylabel(,angle(0))	
		
graph2tex, epsfile(${figs}/FIG3)	

restore

********* check results:

* for comparison: construct same figures as in table directly above

* time=-1
tabstat homeliq [aw=rwtcrnh] if (F1.ragey_b>=65 & F1.ragey_b<.) & (rnhmliv==0 & F1.rnhmliv==1), c(s) s(n mean)

* time=0
tabstat homeliq [aw=rwtcrnh] if (ragey_b>=65 & ragey_b<.) & (L.rnhmliv==0 & rnhmliv==1), c(s) s(n mean)

* time=+1
tabstat homeliq [aw=rwtcrnh] if (L1.ragey_b>=65 & L1.ragey_b<.) & (L2.rnhmliv==0 & L1.rnhmliv==1), c(s) s(n mean)

// -> result match. 

********************************************************************************
