/*******************************************************************************

************
** Function: 
************

  Computes calibration targets for the model. 

************
** Inputs  :
************
	
	- core_sample.dta	(see CoreIWs_GetSample.do)
	
************
** Outputs :
************
	
  - CALIBRATION_TARGETS_AND_TAB6.txt

*******************************************************************************/

clear *
macro drop _all
set more off 

********************************************************************************
** Directory names
********************************************************************************

do GetDirNames.do

********************************************************************************
** DATA
********************************************************************************

use ${save}/core_sample, clear

********************************************************************************
***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   **
   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***  
***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   ***   **   
********************************************************************************

********************************************************************************
** Targets
********************************************************************************

// NOTES: 

//    *) Targets are computed using core data for waves 4-10 (1998-2010).

//    *) Transfers are conditional on health status, which is defined using care hours.
//       Disability means receiving 21+ hours/week of LTC.
//       Hours are imputed (predictive mean nearest neighbor match) for those receiving NH care.

//    *) We use individual weights. There are no household weights corrected for NH residents.

//    *) In order not to double-count couples, we use RAND variable h*pickhh==1 to pick
//       one member from each sub-household. This variable selects the FINR in couples 
//       if available.

//    *) When applying age criteria, we include households in which the eldest member
//       is age eligible.

//    *) All dollar amounts are in 2010 dollars.

//    *) Transfers are two-year amounts.

//    *) When calculating unconditional transfer means, we trim the top 5% of the wealth
//       distribution using the 95th percentile of wealth among HRS households in the core
//       1998-2010 data.
//       This figure can be calculated using: summ atotb10 [aw=rwtall] if (hpickhh==1), det
//       The result is 1,684,282, which we round to 1.684e6.

//    *) We include parents only.  Parents are defined using (hchild>0 & hchild<.).
//       Results are similar with hnkid (from family file). We use hchild to be consistent
//       with our other data work (e.g., regressions).

//    *) Asterisks (***) indicate which targets we are using.

****************************************

* turn on log
cap log close
log using ${logs}/CALIBRATION_TARGETS_AND_TAB6.txt, text replace

****************************************
** Home ownership, ages 65+
****************************************

* parent households
* eldest member ages 65+
* respondent-level weights corrected for NH residents
* pick one member per household to not double-count
summ rownhm [aw=rwtall] if (ageEldest>=65 & ageEldest<.) & (hpickhh==1) & (hchild>0 & hchild<.)
// -> 75.2% *** (TABLE 5: Homeownership rate (ages 65+))
//   (0.7516016)

****************************************
** Median wealth, ages 65-69
****************************************

* parent households
* eldest member ages 65-69
* respondent-level weights corrected for NH residents
* pick one member per household to not double-count
summ atotb10 [aw=rwtall] if (inrange(ageEldest,65,69)) & (hpickhh==1) & (hchild>0 & hchild<.), det
// -> 206,000 *** (TABLE 5: Median household wealth (ages 65-69))

****************************************
** Unconditional inter vivos transfers to children from healthy parents
****************************************

** NOTE: model defines healthy as EITHER (not disabled) OR (coupled)
**                       sick  as  BOTH  (disabed) AND (not coupled)

summ tcamt10 [aw=rwtall] if ///
	  (disablednew==0 | hcpl==1) ///     - healthy (coupled, or not disabled)
	& (ageEldest>=65 & ageEldest<.) ///  - eldest member 65+
	& (hpickhh==1) ///                   - pick one member per household
	& (hchild>0 & hchild<.) ///          - parents
	& (atotb10 <  1.684e6)   //          - trim top 5% of wealth (HRS 1998-2010)
	
// -> 3,356 *** (TABLE 5: Mean (annual) gift: (healthy)-parent-to-child)
	
****************************************
** Care arrangements, single, disabled, 65+
****************************************

* NOTE:

* Care arrangements of sick parents ages 65+
* Sick = single + disabled

* IHC defined as >50% hours IHC and No NHC
* FHC defined as >=50% hours FHC and No NHC
* NHC defined as Any NHC

* To obtain MA uptake rate, we use the following information:
* BK2018 report that 47.9% of disabled FHC individuals are MA-financed. 
* Among nursing-home residents, BK 2018 report that 56.1\% are fully or mostly covered by MA.

preserve

keep if ///
    (hcpl==0 & disablednew==1)   /// - single and disabled
	& (ragey_b>=65 & ragey_b<.)    /// - age 65+
	& (hchild>0 & hchild<.)        //  - parent

tab caretype50new, gen(ct)
descr ct?

qui summ ct1 [aw=rwtall]
local ihc = 100 * `=r(mean)'

qui summ ct2 [aw=rwtall]
local fhc = `=r(mean)'

local fhc_pp = 100 * `fhc' * (1 - 0.479)
local fhc_ma = 100 * `fhc' * (0.479)

qui summ ct3 [aw=rwtall]
local nhc = `=r(mean)'

local nhc_pp = 100 * `nhc' * (1 - 0.561)
local nhc_ma = 100 * `nhc' * (0.561)

local ma = `fhc_ma' + `nhc_ma'

* print results:

di "ihc    = " %4.2f `ihc' "%"
di "fhc-pp = " %4.2f `fhc_pp' "%"
di "nhc-pp = " %4.2f `nhc_pp' "%"
di "ma     = " %4.2f `ma' "%"
di "(      = fhc-ma + nhc-ma = " %4.2f `fhc_ma' "% + " %4.2f `nhc_ma' "% )"

restore

/*

* Results:

ihc    = 48.66% *** (TABLE 5: Informal care)
fhc-pp =  8.40%
nhc-pp = 15.46%
ma     = 27.48% *** (TABLE 5: Medicaid uptake rate)
(      = fhc-ma + nhc-ma = 7.72% + 19.76% )

* From these results, we also construct Table 6:

// IHC   FHC-PP  NHC-PP    MA 
// 48.7   8.4     15.5    27.5

*/

********************************************************************************

cap log close

********************************************************************************
