% Creates (purely illustrative) figures that explain concept of dynasty 
% wealth equivalent variation (dynasty WEV).
%
% Runtime: about 1 second.
% Needed toolboxes: none (only standard Matlab)
% 
% To generate FIGURE C.1 (CEV: standard case)  in paper, run script with
% default values (i.e. dVk=1).

% To generate FIGURE C.2 (CEV: corner solution) in paper, run with default 
% values, but setting dVk=0.2 (then da^p>=0 binds and WEV_p = WEV_{dyn}).


%% Set parameters:
dd = 0.01;                      % Small distance to shift lines. 
                                % Default: 0.01.

ak = 0;                         % Initial parent and child assets: 
ap = 0;                         % normalize to zero (scale not shown).
                                % Default: ak=ap=0.

Vkak = 1;                       % Normalize own derivatives to 1. 
Vpap = 1;                       % Defaults: Vkak=Vpap=1.

Vkap = 0.5;                     % Set valuation of other's assets to half that:
Vpak = 0.5;                     % Default: Vkap=Vpak=0.5.

dVk  = 0.2;                       % Value differential of owning for kid:
                                % a) standard case: 1 (default)
                                % b) non-standard case: 0.2
dVp  = 1;                       % Value differential parent: default 1.


%% Calculations:
% Agents' private WEVs (fixing the other's wealth):
wevk= dVk/Vkak;                 % Kid's private WEV (fixing ap).
wevp= dVp/Vpap;                 % Parent's private WEV (fixing ak).

slopek = Vkap/Vkak;             % Slope of child's iso-value lines: <=1.                                     
slopep = Vpap/Vpak;             % Slope of parent's iso-vallue lines: >=1.

% Set up matrices and vectors to solve system jac*dastar = dV, which are
% the linearized constraints saying that both agents are indifferent:
jac = [ Vkak, Vkap; Vpak, Vpap];% Jacobian matrix: 2-by-2.
dV = [dVk; dVp ];               % Value differentials: 2-by-1.

dastar = jac\dV;                % Solve linear system of equations to get
                                % "unconstrained solution": this is the
                                % solution to the problem if both
                                % components are non-negative.
wevstar = sum(dastar);          % Associated WEV is sum of the two.

xmin = -0.5;    xmax = 1.5;     % Set limits for x=axis.
if dVk>0.9                      % For y-axis:
    ymin = -0.5;    ymax = 1.5; % Standard case
else
    ymin = -0.7;    ymax = 1.1; % Shift them for non-standard case.
end
xvec = [xmin,xmax];             % Put them into vectors: 1-by-2.
yvec = [ymin,ymax];

% Create N-by-N matrix that tells us at which points on the graph both are
% better off:
N = 100;
xfine = linspace(xmin,xmax,N) ; % Fine grid for x-axis: 1-by-N vector.
yfine = linspace(ymin,ymax,N)'; % Fine grid for y-axis: N-by-1 vector.
xmat  = xfine(ones(1,N),:);     % N-by-N matrices with x-coordinate and
ymat  = yfine(:,ones(1,N));     % y-coordinate of all points.
Vkmat = Vkap*xmat + Vkak*ymat;  % N-by-N matrices with value functions
Vpmat = Vpap*xmat + Vpak*ymat;  % on all points (normalize value to 0 at 
                                % x=y=0).
BothBetter = (Vkmat>dVk) & (Vpmat>dVp);
                                % Create logical matrix telling us if
                                % both are better off: N-by-N.

%% Make figure.                                    
%  Create figure name according to which case is run:                                   
if dVk==1      
    figName = 'WEV: regular case (Fig.C1 in paper)';
elseif dVk==0.2
    figName = 'WEV: corner solution (Fig.C2 in paper)'; 
else 
    figName = sprintf('WEV: other case (dVk=%1.2f)',dVk); 
end

figure('Name',figName)
hold on
% First, plot the lines that we mark in legend:
% Plot iso-value lines for renting: 
plot(xvec,-slopek*xvec,'-.r')   % child,
plot(xvec,-slopep*xvec,'--b')   % parent.

% Now, plot iso-dynasty-wealth line (slope -1) that goes through da*:
plot([xmin        ,wevstar-ymin],...
     [wevstar-xmin,ymin        ],   '-k')

% Mark points at which both are better off in light grey:
LightGrey = 0.9*[1,1,1];
scatter(xmat(BothBetter),ymat(BothBetter),'.','MarkerEdgeColor',LightGrey)

% Now, plot second iso-dynasty-wealth line that goes through origin:
plot([ xmin,-ymin],... 
     [-xmin, ymin]    ,'-k')
% Mark slope of iso-dynasty-wealth line:
plot([-10*dd,-10*dd,0],[wevstar+10*dd,wevstar,wevstar],'-k')
text( -5*dd , wevstar-dd  , '1','HorizontalAlignment','center','VerticalAlignment','top'   )
text( -10*dd, wevstar+5*dd,'-1','HorizontalAlignment','right' ,'VerticalAlignment','middle')
    
% Iso-value lines for owning:
plot(xvec,wevk-slopek*xvec,'-.r')   % child.
text(xvec(end)+dd,wevk-slopek*xvec(end),{'child indiff.','to own'},'Color','r') 
plot(wevp-yvec/slopep,yvec,'--b')   % parent.
text(wevp-yvec(end)/slopep,yvec(end)+dd,'parent indiff. to own','Color','b',...
     'HorizontalAlignment','center','VerticalAlignment','bottom')

% Plot zero constraints:
plot(xvec,[0,0],'-k')
plot([0,0],yvec,'-k')
% Mark renting allocation:
plot(0,0,'ok')
text(-0.05,-0.05,'renting','Interpreter','Latex',...
     'HorizontalAlignment','right','VerticalAlignment','top')

plot([-5*dd,-5*dd],[0,wevk],'-r','Marker','_')     
                                % Mark kid's own WEV
text(-5*dd,wevk/2,'$WEV_k$','Interpreter','Latex','Color','r',...
     'HorizontalAlignment','right','VerticalAlignment','middle')
                                % Put text box.
plot([0,wevp] ,[-5*dd,-5*dd],'-b','Marker','|')
text(wevp/2,-5*dd,'$WEV_p$','Interpreter','Latex','Color','b',...
     'HorizontalAlignment','center','VerticalAlignment','top')

txt = {'$\Delta a^*$','both indiff.','to own'};

if dastar(1)>0 && dastar(2)>0   % If da* is the solution:
    plot([5*dd,5*dd],[0,wevstar],'-k','Marker','_')
    text(5*dd,wevstar/2,'$WEV_{dyn}$','Interpreter','Latex',...
         'HorizontalAlignment','left','VerticalAlignment','middle')
    txt{4} = '= solution';      % Enlarge text for da*.
else                            % In the case that parent WEV is the
                                % solution (don't consider kid case here):
    % Mark dynasty WEV:
    plot([0,wevp],[5*dd,5*dd],'-k','Marker','|')
    text(wevp/2,5*dd,'$WEV_{dyn}$','Interpreter','Latex',...
         'HorizontalAlignment','center','VerticalAlignment','bottom')
    % Mark solution:
    plot(wevp,0,'ok')
    text(wevp+dd,dd,'solution','Interpreter','Latex',...
         'HorizontalAlignment','left','VerticalAlignment','bottom')
    % Add another iso-wealth line that cuts through WEV^dyn:
    plot([xmin     ,wevp-ymin  ],...
         [wevp-xmin,ymin       ],   '-k')
end

% Mark point at which both are exactly indifferent:
plot(dastar(2),dastar(1),'ok')
text(dastar(2)-dd,dastar(1)-dd,txt,...
     'HorizontalAlignment','right','VerticalAlignment','top',...
     'Interpreter','Latex')

xlim(xvec);  ylim(yvec);
xlabel('$\Delta a^p$','Interpreter','Latex')
ylabel('$\Delta a^k$','Interpreter','Latex')
xticks(0), yticks(0)
legend('child iso-value lines','parent iso-value lines',...
        'iso-dynyasty-wealth lines','both better off than own')
