function out = SSG_SolveModel(par,opt)
% Solves the model for parameters given in par and options in opt.
%
% Inputs: STRUCTURES 
% par:          Structure. Model parameters (see SSG_MotherShip.m).
% opt:          Structure. Algorithm options (see SSG_MotherShip.m).
% 
% Output: STRUCTURE out=res with fields (search this file for 'res' to find
% exact definitions):  
% Nj:           Scalar. Number of time intervals in algorithm.
% store.tp:     Nj-by-1 vector. Age of parent at end points of time,
%               starting with lowest (tRet+dt) and ending with highest (tDth)
% store.dt:     Nj-by-1 vector. Length of time intervals.
% pan:          Structure containing results from artificial panel.
%       n:      Scalar. Number of panel members, equals par.nPan.
%       T:      Scalar. Number of panel periods (usually 30/2 = 15).
%       agevec: 1-by-T vector. Exact age at interview, roughly [67,68...90].
%       nCln:   Scalar. Number of clones created, is now 3. Main household
%               is index 1 in this dimension.
%       indCln65: Scalar, now 2. Index that refers to the clone that is
%               created at age 65.
%       indClnLtc: Scalar, now 3. Index that refers to the clone created at
%               entry into LTC.
%       stocks: Structure. Stored stock variables at time of interview.
%               Sub-fields: ap, ak, h, ind (linear index of state).
%       flows:  Structure. Stored flows, sub-fields indicate variables. For
%               example, res.pan.flows.gp.val is for parent gift 'gp'.
%               Variables are gifts given by parent ('gp') and kid ('gk'),
%               exchange-motivated transfers by each ('Qp,Qk'), gifts to
%               unconstrained recipients ('gpUnc,gkUnc' -- a subset of
%               gp,gk) and parent expenditures ('ep'). Suffix meanings:
%               .val:   (nPan,TPan,nCln) array with cumulative flow measured 
%                       over the interview period (usually two years) for
%                       all clones.
%               .ccl:   (nPan,TPan) array with cumulative flow measured
%                       as in .val, but for contemporaneous clone (at each
%                       point in time) of the main clone.
%               .polval:(nPan,TPan,nCln) array with snapshot of the flow
%                       (or policy) at the time of the interview, in annual
%                       terms.
%               .polccl:(nPan,TPan) array with snapshot of the policy of
%                       the main household's contemporaneous clone at time
%                       of interview.
%       binary: Structure. Stores information on binary variables (care
%               arrangements IC, FHC, NH, MA and Htm, i.e. hand-to-mouth).
%               Suffix meanings:
%               .val:   (nPan,TPan,nCln) array. Value that the binary
%                       variable takes at time of interview.
%               .ccl:   (nPan,TPan) array. Value that the binary variable
%                       takes at time of interview for the main household's
%                       contemporaneous clone.
%               .time:  (nPan,TPan,nCln) array. Cumulative time over interview
%                       period (2 years) that each clone has spent in the
%                       state during which the binary variable was true.
%               .ever:  (nPan,1,nCln) array. If a clone ever experienced
%                       the state during his life.
%       age:    Structure. At which age an event occurred for a panel
%               member. Is NaN if event never occurred. Sub-fields:
%               Liq:    Liquidated house.
%               EnterLtc: Entered LTC.
%               Died:   Died.
%       once:   Structure. Variables that are only recorded once for each
%               panel member. 
%               - (nPan,1,nCln) arrays: Nwp65 (net worth of parent at age
%                 65), NwpLtc (net worth at entry into LTC), Beq (total
%                 bequest), HBeq (housing bequest).
%               - (nPan,1) arrays: h65, EpInd65 (parent's house size and
%                 prod. index at 65), OwnLtc, KeepLtc (if parent owned
%                 house/kept house at entry into LTC), Surv95 (if parent
%                 survived up to age 95).

addpath('HelperFct')  

% Read out parameters from structure 'par' 
alphap= par.alphap; alphak= par.alphak; 
r     = par.r;      rho   = par.rho;    g    =par.g;     Cma  = par.Cma;
sigmaA= par.sigmaA; deltaHouse = par.deltaHouse;   
epsHaz1D = par.epsHaz1D;
tauLTC= par.tauLTC; tauEst= par.tauEst; valNoConsFeasible = par.valNoConsFeasible;
% Grid-related parameters:
linGrid = par.linGrid;
N     = par.N;      Nw     = par.Nw;     Nh    = par.Nh;   Ns = par.Ns;     % Grid sizes:
ClosestAssetInd = par.ClosestAssetInd;
aup   = par.aup;   avec  = par.avec;   dap=par.dap;   dak=par.dak;   % asset grid,  
apgrid= par.apgrid; akgrid = par.akgrid; startA= par.startA; avecExt = par.avecExt;
epsvec= par.epsvec; 
Pvec  = par.Pvec;   
epsk  = par.epsk;   Pp    = par.Pp;                          % income & pensions
hvec  = par.hvec;   h0     = par.h0;       % housing.
hgrid = par.hgrid;   
ClosestHouseInd = par.ClosestHouseInd;
BKwp  = par.BKwp;  ExtrapHouse = par.ExtrapHouse;  SCmustSell = par.SCmustSell;
dGen  = par.dGen;  tDth   = par.tDth;   tRet  = par.tRet;   % Life cycle.
nPan  = par.nPan;   dtPan  = par.dtPan;  BurnInPan = par.BurnInPan; % simulated panel
% Transition matrices and functions and for life cycle:
nInit       = par.nInit;          HHeffUnits = par.HHeffUnits; 
tauSS       = par.tauSS;          MatchProb  = par.MatchProb;  
polyMenAlv  = par.polyMenAlv;      
medMu       = par.medMu;          medSig     = par.medSig;
InfC = par.InfC;
qnet = par.qnet;      
                  
% Read out the options for the algorithm from structure 'opt':
dt          = opt.dt; % Read out time increment: Fixed throughout.
tpStop      = opt.tpStop;       Mextrap      = opt.Mextrap;   
tolOLG      = opt.tolOLG;       maxIterVF    = opt.maxIterVF;
DisplayEvery = opt.DisplayEvery; 
noKid        = opt.noKid;
CloneWars   = opt.CloneWars;

indHlth= 1;
indBC  = 2;                     % Indeces of basic care and skilled care
indSC  = 3;                     % in dim. 3: Define these as variables so we
                                % we can search for them when making
                                % changes.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Find value-function guess %%   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Solve the model for an agent who is alone in the world (marker 'Aln') to
% get an initial value-function guess. This will be the kid of the parent
% that we consider in the first OLG value-function iteration, so we will
% refer to the agent here as 'kid'. For counterfactuals without children
% (noKids=true), this code is used to obtain results for childless agents.

% First, set up grids for the state space with four dimensions:
% 1: assets
% 2: LTC state
% 3. productivity 
% 4: housing 
% --> total size is (N,Ns,Nw,Nh).
akGridAln  = reshape(avec ,[N,1, 1,1]);  % Asset grid.
PkAln      = reshape(Pvec   ,[1,1,Nw,1]);  % Pension income.
epskIndAln = reshape(1:Nw   ,[1,1,Nw,1]);  % Pension/productivity indices.
hgridAln   = reshape(hgrid  ,[1,1,1,Nh]);  % Housing grid.

% Now, set up arrays that will be used in the loop for the val.-fct. guess:                                        
VarAkAln   = (sigmaA*(akGridAln)).^2;   % Variance of assets on grid.
                                        % (N,1,1,1) array.
nnkAlnArr = ones(1,Ns,Nw,1);            % Array with number of persons in 
                                        % kid household. 1 if sick, healthy
                                        % dimension is set in loop. 
                                        % (1,Ns,Nw,1) array.
% Formal-care (FC) costs (since agent is alone, she MUST receive FC):
qnetAln  = reshape(qnet,[1,Ns,1,Nh]);   % Re-shape qnet to the alone dimensions.

% Now, set terminal value function when kid is alone on world:
% (N,Ns,Nw,Nh)-array Zgss, where 'gss' is for 'guess'.
% Dimensions (N x Ns x Nw x Nh):
% ay: wealth of kid
% sy: kid health state
% w:  wage (or pension) of kid
% h:  house size
% We will fill in Zgss in the following if-statement for the two cases.
% Also, set up storage (only needed in noKid-counterfactuals).

if noKid               % In counterfactuals in which there are no kids:
    % "Limbo state": Final value function for noKid economy. Calculated 
    % based on a period of length one-year (dt=1). This ensures a well- 
    % behaved final value function and reasonable consumption rates.
    hhvec = hgridAln;   % Assume all are disabled in the limbo. Renters       
    hhvec(1) = 0;       % have to pay NH and homeowners pay FHC, which are 
                        % both included in qnetAln.
    qnetLimbo = qnetAln;    % Create care cost for limbo: Give basic-care
    CashOnHand = akGridAln + hhvec ...
                 +NetIncPar(0,PkAln,1,0,par) - qnetLimbo;             
                        % Cash-on-hand in the limbo: Eat up all wealth
                        % (assets plus housing) plus the year's income
                        % minus the formal-care costs: (N,Ns,Nw,Nh) array.                       
    CLimbo    = max(Cma,CashOnHand);        % Kids take Medicaid if this is
                                            % better than consuming
                                            % cash-on-hand: (N,Ns,Nw,Nh).
    ULimbo    = utilde(CLimbo,h0,1,par);    % Flow dutility in limbo:
                                            % (N,Ns,Nw,Nh) array.
    Zgss      = ULimbo;                     % Final value function in limbo:
                                            % (N,Ns,Nw,Nh) array.
                                            % Contains same value for
                                            % all health states.
    DeathValAln= 0;                         % Value for agent when dying 
                                            % before 95 is zero.
    % Set up storage cells of length Nj: Each cell will contain an array 
    % for each age j.
    Nj                 = round(dGen/dt);
    aDotNoKidStore     = zeros(N,Ns  ,Nw,Nh,Nj);
    MAdecNoKidStore    = zeros(1,Ns-1,Nw,Nh,Nj);   % MA dec. for LTC states.
    xstarNoKidStore    = zeros(N,Ns  ,Nw,Nh,Nj);   % House-selling.
        eNoKidStore    = zeros(N,Ns  ,Nw,Nh,Nj);   % Consumption expenditure: new convention
                                                   % is to store all.
    
else                    % If we are NOT in a noKid-counterfactual:
    % Assume that a grandchild is born when kid dies, which lives forever with 
    % median wage (eps=0) at age 25. House is sold upon bequest and grandchild 
    % rents always. Altruism of young towards grandchild is alphap.
    HouseBeq = hgridAln;                    % Bequest value of house.
    HouseBeq(:,:,:,1) = 0;                  % Renters don't leave house.
    HHgrCh = (1-tauSS)*HHeffUnits(25,0) + r*(akGridAln+HouseBeq)*(1-tauEst);
                                            % (N,1,1,Nh) array with permanent
                                            % income of grandchild.                                    
    % Get flow utility of grandchild with fixed household size 2:
    flowUgrCh = utilde( HHgrCh.*(1-tax(HHgrCh,tauLTC)) , h0, 2, par);
    DeathValAln = alphap*flowUgrCh / rho;   % Value for agent when dying.
                                            % Assume here that grandchild
                                            % enters world exactly when
                                            % parent dies (at age 95 or
                                            % earlier). 
    clear HouseBeq HHgrCh flowUgrCh         % grandchild obtaining the bequest.
                                            % (N,1,1,Nh) array.
    Zgss = DeathValAln.*ones(N,Ns,Nw,Nh);    % Extend to full size to get 
                                            % final value function:
end                                         % (N,Ns,Nw,Nh) array. 
                                         
t = tRet+dGen;                          % Start kids off at age 95.
j = 0;                                  % Count iterations.
optGss = opt;                           % Take same options for value-fct.
optGss.reflect = false;                 % guess as for the main loop, but
                                        % DON'T reflect: Want linear
                                        % consumption functions (they bend
                                        % up at upper with reflect=true).

                                     
% Now, start the loop to find the val.-fct. guess:                                         
while t>tRet                             % Loop back until age 65.
    menAlvAln = RatePoly2(t,epskIndAln,polyMenAlv,0,par);
                                        % Measure of men alive in
                                        % households with healthy woman:
                                        % (1,1,Nw,1) array.
    nnkAlnArr(:,1,:,:) = 1 + menAlvAln; % This gives number of persons in
                                        % healthy household.
    NetIncGss  = NetIncPar(akGridAln,PkAln,nnkAlnArr,hgridAln,par) - qnetAln;
                                        % Net household income (N,Ns,Nw,Nh) is:
                                        %   net capital income
                                        % + pension income (times fraction
                                        %   of initial couple surviving), 
                                        %   neet to take times w... matters
                                        %   for Ay>1 counterfactual!)
                                        % - expenditures on LTC for woman
                                        % - (fixed) medical expenditures
                                        %   per survivor
                                        % - housing depreciation cost
    [ekGss, ZalnAk, ZgssI, akDotGss, MAGss] = ConsumptionAln(Zgss,hgridAln,nnkAlnArr, NetIncGss,par);  
                                        % Obtain optimal consumption deci-
                                        % sion (ekGss), extrapolated value
                                        % function (ZgssI), the law of
                                        % motion (akDotGss), and Medicaid
                                        % decision.

    llGss.tk       = t;          llGss.epskIndAln = epskIndAln;                               
    llGss.akDotAln = akDotGss;   llGss.VarAkAln   = VarAkAln; 
    llGss.nnkAlnArr= nnkAlnArr;         % Pack structure llGss (changes on
                                        % each loop iteration)...
    hGss = JumpHazardsAln(llGss,par,optGss); 
                                        % ... to get jump hazards on the 
                                        % grid in structure hGss. Don't
                                        % reflect --> use optGss.
    ZalnTom = ContinuationValueAln(ZgssI,hGss,DeathValAln,dt);
                                        % Get continuation value for kids who
                                        % are alone on world: DeathValAln
                                        % is their value when they die.

                                        % Then update value function Z:
    uGss = utilde(ekGss,hgridAln,nnkAlnArr,par);
    % Replace flow utility of MA cases with utility of Cma as RENTER:
    UGss0 = uGss(1,indBC:indSC,:,:);              
    UGss0(MAGss) = utilde(Cma,h0,1,par);
    uGss(1,indBC:indSC,:,:) = UGss0;
    Zgss = uGss*dt + (1-rho*dt)*ZalnTom; 
                                        % Note: child's household is of 
                                        % size 1 when sick and of size nnk
                                        % when healthy
       
    if Nh>1                             % If there is a housing choice: 
        % Decision if to sell house:
        Zrent = Zgss(:,:,:,1);      % Read out value for renters:
                                    % (N,Ns,Nw,1) array.
        sellNewAk = akGridAln + hgridAln;
                                    % Assets after selling the house,
                                    % only depends on child assets and
                                    % house size:
                                    % (N,1,1,Nh) array.
        sellAkRefl  = min(sellNewAk, aup);
                                    % Maximal asset value upon selling
                                    % under reflection: don't go above
                                    % aup.
                                    % (N,1,1,Nh) array.
        % Set up sales value: First, interpolate Zrent inside grid (this
        % just caps assets at aup).
        Zsale = InterpSellAln(Zrent,sellAkRefl,par);
                                    % This is the sales value when we don't
                                    % extrapolate for housing. If we do
                                    % extrapolate, this will only be used
                                    % when falling inside grid).
                                    % (N,Ns,Nw,Nh) array.
                                   
        if ExtrapHouse                  % If housing extrapolation is on:
            ZNrent  = Zrent(N,:,:);     % Read value function of renters at
                                        % outermost grid point:
                                        % (1,Ns,Nw) array.
            cc= Duinvtilde( diff( Zrent )./dak, h0, 2, par);
                                        % (N-1,Ns,Nw) array with "pseudo"
                                        % consumption implied by
                                        % value-function derivatives.
            eNrent  = cc(end,:,:) + ( cc(end,:,:) - cc(end-1,:,:) )/2 ;
                                        % Find best approximation for
                                        % pseudo consumption at aup: 
                                        % (1,Ns,Nw) array. COULD MAKE THIS
                                        % EVEN MORE PRECISE, BUT EFFECT
                                        % WILL BE 2ND-ORDER.
            eMrent  = ekGss(Mextrap,:,:,1); % Read out consumption of renters
                                        % at grid point Mextrap (fixed as
                                        % an option of the algorithm):
                                        % (1,Ns,Nw) array.
            daM = avec(N) - avec(Mextrap);  % Distance from Mextrap'th to
                                        % last asset grid point: scalar.
            eSlope  = (eNrent-eMrent)/daM;
                                        % Get approximation for slope of
                                        % consumption function at aup:
                                        % (1,Ns,Nw) array.
            Arent = Ash(nnkAlnArr(:,:,:,1),h0,par);
                                        % Get utility shifter for renters:
                                        % (1,Ns,Nw) array.
            thetarent = thetah(h0,par); % Get curvature of utility: scalar.
            sellAkOut = max(sellNewAk - aup,0);
                                    % By how much new assets after sale go
                                    % outside the ak-grid:
                                    % (N,1,1,Nh) array.
            Zextrap = ExtrapValFctCRRA(ZNrent,eNrent,eSlope,sellAkOut,...
                        thetarent,Arent);
                                        % Do extrapolation (will only be
                                        % used for ak-values above aup):
                                        % (N,Ns,Nw,Nh) array.
            sellOut = sellAkOut.*ones(1,Ns,Nw,1)>0;      % Logical variable: If selling house
                                        % takes us outside grid.
                                        % (N,Ns,Nw,Nh) array.
            Zsale(sellOut) = Zextrap(sellOut);
                                        % Replace sales value with
                                        % extrapolated one if sales value
                                        % falls above aup.
        end                             % Just copy the last values.

        xgss = Zsale>=Zgss;             % Sell if it's better than staying.
                                        % (N,Ns,Nw,Nh) array. 
%         %Can compare selling and keeping value functions here:
%         hhind = 4;
%        plot(par.avec(2:end),squeeze(Zsale(2:end,1,2,hhind)),'-xr',...    % plot sales value in red
%              par.avec(2:end),squeeze( Zgss(2:end,1,2,hhind)),'-xb'    )   % keep in blue.
%         legend('value sell','value keep')
%          plot(par.avec(2:5),squeeze(Zsale(2:5,3,2,:)-Zgss(2:5,3,2,:)));
        xgss(:,:,:,1) = 0;              % Renters don't sell.
%        GetMarginal(~xgss,1)'          % Count how many don't sell by
%        asset level   
        if SCmustSell                   % If we force skilled LTC to sell,
            xgss(:,indSC,:,2:Nh) = 1;   % do this here for all houses.
        end
        Zgss = xgss.*Zsale + ~xgss.*Zgss; % New value function is the max over
                                          % selling and staying.
    end 
   
    t = t-dt;                           % Update age.
    j = j+1;                            % Count iterations.
     

    if noKid && j<(Nj+1)                 % In no-kids counterfactuals: Save policies,
                                        % these are what childlesss
                                        % optimally do.
         aDotNoKidStore(:,:,:,:,Nj-j+1) = llGss.akDotAln;
        MAdecNoKidStore(:,:,:,:,Nj-j+1) = MAGss;
        if Nh >1
        xstarNoKidStore(:,:,:,:,Nj-j+1) =  xgss; 
        end
            eNoKidStore(:,:,:,:,Nj-j+1) = ekGss;
    end
end
 

if Nh==1                                    % In liquid-housing counterfactual, 
    Zfin = Zgss(:,1,:,:);                   % we already have the final value:
                                            % Read out values for the
                                            % healthy, (N,1,Nw) array.
else                                        % If there is a housing choice to make:
    [Zfin,hpolAln] = BuyHouseValue(Zgss(:,1,:,:),par,0);
                                            % Obtain final value Zfin: This is the
                                            % value for a healthy 65-year old
                                            % without a house when entering
                                            % retirement and deciding if to 
                                            % buy a house: (N,1,Nw) array.
                                            % hpolAln: cell with
                                            % house-buying policies in
                                            % alone economy.
close all                                            
% Also, get threshold at which households start to buy, (1,1,Nw) array, and
% (1,Nw) cell hBuyPol with the buying policy functions (i.e. optimal house
% size conditional on buying).
dvAln     = nan(Nw,Nh);
dvAln(:,1)= 0;
wevAln        = nan(Nw,Nh);
wevAln(:,1)   = 0;
nwpInitAln    = nan(Nw,Nh);
nwpInitAln(:,1)=0;
da = 0.001; 

for iw=1:Nw
hAln       = hpolAln{iw};
hAlnstar   = hAln(akgrid);                  % optimal housing policy for given productivity
nwpInitAln(iw,:) = interp1(hAlnstar,akgrid,hvec); % initial net worth levels for which hvec is optimally chosen
    % % level: find initial net worth that makes landing precisely
    % % on asset grid points optimal. Have to write InvertHpol(.,.).
    % % after buying house.
    for ih=2:Nh                  % Loop over all houses.
        if nwpInitAln(iw,ih)<aup/2     % Only look at starting assets below 2m.
            ap      = nwpInitAln(iw,ih)-hvec(ih); % Financial wealth
            VeqmAln =       interp1(apgrid,Zgss(:,1,iw,ih),ap);
            VRentAln = @(ap)interp1(apgrid,Zgss(:,1,iw,1),ap+hvec(ih),'spline');
            VAlnj    =  VRentAln(ap);    
            VAlnaj   = (VRentAln(ap+da) - VAlnj)/da;
            
%%%%%% WEV based on first-order approximation: one Newton step
            dvAln(iw,ih)  = VeqmAln - VRentAln(ap);
            wevAln(iw,ih) = (1/VAlnaj)*dvAln(iw,ih);
        end
    end
end

dvAln     = nan(Nw,Nh);
dvAln(:,1)= 0;
wevAln        = nan(Nw,Nh);
wevAln(:,1)   = 0;
nwpInitAln    = nan(Nw,Nh);
nwpInitAln(:,1)=0;
da = 0.001; 

            
for iw=1:Nw
hAln             = hpolAln{iw};
hAlnstar(:,iw)   = hAln(akgrid);                  % optimal housing policy for given productivity
nwpInitAln(iw,:) = interp1(hAlnstar(:,iw),akgrid,hvec); % initial net worth levels for which hvec is optimally chosen
iwNaN = isnan(nwpInitAln(iw,:));
nwpInitAln(iw,iwNaN)=aup;
    for ih=2:Nh                 % Loop over all houses.
        if nwpInitAln(iw,ih)<=aup    
            ap          = nwpInitAln(iw,ih)-hvec(ih); % Financial wealth
            apInd       = ClosestAssetInd(ap);  % Find closest financial wealth index.
            VeqmAln     = Zgss(apInd,1,iw,ih);
            VSellAln    = Zsale(apInd,1,iw,ih);
            CEValn(iw,ih) = (VSellAln./VeqmAln).^(1/(g-1)) - 1; %#ok
                                  %Calculate CEV (equiv. var. in all goods)
                                % 
            VaAln(iw,ih) = interp1(avec,ZalnAk(:,1,iw,ih),nwpInitAln(iw,ih),'linear'); %#ok
%%%%%% WEV based on first-order approximation: one Newton step
            dvAln(iw,ih) = VeqmAln - VSellAln;
            wevAln(iw,ih)= (1/VaAln(iw,ih))*dvAln(iw,ih);
        end
    end
end


if noKid                                % In no-kid counterfactuals: 
    ZhousWEV = Zfin;                    %#ok %Store these values for WEV 
    ZrentWEV = Zgss(:,1,:,1);           %#ok %calculations: (N,1,Nw) arrays.
end
   
end % close liquid counterfactual


NetIncFin = NetIncGss(:,1,:,1);         % Net income at final stage (need this
                                        % later to check convergence):
                                        % (N,1,Nw) array.
eFin = ConsumptionAln(Zfin,h0,2,NetIncFin,par);         
                                        % Get optimal consumption expenditure 
clear Zgss PkAln sickAln ...            % given this value function to check 
      NetCplIncAln nnkAlnArr...         % convergence free of scale: (N,1,Nw) array.
      DeathValAln NetIncGss llGss akDotGss hGss ekGss ZalnTom ZgssI
                                        % Free up memory.
fprintf('Value-function guess found in %1.0f iterations.\n',j);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Set up storage for main loop %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Nj         = round(dGen/dt);        % Storage set up in age dimension: One
                                    % grid point for each time interval. 
logCold    = NaN;                   % Old log-consumption: initialize loop.

fdFullStore = {'ep','gp','gk','V','W','akDot','apDot',...
               'xstar','Qstar','QPaltrComp','QKaltrComp'};
                                    % Names of fields for which we store
                                    % policies of full size.
for ifd=1:numel(fdFullStore)        % Set up these standard fields of the 
    store.(fdFullStore{ifd}) = zeros(N,N,Ns,Nw,Nw,Nh,Nj);   % store-structure.
end
% Now, set up fields with non-standard size for storage.
store.tp    = zeros(Nj,1);           % Age of parent.
store.minPm = zeros(Nj,1);           % Store the minimal P_m (probability of
store.dt    = zeros(Nj,1);           % staying at the same grid point:
store.check = zeros(Nj,1);           % should not get negative!)
store.Cdist = zeros(Nj,1);           % Distance of consumption functions. 
store.MA    = zeros(N,1,Ns-1,Nw,Nw,Nh,Nj); % MA decision: both LTC states.
store.istar = zeros(N,N,1   ,Nw,Nw,Nh,Nj); % informal care: only basic LTC state.
store.FHC   = zeros(N,N,1   ,Nw,Nw,Nh,Nj); % FHC in basic LTC state.

nStsOLG    =       N*N*Ns*Nw*Nw*Nh    ; % Number of states in state space.

% Also store hazards:
store.hazAkp= zeros(N,N,Ns,Nw,Nw,Nh,4,Nj);
% Store hazards of moving in the two asset dimensions.
% dim. 1-6: the usual state space
% dim. 7:   which hazard (kid/parent, up/down)
%   index 1: akUp, hazard that kid's assets go up.
%   index 2: akDn, kid's assets down.
%   index 3: apUp, parent's assets up.
%   index 4: apDn, parent's assets down.
% dim. 8:   age
store.haz345 = cell(Nj,1);           % Store hazards for medical, LTC, and
                                    % death shocks in a cell. For each age,
                                    % there is one structure containing the
                                    % hazards. See documentation for haz345
                                    % below what this structure looks like.
store.haz345Arr = zeros(Ns,Nw,Nw,4,Nj);% Also create an array containing these     
                                    % hazards since we need this structure
                                    % for GetMoments2.m. Not problematic
                                    % for storage since only few
                                    % dimensions. 
                                    % Dim.1: health
                                    % Dim.2: child prod.
                                    % Dim.3: parent prod.
                                    % Dim.4: which hazard:
                                    %        index 1: medical shock
                                    %        index 2: basic-LTC shock
                                    %        index 3: skilled-LTC shock
                                    %        index 4: death
                                    % Dim.5: age
store.hazAkpAln= zeros(N,  1,Nw,   2,Nj);  % Store hazard rates of going 
% up or down in kid's assets (due to savings and Brownian Motion) in the 
% case where kid is alone. Dimensions: 1-3 as usual for alone case. Dim. 5
% is age. Dimension 4:
% dim. 4, index 1: akAlnUp, assets up.
% dim. 4, index 2: akAlnDn, assets down.

% Set up some arrays we need during the OLG loop:
epspInd = reshape( 1:Nw, [1,1,1,1,Nw,1] );
                                        % Indeces of parent's productivity 
                                        % on 6D grid: (1,1,1,1,Nw,1) array.
nnpArr    = ones(1,1,Ns,1,Nw,1);        % Array with number of persons in
                                        % parent household: 1 if sick, 
                                        % healthy dimension is set in loop.
                                        % Depends only on health state and
                                        % parent productivity:
                                        % (1,1,Ns,1,Nw,1) array.
% Now, when alone on world, we only need to keep track of kids in healthy
% state and when they are renting. Set up some arrays for this, taking the
% helper arrays from before and taking only the healthy renting state.
akGridAln = reshape(  avec,[N,1,1 ]);   % Child wealth when alone:
                                        % (N,1,1) array.
epskAln   = reshape(epsvec,[1,1,Nw]);   % Child productivity when alone:
                                        % (1,1,Nw) array.
VarAkAln  = (sigmaA*(akGridAln)).^2;    % Return variance on grid: 
                                        % (N,1,1) array.
                                        
dumpInd = round(N/4);                   % To check transfer motives along the 
                                        % way, dump the upper 4th of indeces in 
                                        % asset grid when displaying
                                        % transfer motive.
% Set up logical (N,N,Ns,Nw,Nw,Nh) array that tells us where we want to
% check the parent's transfer motive for stability message.
taupEval = false(N,N,Ns,Nw,Nw,Nh);
taupEval(1:N-dumpInd,2:N-dumpInd,1:Ns-1,:,:,:) = true;  % Dump indices above dumpInd.
taupEval(1:N-dumpInd,2:N-dumpInd,  Ns  ,:,:,1) = true;  % Only check renters for skilled LTC state.    
% Same for kid: logical array where to evaluate kid's tau.
taukEval = false(N,N,Ns,Nw,Nw,Nh);
taukEval(2:N-dumpInd,1:N-dumpInd,1:Ns-1,:,:,:) = true;
taukEval(2:N-dumpInd,1:N-dumpInd,  Ns  ,:,:,1) = true; 
                                        
% Set up two transition matrices between the large grid (N,N,Ns,Nw,Nw,Nh) and 
% the small grid (N,1,Nw), where households are alone: one for the death
% case, and one for the case where new children are matched.
% First, we need the following functions to transform indices:
getVecInd = @(aki,api,si,eki,epi,hi) Nw*Nw*Ns*N*N*(hi-1) + ...
                                     Nw*Ns*N*N*(epi-1) + Ns*N*N*(eki-1) + ...
                                     N*N*(si-1) + N*(api-1) + aki;
                                % Function that gives back the index for
                                % the vectorized state space given the five
                                % indeces for the conventional dimensions.
getAlnVecInd = @(ai,ei,hi) N*Nw*(hi-1) + (ei-1)*N + ai;                              
                                % Similarly, this function returns the 
                                % linear index for the small state space 
                                % (where kid is alone) given the asset (ai),
                                % productivity index (ei), and housing
                                % index (hi).
                                
% Start with death matrix: N*N*Nw*Nh-by-N*Nw sparse matrix with transitions 
% that occur upon parent's death (including the bequest that takes place)
% from: kids wealth (ak), parent's wealth (ap), kid's productivity (epsk),
%       and housing (h)
% to:   kid's wealth (ak) and kid's productivity (epsk)
% (Cannot treat kid's productivity with Kronecker product here since it
%  lies between the asset and the housing dimensions).
hhvec = hvec;
hhvec(1) = 0;
Beq126 = apgrid + akgrid + reshape(hhvec,[1,1,1,1,1,Nh]);  % (1-HouseTransCost) (1-tauEst)
                                % Create (N,N,1,1,1,Nh) array with the
                                % different bequests that are possible.
                                % '126' indicates that this varies along
                                % dimensions 1, 2, and 6.
                                % Have to take transaction cost out of
                                % estate if turning this feature on, but
                                % would have to re-program stuff below
                                % since we would not hit grid points
                                % exactly any more and would have to
                                % interpolate.
Beq1246 = Beq126 .* ones(1,1,1,Nw,1,1);   
                                % Extend in the dimension of kid's
                                % productivity (dim.4): Get
                                % (N,N,1,Nw,1,Nh)  array. All '1246'
                                % variables will have this size.                                
Beq1246refl = min(Beq1246,aup); % Get bequest that occurs when forcing 
                                % assets inside the grid ('reflect').
Beq1246dOut = Beq1246 - Beq1246refl;
                                % Get how much we jump out of the grid
                                % (need this for extrapolation later).
Beq1246isOut = (Beq1246dOut>0); % Create logical variable capturing if the 
                                % bequest lies outside the grid.
                                
epsk1246 = reshape((1:Nw),[1,1,1,Nw,1,1]) .* ones(N,N,1,1,1,Nh);    
                                % Make (N,N,1,Nw,1,Nh) array with
                                % productivity index of kid. 
if linGrid==1                                
    indAkBeq = 1 + round(Beq1246refl/dak(1));
                                % Get index of kid assets to which we are
                                % jumping upon death.
    n1246 = N*N*Nw*Nh;              % Number of grid points taking together dim. 1,2,4,6.                                
    toInd   = indAkBeq(:) + ( epsk1246(:) - 1 )*N ;
                                % Get indeces of points to which we are going
                                % in the alone-space: n1246-by-1 vector.
    fromInd = (1:n1246)';           % Indices of points from which we come in
                                % space of dim. 1,2,4,6: n1246-by-1 vector.                                
% Finally, create the sparse transition matrix:
    par.BeqMat1246toAln = sparse(fromInd,toInd,1,n1246,N*Nw);                         
    clear Beq126 epsk1246 loIndAkBeq hiWgt fromInd toInd    
                               
% Second, construct the matching matrix: With which kid are the new parents
% matched (after they have bought a house)? Cannot use Kronecker product
% because kid's assets (dim.1) depend on kid's productivity (dim.4), thus
% create N*Nw*Nh-by-N*N*Ns*Nw*Nw*Nh matrix.
end
MatchMat= spalloc(N*Nw*Nh,nStsOLG,N*Nw*Nh*2*Nw);
                                % Set up matrix for matching: from each
                                % state alone (dim.1, in rows), can jump to
                                % 2*Nw states in the large state space (Nw
                                % different productivities of kid, 2 for
                                % each prod. due to interpolation of 
                                % kid's starting wealth on asset grid).
for ik=1:Nw                     % Loop over all productivities of kid:             
    iA  = startA(ik);           % Read out starting wealth of kid 
    flrInd = max(find(avec>iA,1)-1,1);  
   % flrInd = 1+floor(iA/dak(1));    % Interpolate : lower index the kid can 
    flrWgt = 1-mod(iA,dak(flrInd))/dak(flrInd);   % be inserted, and the weight on it.
    for ip=1:Nw                 % Loop over all productivities of parent:
        pkProb   = MatchProb(ip,ik);
                                % Read out probability the two are matched
        fromInd  = getAlnVecInd((1:N)',ip,1:Nh);
                                % Jumps: from all combinations of asset and
                                % housing levels for this parent prod.
                                % level -- get N-by-Nh matrix.
        toInd    = getVecInd(flrInd,(1:N)',1,ik,ip,1:Nh);
                                % Jump to assets of kid (floor level), same
                                % assets of parent, healthy, productivity
                                % levels: same for parent, fix for kid, 
                                % parents are renters for now (choose
                                % later). Another N-by-Nh matrix.
        MatFlr   = sparse(fromInd(:),toInd(:)  ,pkProb*  flrWgt  ,N*Nw*Nh,nStsOLG);
                                % Sparse transition matrix for going to the
                                % lower grid point in kid's wealth.
        MatClg   = sparse(fromInd(:),toInd(:)+1,pkProb*(1-flrWgt),N*Nw*Nh,nStsOLG);
                                % Same for upper grid point.
        MatchMat = MatchMat + MatFlr + MatClg;
    end                         % Add them all up cumulatively to get 
end                             % large matrix.
clear MatFlr MatClg flrInd flrWgt iA pkProb fromInd toInd


%%%%%%%%%%%%%%
%% Main loop %
%%%%%%%%%%%%%%
Zaln = Zfin;                            % Set final value function for kids
distOLG = 1;                            % and start counters.                      
iterOLG = 0;

while distOLG>tolOLG && iterOLG<maxIterVF % OLG loop: until converged or
                                        % aborted
iterOLG = iterOLG+1;                    % Count OLG iterations.
disp('-----------------------------------------------------------------')
fprintf('OLG VALUE-FUNCTION ITERATION %1.0f:\n',iterOLG) % Show progress.
disp('Entering BK model on enhanced steroids...')

tp        = tRet+dGen;                  % Start off parents at age 95.   
ll.tp     = tp;                         % Note that in counterfactual with 
                                        % certain death age, none of the
                                        % parents is alive in the last
                                        % years.
                                        
j         = 0;                          % Count age iterations.
ParAlive  = false;                      % Keep track if parent is alive.                 
                                        
% NOTE: IMPORTANT VARIABLES THAT HAVE TO BE HANDED OVER TO FUNCTIONS AND 
% CHANGE IN EACH LOOP ITERATION ARE KEPT IN STRUCTURES WITH NAMES 'll...'
% --> HAVE TO HAND OVER FEWER FUNCTION ARGUMENTS THIS WAY.

while j<Nj                              % Start age loop:
tp = tp-dt;                             % Update age of parents: tp is the age
                                        % in the beginning of period
                                        % [tp,tp+dt).
j      = j+1;                           % Count up age iterations.
ll.tp  = tp;                            % Record age of parents in ll.
tk     = tp-dGen;                       % Get age of kids, and
ll.tk  = tk;                            % record in ll-structure.
nnk    = 2;                             % Always two household members in  
ll.nnk = nnk;                           % child household since tk<tRet.
menAlv= RatePoly2(tp,epspInd(:,:,1,:,:,:),polyMenAlv,0,par);
nnpArr(:,:,1,:,:,:) = 1 + menAlv;       % Get number of alive men in healthy
ll.nnpArr         = nnpArr;             % households: (1,1,1,1,Nw,1) array.
% clear menAlv                          % Don't clear loop variables: re-
                                        % allocating and clearing them
                                        % every time is costly.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%     % First, handle case where the old
%% 'ALONE' CASE: ONLY YOUNG ALIVE %     % are already dead (fewer state variables)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
NetIncAln= NetIncKid(tk,akGridAln,epskAln,0,par);
                                        % Get net income of kid houshehold 
                                        % when alone: (N,1,Nw) array.
[ekAln, ~, ZalnI, akDotAln] = ConsumptionAln(Zaln,h0,nnk,NetIncAln,par);
                                        % Get optimal consumption (ekAln), -
                                        % extrapolated value function (ZalnI) 
                                        % and law of motion (akDotAln):
                                        % (N,1,Nw) arrays.
llAln.tk       = tk;         llAln.epskIndAln = [];                               
llAln.akDotAln = akDotAln;   llAln.VarAkAln   = VarAkAln; 
llAln.nnkAlnArr= [];                    % Pack structure llAln (changes on
                                        % each loop iteration)...
hAln = JumpHazardsAln(llAln,par,opt);   % ... to get jump hazards on the 
                                        % grid in structure hAln.
minPm = 1-max(hAln.leave(:))*dt;        % Calculate the minimal probability 
                                        % of staying at the same grid point
                                        % (this number should not go below
                                        % zero). Will later be over-written
                                        % by the dynasty code if there are
                                        % lower Pms there.
ZalnTom = ContinuationValueAln(ZalnI,hAln,0,dt);
                                        % Get continuation value for kids 
                                        % who are alone on world: hand over
                                        % 0 death value since death hazard 
                                        % is zero for them now.
Zaln = utilde(ekAln,h0,nnk,par)*dt + (1-rho*dt)*ZalnTom;
                                    % Update value function.
% Store objects when kid alone:
  store.hazAkpAln(:,:,:,1,Nj-j+1) = hAln.upAk;
  store.hazAkpAln(:,:,:,2,Nj-j+1) = hAln.dnAk;
    store.Zaln(:,:,:,  Nj-j+1) = Zaln;   
store.akDotAln(:,:,:,  Nj-j+1) = akDotAln;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%           
%% OLG CASE: BK ON STEROIDS %           % Main part: Both are alive.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 

if tp<=tDth                             % If the parent is alive at this tp
                                        % with positive probability:
ll.NetIncPar = NetIncPar(apgrid,Pp,nnpArr,hgrid,par);  
                                        % Get net income of parents:
                                        % (1,N,Ns,1,Nw,Nh) array (does not
                                        % depend on child's states; depends 
                                        % on parent health state since hh 
                                        % size varies).
% NOTE: NetIncPar.m and NetIncKid.m compute agent's net income from both
% labor/pension and capital and alreay takes the following out of income:
% - housing-depreciation expenditures                                        
% - fixed medical expenditure 

% Get kid's net income when giving informal care (function returns income 
% already net of housing-depreciation expenditures): includes subsidy (parents
% will pay equipment for IC) -- this depends only on kid's states:
ll.NetIncKidIC = NetIncKid(tk,akgrid,epsk,1,par);
                                        % Net income of kids when giving
                                        % informal care:
                                        % (N,1,1,Nw,1,1) array.
% ... and when not giving informal care;
ll.NetIncKidFC  = NetIncKid(tk,akgrid,epsk,0,par);
                                        % For formal care: (N,1,1,Nw,1,1)
                                        % array.
% Now, obtain value for children when parents die ('beq' stands for 'bequest' 
% and '1246' refers to the fact that this value changes along dimensions 1, 
% 2, 4, and 6):
if linGrid==1
    WalnExt1246vec = par.BeqMat1246toAln * Zaln(:);
                                        % Create vector of length N*N*Nw*Nh
                                        % (vectorized state over both
                                        % assets dim., kid's prod. and
                                        % housing) with the kid's value 
                                        % upon the parent's death.
    WalnExt1246 = reshape( WalnExt1246vec, [N,N,1,Nw,1,Nh] );
elseif linGrid==0                       % Non-linear grid: call function to update.
    WalnExt1246=BequestValue(Zaln,Beq126,par);
end
                                        % Bring array to 6D.
ExtrapW1246 = zeros(N,N,1,Nw,1,Nh);     % Now, calculate extrapolated values:                                        
                                        % points with bequest outside grid.
                                        % Set up array that we fill now:
if g==1                                 % Extrapolation for log utility:
    for i=1:Nw                          % Loop over all wages of the kid.
        iWbeqSlope = (Zaln(end,1,i)-Zaln(end-1,1,i))/(log(aup)-log(aup-da));
                                        % Get slope at last grid point and
                                        % extraploate: scalar.
        ExtrapW1246(:,:,:,i,:,:) = InterpBeqi + (log(Beqi)-log(aup))*iWbeqSlope; 
    end
else                                % For power utility:
    % Tried Matlab's cubic and spline extrapolation, but did not work
    % well. Also tried to use homogeneity (see below), but still not
    % precise enough. Best is to rely on linearity of consumption
    % function.                                       
    cc  = Duinvtilde( diff( Zaln )./dak, h0, 2, par);
                                    % (N-1,1,Nw) array. 
    cN  = cc(end,:,:) + ( cc(end,:,:) - cc(end-1,:,:) )/2 ;
                                    % (1,1,Nw) array
    % Calculate pseudo-consumption cci here based on numerical
    % derivative. This is more precise than reading out ekAln since
    % ekAln may sometimes contain forward and sometimes backward
    % derivatives (e.g. if UpWind=true).
            % Best guess of consumption right at the grid boundary aup:
            % When using the backward difference to calculate consumption 
            % at the upper-most grid point we obtain consumption at 
            % "grid point" N-1/2 (where the derivative comes from:
            % difference between grid points N and N-1) which is
            % ekAln(N,.). This is often a good enough approximation for
            % consumption at c(N). For the extrapolation we found that
            % a kink results when relying on this consumption level
            % since it is lower than the one c(N). To get a better
            % approximation we interpolate:
            %         c(N) = c(N-1/2)+[[C(N-1/2)-C(N-3/2)]/da]da/2
            %              = c(N-1/2)+(1/2)[C(N-1/2)-C(N-3/2)]
    daM = avec(end) - avec(Mextrap);    % Difference in assets from Mextrap'th
            % to last grid point: scalar. 
    cSlope = (cc(end,:,:) - cc(Mextrap,:,:) ) / daM; % (1,1,Nw) array.
                                    % Get slope of consumption
                                    % function.
    uShift = Ash(2,h0,par);       % Utility shifter: Kids have two 
                                    % persons in household and house of
                                    % size h0.
    Beq126dOut = reshape(Beq1246dOut(:,:,1,1,1,:),[N,N,1,Nh]);
                                    % Create (N,N,1,Nh) array with possible
                                    % bequests. Send prod. to 3rd
                                    % dimension, which does not matter for 
                                    % bequest. Zaln(N,1,:), cN, cSlope only
                                    % vary on prod. dimension. Can combine
                                    % these to extrapolate in one batch:
    ExtrapW1246 = ExtrapValFctCRRA(Zaln(N,1,:), cN, ...
                  cSlope, Beq126dOut, thetah(h0,par) , uShift);            
                                    % Get extrapolated values when off
                                    % the grid: (N,N,Nw,Nh) array.
	ExtrapW1246 = reshape(ExtrapW1246,[N,N,1,Nw,1,Nh]); % Bring to full 6D dimension.

end

if linGrid==1
    WalnExt1246(Beq1246isOut) = ExtrapW1246(Beq1246isOut);
                                        % Replace bequest value by the
                                        % extrapolated one for points with
                                        % a bequest that jumps off the grid.
    WalnExt =  ones(1,1,Ns,1,Nw,1) .* WalnExt1246;   
                                        % Finally, bring to full 6D size: 
                                        % (N,N,Ns,Nw,Nw,Nh) array.                                        
elseif linGrid==0
    WalnExt = (1-Beq1246isOut).*WalnExt1246 + Beq1246isOut.*ExtrapW1246;
    WalnExt = ones(1,1,Ns,1,Nw,1) .* WalnExt;
end
                                      
if ParAlive==false                      % If this is the parent's last period
    ParAlive =           true;          % alive (dies with certainty at tDth): 
    indDth   =         Nj-j+1;          % Update marker and record age index.
    ll.Wold  =        WalnExt;          % Set continuation value for kids
    ll.Vold  = alphap*WalnExt;          
end                                 

ll = Consumption2(ll,par,opt);          % From last value-function
                                        % iteration (ll.Vold, Wold), get
                                        % unconstrained consumption (epUnc,
                                        % ekUnc), extrapolated derivatives of
                                        % the value function (Vap,Vak, 
                                        % Wap,Wak) and extended arrays for
                                        % the value function (VIold, WIold)
                                        % for updating --> are appended to
                                        % structure ll.

% Set up (N,N,Ns,Nw,Nw,Nh) arrays for laws of motion:
ll.apDot = zeros(N,N,Ns,Nw,Nw,Nh);          % For parent's and  
ll.akDot = zeros(N,N,Ns,Nw,Nw,Nh);          % for kid's assets.


%% Check transfer motives (good indicator for stability):
% Wedge-measure of transfer derivative tau: Tax on transfer (negative means
% subsidy) needed for donor to be indifferent about giving a marginal $ to
% the other. A positive tau means the donor wants to give.
% Get tau for parent and child at indeces where it interests us:
check = [ 1 - ll.Vap(taupEval)./ll.Vak(taupEval); ...
          1 - ll.Wak(taukEval)./ll.Wap(taukEval)     ];
% Get maximal transfer motive: Is a good indicator if code runs smoothly.
% Bad sign if this goes above 1.
% check  = cat(6,taup,tauk);
ll.Max_check  = max(check(:));

   
%%%%%%%%%%%%%%%%%%%
%% OLD IS HEALTHY %
%%%%%%%%%%%%%%%%%%%
% Have to compute transfers as in standard BK. However, we add WP transfers
% (analogous to how we deal with WP regions under bargaining). What we
% calculate in this block for owners is the outside option: What happens if
% house is kept (no bargain reached):
% Fill in structure llHlth (variables for healthy state)

% Make a copy of ll-structure, only healthy state:
llHlth = CopyStruct(ll,indHlth);
% Now, create additional fields that GetBKalloc2.m needs:
% First, create fields IncP and IncK: 
llHlth.IncP  = ll.NetIncPar(:,:,indHlth,:,:,:); 
llHlth.IncK  = ll.NetIncKidFC(:,:,indHlth,:,:,:);
llHlth.NetIncPar = [];   % For clarity, kick out other income fields that
llHlth.NetIncKidFC = []; % CopyStruct.m created.
llHlth.QbarL = -par.QmaxFact*ll.NetIncKidFC(:,:,indHlth,:); % WP transfer from kid to parent
llHlth.QbarU = par.QmaxFact*(ll.NetIncPar(:,:,indHlth,:,:,:)...
               +(r+deltaHouse)*hgrid(:,:,indHlth,:,:,:));   % WP transfer from parent to kid
% Need transfer wedges:                                  
llHlth.taup  = 1 - ll.Vap(:,:,indHlth,:,:,:)./ll.Vak(:,:,indHlth,:,:,:);     % Get tau_p and tau_k: Transfer measured 
llHlth.tauk  = 1 - ll.Wak(:,:,indHlth,:,:,:)./ll.Wap(:,:,indHlth,:,:,:);     % as wedge, i.e.: tau_p is the tax one can 
                          % impose on a transfer given by the parent
                          % (or subsidy, for negative tau_p) for the
                          % parent to be indifferent to transfer to
                          % the child.
llHlth.Ap    = Ash( ll.nnpArr(:,:,indHlth,:,:,:), hgrid, par);   % marginal-utility shifter of parent 
llHlth.Ak    = Ash(nnk,h0,par);                   % and of kid
llHlth.thetap= thetah(hgrid,par);       % (1,1,1,1,1,Nh) array with curvature
                                        % of parent felicity.
llHlth.thetak= thetah(h0,par);  
llHlth.valNoConsFeasible = valNoConsFeasible;  % no consumption is feasible if
                                               % total dynasty income is
                                               % negative and cannot cover 
                                               % housing depreciation for broke 
                                               % dynasties  
llHlth = GetBKalloc2(llHlth,par,BKwp);  % Calculate equilibrium transfers
                                        % and consumption:
                                        % (N,N,1,Nw,Nw,Nh) arrays.
% Still need to calculate drifts resulting from choices:                                        
llHlth.apDot = zeros(N,N,Ns,Nw,Nw,Nh);  % Set up arrays of full size here
llHlth.akDot = zeros(N,N,Ns,Nw,Nw,Nh);  % (but LTC pages won't be used).
llHlth.akDot(:,:,indHlth,:,:,:) = GetAkDot(llHlth.IncK,llHlth.ek,llHlth.gk,llHlth.gp);                                       
llHlth.apDot(:,:,indHlth,:,:,:) = GetApDot(llHlth.IncP,llHlth.ep,llHlth.gk,llHlth.gp);

[hazAkpH, haz345H, maxLeavHaz] = JumpHazards2(llHlth,par,opt);
                                        % Get jump hazards in state space.
% Obtain the value for parent in case she dies:
BeqVal  = alphap*WalnExt;    
                                   % Get value of parent upon death: have
                                    % to adjust bequest value by altruism
                                    % parameter and take into account warm
                                    % glow.
% VHtom: next-period value function conditional on entering healthy 
VpHtom   = ContinuationValue2(hazAkpH, haz345H, ll.VIold, BeqVal , dt, par);
VkHtom   = ContinuationValue2(hazAkpH, haz345H, ll.WIold, WalnExt, dt, par); 
                                    % Calculate the continuation values for
                                    % both agents after interval dt:
                                    % (N,N,Ns,Nw,Nw,Nh) arrays -- full 6D.
% VHlth: current-period value function conditional on remaining healthy  
llHlth.Vp = llHlth.Ucp*dt + alphap*llHlth.Uck*dt + (1-rho*dt).*VpHtom(:,:,indHlth,:,:,:);  
llHlth.Vk = llHlth.Uck*dt + alphak*llHlth.Ucp*dt + (1-rho*dt).*VkHtom(:,:,indHlth,:,:,:); 
                                    % Get stage-2 value functions: 
                                    % (N,N,Ns,Nw,Nw,Nh) arrays -- full 6D.



%%%%%%%%%%%%%%%% 
%% FORMAL CARE %
%%%%%%%%%%%%%%%%
% Case in which the family has opted for formal care. 

% Create copy of ll-structure.
llFC.tp    =    tp;
llFC.nnpArr=  nnpArr(:,:,indBC:indSC,:,:  );  
llFC.epUnc = ll.epUnc(:,:,indBC:indSC,:,:,:); % Fill structure in order to calculate
llFC.ekUnc = ll.ekUnc(:,:,indBC:indSC,:,:,:); % transfers, consumption and MA probability
llFC.Vap   =   ll.Vap(:,:,indBC:indSC,:,:,:); % for FC scenarios (both basic and  
llFC.Vak   =   ll.Vak(:,:,indBC:indSC,:,:,:); % skilled FC, thus have two health 
llFC.Wap   =   ll.Wap(:,:,indBC:indSC,:,:,:); % states.
llFC.Wak   =   ll.Wak(:,:,indBC:indSC,:,:,:); 
llFC.nnk   =                             nnk;
llFC.hp    =                           hgrid; % FC can be renters (NH) or owners (FHC)

% Calculate net income of both agents under formal care:
% Parents have to pay nursing home when renters and FHC when owners
NetIncParFCSick = ll.NetIncPar(:,:,indBC:indSC,:,:,:) - qnet(:,:,indBC:indSC,:,:,:);
% Parent's income in FC: (1,N,2,1,Nw,Nh) array (both basic and skilled FC).
NetIncKidFCSick = ll.NetIncKidFC; % Need to create these variables because we index them
                                  % when computing transfers when constrained
% Kid's income in FC: (N,1,1,Nw,1,1) array. Don't need to distinguish
% basic and skilled FC, same income.

% Create structure for formal care
llFC.IncP =                 NetIncParFCSick; % Under formal care
llFC.IncK =                 NetIncKidFCSick;
% Also need utility shifters/exponents:
llFC.Ap   =              Ash(1  ,hgrid,par); 
llFC.Ak   =              Ash(nnk,h0 ,par);
llFC.thetap =             thetah(hgrid,par);
llFC.thetak =             thetah(h0 ,par);
llFC.valNoConsFeasible =  valNoConsFeasible;
% We add WP transfers to the standard BK regions (analogous on how we deal 
% with WP regions under bargaining):
llFC.QbarL = min(-par.QmaxFact*ll.NetIncKidFC, ...
                 1.1*(ll.NetIncPar(:,:,indBC:indSC,:,:,:) - Cma ) ); % WP transfer from kid to parent
    % Make sure that this transfer at least keeps parent out of MA
llFC.QbarU = par.QmaxFact*(ll.NetIncPar(:,:,1,:,:,:)+(r+deltaHouse)*hgrid);
   % WP transfer from parent to kid. Take this as for healthy for
   % consistency and since it's guaranteed to be positive.                                  
llFC.taup  = 1 - ll.Vap(:,:,indBC:indSC,:,:,:)./ll.Vak(:,:,indBC:indSC,:,:,:);    
llFC.tauk  = 1 - ll.Wak(:,:,indBC:indSC,:,:,:)./ll.Wap(:,:,indBC:indSC,:,:,:);     
                            % Get tau_p and tau_k: Transfer measured 
                            % as wedge, i.e.: tau_p is the tax one can 
                            % impose on a transfer given by the parent
                            % (or subsidy, for negative tau_p) for the
                            % parent to be indifferent to transfer to
                            % the child.

llFC = GetBKallocFloor2(llFC,par,BKwp);  % Obtain equilibrium allocation.

ProbMAf = llFC.ProbMA;              % Read out MA probability.
llFC.ek = MAaverage(ProbMAf,llFC.ekMA ,llFC.ekNoMA );
llFC.ep = MAaverage(ProbMAf,Cma       ,llFC.epNoMA );
llFC.gk = MAaverage(ProbMAf,0         ,llFC.gkNoMA );   % Get gifts of size (N,N,Ns-1,Nw,Nw,Nh) 
llFC.gp = MAaverage(ProbMAf,0         ,llFC.gpNoMA );   % and write into ll.

% Make compatible with the other cases:                                                    
UckF    = MAaverage(ProbMAf,llFC.UckMA,llFC.UckNoMA);
UcpF    = MAaverage(ProbMAf,llFC.UcpMA,llFC.UcpNoMA);                                      
                                    % Take averages of felicity in each
                                    % state of world and average by prob.
% NOTE: Will create laws of motion llFC.akDot and llFC.apDot below together 
% with no-MA case. 
                                    
% Now, create another structure llFCnoMA that tells us what happens in formal 
% care if MA is not chosen at the baseline of Q=0. Will need this for 
% bargaining on the house-selling decision under skilled care (IC
% impossible).
llFCnoMA = CopyStruct(llFC,indBC-1:indSC-1);    % Copy ll.FC structure.
% First, copy fields that were calculated for allocations by
% GetBKallocFloor2.m:
llFCnoMA.ek = llFC.ekNoMA ;     llFCnoMA.ep = llFC.epNoMA;
llFCnoMA.gk = llFC.gkNoMA ;     llFCnoMA.gp = llFC.gpNoMA;
llFCnoMA.Uck= llFC.UckNoMA;     llFCnoMA.Ucp= llFC.UcpNoMA;

% Finally, calculate laws of motion for formal-care (llFC) and
% formal-care-no-MA (llFCnoMA) scenario:
% Set up arrays for laws of motion (need full size here to get hazards):                                    
llFC.akDot    = zeros(N,N,Ns,Nw,Nw,Nh);                                    
llFC.apDot    = zeros(N,N,Ns,Nw,Nw,Nh);  
llFCnoMA.akDot= zeros(N,N,Ns,Nw,Nw,Nh);  
llFCnoMA.apDot= zeros(N,N,Ns,Nw,Nw,Nh);  
% Get laws of motion in private-pay care, i.e. outside MA:
% (N,N,Ns-1,Nw,Nw,Nh) arrays.  
llFCnoMA.akDot(:,:,indBC:indSC,:,:,:) = ...
        GetAkDot(NetIncKidFCSick,llFC.ekNoMA,llFC.gkNoMA,llFC.gpNoMA);
llFCnoMA.apDot(:,:,indBC:indSC,:,:,:) = ...
        GetApDot(NetIncParFCSick,llFC.epNoMA,llFC.gkNoMA,llFC.gpNoMA);
% Get laws of motion in Medicaid:           
akDotMA = NetIncKidFCSick - llFC.ekMA;         % (N,1,Ns-1,Nw,Nw,Nh) array.
apDotMA = 0;
% Get law of motion when parent broke in FC:(N,1,Ns-1,Nw,Nw,Nh) arrays.
akDotFCbroke = MAaverage(ProbMAf,akDotMA,llFCnoMA.akDot(:,1,indBC:indSC,:,:,:));
apDotFCbroke = MAaverage(ProbMAf,apDotMA,llFCnoMA.apDot(:,1,indBC:indSC,:,:,:));
% Write in laws of motion when parent not broke from no-MA case:
llFC.akDot(:,2:N,indBC:indSC,:,:,:) = llFCnoMA.akDot(:,2:N,indBC:indSC,:,:,:);
llFC.apDot(:,2:N,indBC:indSC,:,:,:) = llFCnoMA.apDot(:,2:N,indBC:indSC,:,:,:);
% Write in laws of motion when parent broke:
llFC.akDot(:,  1,indBC:indSC,:,:,:) = akDotFCbroke;
llFC.apDot(:,  1,indBC:indSC,:,:,:) = apDotFCbroke;
% Now, write drifts in no-MA scenario into structure llFCnoMA. Need not
% sepearate broke from non-broke case since this is only necessary under MA
% option:
% llFCnoMA.akDot = akDotPP;   llFCnoMA.apDot = apDotPP;

% Get all hazards of leaving a grid point in the Markov-chain approximation
% method: 
% hazAkp:    (N,N,Ns,Nw,Nw,Nh,4) array: move up or down in the asset dimensions
%            due to savings and Brownian Motion.
% Dimension 7:
%   index 1: akUp, hazard that kid's assets go up.
%   index 2: akDn, kid's assets down.
%   index 3: apUp, parent's assets up.
%   index 4: apDn, parent's assets down.
% haz345: Structure with fields Med, LtcBasic, LtcSkill, Death, each 
%         containing a (Ns,Nw,Nw) array with hazards of medical shock,
%         basic and skilled-LTC shock and death, which depend only on sickness
%         and productivity.
% maxLeavHaz:scalar. maximal hazard rate of leaving a grid point.
[hazAkpF, haz345F, maxLvHazFC] = JumpHazards2(llFC,par,opt);
maxLeavHaz = max(maxLeavHaz,maxLvHazFC);
                                    % Update what the maximal hazard of
                                    % leaving a grid point is in our
                                    % calculations so far. 
% Now, calculate hazards under no-MA scenario:
% llFCnoMA.tp = tp;       llFCnoMA.nnpArr = nnpArr;  !!! WE READ THIS IN
% HERE INCORRECTLY AGAIN
[hazAkpFCnoMA, haz345FCnoMA, ~] = JumpHazards2(llFCnoMA,par,opt);                                   

BeqVal  = alphap*WalnExt;   % + WarmGlow(apgrid,w*HHeffUnits(25,0),par) 
                                    % Get value of parent upon death: have
                                    % to adjust bequest value by altruism
                                    % parameter and take into account warm
                                    % glow. (N,N,Ns,Nw,Nw,Nh) array.
% The continuation value depends on currently receiving FC because the
% price of FC enters the law of motion which changes the jump
% probabilities in regards to the financial asset.
VpFtom   = ContinuationValue2(hazAkpF, haz345F, ll.VIold, BeqVal , dt, par);
VkFtom   = ContinuationValue2(hazAkpF, haz345F, ll.WIold, WalnExt, dt, par); 
% clear BeqVal WalnExt              % Calculate the continuation values for
                                    % both agents after interval dt:
                                    % (N,N,Ns,Nw,Nw,Nh) arrays.
% Current-period value function conditional on currently receiving FC.                                   
llFC.Vp = UcpF*dt + alphap*UckF*dt + (1-rho*dt).*VpFtom(:,:,indBC:indSC,:,:,:);  
llFC.Vk = UckF*dt + alphak*UcpF*dt + (1-rho*dt).*VkFtom(:,:,indBC:indSC,:,:,:);  
                                    % Get Stage-2 values:
                                    % (N,N,Ns-1,Nw,Nw,Nh) arrays.
% Do the same updating for no-MA scenario:                                    
VpFCnoMAtom = ContinuationValue2(hazAkpFCnoMA, haz345FCnoMA, ll.VIold, BeqVal , dt, par);
VkFCnoMAtom = ContinuationValue2(hazAkpFCnoMA, haz345FCnoMA, ll.WIold, WalnExt, dt, par);                                   
llFCnoMA.Vp = llFCnoMA.Ucp*dt + alphap*llFCnoMA.Uck*dt ...
             + (1-rho*dt).*VpFCnoMAtom(:,:,indBC:indSC,:,:,:);  
llFCnoMA.Vk = llFCnoMA.Uck*dt + alphak*llFCnoMA.Ucp*dt ...
             + (1-rho*dt).*VkFCnoMAtom(:,:,indBC:indSC,:,:,:);  
                                    

%%%%%%%%%%%%%%%%%%%
%% INFORMAL CARE %%
%%%%%%%%%%%%%%%%%%%
% Now, do case in which family has opted for informal care. We don't have
% to calculate this in the no-IC counterfactual. Always operate on health
% index 2 here: IC is only an option when requiring basic care.

% Calculate net income of both agents under informal care:
NetIncParICSick = ll.NetIncPar(:,:,indBC,:,:,:);  
NetIncKidICSick = ll.NetIncKidIC;
% Net income for the parent in IC does not include any transfers.
% Kid's income IC: (N,1,1,Nw,1,1) array. Includes opportunity cost of
% providing care in the form of lost labor income of the marginal worker. 

if InfC                             % Unless we are in no-IC counterfactual:
    llIC = CopyStruct(ll,indBC);    % Create copy of ll-structure only with 
                                    % basic-care health state.
    llIC.IncP  =    NetIncParICSick;
    llIC.IncK  =    NetIncKidICSick;
    llIC.h     =              hgrid;    % Fill structure llIC.
    llIC.Ap    = Ash(1  ,hgrid,par) .* ones(1,1,1,1,Nw,1);  % Have to extend
    llIC.Ak    = Ash(nnk, h0,par);    % by housing dimension since GetBKalloc2 will require this.
    llIC.thetap= thetah( hgrid,par);
    llIC.thetak= thetah(  h0,par);
    llIC.nnk   = nnk; 
    llIC.valNoConsFeasible = valNoConsFeasible;

    llIC = GetBKalloc2(llIC,par,0); % Get consumption and gift allocation 
                                    % that would  occur under IC if there
                                    % are no transfers (Q=0).
    % Get drift under IC and zero transfers (need full size here to get hazards):
    llIC.akDot = zeros(N,N,Ns,Nw,Nw,Nh); % Leave zeros on healthy page -- won't be used.         
                                         % Same on skilled FC page.
    llIC.akDot(:,:,indBC,:,:,:) = GetAkDot(llIC.IncK,llIC.ek,llIC.gk,llIC.gp);
    % NOTE: This is for zero exchange-motivated transfers -- these will come in
    % later!
    llIC.apDot = zeros(N,N,Ns,Nw,Nw,Nh);   
    llIC.apDot(:,:,indBC,:,:,:) = GetApDot(llIC.IncP,llIC.ep,llIC.gk,llIC.gp);
    % Get jump hazards under IC and zero transfers:
    [hazAkpI, haz345I, maxLvHazIC] = JumpHazards2(llIC,par,opt);
    maxLeavHaz = max(maxLeavHaz,maxLvHazIC);    % Update the maximal hazard rate of leaving.

    BeqVal  = alphap*WalnExt;           % Get value of parent upon death: have
                                        % to adjust bequest value by altruism
                                        % parameter and take into account warm
                                        % glow.
    % Continuation value conditional on receiving IC in the current period which
    % influences the laws of motion and therefore the jump probabilities in the
    % asset grid.
    VpItom   = ContinuationValue2(hazAkpI, haz345I, ll.VIold, BeqVal , dt, par);
    VkItom   = ContinuationValue2(hazAkpI, haz345I, ll.WIold, WalnExt, dt, par); 
                                        % Calculate the continuation values for
                                        % both agents after interval dt:
                                        % (N,N,Ns,Nw,Nw,Nh) arrays.
    % Get value functions for current IC under zero transfers (Q=0):                                    
    llIC.Vp = llIC.Ucp*dt + alphap*llIC.Uck*dt + (1-rho*dt).*VpItom(:,:,indBC,:,:,:); 
    llIC.Vk = llIC.Uck*dt + alphak*llIC.Ucp*dt + (1-rho*dt).*VkItom(:,:,indBC,:,:,:);  
end                                     % (N,N,1,Nw,Nw,Nh) arrays.

%% IC Bargaining of renters
if InfC
    % Scenario 0:
    % Disabled non-home-owning parent and kid bargain over IC decision. The 
    % value of the outside option is given by llFC.Vp and llFC.Vk and is
    % the result of the parent having to get FC. Renters go into nursing home
    % since this dominates FHC (is more expensive).
    llDisabRent = llIC;                             % Set up structure for bargaining.
    % NOTE: There is no harm in also copying the housing states since
    % GetBargainSolIC only reads out the renting page (first index of
    % dim.6).
    llDisabRent.VpOut    =    llFC.Vp(:,:,indBC-1,:,:,1); % Read out basic-care renter 
    llDisabRent.VkOut    =    llFC.Vk(:,:,indBC-1,:,:,1); % values: outside options.
    llDisabRent.akDotOut = llFC.akDot(:,:,:      ,:,:,1);%  Drift under basic care is on
    llDisabRent.apDotOut = llFC.apDot(:,:,:      ,:,:,1);%  index 2 since the adots have 
                                                    % full size. 
    llDisabRent.akDot    = [];      % Remove drifts from llIC for clarity;
    llDisabRent.apDot    = [];      % this field won't be used any more.
    llDisabRent = GetBargainSolIC(llDisabRent,dt,par,opt);
    % Structure contains post-bargaining objects: (N,N,1,Nw,Nw,1) arrays.
    % VkIn, VpIn: value functions under the inside option whether bargainig 
    %             is successful or not.
    % ekIn, epIn: expenditures under the inside option whether bargainig 
    %             is successful or not.
    % gkIn, gpIn: gifts under the inside option whether bargainig 
    %             is successful or not.
    % akDotIn, apDotIn: drifts under inside option.
    % ibarg     : indicator whether bargaining was successful or not.
    % QiIn      : (non-negative) transfer from parent to kid in return for IC.

    % Apply the smoother to the bargaining decision:
    ProbICBarg  = SmoothX(llDisabRent.ibarg,par); % probabilities of bargaining success
    % Get equilibrium in IC-subgame by averaging over the inside value and
    % the outside value with this probability
    llDisabRent.QiBarg    = ProbICBarg.*llDisabRent.QiIn      ;
    llDisabRent.QPaltrComp= ProbICBarg.*llDisabRent.QPaltrComp;
    llDisabRent.QKaltrComp= ProbICBarg.*llDisabRent.QKaltrComp;
    llDisabRent.VpBarg    = ProbICBarg.*llDisabRent.VpIn    + (1-ProbICBarg).*   llFC.Vp(:,:,indBC-1,:,:,1); % Value of outside  
    llDisabRent.VkBarg    = ProbICBarg.*llDisabRent.VkIn    + (1-ProbICBarg).*   llFC.Vk(:,:,indBC-1,:,:,1); % option is on page
    llDisabRent.apDotBarg = ProbICBarg.*llDisabRent.apDotIn + (1-ProbICBarg).*llFC.apDot(:,:,      :,:,:,1); % 1 in llFC object,
    llDisabRent.akDotBarg = ProbICBarg.*llDisabRent.akDotIn + (1-ProbICBarg).*llFC.akDot(:,:,      :,:,:,1); % but page 2 for 
    llDisabRent.epBarg    = ProbICBarg.*llDisabRent.epIn    + (1-ProbICBarg).*   llFC.ep(:,:,indBC-1,:,:,1); % adots since they
    llDisabRent.ekBarg    = ProbICBarg.*llDisabRent.ekIn    + (1-ProbICBarg).*   llFC.ek(:,:,indBC-1,:,:,1); % have full size by
    llDisabRent.gpBarg    = ProbICBarg.*llDisabRent.gpIn    + (1-ProbICBarg).*   llFC.gp(:,:,indBC-1,:,:,1); % convention.
    llDisabRent.gkBarg    = ProbICBarg.*llDisabRent.gkIn    + (1-ProbICBarg).*   llFC.gk(:,:,indBC-1,:,:,1); % 
    llDisabRent.MABarg    = (1-ProbICBarg(:,1,:,:,:,:)).*llFC.ProbMA(:,:,indBC-1,:,:,1);
else                            % If in No-IC counterfactual:
    ProbICBarg             = zeros(N,N,1,Nw,Nw);    % There is no IC.
    llDisabRent.ibarg      = ProbICBarg;
    llDisabRent.QiBarg     = zeros(N,N,1,Nw,Nw);    % Transfers are zero.
    llDisabRent.QPaltrComp = zeros(N,N,1,Nw,Nw);
    llDisabRent.QKaltrComp = zeros(N,N,1,Nw,Nw);
    llDisabRent.VpBarg     =    llFC.Vp(:,:,:,:,:,1);   % Set FC scenario
    llDisabRent.VkBarg     =    llFC.Vk(:,:,:,:,:,1);   % as outcomes.
    llDisabRent.apDotBarg  = llFC.apDot(:,:,:,:,:,1);   % Indexing as above.
    llDisabRent.akDotBarg  = llFC.akDot(:,:,:,:,:,1);   
    llDisabRent.epBarg     =    llFC.ep(:,:,:,:,:,1);
    llDisabRent.ekBarg     =    llFC.ek(:,:,:,:,:,1);
    llDisabRent.gpBarg     =    llFC.gp(:,:,:,:,:,1);
    llDisabRent.gkBarg     =    llFC.gk(:,:,:,:,:,1);
end


%% Now, do bargaining for owners/force house-selling for skilled-LTC cases:
if Nh==1   % If there are no owners (in liquid-housing 
           % counterfactuals), we can already record the results:
    % (N,N,Ns,Nw,Nw,1) arrays.
    ll.FHC  = false(N,N,1,Nw,Nw,Nh);
    ll.Vold = cat(3,  llHlth.Vp, llDisabRent.VpBarg, llFC.Vp(:,:,indSC-1,:,:) );
    ll.Wold = cat(3,  llHlth.Vk, llDisabRent.VkBarg, llFC.Vk(:,:,indSC-1,:,:) );
    ll.ek   = cat(3,  llHlth.ek, llDisabRent.ekBarg, llFC.ek(:,:,indSC-1,:,:) );
    ll.ep   = cat(3,  llHlth.ep, llDisabRent.epBarg, llFC.ep(:,:,indSC-1,:,:) );  
    ll.gk   = cat(3,  llHlth.gk, llDisabRent.gkBarg, llFC.gk(:,:,indSC-1,:,:) );                        
    ll.gp   = cat(3,  llHlth.gp, llDisabRent.gpBarg, llFC.gp(:,:,indSC-1,:,:) );
    ll.barg = cat(3,  false(N,N,1,Nw,Nw,1),   llDisabRent.ibarg, false(N,N,1,Nw,Nw,1) );   
    ll.akDot = cat(3,          llHlth.akDot(:,:,indHlth,:,:,1),...
                      llDisabRent.akDotBarg(:,:,indBC  ,:,:  ),...
                                 llFC.akDot(:,:,indSC  ,:,:  )    );
    ll.apDot = cat(3,          llHlth.apDot(:,:,indHlth,:,:,1),...
                      llDisabRent.apDotBarg(:,:,indBC  ,:,:  ),...
                                 llFC.apDot(:,:,indSC  ,:,:  )    ); 
    % House is sold if there is no bargaining solution and house sold under 
    % outside option for the healthy; for disabled owners, house is sold
    % if there is no bargaining solution and they don't sell under outside
    % option either:
    ll.xstar = false(N,N,Ns,Nw,Nw,1);        % Nobody can sell.
    ll.Qstar = cat(3, zeros(N,N,1,Nw,Nw,1), llDisabRent.QiBarg,     zeros(N,N,1,Nw,Nw,1));
    ll.QPaltr= cat(3, zeros(N,N,1,Nw,Nw,1), llDisabRent.QPaltrComp, zeros(N,N,1,Nw,Nw,1));
    ll.QKaltr= cat(3, zeros(N,N,1,Nw,Nw,1), llDisabRent.QKaltrComp, zeros(N,N,1,Nw,Nw,1));
                                            % Bargaining transfers.
    ll.istar = llDisabRent.ibarg;           % Informal-care decision.
    ll.ProbMA = cat(3, zeros(N,1,1,Nw,Nw,1), llDisabRent.MABarg, llFC.ProbMA(:,:,indSC-1,:,:)  );
  %  fc = 1 - llDisabRent.ibarg;             % Formal-care decision.

else                            % Now, do case of owners: Nh>1
% We have to extrapolate value
% functions for values outside the parent's wealth grid. We should do this
% based on the observation that consumption functions are close to linear
% for high wealth levels. There is one exception: There is a kink between
% the self-sufficiency region and the kid's over-consumption region. We
% build on the insight that for high wealth levels, the asset share
% P=a^k/(a^p+a^k) determines in which region we are. So we will
% extrapolate consumption functions along rays that emanate from the origin.
% Proceed as follows:

%   THEOREM ON TWO-STAGE BARGAINING:
%   1. GET VALUE FUNCTION FOR DISABLED RENTERS (llDisabRent). 
%   2. EXTRAPOLATE THIS VALUE FUNCTION TO OBTAIN 2ND-STAGE VALUES FOR 
%      DISABLED OWNERS WHO SELL (I.E. EITHER OBTAIN IC OR FC). 
%   3. OBTAIN BARGAINING SOLUTION ON SELLING.

% 1. Create grid with all levels of a^p that the parent can jump to outside
%   the grid:
% For extrapolation of value functions combine healthy renters, basic-care
% renters and skilled-care renters on the 3rd dimension: (N,N,Ns,Nw,Nw) arrays.
VpRent = cat(3,llHlth.Vp(:,:,:,:,:,1),llDisabRent.VpBarg(:,:,:,:,:,1),llFC.Vp(:,:,indSC-1,:,:,1)); % Skilled value: page 2
VkRent = cat(3,llHlth.Vk(:,:,:,:,:,1),llDisabRent.VkBarg(:,:,:,:,:,1),llFC.Vk(:,:,indSC-1,:,:,1)); % in llFC object.                           

if ExtrapHouse                  % If extrapolation for houses is asked for:                             
%   apGridExtrap = (aup+da):da:(aup+Nextrap*da);
                                    % 1-by-Nextrap vector: extrapolation grid
                                    % for parent wealth.
    % 2. Determine what the kid's wealth share is for the extrapolation points:
    nExt = numel(avecExt);          % Read out size of extended grid.
    nExtrap = nExt-N;               % How many extra grid points we have.
    apExtrap = avecExt(1,N+1:nExt); % 1-by-nExtrap vector with the extension
                                    % grid points.
    Pextrap = akgrid./(akgrid+apExtrap); % N-by-nExtrap matrix.
    % Pextrap = akgrid./(akgrid+apGridExtrap);  % old code.
    % 3. By interpolation, find what consumption is for each P in Pextrap for
    %    the following levels of parent's wealth:
    %    a) the boundary of the regular grid: a^p = topA.
    %    b) at a point inside the boundary: a^p = 0.75*topA (but make sure it
    %       lies on grid)
    % 4. Extrapolate consumption linearly on the rays from the origin.
    M = Mextrap;                    % Shorter name for M.
    apM= apgrid(M);                 % a^p at that grid point.

    % Set up arrays for the extrapolated functions for renters who have sold.
    % Lives on dimensions 1-5, extended in the second dimension.
    cpExtrap   = zeros(N,nExtrap,Ns,Nw,Nw); 
    ckPsExtrap = zeros(N,nExtrap,Ns,Nw,Nw); 

    akN= aup*Pextrap./(1-Pextrap);  % Get kid's assets for the P's we have from 
    akM= apM*Pextrap./(1-Pextrap);  % the extrapolation grid on both points:
                                    % N-by-nExtrap matrices.
    for is=1:Ns                     % Loop over all health states.
        for ik=1:Nw                 % Loop over kid's and parent's productivity
           for ip=1:Nw              % levels.
     % Get N-by-1 vector of parent's consumption (when looking left in a^p), 
     % which is a transformation of the a^p-derivative.
               cN    = ( (VpRent(:,N,is,ik,ip)- VpRent(:,N-1,is,ik,ip))/dap(1,N-1) ).^(-1/g);
               cPsN  = ( (VkRent(:,N,is,ik,ip)- VkRent(:,N-1,is,ik,ip))/dap(1,N-1) ).^(-1/g);
                                        % Get pseudo-consumption for kid.
                                        % N-by-1 vectors.
                                        % NOTE: THIS IS THE KID'S DESIRED 
                                        % CONSUMPTION FOR THE PARENT, WITHOUT
                                        % DIVIDING BY ALPHAk SINCE IT IS
                                        % IRRELEVANT.              
               cM    = ( (VpRent(:,M,is,ik,ip)- VpRent(:,M-1,is,ik,ip))/dap(1,M-1) ).^(-1/g);                         
               cPsM  = ( (VkRent(:,M,is,ik,ip)- VkRent(:,M-1,is,ik,ip))/dap(1,M-1) ).^(-1/g);
                                        % Same at point M: N-by-1 vectors.
               cpN = interp1(akgrid,cN,akN,'linear');
                                        % 3a) Get interpolated consumption at a_N
                                        % N-by-nExtrap matrices.
               cpM = interp1(akgrid,cM,akM,'linear');
                                        % 3b) and at a_M.
               ckN = interp1(akgrid,cPsN,akN,'linear');
               ckM = interp1(akgrid,cPsM,akM,'linear');
               
               cpExtrap(:,:,is,ik,ip) = cpN + (apExtrap-aup).*(cpN-cpM)/(aup-apM);
                                        % N-by-nExtrap matrix with
                                        % extrapolated consumption.
               % surf(apExtrap,akgrid,cpExtrap(:,:,is,ik,ip))
               % Checked here, looks very nice and linear.
               % cpExtrap(:,:,is,ik,ip) = cpN + (apGridExtrap-aup).*(cpN-cpM)/(aup-apM);
               % NOTE: The change in consumption cpN-cpM is actually divided by
               % the change in total assets given by (topA-apM)/(1-P). The (1-P)
               % however, cancels out with the extrapolation piece
               %                                      (apGridExtrap-topA)/(1-P),
               % which is why (1-P) does not show up in the extrapolation. For
               % a given P, total assets are ap/(1-P) or ak/P. The change in total
               % assets is thus (topA-apM)/(1-P).
                                        % 4) Extrapolate consumption on the ray.
              ckPsExtrap(:,:,is,ik,ip) = ckN + (apExtrap-aup).*(ckN-ckM)/(aup-apM);                         
             %  surf(apExtrap,akgrid,ckPsExtrap(:,:,is,ik,ip));
             % Checked, looks very nice.
           end                          % Can do surface plot to see if 
        end                             % extrapolation worked --> beautiful in 1st iteration.
    end

    % 5. With the extrapolated consumption function, we can then proceed as in
    %    the function ExtrapValFctCRRA.m: We read off dV^p/da^p from the FOC
    %    and then integrate up the value function from the boundary of the
    %    grid. (For the kid's value function, use a 'pseudo-consumption
    %    function' with FOC: ckPseudo^(-g) = d V^k/d a^p. Extrapolate ckPseudo
    %    the same way as parent's consumption and then integrate up d V^k/d a^p
    %    along the a^p-axis on the extrapolation grid)
    % Use the formula:
    % V({a_1}) = V({a_0}) + \int_{a_0}^{a_1} V'(a) da  
    %       = V_0    + \int_{a_0}^{a_1} A*c(a)^{-g} da 

    VapExtrap  = cpExtrap.^(-g);        % Get slope of value function from 
                                        % FOC for consumption. Array of size
                                        % (N,nExtrap,Ns,Nw,Nw).
                                        % matrix.
    dapExtrap = avecExt(1,N+1:nExt) - avecExt(1,N:nExt-1);
                                        % 1-by-nExtrap vector with
                                        % differences between ap-points on
                                        % extrapolation grid.    
    % Now, Integrate up parent's value function along a^p (maintaining a^k
    % fixed):
    VpExtrap = VpRent(:,N,:,:,:) + cumsum( VapExtrap.*dapExtrap, 2 );                                   
                                        % (N,nExtrap,Ns,Nw,Nw) array
    % surf( [ V2(:,:,2,ik,ip,1),  Vextrap(:,:,1,ik,ip) ] );     % Can check here.   
    WapExtrap  = ckPsExtrap.^(-g);      %  (N,nExtrap,Ns,Nw,Nw) array with 
                                        % kid's value function derivative
                                        % in a^p.
    VkExtrap = VkRent(:,N,:,:,:) + cumsum( WapExtrap.*dapExtrap, 2);
                                        % Finally, get extrapolated kid
                                        % value function: 
                                        % (N,nExtrap,Ns,Nw,Nw) array.    
else                            % If no extrapolation is wanted for houses:
    VpExtrap = VpRent(:,end,:,:,:).*ones(1,nExtrap); % Cannot cash in the 
    VkExtrap = VkRent(:,end,:,:,:).*ones(1,nExtrap); % house -- just copy the
end                             % last values

Vp2C = [VpRent, VpExtrap];      % Add extrapolated values to grid:
Vk2C = [VkRent, VkExtrap];      % (N,nExt,Ns,Nw,Nw) arrays, where we recall
                                % that nExt=N+nExtrap.
sellNewAp = apgrid + hgrid;         % New parent assets upon selling:
                                    % (1,N,1,1,1,Nh) array.
% Now, interpolate sales value. Use the extended value functions just created,
% evaluate at the ap-positions after selling:
VpSaleC = InterpSell(Vp2C,sellNewAp,par);   % (N,N,Ns,Nw,Nw,Nh) arrays.
VkSaleC = InterpSell(Vk2C,sellNewAp,par);
                                
% Now, find values under the outside option: parent unilaterally decides if
% to sell. 
% Combine healthy and FC outcomes to get the value of keeping the house:
% (N,N,Ns,Nw,Nw,Nh) arrays (don't take seriously the renting state!).
VpKeep = cat(3, llHlth.Vp, llFC.Vp); 
VkKeep = cat(3, llHlth.Vk, llFC.Vk);
% Obtain selling decision under outside option either for basic LTC and 
% healthy or ALL health states if home-bargaining also takes place under
% the skilled LTC case: (N,N,Ns,Nw,Nw,Nh) array (don't take renting 
% dimension seriously, we want full-size arrays here since GetBargainSolSell.m 
% wants it that way).
xOut  =VpSaleC  > VpKeep;
VpOut = xOut.*VpSaleC + (~xOut).*VpKeep; 
VkOut = xOut.*VkSaleC + (~xOut).*VkKeep; 

% Now, create structures to obtain bargaining solution for FHC for both 
% basic and skilled LTC states:
llBasicFHC = CopyStruct(llFCnoMA,indBC-1); % Basic LTC state.
% To be safe, set fields akDot and apDot with drifts from before empty:
% These should not be used (only apDotIn, apDotBarg, etc. are used):
llBasicFHC.akDot = [];
llBasicFHC.apDot = [];

% Same for skilled care:
llSkill    = CopyStruct(llFCnoMA,indSC-1);
llSkill.akDot = [];
llSkill.apDot = [];

% Set value of outside option for the skilled LTC: (N,N,1,Nw,Nw,Nh) arrays.                                   
% Under basic care:   
llBasicFHC.xOut  =  xOut(:,:,indBC,:,:,:);  
llBasicFHC.VpOut = VpOut(:,:,indBC,:,:,:);
llBasicFHC.VkOut = VkOut(:,:,indBC,:,:,:);

% Under skilled care:   
llSkill.xOut  =  xOut(:,:,indSC,:,:,:);
llSkill.VpOut = VpOut(:,:,indSC,:,:,:);
llSkill.VkOut = VkOut(:,:,indSC,:,:,:);

if SCmustSell                               % If parent must sell under skilled care,
                                            % only need outside options for
                                            % healthy and basic LTC state:
    SCfdNames = {'xbarg','QxIn','gpIn','gkIn','QPaltrComp','QKaltrComp','VpIn','VkIn',...
                 'epIn','ekIn','apDotIn','akDotIn'};
    for ifd=1:numel(SCfdNames) 
        llSkill.(SCfdNames{ifd}) = zeros(N,N,1,Nw,Nw,Nh-1);% Set bargaining solution to zero,
    end                                     % as well as all other fields 
                                            % (won't be used)

else                                        % Otherwise, get outside option 
                                            % for all three states.    
    llSkill.h    = par.hgrid;
    llBasicFHC.h = par.hgrid;
    llSkill      = GetBargainSolSell(llSkill,   1,2:Nh,dt,par,opt);
    llBasicFHC   = GetBargainSolSell(llBasicFHC,1,2:Nh,dt,par,opt);
    % Here we solve the game that occurs if home is not sold and parent 
    % gets FHC. For this, the 'baseline' scenario is formal care
    % with bargaining transfer Q=0.
end

ProbSkillBarg = SmoothX(llSkill.xbarg,par);     % Smooth bargaining decision,
                                                % if smoothing on
                                                % (otherwise, have 0-1
                                                % array here).
ProbBasicFHCBarg = SmoothX(llBasicFHC.xbarg,par);  
llBasicFHC.xbarg = ProbBasicFHCBarg;
llSkill.xbarg    = ProbSkillBarg;
% Smooth value functions and other outcomes by taking the average between 
% the inside value and the outside value: (N,N,1,Nw,Nw,Nh-1) arrays.
llSkill.QxBarg     = ProbSkillBarg.*llSkill.QxIn      ;
llSkill.QPaltrComp = ProbSkillBarg.*llSkill.QPaltrComp;
llSkill.QKaltrComp = ProbSkillBarg.*llSkill.QKaltrComp;
llSkill.VpBarg     = ProbSkillBarg.*llSkill.VpIn + (1-ProbSkillBarg).*llSkill.VpOut(:,:,:,:,:,2:Nh);
llSkill.VkBarg     = ProbSkillBarg.*llSkill.VkIn + (1-ProbSkillBarg).*llSkill.VkOut(:,:,:,:,:,2:Nh);

llBasicFHC.QxBarg     = ProbBasicFHCBarg.*llBasicFHC.QxIn      ;
llBasicFHC.QPaltrComp = ProbBasicFHCBarg.*llBasicFHC.QPaltrComp;    
llBasicFHC.QKaltrComp = ProbBasicFHCBarg.*llBasicFHC.QKaltrComp;
llBasicFHC.VpBarg     = ProbBasicFHCBarg.*llBasicFHC.VpIn + (1-ProbBasicFHCBarg).*llBasicFHC.VpOut(:,:,:,:,:,2:Nh);
llBasicFHC.VkBarg     = ProbBasicFHCBarg.*llBasicFHC.VkIn + (1-ProbBasicFHCBarg).*llBasicFHC.VkOut(:,:,:,:,:,2:Nh);
% For consumption, gifts etc.: Write in what happens under inside option
% when bargaining successful; write in what happens in scenario that house
% is not sold (i.e.: FHC takes place) for outside option. Note that for the
% case when bargaining not successful and the parent then SELLS, we
% will have zero density and the relevant policies will be taken from 
% when it comes to calculating moments and when mapping the density
% forward. 
llSkill.apDotBarg  = ProbSkillBarg.*llSkill.apDotIn + (1-ProbSkillBarg).*llFC.apDot(:,:,indSC  ,:,:,2:Nh); % Outside option: health index 2 
llSkill.akDotBarg  = ProbSkillBarg.*llSkill.akDotIn + (1-ProbSkillBarg).*llFC.akDot(:,:,indSC  ,:,:,2:Nh); % for adots since they have full dimension.
llSkill.epBarg     = ProbSkillBarg.*llSkill.epIn    + (1-ProbSkillBarg).*   llFC.ep(:,:,indSC-1,:,:,2:Nh); % In llFC, basic care is 
llSkill.ekBarg     = ProbSkillBarg.*llSkill.ekIn    + (1-ProbSkillBarg).*   llFC.ek(:,:,indSC-1,:,:,2:Nh); % index 1.
llSkill.gpBarg     = ProbSkillBarg.*llSkill.gpIn    + (1-ProbSkillBarg).*   llFC.gp(:,:,indSC-1,:,:,2:Nh); 
llSkill.gkBarg     = ProbSkillBarg.*llSkill.gkIn    + (1-ProbSkillBarg).*   llFC.gk(:,:,indSC-1,:,:,2:Nh); 
llSkill.MABarg     =                                  (1-ProbSkillBarg(:,1,:,:,:,:)).*llFC.ProbMA(:,:,indSC-1,:,:,2:Nh);

llBasicFHC.apDotBarg  = ProbBasicFHCBarg.*llBasicFHC.apDotIn + (1-ProbBasicFHCBarg).*llFC.apDot(:,:,indBC  ,:,:,2:Nh); % Outside option: health index 2 
llBasicFHC.akDotBarg  = ProbBasicFHCBarg.*llBasicFHC.akDotIn + (1-ProbBasicFHCBarg).*llFC.akDot(:,:,indBC  ,:,:,2:Nh); % for adots since they have full dimension.
llBasicFHC.epBarg     = ProbBasicFHCBarg.*llBasicFHC.epIn    + (1-ProbBasicFHCBarg).*   llFC.ep(:,:,indBC-1,:,:,2:Nh); % In llFC, BasicFHC care is 
llBasicFHC.ekBarg     = ProbBasicFHCBarg.*llBasicFHC.ekIn    + (1-ProbBasicFHCBarg).*   llFC.ek(:,:,indBC-1,:,:,2:Nh); % index 1.
llBasicFHC.gpBarg     = ProbBasicFHCBarg.*llBasicFHC.gpIn    + (1-ProbBasicFHCBarg).*   llFC.gp(:,:,indBC-1,:,:,2:Nh); 
llBasicFHC.gkBarg     = ProbBasicFHCBarg.*llBasicFHC.gkIn    + (1-ProbBasicFHCBarg).*   llFC.gk(:,:,indBC-1,:,:,2:Nh); 
llBasicFHC.MABarg     =                                        (1-ProbBasicFHCBarg(:,1,:,:,:,:)).*llFC.ProbMA(:,:,indBC-1,:,:,2:Nh);

% MA does not take place if bargain successfull; take MA prob. from FC-
% scenario if bargain not successful. Only store for broke parents.
llHlth.xOut =  xOut(:,:,indHlth,:,:,:);
llHlth.VpOut= VpOut(:,:,indHlth,:,:,:);
llHlth.VkOut= VkOut(:,:,indHlth,:,:,:);
llHlth.h    = hgrid;

% Scenario 1:
% Healthy home-owning parent and kid bargain over house-selling decision. 
% The value of the outside option is given by llHlth.VpOut and llHlth.Vkout 
% and is the result of whatever the parent would do unilaterally (keep or sell).  
llHlth = GetBargainSolSell(llHlth,1,2:Nh,dt,par,opt);
% Structure now also contains post-bargaining objects:
% (N,N,1,Nw,Nw,Nh-1) arrays.
% VkIn, VpIn: Value functions under the inside option whether bargainig 
%             is successful or not.
% ekIn, epIn: Expenditures under the inside option whether bargainig 
%             is successful or not.
% gkIn, gpIn: gifts under the inside option whether bargainig 
%             is successful or not.
% akDotIn, apDotIn: Drifts under inside option.
% xbarg     : Whether bargaining was successful or not.
% QxIn      : (Non-positive) transfer from kid to parent in return for 
%             parent keeping the house.
% Apply the smoother to the bargaining decision:
ProbHlthyBarg  = SmoothX(llHlth.xbarg,par); % Probabilities of bargaining success
llHlth.xbarg   = ProbHlthyBarg;
% Smooth value functions by taking the average between the inside value and
% the outside value:
% In Scenario 1, renters do not bargain since they are healthy and have no
% home. So the average is only taken over homeowners
llHlth.QxBarg    = ProbHlthyBarg.*llHlth.QxIn      ; % Q is zero under outside option.
llHlth.QKaltrComp= ProbHlthyBarg.*llHlth.QKaltrComp; % Altruistic components.
llHlth.QPaltrComp= ProbHlthyBarg.*llHlth.QPaltrComp;
llHlth.VpBarg    = ProbHlthyBarg.*llHlth.VpIn + (1-ProbHlthyBarg).*llHlth.VpOut(:,:,:,:,:,2:Nh);
llHlth.VkBarg    = ProbHlthyBarg.*llHlth.VkIn + (1-ProbHlthyBarg).*llHlth.VkOut(:,:,:,:,:,2:Nh);
% Assign the inside actions for those who stay as owners. Assign the outside
% option for the others: the relevant case here is if parent does not sell
% under outside option, for these we need the right outcomes. For those who
% really end up selling, it does not matter what we write in here since
% density will be zero/no panel member will ever reach this state.
llHlth.apDotBarg = ProbHlthyBarg.*llHlth.apDotIn + (1-ProbHlthyBarg).*llHlth.apDot(:,:,1,:,:,2:Nh);  % apDot, akDot, ep, ek, gp, gk were set
llHlth.akDotBarg = ProbHlthyBarg.*llHlth.akDotIn + (1-ProbHlthyBarg).*llHlth.akDot(:,:,1,:,:,2:Nh);  % to the outside option values before, 
llHlth.epBarg    = ProbHlthyBarg.*llHlth.epIn    + (1-ProbHlthyBarg).*llHlth.ep(   :,:,:,:,:,2:Nh);  % so use these.
llHlth.ekBarg    = ProbHlthyBarg.*llHlth.ekIn    + (1-ProbHlthyBarg).*llHlth.ek(   :,:,:,:,:,2:Nh);
llHlth.gpBarg    = ProbHlthyBarg.*llHlth.gpIn    + (1-ProbHlthyBarg).*llHlth.gp(   :,:,:,:,:,2:Nh);
llHlth.gkBarg    = ProbHlthyBarg.*llHlth.gkIn    + (1-ProbHlthyBarg).*llHlth.gk(   :,:,:,:,:,2:Nh);


if InfC                                     % If IC is feasible (normal case):
    % Scenario 2:
    % Disabled home-owning parent and kid bargain over house-selling decision.
    % The value of the outside option is determined by SCENARIO 0 (IC & rent, 
    % or FC).
    llDisabOwn = llIC;                          % Create structure for bargaining:
    % Have to take llIC since this gives us the policies under inside
    % option (IC) for the scenario Q=0. GetBargainSolSell.m then calculates
    % given those what values would be for non-zero Q-transfers.
    llDisabOwn.VpOut    = VpOut(:,:,indBC,:,:,:);   % Tell it the outside values: 
    llDisabOwn.VkOut    = VkOut(:,:,indBC,:,:,:);   % (N,N,1,Nw,Nw,Nh).
    llDisabOwn.xOut     =  xOut(:,:,indBC,:,:,:);
    % Note: We don't have to tell GetBargainSolSell.m if house is sold
    % under outside option since there is ALWAYS something to bargain on:
    % IC. 
  
    % Get bargaining solution: The function adds fields epIn, ekIn, VpIn etc.
    % for what happens under the inside option. Field xbarg tells us if
    % bargaining was successful.
    llDisabOwn = GetBargainSolSell(llDisabOwn,2,2:Nh,dt,par,opt);

else  % ??? POSTPONED THIS BLOCK.               % If we are in the no-IC counterfactual: 
    % First, we have to solve the game that occurs if home is not sold
    % and parent gets FHC. For this, the 'baseline' scenario is formal care
    % with bargaining transfer Q=0.
    llFHC = llFC;               % Inherits most properties from FC...
    llFHC.Ap = llFC.Ap .* ones(1,1,1,1,Nw,1);   % Bring utility shifter to larger size.
    llFHC.h  = hgrid(:,:,1,:,:,:);
    llFHC = GetBKalloc2(llFHC,par,1);           % Solve for what transfers are.
    llDisabOwn = llFHC;          % Now, formal home care is the inside option.
                                % Under outside option, sell home and go to NH.
    llDisabOwn.xOut     =  xOut(:,:,2,:,:,:);   % Set outside options for IC
    llDisabOwn.VpOut    = VpOut(:,:,2,:,:,:);   % Set outside values and
    llDisabOwn.VkOut    = VkOut(:,:,2,:,:,:);   % drifts, as before.
                                % Do bargaining ONLY on selling, not IC
                                % now: Scenario 1                                
    % No IC bargaining solution in this case:
    llDisabOwn.xbarg = zeros(N,N,1,Nw,Nw,Nh-1);
    llDisabOwn.VpIn  = zeros(N,N,1,Nw,Nw,Nh-1);
    llDisabOwn.VkIn  = zeros(N,N,1,Nw,Nw,Nh-1);
    llDisabOwn.epIn  = zeros(N,N,1,Nw,Nw,Nh-1);
    llDisabOwn.ekIn  = zeros(N,N,1,Nw,Nw,Nh-1);
    llDisabOwn.QxIn  = zeros(N,N,1,Nw,Nw,Nh-1);
    llDisabOwn.apDotIn    = zeros(N,N,1,Nw,Nw,Nh-1);
    llDisabOwn.akDotIn    = zeros(N,N,1,Nw,Nw,Nh-1);
    llDisabOwn.QPaltrComp = zeros(N,N,1,Nw,Nw,Nh-1);
    llDisabOwn.QKaltrComp = zeros(N,N,1,Nw,Nw,Nh-1);
end

% Apply the smoother to the bargaining decision:
ProbDisabBarg  = SmoothX(llDisabOwn.xbarg,par); % Probabilities of bargaining success for owners: (N,N,1,Nw,Nw,Nh-1) array.
%llDisabOwn.xbarg = ProbDisabBarg;
% We now have two inside options: IC & keep and FHC & keep
noBargSol  = (1-ProbDisabBarg).*(1-ProbBasicFHCBarg); % no inside option is chosen
ICbargSol  =    ProbDisabBarg .*(1-ProbBasicFHCBarg); % only IC is chosen
FHCbargSol = (1-ProbDisabBarg).*   ProbBasicFHCBarg ; % only FHC is chosen
BOTHbargSol=    ProbDisabBarg .*   ProbBasicFHCBarg ; % both IC and FHC are chosen over outside option 
% ************************
if strcmp(par.bargainMode,'xout')|| strcmp(par.bargainMode,'kid')
    ICpref     = (llDisabOwn.VkIn >=llBasicFHC.VkIn).*BOTHbargSol; % value comparison: IC preferred to FHC
    FHCpref    = (llDisabOwn.VkIn < llBasicFHC.VkIn).*BOTHbargSol; % value comparison: FHC preferred to IC
elseif strcmp(par.bargainMode,'parent')
    ICpref     = (llDisabOwn.VpIn >=llBasicFHC.VpIn).*BOTHbargSol; % value comparison: IC preferred to FHC
    FHCpref    = (llDisabOwn.VpIn < llBasicFHC.VpIn).*BOTHbargSol; % value comparison: FHC preferred to IC
end
% These two lines are correct under the assumption bargainMode=xout (what
% we currently have) and bargainMode=kid, but not for the other bargaining
% modes. The following explains why:
% 1. If parent KEEPS under outside option, then FHC is never a bargaining 
%    solution -- only ICbargSol=true can occur, not BOTHbargSol=true.
% 2. If parent SELLS under outside option, then (and only then)
%    BOTHbargSol=true can occur. Under bargaining mode 'xout' (and also
%    'kid') this means that kid has bargaining power in IC bargaining. Since
%    kid also always has bargaining power in FHC bargaining, it means the
%    power sits with the kid in both situations and thus it is OK to just
%    compare child surplus. Under
%    bargaining modes 'parent' and 'adhoc', though, bargaing power sits
%    with parent in IC negotiation, so the above is NOT correct.
% ************************
llDisabOwn.xbarg = 1-noBargSol; % record if there is any bargaining solution
% Have three outcomes:
ICoutc =  ICbargSol+ ICpref;        % Prob. of IC outcome.
FHCoutc= FHCbargSol+FHCpref;        % Prob. of FHC outcome.
% Prob. of outside option is noBargSol.
llDisabOwn.QxBarg     = ICoutc.*llDisabOwn.QxIn      + FHCoutc.*llBasicFHC.QxIn; 
llDisabOwn.QPaltrComp = ICoutc.*llDisabOwn.QPaltrComp+ FHCoutc.*llBasicFHC.QPaltrComp;
llDisabOwn.QKaltrComp = ICoutc.*llDisabOwn.QKaltrComp+ FHCoutc.*llBasicFHC.QKaltrComp;
llDisabOwn.VpBarg     = ICoutc.*llDisabOwn.VpIn      + FHCoutc.*llBasicFHC.VpIn      + noBargSol.*llDisabOwn.VpOut(:,:,:,:,:,2:Nh);
llDisabOwn.VkBarg     = ICoutc.*llDisabOwn.VkIn      + FHCoutc.*llBasicFHC.VkIn      + noBargSol.*llDisabOwn.VkOut(:,:,:,:,:,2:Nh);

% For consumption, gifts etc.: Write in what happens under inside option
% when bargaining successful; write in what happens in scenario that house
% is not sold (i.e.: FHC takes place) for outside option. Note that for the
% case when when bargaining not successful and the parent then SELLS, we
% will have zero density and the relevant policies will be taken from 
% when it comes to calculating moments and when mapping the density
% forward. 
llDisabOwn.apDotBarg  = ICoutc.*llDisabOwn.apDotIn + FHCoutc.*llBasicFHC.apDotIn + noBargSol.*llFC.apDot(:,:,indBC  ,:,:,2:Nh); % Outside option: health index 2 
llDisabOwn.akDotBarg  = ICoutc.*llDisabOwn.akDotIn + FHCoutc.*llBasicFHC.akDotIn + noBargSol.*llFC.akDot(:,:,indBC  ,:,:,2:Nh); % for adots since they have full dimension.
llDisabOwn.epBarg     = ICoutc.*llDisabOwn.epIn    + FHCoutc.*llBasicFHC.epIn    + noBargSol.*llFC.ep(   :,:,indBC-1,:,:,2:Nh); % In llFC, basic care is 
llDisabOwn.ekBarg     = ICoutc.*llDisabOwn.ekIn    + FHCoutc.*llBasicFHC.ekIn    + noBargSol.*llFC.ek(   :,:,indBC-1,:,:,2:Nh); % index 1.
llDisabOwn.gpBarg     = ICoutc.*llDisabOwn.gpIn    + FHCoutc.*llBasicFHC.gpIn    + noBargSol.*llFC.gp(   :,:,indBC-1,:,:,2:Nh); 
llDisabOwn.gkBarg     = ICoutc.*llDisabOwn.gkIn    + FHCoutc.*llBasicFHC.gkIn    + noBargSol.*llFC.gk(   :,:,indBC-1,:,:,2:Nh); 
llDisabOwn.MABarg     = noBargSol(:,1,:,:,:,:).*llFC.ProbMA(:,:,indBC-1,:,:,2:Nh); 

% No MA if bargain successful, take MA prob. from FC scenario if bargaining
% not successful. Only store for broke parents.

% Function that combines all outcomes: Concatenate renter and owner
% outcomes in housing dimension (6) and then concatenate healthy and LTC
% outcomes in LTC dimension (3):
CombineBargOutcomes = @(outcHlthRent    , outcHlthOwn    ,...
                        outcLtcBasicRent, outcLtcBasicOwn, ...
                        outcLtcSkillRent, outcLtcSkillOwn      ) ...
    cat(3, cat(6,outcHlthRent    ,outcHlthOwn    ) , ...
           cat(6,outcLtcBasicRent,outcLtcBasicOwn) , ...
           cat(6,outcLtcSkillRent,outcLtcSkillOwn)        );
% Now, use function to assign final outcomes to structure ll: 
% (N,N,Ns,Nw,Nw,Nh) arrays.
ll.Vold = CombineBargOutcomes(llHlth.Vp(:,:,1,:,:,1), llHlth.VpBarg, ...             % healthy renters and owners (renter/owners concatenate on dim 6)
                              llDisabRent.VpBarg, llDisabOwn.VpBarg, ...             % basic LTC renters and owners
                              llFC.Vp(:,:,2,:,:,1),  llSkill.VpBarg     );      % skilled LTC renters and owners (owners must sell)
ll.Wold = CombineBargOutcomes(llHlth.Vk(:,:,1,:,:,1), llHlth.VkBarg, ...
                              llDisabRent.VkBarg, llDisabOwn.VkBarg, ...
                              llFC.Vk(:,:,2,:,:,1),  llSkill.VkBarg     );
ll.ek   = CombineBargOutcomes(llHlth.ek(:,:,1,:,:,1), llHlth.ekBarg, ...
                              llDisabRent.ekBarg, llDisabOwn.ekBarg, ...
                              llFC.ek(:,:,2,:,:,1),  llSkill.ekBarg     );
ll.ep   = CombineBargOutcomes(llHlth.ep(:,:,1,:,:,1), llHlth.epBarg, ...
                              llDisabRent.epBarg, llDisabOwn.epBarg, ...
                              llFC.ep(:,:,2,:,:,1),  llSkill.epBarg     );  
ll.gk   = CombineBargOutcomes(llHlth.gk(:,:,1,:,:,1), llHlth.gkBarg, ...
                              llDisabRent.gkBarg, llDisabOwn.gkBarg, ...
                              llFC.gk(:,:,2,:,:,1),  llSkill.gkBarg     );                  
ll.gp   = CombineBargOutcomes(llHlth.gp(:,:,1,:,:,1), llHlth.gpBarg, ...
                              llDisabRent.gpBarg, llDisabOwn.gpBarg, ...
                              llFC.gp(:,:,2,:,:,1),  llSkill.gpBarg     );
ll.barg = CombineBargOutcomes(false(N,N,1,Nw,Nw,1),    llHlth.xbarg, ...
                              llDisabRent.ibarg,   llDisabOwn.xbarg, ...
                              false(N,N,1,Nw,Nw,1),   llSkill.xbarg     ); % No bargaining for skilled-LTC cases and healthy renters.
ll.akDot = CombineBargOutcomes(llHlth.akDot(:,:,1,:,:,1),           llHlth.akDotBarg,...  % akDot and apDot have full health dimension
                               llDisabRent.akDotBarg(:,:,1,:,:),llDisabOwn.akDotBarg,...
                               llFC.akDot(:,:,Ns,:,:,1),           llSkill.akDotBarg    ); 
ll.apDot = CombineBargOutcomes(llHlth.apDot(:,:,1,:,:,1),           llHlth.apDotBarg,...
                               llDisabRent.apDotBarg(:,:,1,:,:),llDisabOwn.apDotBarg,...
                               llFC.apDot(:,:,Ns,:,:,1),           llSkill.apDotBarg    ); 
% House is sold if there is no bargaining solution and house sold under 
% outside option for the healthy; for basic-LTC owners, house is sold
% if there is no bargaining solution and they don't sell under outside
% option either. Skilled-LTC owners are forced to sell.
ll.xstar = CombineBargOutcomes( false(N,N,1,Nw,Nw,1),     ~llHlth.xbarg & xOut(:,:,1,:,:,2:Nh),...                               
                                false(N,N,1,Nw,Nw,1), ~llDisabOwn.xbarg & xOut(:,:,2,:,:,2:Nh),...
                                false(N,N,1,Nw,Nw,1),    ~llSkill.xbarg & xOut(:,:,3,:,:,2:Nh)    );           
% Bargaining transfer: Zeros for healthy renters and skilled-LTC renters 
% since there is no house and no IC.                          
ll.Qstar=CombineBargOutcomes(zeros(N,N,1,Nw,Nw,1),     llHlth.QxBarg,...
                               llDisabRent.QiBarg, llDisabOwn.QxBarg,...
                             zeros(N,N,1,Nw,Nw,1),    llSkill.QxBarg    );
ll.QPaltr = CombineBargOutcomes(zeros(N,N,1,Nw,Nw,1), llHlth.QPaltrComp,...
                               llDisabRent.QPaltrComp, llDisabOwn.QPaltrComp,...
                              zeros(N,N,1,Nw,Nw,1),    llSkill.QPaltrComp    );                     
ll.QKaltr = CombineBargOutcomes(zeros(N,N,1,Nw,Nw,1), llHlth.QKaltrComp,...
                               llDisabRent.QKaltrComp, llDisabOwn.QKaltrComp,...
                             zeros(N,N,1,Nw,Nw,1),    llSkill.QKaltrComp    );  
% MA prob.: Only store for broke parents.
ll.ProbMA = CombineBargOutcomes(zeros(N,1,1,Nw,Nw,1), zeros(N,1,1,Nw,Nw,Nh-1),...   % zero for the healthy.
                               llDisabRent.MABarg,       llDisabOwn.MABarg, ...
                               llFC.ProbMA(:,:,indSC-1,:,:,1), llSkill.MABarg   );
% (N,N,1,Nw,Nw,Nh) array with IC decision (only record for basic-LTC
% state):
if InfC                         % In standard case: occurs if bargaining is successful.
   %ll.istar = cat( 6, llDisabRent.ibarg, llDisabOwn.xbarg);
   ll.istar = cat( 6, llDisabRent.ibarg,   ICbargSol +ICpref);
   ll.FHC   = cat(6,  false(N,N,1,Nw,Nw,1),FHCbargSol+FHCpref);
else                                    % No-IC counterfactual:
   ll.istar = zeros(N,N,1   ,Nw,Nw,Nh); % IC never occurs.
%   fc       =  ones(N,N,Ns-1,Nw,Nw,Nh); % Formal care always occurs.
end

end
% Now, we go back to general case: 
  
% Get the jump hazards for the entire equilibrium object (need these for
% density iteration -- could piece together from the other ll-objects, but
% computing them anew is eallFC.ProbMAsier to code and does not lose much time).
[hazAkp, haz345, hazLvEqm] = JumpHazards2(ll,par,opt);                                 
% hazAkp:    (N,N,Ns,Nw,Nw,Nh,4) array: move up or down in the asset dimensions
%            due to savings and Brownian Motion.
% haz345:    Structure with fields containing hazard rates: 
%            Med:        medical shock to parent
%            LtcBasic:   disabled, need basic care services
%            LtcSkill:   disabled, need skilled care services
%            Death:      death
%            All hazards are (Ns,Nw,Nw) arrays. Dimensions:
%                       dim. 1: disability (healthy, LTC basic, LTC sk.)
%                       dim. 2: kid's productivity
%                       dim. 3: parent's productivity(2,Nw,Nw,3) 
maxLeavHaz = max(maxLeavHaz,hazLvEqm);  % Update maximal hazard rate at which 
% a point is left. Note: We neglect scenarios off the eq'm path here for
% which stability condition may not hold -- do not calculate these for now,
% should work. 


%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% SAVE RESULTS FOR LATER %
%%%%%%%%%%%%%%%%%%%%%%%%%%%

logCnew = log( [ll.ep(:); ll.ek(:)] );          % Get largest percentage change (yearly)
Cdist   = max(abs( (logCnew-logCold) / dt ));   % in consumption.
logCold = logCnew;                              % Update loop variable.
minPm      = min(1-maxLeavHaz*dt,minPm);        % Get minimal P_middle: The 
                                                % PDE stability criterion.

%% Store results every age iteration:
   store.tp(            Nj-j+1) =                     tp;
store.check(            Nj-j+1) =           ll.Max_check;
store.Cdist(            Nj-j+1) =                  Cdist;
% Now, regular-size variables: Don't do loop since Vold and Wold have
% special names...
   store.ep(:,:,:,:,:,:,Nj-j+1) =                  ll.ep; % Store consumption of parent 
   store.gp(:,:,:,:,:,:,Nj-j+1) =                  ll.gp; % when sick: need this to get 
   store.gk(:,:,:,:,:,:,Nj-j+1) =                  ll.gk; % private nursing-home spending.
   store.MA(:,:,:,:,:,:,Nj-j+1) = ll.ProbMA(:,:,indBC:indSC,:,:,:); % For ProbMA, only store
store.Qstar(:,:,:,:,:,:,Nj-j+1) =               ll.Qstar; % in LTC states and for broke 
store.QKaltrComp(:,:,:,:,:,:,Nj-j+1) =         ll.QKaltr; % parent.
store.QPaltrComp(:,:,:,:,:,:,Nj-j+1) =         ll.QPaltr;
store.istar(:,:,:,:,:,:,Nj-j+1) =               ll.istar;
store.FHC(:,:,:,:,:,:,Nj-j+1)   =               ll.FHC;
store.xstar(:,:,:,:,:,:,Nj-j+1) =               ll.xstar;
    store.V(:,:,:,:,:,:,Nj-j+1) =                ll.Vold;
    store.W(:,:,:,:,:,:,Nj-j+1) =                ll.Wold;
store.akDot(:,:,:,:,:,:,Nj-j+1) =               ll.akDot;
store.apDot(:,:,:,:,:,:,Nj-j+1) =               ll.apDot;
store.haz345Arr(:,:,:,:,Nj-j+1) = cat(4,haz345.Med,haz345.LtcBasic, haz345.LtcSkill,haz345.Death);  
   store.haz345{        Nj-j+1} =                 haz345;    % Put structure into field of cell. 
% hazAkp has 8 dimensions:           
store.hazAkp(:,:,:,:,:,:,:,Nj-j+1)=               hazAkp;


end                                         % End of if-statement: parent alive.

                                       
store.minPm(Nj-j+1) = minPm;            
   store.tp(Nj-j+1) = tp;
   store.dt(Nj-j+1) = dt;

if mod(j,DisplayEvery)==0 && ParAlive       % Display progress.          
    fprintf('Iteration %1.0f, old is %1.1f\n', ...
                     [ j             tp    ])
end

if tp < tpStop                      % Stop if user asked to do so.
    fprintf('Stopped in iteration %1.0f\n',j)
    keyboard
end


end                                 % End of age-iteration loop
                                    % inside one OLG iteration.
fprintf('Value-function age loop done in %1.0f iterations.\n',j)


ZBefMatchVec = MatchMat*ll.Vold(:); % Value of new parents at age 65 before
                                    % being matched to kids, but after having 
                                    % bought a house, in vectorized
                                    % form. Use transition matrix constructed 
                                    % before loop.
                                    % N*Nw*Nh-by-1 column vector.
ZBefMatch = reshape(ZBefMatchVec,[N,1,Nw,Nh]);
                                    % Bring to array form:
                                    % (N,1,Nw,Nh) array.
if Nh==1                            % For liquid-housing counterfactual:
    ZfinNew = ZBefMatch;            % We already have the new final value.
else                                % If people can buy houses:
    [ZfinNew, hpol] = BuyHouseValue(ZBefMatch,par,iterOLG);
end                                 % Now, get value of the new retirees
                                    % before buying the house:
                                    % (N,1,Nw) array. 
                                    % HouseTransMat is a N*Nw-by-N*Nw*Nh
                                    % sparse matrix containing the housing
                                    % decisions.
close all;                                    
eFinNew = ConsumptionAln(ZfinNew,h0,2,NetIncFin,par);
                                    % Finally, get optimal consumption at this
                                    % stage (household size = 2 at tRet).
distOLG = max(abs( log(eFinNew(:)) - log(eFin(:)) ));
                                    % Get distance between final value 
                                    % functions measured in percentage of 
                                    % optimal consumption.
fprintf('MAX. DISTANCE OF CONSUMPTION FUNCTIONS: %1.1f%%.\n',100*distOLG);                             
                             
Zfin = ZfinNew;                     % Update loop variables; set final 
eFin = eFinNew;                     % value as the new starting condition.
Zaln = Zfin;                       

end                                 % End of OLG iteration loop.

% if Nh>1                             % Nh=1 is the lquid counterfactual
%     if ~noKid                           % For counterfactuals with kids:
%         disp('Obtaining wealth equivalent variation (WEV) for housing...')
%         res.wev = HouseWEV(Zfin,ZBefMatch(:,1,:,1),akGridAln,0,1,HumanW65);
%         % Get wealth equivalent variation for
%         % having housing and plot.
%         ZhousWEV = Zfin;                %#ok % Store these values for WEV
%         ZrentWEV = ZBefMatch(:,1,:,1);  %#ok % calculations: (N,1,Nw) arrays.
%     end
% end


disp('--------------------------------------------------------------')
if distOLG<tolOLG                   % Report what happened:
   fprintf('OLG value-function loop converged after %1.0f iterations.\n',iterOLG)
else
   fprintf('OLG VALUE-FUNCTION LOOP ABANDONED AFTER %1.0f ITERATIONS!!!.\n',iterOLG)
end
clear epsk    Pp             logCold
clear akGridAlnx           epskAln VarAkAln Zaln PrI     PrTom
 
%% Get wealth equivalent variation for housing.
% The most robust way we found to calculate this (since closest to how we
% actually determine the house-selling decision in the code):
% 1. Only evaluate WEVs on housing grid points, i.e. for values of
%    ak,ap and h on grid. This gets rid of many interpolations.
% 2. Don't do full Newton method but only the first step. For this,
%    we can use the derivatives Vap, Vak, Wap, Wak which we can
%    store in the last OLG iteration. This is what is actually used
%    in the bargaining code, so it's more likely to give accurate
%    answers.
if Nh > 1    % Nh=1 is the liquid housing counterfactual
WEVk = nan(Nw,Nh);          % Set up matrices for WEV (kid & parent)
WEVp = nan(Nw,Nh);          % for all prod. levels and all house sizes.
                            % Leave default at NaN: Won't calculate it for
                            % largest houses.
CEVp = nan(Nw,Nh);          % Create "goods"-equivalent variation.
CEVk = nan(Nw,Nh);
CEVp(:,1) = 0;
CEVk(:,2) = 0;
                            
dVp = nan(Nw,Nh);  
dVk = nan(Nw,Nh);

slopeIndK = nan(Nw,Nh);     % Set up matrices for slopes of parent
slopeIndP = nan(Nw,Nh);     % and child iso-value lines: can check 
                            % if close to
                            % wealth-pooling/singularity of
                            % Jacobian matrix.
WEVk(:,1) = 0;              % Renters have zero WEVs since
WEVp(:,1) = 0;              % indifferent when offered housing (interpret
                            % (these indeces as those who cannot afford house). 
dVp(:,1) = 0;  
dVk(:,1) = 0;

nwpInit    = nan(Nw,Nh);
nwpInit(:,1)=0;

if ~noKid                                    

hstar = []; nwpInit=[];
for iw=1:Nw
    % Since ep and ek are perfectly correlated at 65, only need to look
    % at prod. combinations (1,1), (2,2), ... 
    % FOR EACH HOUSE SIZE, SHOULD OBTAIN THE ASSETS THAT A PARENT
    % WOULD HAVE FOR WHOM PRECISELY THIS HOUSE IS OPTIMAL. THEN GO
    % TO THE CLOSEST ASSET GRID POINT.
    % nwInit = InvertHpol(hvec,iw);     % Invert for this prod.
    h            = hpol{iw};
    hstar(:,iw)  = h(akgrid);                  %#ok %optimal housing policy for given productivity
    nwpInit(iw,:)= interp1(hstar(:,iw),akgrid,hvec); %#ok %initial net worth levels for which hvec is optimally chosen
    iwNaN = isnan(nwpInit(iw,:));
    nwpInit(iw,iwNaN)=aup; %#ok 
    akInit = startA(iw);                 % child starting financial wealth as function of productivity
    for ih=2:Nh                         % Loop over all houses.
        if nwpInit(iw,ih)<=aup           % Only look at starting assets below 2m.
            ap    = nwpInit(iw,ih)-hvec(ih); % Financial wealth
            apInd = ClosestAssetInd(ap);  % Find closest financial wealth index.
            akInd = ClosestAssetInd(akInit);
            % Read out last values under bargaining (equilibrium outcome at age
            % 65) for this house and this prod. level:
            Vkeqm   = llHlth.VkBarg(akInd,apInd,1,iw,iw,ih-1);  % Scalars.
            Vpeqm   = llHlth.VpBarg(akInd,apInd,1,iw,iw,ih-1);
            % Read out what would happen if they sold:
            VkSell = VkSaleC(akInd,apInd,1,iw,iw,ih);      % (N,N,1,Nw,Nw,Nh-1) arrays.      
            VpSell = VpSaleC(akInd,apInd,1,iw,iw,ih);      %
            dVk(iw,ih) = Vkeqm - VkSell;
            dVp(iw,ih) = Vpeqm - VpSell;
            CEVp(iw,ih) = (VpSell./Vpeqm).^(1/(g-1)) - 1;
            CEVk(iw,ih) = (VkSell./Vkeqm).^(1/(g-1)) - 1;
            % Now, find marginal values of ak/ap for both agents:
            % Interpolate linearly between ap grid points.
            Vpapii = interp1(avec,ll.Vap(akInd,:,1,iw,iw,1),nwpInit(iw,ih),'linear');
            Vpakii = interp1(avec,ll.Vak(akInd,:,1,iw,iw,1),nwpInit(iw,ih),'linear');
            Vkapii = interp1(avec,ll.Wap(akInd,:,1,iw,iw,1),nwpInit(iw,ih),'linear');
            Vkakii = interp1(avec,ll.Wak(akInd,:,1,iw,iw,1),nwpInit(iw,ih),'linear');
            % Record slopes of iso-value lines:
            slopeIndK(iw,ih) = -Vkapii/Vkakii;
            slopeIndP(iw,ih) = -Vpapii/Vpakii;                    
            % Set up Jacobian (2-by-2) and vector with value
            % differences (2-by-1):
            jac = [Vkakii, Vkapii;  Vpakii, Vpapii];
            dV  = [dVk(iw,ih); dVp(iw,ih)];
            % Solve system of equations to obtain 2-by-1 vector
            % with wealth equivalent variation:
            wev = jac\dV;
            WEVk(iw,ih) = wev(1);       % Record results.
            WEVp(iw,ih) = wev(2);
            % Obtain agent's individual variation (keeping the other party
            % constant):
            WEVkIndv(iw,ih)  = dVk(iw,ih)/Vkakii; %#ok % child's individual wev
            WEVpIndv(iw,ih)  = dVp(iw,ih)/Vpapii; %#ok % parent's individual wev
            WEVmaxIndv(iw,ih)= max(WEVkIndv(iw,ih),WEVpIndv(iw,ih)); %#ok
            % Max is the right operator to combine them: Party with higher
            % surplus will determine if house sold.
        end
    end
end
WEVdyn            = WEVk+WEVp; % Dynasty WEV is the sum of individual dynasty WEVs  
% THE FOLLOWING WEVdynNoRedistrb IS THE WEV FOR DYNASTIES THAT WE FINALLY SETTLED ON:
WEVdynNoRedistrb  =(WEVp>=0).*(WEVk>=0).*WEVdyn     +... % no redistribution motive
                   (WEVp<0) .*(WEVk>0) .*WEVmaxIndv +... % parent has redistribution motive
                   (WEVp>0) .*(WEVk<0) .*WEVmaxIndv +... % child has redistribution motive   
                   (WEVp<0) .*(WEVk<0) .*WEVmaxIndv;     % both have redistribution motive
                  % Measure an alternative dyansty WEV where redistribution is not included. 
                  % Take the previous dynasty WEV if there no one has a
                  % redistribution  motive. 
                  % but if either the parent or the child have a
                  % redistribution motive then (WEVp<0 or WEVk<0) then
                  % take the largest INDIVIDUAL wev as the most informative
                  % measure.

% Make figure that compares willingness to pay between dynasty and alone 
% for people entering retirement with the same net worth (separate by prod. levels)
figure, hold on                         
for iw=1:Nw                             % Loop over productivities.
    subplot(Nw,1,iw)
    plot(nwpInit(iw,:),WEVdyn(iw,:),'-*k',nwpInitAln(iw,:),wevAln(iw,:),'-*b')
    ylabel('WEV')                       % Plot dynasty WEV in black and
    title(sprintf('Prod. %1.0f',iw))    % alone WEV in blue.
    xlim([0 1000])
end
xlabel('net worth at 65')          % Put x-label and legend only in last plot.
legend('dynasty','alone')

figure, hold on                         
for iw=1:Nw                             % Loop over productivities.
    subplot(Nw,1,iw)
    plot(hvec',WEVdyn(iw,:),'-*k',hvec',wevAln(iw,:),'-*b')
    ylabel('WEV')                       % Plot dynasty WEV in black and
    title(sprintf('Prod. %1.0f',iw))    % alone WEV in blue.
end
xlabel('house size h')          % Put x-label and legend only in last plot.
legend('dynasty','alone')
end
end  % close nonliquid restriction

close all;

%% Draw simulated panel.
TPan      = ceil(dGen/dtPan);   % Length of simulated panel: 15.  

fprintf('Drawing simulated panel (N=%1.0f, T=%1.0f)...\n',[nPan, TPan])
% Set up matrices to store data on our panel households:

if noKid                            % In no-kid counterfactual:
    cap    = par.apInit;            % Take initial state from what was provided
    cEpInd = par.EpInit;            % in input structure par.                      
else                                % In case of economy with kids:
    startInd = GetDrawProbDist(nInit,nPan); % Get draw from empirical starting 
                                    % distribution on (N,1,Nw) state space,
                                    % before buying house. These are
                                    % linearized indeces on the (N*Nw-by-1)
                                    % linearized state vector. The same for all
                                    % clones: nPan-by-1 vector.
    cEpInd = ceil(startInd/N);      % Get current ("c") parent ("p") prod. ("E")
                                    % index. The same for all clones, thus
                                    % nPan-by-1 vector.                                 
    cApInd = startInd - (cEpInd-1)*N;
                                    % Also obtain current asset index. Since there
                                    % is no cloning in burn-in runs, this is:which is 
                                    % nPan-by-1 vector.                                
    % histogram(cApInd), min(cApInd), max(cApInd): checked, looks fine.                                
    cap     = reshape( avec(cApInd), [nPan,1] );
                                    % Assign starting value for current ("c")
                                    % parent assets ("ap"). Let them start
                                    % exactly on grid points for now. Since
                                    % there are no clones in burn-in runs:
                                    % nPan-by-1 vector. 
end
eTransMat = eye(Nw) + dt*epsHaz1D;
                                % Get Nw-by-Nw transition matrix for 
                                % child productivity over dt.
                                
jPan = 1;                       % Count OLG iterations of panel.
apStartMom = zeros(5,BurnInPan+2);
                                % Set up matrix to store moments of the 
                                % parent asset distribution when entering
                                % retirement to check convergence.
qtl = [0.1; 0.25; 0.5; 0.75; 0.9];
                                % Which quantiles we want.
apStartMom(:,1) = GetQuantProbDist(cap,qtl);
                                % Record quantiles of starting
                                % distribution.
disp('Starting distribution:')                                
fprintf('Quantiles when entering retirement: %1.0f (0.1), %1.0f (0.25), %1.0f (0.5), %1.0f (0.75), %1.0f (0.9)\n',...
            apStartMom(:,1) );     
% Will store results in structure res.pan. First, put in some general panel
% info:
res.pan.n      =          nPan;
res.pan.T      =          TPan;
res.pan.agevec = zeros(1,TPan); % Set up vector with precise age at interview. 

while jPan<=BurnInPan+1             % Loop over burn-in plus final run.
    FinalRun = jPan>BurnInPan;      % Logical if we are in final run.
    disp('--------------------------')

    
    if FinalRun                     % Tell user which run we are in.
        disp('Panel: Final run...');
        if Nh>1
        disp('Obtaining housing WEV...')
        res.pan.once.WEVdyn65          = zeros(nPan,1);   % Set up vector for dynasty WEV and
        res.pan.once.WEVdyn65NoRedistrb= zeros(nPan,1);   % Set up vector for dynasty WEV and
        res.pan.once.WEVmax65Indv      = zeros(nPan,1);   % Set up vector for dynasty WEV and

        res.pan.once.WEVp65  = zeros(nPan,1);   % parent WEV for main clone.
        for i=1:Nw                  % Loop over prod. levels and interpolate.
            isi = cEpInd==i;        % Logical marker for prod. level i:
                                    % nPan-by-1 vector.
            nwi = cap(isi);        % Read out net worth for hh with this
                                    % prod.: Ni-by-1 vector.
            if noKid                % In no-kid counterfactual:
                %nonNan= ~isnan(nwpInitAln(i,:)) & ~isnan(wevAln(i,:));
                                    % Get 1-by-Nh vector for valid values
                                    % we have to interpolate. 
                maxNWP = find(nwpInitAln(i,:)==aup, 1 );                  
                res.pan.once.WEVp65(isi) = interp1(nwpInitAln(i,1:maxNWP),wevAln(i,1:maxNWP),nwi,'linear','extrap');
                                    % Get WEV by linear interpolation.              
            else                    % In counterfactual with kids:
                %nonNan= ~isnan(nwpInit(i,:)) & ~isnan(WEVdyn(i,:));
                                    % Get 1-by-Nh vector for valid values
                                    % we have to interpolate (recall: the
                                    % largest house is not bought, thus net
                                    % worth jumps off grid).
                maxNWP = find(nwpInit(i,:)==aup, 1 );                  
                res.pan.once.WEVdyn65(isi)           = interp1(nwpInit(i,1:maxNWP),WEVdyn(i,1:maxNWP),          nwi,'linear','extrap');
                %   THE FOLLOWING WEVdyn65NoRedistrb IS THE WEV WE SETTLED
                %   ON:
                res.pan.once.WEVdyn65NoRedistrb(isi) = interp1(nwpInit(i,1:maxNWP),WEVdynNoRedistrb(i,1:maxNWP),nwi,'linear','extrap');
                res.pan.once.WEVmax65Indv(isi)       = interp1(nwpInit(i,1:maxNWP),WEVmaxIndv(i,1:maxNWP),      nwi,'linear','extrap');
                                    % Obtain dynasty WEV in $ for all panel
                                    % members by interpolating WEVdyn
                                    % between the grid points we
                                    % calculated. Take initial net worth of
                                    % on x-axis. Ni-by-1 vector.            
            % NOTE: for those who sell immediately, we will set WEVs to 0
            % below, replacing the wrong values we have interpolated at the
            % bottom.
                res.pan.once.WEVp65(isi)   = interp1(nwpInit(i,1:maxNWP),WEVdyn(i,1:maxNWP),nwi,'linear','extrap');
                                    % Also get parent WEV to decompose if
                                    % we want to.
            end
        end
        
        res.pan.once.WEVp65(  cap<par.hvec(2))             = 0;


                                    % Correct for those who can't buy house:
                                    % WEV must be zero.
        if noKid                    % There is no dynasty WEV without kids.
            res.pan.once.WEVdyn65(:) = NaN;
        else                        % Correct parent WEV also.
            res.pan.once.WEVdyn65(          cap<par.hvec(2)) = 0;
            res.pan.once.WEVdyn65NoRedistrb(cap<par.hvec(2)) = 0;
            res.pan.once.WEVmax65Indv(      cap<par.hvec(2)) = 0;
        end
        end % close non-liquid scenarios                                    
%         % Calculate some statistics of interest:
%         res.wevQuant = GetQuantProbDist(wevPan(~isnan(wevPan)),[.1,.25,.5,.75,.9,.95]);
%         res.wevMean  = mean(wevPan(~isnan(wevPan)));
        
        if CloneWars                % If clones are needed:
            disp('Commencing clone wars...')
            nCln      = 3;          % How many clones there are: Scalar.
                                    % The main clone (regular hh): index 1,
            indCln65  = 2;          % clones taken at age 65 (index 2),
            indClnLtc = 3;          % clones taken at entry into LTC (index 3),
                                    % clones taken of those who keep house when 
                                    % entering LTC.
            res.pan.indCln65  = indCln65 ;  % Put these indeces into results
            res.pan.indClnLtc = indClnLtc;  % structure.
            cap = cap.*ones(1,1,nCln);
                                    % Have to extend parent assets to have 
                                    % them vary across clones now:
                                    % (nPan,1,nCln) array.
        else                        % Burn in runs: 
            nCln      = 1;          % No cloning: just have main clone on index 1.
        end       
        res.pan.nCln = nCln;        % Record number of clones in panel.
        
        % 1. Set up some fields in res.pan that are recorded only once:
        res.pan.once.EpInd65 =             cEpInd;  % Parent prod. at 65.
        res.pan.once.Nwp65   =                cap;  % Net worth at 65.
        res.pan.once.h65     =    zeros(nPan,1,1);  % Housing at age 65. Only for
        % main clone, can easily be inferred for other clones.
        res.pan.once.NwpLtc  =   nan(nPan,1,nCln);  % Net worth upon entering LTC.
        res.pan.once.OwnLtc  = false(nPan,1,   1);  % If owner when LTC hits.
        res.pan.once.KeepLtc = false(nPan,1,   1);  % If house kept when LTC hits.
        res.pan.once.Beq     =   nan(nPan,1,nCln);  % Bequest.
        res.pan.once.HBeq    =   nan(nPan,1,nCln);  % Housing bequest.
        res.pan.once.FBeq    =   nan(nPan,1,nCln);  % Financial bequest.
        res.pan.once.Surv95  = false(nPan,1,   1);  % If individual survived
        % until age 95 .Rather record this, is tedious to construct later
        % and takes little storage.
        % Important age variables (also needed only once):
        res.pan.age.Liq      =   nan(nPan,1,nCln);  % House liquidation:
        % depends on which clone.
        res.pan.age.Died     =   nan(nPan,1,   1);  % Age of death and LTC
        res.pan.age.EnterLtc =   nan(nPan,1,   1);  % entry do not depend 
                                                    % on clone.
                                                    
        % 2. Binary variables: Will record current state, time spent in state,
        % and if state ever occurred. Care forms and if hand-to-mouth.
        % In general, panel variables will have the following
        % dimensionality:
        % dim.1:    which panel household
        % dim.2:    which panel period
        % dim.3:    which clone of the household. 
        fdBinary = {'IC','FHC','NH','MA','Htm'};
        for ifd=1:numel(fdBinary)       % Loop over all binary variables.
            iname = fdBinary{ifd};      % Read out name.
            res.pan.binary.(iname).ever = false(nPan,  1 ,nCln);
                                        % If ever in state: record for all 
                                        % three clones.
            res.pan.binary.(iname).val  = false(nPan,TPan,nCln);
                                        % val: records current state at
                                        % interview (or at time of death
                                        % for exit interviews).
            curr.(iname).val            = false(nPan,  1 ,nCln);
                                        % In structure curr, store the
                                        % current value of a variable (need
                                        % this to keep track of last state
                                        % of the dead).
            res.pan.binary.(iname).time = zeros(nPan,TPan,nCln);
                                        % Record time spent in each state
                                        % over interview period.
            curr.(iname).time           = zeros(nPan,  1 ,nCln);
                                        % Keep track of cumulative time
                                        % over interview period.
            % Also record state of contemporaneous clone at interview.
            % Pattern is the same as for val here, but only make a contemp.
            % clone from the main clone, thus only have matrix here:
            res.pan.binary.(iname).ccl  = false(nPan,TPan     );
            curr.(iname).ccl            = false(nPan,  1      );
        end
        
        % 3. Stocks: We only record these at the end of the panel period
        % (2yrs). (nPan,TPan,nCln) arrays with         
        fdStocks = {'ak','ap','h','ind'};   % Fields with stocks for storage.
        % Note: Housing is recorded as initial house bought and liquidation
        % age, housing at each age can then be constructed.
        for ifd=1:numel(fdStocks)           % Set up storage. 
            res.pan.stocks.(fdStocks{ifd}) = zeros(nPan,TPan,nCln);
        end
        % 4. Flows: Sum them up over 2 years of panel interview period with
        %           the fields in structure 'curr', then write into
        %           sub-structure res.pan.flows.
        fdFlows  = {'Qp','Qk','QPaltrComp','QKaltrComp','gp','gk','gpUnc','gkUnc','ep'}; 
        % Need to keep track of Qp and Qk separately since the two may
        % alternate over 2 years in giving to the other and we don't want
        % to net the flows out.
        for ifd=1:numel(fdFlows)            % Loop over all flow fields:
            iname = fdFlows{ifd};       
            res.pan.flows.(iname).val = zeros(nPan,TPan,nCln);
            % Field 'val' is for the panel observations of the 3 clones:
            % These are cumulative flows over 2 years.
            curr.(iname).val          = zeros(nPan,  1 ,nCln);
            % For variables Qp and ep, also store what contemporaneous
            % clone of the main clone is doing:
           % if strcmp(iname,'Qp') || strcmp(iname,'ep')
            res.pan.flows.(iname).ccl = zeros(nPan,TPan);
            curr.(iname).ccl          = zeros(nPan,  1 );
           % end   
            % Finally, also story the current policies (at time of
            % interview), for 3 clones and the contemp. clone:
            if ~(strcmp(iname,'gpUnc') || strcmp(iname,'gkUnc'))
                res.pan.flows.(iname).polval = zeros(nPan,TPan,nCln);
                res.pan.flows.(iname).polccl = zeros(nPan,TPan     );
            end
            % Would have to program extra for gpUnc and gkUnc.
        end                              
                                         
                                         

    else                            % If burn-in run:
        fprintf('Panel: Burn-in run %1.0f...\n',jPan);
        nCln = 1;                   % No clones in burn-in runs.
    end                            
    
    cSInd  = ones(nPan,1);          % All parents start out healthy. Health 
                                    % state common across clones, thus
                                    % (nPan,1,1) array.
    cDeadPan  = false(nPan,1);      % Nobody dead to start with.
    enterLTC  = false(nPan,1);      % Initialize variable: if parent has just
                                    % entered LTC.
    ctIndPan  = 1;                  % Current time index (between 1 and 15,     
                                    % will be counted up) in panel.
    prdPan    = tRet+dtPan;         % Panel period: This is the age of parent 
                                    % when going into the next panel period.
    % Now, do house-buying decision:
    ch     = zeros(nPan,1,nCln);    % Initialize array for houses: 
                                    % (nPan,1,nCln) array.
    if Nh>1                         % If there are houses (not the case in
                                    % LiqHous counterfactual!):
        for i=1:Nw                  % Loop over all parent productivity levels:
            isi = (cEpInd==i);      % Get nPan-by-1 logical vector with dummy
                                    % for prod. level i.
            cap1= cap(:,1,1);       % Get assets of baseline household:
                                    % nPan-by-1 vector.
            capi= cap1(isi);        % Read out asset positions of those with
                                    % prod.i: ki-by-1 vector.
        % NOTE: hpol are the policies for new parents after the last OLG iteration.
        % For alone economy, have to take these for the PARENT of the childless. 
            if noKid && FinalRun    % For childless in final run: Have to  
                hi = hpolAln{i}(capi);% take policies we obtained after the
                                    % value-function guess loop.
            else                    % In all other cases, use policy from 
                                    % regular (joint) loop.
                hi = hpol{i}(capi); % Which house they buy conditional on
            end                     % buying: ki-by-1 vector. Those who
                                    % can't afford house (cap<hmin) get
                                    % hi=0, everybody who can afford gets
                                    % the best house size. Agents who can
                                    % afford a house but prefer renting
                                    % will just sell off at first time
                                    % iteration (is more parsimonious this
                                    % way).
            ch(isi,1,1:nCln) = hi.*ones(1,1,nCln);  
                                    % Put solution into ch, is still common
        end                         % across clones.
    end
    if nCln>1                       % If we do cloning: Take away house
        ch(:,1,indCln65) = 0;       % from clone at age 65 (index 2).
    end
    cHInd = ClosestHouseInd(ch);    % Obtain house index corresponding to 
                                    % closest grid point: (nPan,1,nCln)
                                    % array.
    cap = cap - ch;                 % Deduct house value from assets.
    % Now, match children: (for alone economy in final loop, let ghost kids
    % float around: will not really use them, but don't write extra code to
    % keep it simpler)
    kidMatchProb = MatchProb( cEpInd, :);   
                                    % Get nPan-by-Nw matrix with 
                                    % probabilities of having kids in
                                    % different productivity bins.
    cEkInd  = GetDrawMultProbDist(kidMatchProb')';
                                    % Draw kid prod. indeces.
                                    % Is common across clones:
                                    % nPan-by-1 vector
    cak = startA(cEkInd).*ones(1,1,nCln);
                                    % Let kids start off with starting
                                    % wealth that corresponds to their
                                    % prod. level. Same for all clones:
                                    % (nPan,1,nCln) array.
                                    
    sqrtdt = sqrt(dt);              % Will need this scalar all the time.

    for j=1:Nj                      % Loop over all (small) dt-time periods.
        % First, draw shocks for the simulated panel:
        % Read out (N,N,Ns,Nw,Nw,Nh) arrays with hazards of moving up or
        % down in wealth grid due to savings and Brownian Motion. 
        if rem(j,100)==0
            fprintf('%1.0f of %1.0f time periods done.\n',[j,Nj]);
        end
        
        % First, read out jump hazards in dim. 3 to 5: health and income.
        % These are just exogenously-given functions of age and parent
        % prod., don't have to distinguish between alone and joint case.
        haz345 = store.haz345{j};
        hazMed35      = squeeze(      haz345.Med(:,1,:) ); % Store these as 
        hazLtcBasic35 = squeeze( haz345.LtcBasic(:,1,:) ); % Ns-by-Nw matrices.
        hazLtcSkill35 = squeeze( haz345.LtcSkill(:,1,:) );
        hazDth35      = squeeze(    haz345.Death(:,1,:) );
       
        % Read out laws of motion for the states (in all runs):
        if noKid && FinalRun             % In no-kid counterfactual:
              
            % For house-selling decision, patch 4D policy to full 6D:
            pol.xstar = PatchToArray(xstarNoKidStore(:,:,:,:,j),...
                                     [N,N,Ns,Nw,Nw,Nh], [2,3,5,6]   );
            % Just puts the same policies for all ghost-kid states on dim.
            % 1 and 4.
            pol.apDot  = PatchToArray(aDotNoKidStore(:,:,:,:,j), ...
                                      [N,N,Ns,Nw,Nw,Nh], [2,3,5,6] );
            % For ghost kids, just write in zeros:
            pol.akDot    = zeros(N,N,Ns,Nw,Nw,Nh);
            pol.akDotAln = zeros(N,  Ns,Nw      );
                                                % all LTC cases do FC.
        else   %%%%%%%%%% The regular case WITH kids, or no-kids in 
            % burn-in run with parents %%%%%%%%%%%%%%%
            % Read out policies and drifts at nearest grid points: (N,N,Ns,Nw,Nw,Nh) arrays.
            % Write them into structure 'pol', containing current policy
            % functions.
            pol.xstar  = store.xstar(:,:,:,:,:,:,j);  % ??? COULD CHANGE TO LINEAR
            pol.akDot  = store.akDot(:,:,:,:,:,:,j);  % INTERPOLATION HERE FOR 
            pol.akDotAln = store.akDotAln(:,:,:, j);  % THE CONTINUOUS VARIABLES
            pol.apDot  = store.apDot(:,:,:,:,:,:,j);    
        end  
        
        if FinalRun                     % In final run, need more policies:
            if noKid                    % First, no-kid economyL
                % No transfers in no-kid world:
                pol.gp    = zeros(N,N,Ns,Nw,Nw,Nh);
                pol.gk    = zeros(N,N,Ns,Nw,Nw,Nh);
                pol.Qstar = zeros(N,N,Ns,Nw,Nw,Nh);
                pol.QKaltrComp = zeros(N,N,Ns,Nw,Nw,Nh);
                pol.QPaltrComp = zeros(N,N,Ns,Nw,Nw,Nh);
                % Now, read out other policies and bring to full size
                % (are the same for all ghost-kid states):
                pol.ep     = PatchToArray(eNoKidStore(:,:,:,:,j), ...
                                     [N,N,Ns,Nw,Nw,Nh], [2,3,5,6] );
                ma = PatchToArray( MAdecNoKidStore(:,:,:,:,j), ...
                                   [N,1,2,Nw,Nw,Nh], [2,3,5,6]    );
                                            % Bring MA to
                                            % (N,1,2,Nw,Nw,Nh) array.
                ic = zeros(N,N,1,Nw,Nw,Nh); % Never informal in basic state.

            else                          % For usual economy:
                pol.gp         = store.gp(:,:,:,:,:,:,j);  % Read out current gift flows
                pol.gk         = store.gk(:,:,:,:,:,:,j);
                pol.Qstar      = store.Qstar(:,:,:,:,:,:,j);  % and transfer flows.
                pol.QKaltrComp = store.QKaltrComp(:,:,:,:,:,:,j);
                pol.QPaltrComp = store.QPaltrComp(:,:,:,:,:,:,j);
                pol.ep         = store.ep(:,:,:,:,:,:,j);
                % The following are read out, but must be increased to full
                % size to write them into structure 'pol':
                ic         = store.istar(:,:,:,:,:,:,j);  % (N,N,1,Nw,Nw,Nh) array
                %fhc        = store.FHC(:,:,:,:,:,:,j);    % FHC inside option in basic care
                ma         = store.MA(:,:,:,:,:,:,j);  % (N,1,2,Nw,Nw,Nh) array: 
                % MA in basic and skilled LTC state, only broke parents.
                % Now, construct binary variables of full 6D size:
            end
            
            % Now, bring care decisions to full size:
            fc = 1-ic;                       % Formal care in basic state: (N,N,1,Nw,Nw,Nh) array
            MAdec23 = zeros(N,N,2,Nw,Nw,Nh);     % Bring MA decision to (N,N,2,Nw,Nw,Nh) array
            MAdec23(:,1,:,:,:,:) = ma;           % (for the two disabled states, "2" and "3").
            % Formal home care: 
            % First in basic-LTC state: (N,N,1,Nw,Nw,Nh) array.
            FHCdec2 = cat(6, zeros(N,N,1,Nw,Nw,1), ...
                     fc(:,:,:,:,:,2:end) .* (1-MAdec23(:,:,indBC-1,:,:,2:end))) ;
                 % for owners it's fc but not MA.
            % Then in skilled-LTC state: (N,N,1,Nw,Nw,Nh) array.
            FHCdec3 = cat(6, zeros(N,N,1,Nw,Nw,1), (1-MAdec23(:,:,indSC-1,:,:,2:end)) );
            % FHC in skilled state ("3"): 0 in renting state (NH!), in
            % housing only if no MA received.
            % Create IC variable of full 6D size (need also the healthy page
            % for our indexing strategy!)
            pol.IC = cat(3, zeros(N,N,1,Nw,Nw,Nh), ic, zeros(N,N,1,Nw,Nw,Nh) );
            % IC can take place only in basic LTC health state.
            pol.FC = cat(3, zeros(N,N,1,Nw,Nw,Nh), fc,  ones(N,N,1,Nw,Nw,Nh) );
            % FC can take place in basic LTC health state and must take
            % place in skilled LTC state.
            pol.MA = cat(3, zeros(N,N,1,Nw,Nw,Nh), MAdec23         );
            % MA can take place in basic and skilled LTC health state.                     
            pol.FHC= cat(3, zeros(N,N,1,Nw,Nw,Nh), FHCdec2, FHCdec3);
            % FHC can only take place in basic LTC health state
            % NH: (N,N,Ns,Nw,Nw,Nh) array if in nursing home (can be
            % privately-paid or MA).
            pol.NH = cat(3, zeros(N,N,1,Nw,Nw,Nh), ...
                     pol.FC(:,:,indBC:indSC,:,:,:) & ~pol.FHC(:,:,indBC:indSC,:,:,:) );
            % NH: disabled who are in formal care but not home care.
            % (both MA and PP-NH).
            % basic LTC state: Those in FC but not FHC are NH.
            % skilled LTC state: All must be in NH.
        end
        
        % Now, obtain hazard of health shocks, which depend only on dim. 3
        % and 5, i.e. parent health and parent productivity. 
        cInd35Pan = (cEpInd-1)*Ns + cSInd;  % First, create linear indices 
        % on the matrices ...35Pan (size Ns-by-Nw). Identical across clones:
        % nPan-by-1 vector.
        
        % Now, do stages: 1.death, 2.sell house, 3.all others.
        % First, draw shocks and determine which parents die:
        % Draw discrete jumps in one go, only one jump can happen at a
        %t ime (is first-order accurate, OK since probabilities very small):
        panJumpProb = dt* [     hazMed35(cInd35Pan'); hazLtcBasic35(cInd35Pan'); ...
                           hazLtcSkill35(cInd35Pan');      hazDth35(cInd35Pan'); ...
                                                             zeros(1,nPan )         ]; 
        % Read out hazards from Ns-by-Nw matrices ...35Pan by linear
        % indexing: Combine five 1-by-nPan vectors to 5-by-nPan matrix.
        panJumpProb(:,cDeadPan) = 0;% For dynasties with dead parents, there
                                    % are no such shocks any more.
        medInd   = 1;               % Which index refers to medical shock,
        basicInd = 2;               % which to basic-LTC shock,
        skillInd = 3;               % which to skilled-LTC shock and
        dthInd   = 4;               % which index refers to death.
        panJumpProb(end,:) = 1 - sum(panJumpProb(1:end-1,:),1);
                                    % Get prob. of none of the four shocks
                                    % happening and write into last row.
        panJumpInd = ( GetDrawMultProbDist(panJumpProb) )';
                                    % Draw the event for each family: a
                                    % number between 1 and 5, indexing the
                                    % event. Total: nPan-by-1 vector.
        % Probabilities always fine, turned this off:
%         pmNeg = (panJumpProb(end,:)'<0) & ~cDeadPan;
%         pmNegFracPan(j) = sum(pmNeg)/nPan;
%                                     % For which fraction of panel members 
%                                     % p_middle turned negative, 
%         if pmNegFracPan(j)>0        % If this happened, get average violation:
%             avgViolPan(j) = mean( panJumpProb(10,pmNeg) );
%         end
        % First, let death event occur:
        if j==indDth                % If we are in the period where everybody
            cDied = ~cDeadPan;      % dies: Kill off those still alive
            res.pan.once.Surv95 = cDied;
                                    % nPan-by-1 vector: Mark survivors until age 95.
        else                        % For the usual case:
            cDied = ~cDeadPan & panJumpInd==dthInd;
                                    % Create dummy if individual died this
        end                         % dt: Was alive but drew 4. nPan-by-1 vector, identical across clones.
        cDeadPan = cDeadPan | cDied;% Update if person is dead.
        cDied3D  = cDied(:,:,ones(1,nCln)); %  Extend cDied to all clones: (nPan,1,nCln) array.
        CHBEQ= ch(cDied3D);         % housing bequest (nDied*nCln)-by-1 vector.
        CFBEQ= cap(cDied3D);        % financial bequest (nDied*nCln)-by-1 vector.  
        % Then, get total bequest: 
        CBEQ = cap(cDied3D) + CHBEQ;% (nDied*nCln)-by-1 vector.
        if FinalRun                 % In final run, store bequests:
            % Add bequests to current flow: If a person died this dt, sum up
            % parent assets and housing wealth.
            res.pan.age.Died( cDied  ) = store.tp(j);   % Age at death.
            res.pan.once.Beq( cDied3D) = CBEQ;          % Bequest.
            res.pan.once.HBeq(cDied3D) = CHBEQ;         % Housing bequest.
            res.pan.once.FBeq(cDied3D) = CFBEQ;         % Financial bequest.
        end
        cak(cDied3D) = cak(cDied3D) + CBEQ;
                                    % Pass assets of parents who died on to
                                    % child: (nPan,1,nCln) array.
        cap(cDied3D) = 0;           % Set asset positions for dead parent
         ch(cDied3D) = 0;           % to zero: need 0, not NaN, for 
                                    % calculations below.

        % 2. Second, let panel members sell house:  
        [cIndPan,~,~] = PanelIndices(cak,cap,cSInd,cEkInd,cEpInd,cHInd,...
                                     ClosestAssetInd,pol.akDot,pol.apDot,cDeadPan);
                                    % Obtain linear indices on 6D grid for
                                    % each panel member: (nPan,1,nCln) array.  
        sell = ~cDeadPan & pol.xstar(cIndPan);  % (nPan,1,nCln) array if 
                                    % panel members sell (contains nSell
                                    % true values). Note: Those who died
                                    % are not counted here. Note 2: In
                                    % liquid-housing counterfactual, xstar
                                    % is always zero, code works.
        saleV = ch(sell);           % Sales value of sellers:
                                    % nSell-by-1 vector.
        if FinalRun                 % In final run, store liquidation decision:
            res.pan.once.KeepLtc(enterLTC & ch(:,1,1)>0 & ~sell(:,1,1)) = true;
            if j==1                 % If household sells immediately at
                                    % age 65, interpret as never own, thus 
                res.pan.once.WEVp65(sell(:,1,1)) = 0; % WEVs are zero (may 
                if ~noKid           % have small positive numbers due to
                    res.pan.once.WEVdyn65(sell(:,1,1))           = 0;
                    res.pan.once.WEVdyn65NoRedistrb(sell(:,1,1)) = 0;
                    res.pan.once.WEVmax65Indv(sell(:,1,1))       = 0;
                end                 % interpolation above, correct these).
            else                    % Record sale as liquidation after first                
                res.pan.age.Liq(sell) = store.tp(j);    % period. At 65,
            end                     % we interpret as not buying.
        end
        cHInd(sell) = 1;            % Set house index to 1 for them (rent now).
        ch(sell)    = 0;            % Set house value to renter value. 
                                    % Will add house value to ap below.
        newApSell = cap(sell) + saleV; % New asset position after selling:
                                    % nSell-by-1 vector.
        cap(sell) = newApSell;      % Record this. 
        
        % Update the indeces for assets a^k and a^p, correcting close to
        % the origin if necessary, get linear indeces. We do this to obtain
        % the correct policies here, i.e. AFTER selling the house and AFTER
        % parent has died: 
        [cIndPan,~,~] = PanelIndices(cak,cap,cSInd,cEkInd,cEpInd,cHInd,...
                             ClosestAssetInd,pol.akDot,pol.apDot,cDeadPan);
        if FinalRun                         % In final run, record policies 
                                            % of interest:
            % Separate exchange transfer Qstar in positive and negative 
            pol.Qp    = max(pol.Qstar,0);   % Positive: parent-to-child.
            pol.Qk    = min(pol.Qstar,0);   % Negative: child-to-parent.
            % NOTE: CONVENTION IS THAT Q-VARIABLES ALWAYS ARE NET
            % PARENT-TO-CHILD TRANSFERS, I.E. THEY ARE NEGATIVE IF CHILD
            % GIVES TO PARENT.
            % Determine linear indeces of contemporaneous clones: Strip all main
            % clones of their houses. nPan-by-1 vector.
            capCcl = cap(:,1,1) + ch(:,1,1);    % Get parent assets for 
            % contemporaneous clone: add house value. nPan-by-1 vector.
            [cIndCcl,~,~] = PanelIndices(cak(:,1,1),capCcl,...
                               cSInd,cEkInd,cEpInd,ones(nPan,1,1),...
                               ClosestAssetInd,pol.akDot,pol.apDot,cDeadPan);
            % Binary variables:
            % First, treat hand-to-mouth, which is special:
            nowHtm = (cap<0.001) & (pol.apDot(cIndPan)<0.001);
            % Determine if panel members are hand-to-mouth: (nPan,1,nCln)
            curr.Htm.val = cDeadPan.*curr.Htm.val + ~cDeadPan.*nowHtm;
            % Update current hand-to-mouth status: Take last status for the
            % dead, write in status now for the living.
            % Now, do the same for contemporaneous clone:
            HtmCcl = (capCcl<0.001) & (pol.apDot(cIndCcl)<0.001);
            curr.Htm.ccl = cDeadPan.*curr.Htm.ccl + ~cDeadPan.*HtmCcl;
            % Time hand-to-mouth:
            curr.Htm.time= curr.Htm.time + (~cDeadPan).*nowHtm*dt;
            % Contemporaneous clone is hand-to-mouth if assets zero and
            % drift (upon selling house) is zero.
            % Now, loop over the other binary fields:
            for ifd=1:numel(fdBinary)       % Loop over all binary fields:
                iname = fdBinary{ifd};      % IC, FHC,NH,MA. 
                if ~strcmp(iname,'Htm')     % (Htm is done already).
                    cci = (pol.(iname)(cIndPan)>=0.5); %Read out current value of
                    % binary variable for all panel members: Take cut-off 0.5
                    % for probabilities: (nPan,1,nCln) array.
                    curr.(iname).val = cDeadPan.*curr.(iname).val + ...
                                      ~cDeadPan.*cci;
                    % Take over previous value for dead. For alive, write in
                    % current value of variable.
                    % An example of what this looks like for MA:
                    % MAPAN = ~cDeadPan & (pol.MA(cIndPan)>  0.5 );
                    % cMA = cDeadPan.*cMA + MAPAN;
                    % Now, contemporaneous clone: 
                    ccc = (pol.(iname)(cIndCcl)>=0.5); % policy: nPan-by-1.
                    curr.(iname).ccl = cDeadPan.*curr.(iname).ccl + ...
                                      ~cDeadPan.*ccc;
                    % Now, add time in state for the alive:
                    curr.(iname).time = curr.(iname).time + (~cDeadPan).*cci*dt;
                    % An example:
                    % ctHtmPan= ctHtmPan+ (~cDeadPan).*nowHtm*dt;   % right now and increment time.
                    % Finally, set 'ever' field to true if event occurred:
                    res.pan.binary.(iname).ever(~cDeadPan & cci) = true;                                    
                    % Example:       EverMA(MAPAN) = true;
                end
            end
            
            % Now, do flows:
            for ifd=1:numel(fdFlows)        % For all flows (Qp,Qk,gp,gk,gpUnc,gkUnc,ep):
                iname = fdFlows{ifd};       % Read out name of the field.
                if ~(strcmp(iname,'gpUnc') || strcmp(iname,'gkUnc'))
                                            % Don't do gpUnc and gkUnc
                                            % here -- special.
                    cci = pol.(iname)(cIndPan); % Get current policy of all panel
                                                % members: (nPan,1,nCln) array
                    curr.(iname).val = curr.(iname).val + cci.*(~cDeadPan)*dt;
                    if isfield(curr.(iname),'ccl')  % If field keeps track of 
                        ccc = pol.(iname)(cIndCcl); % contemporaneous clone:
                                                % Read out current policy of
                                                % the contemporaneous clone:
                                                % nPan-by-1 vector.
                        curr.(iname).ccl = curr.(iname).ccl + ccc.*(~cDeadPan)*dt;
                    end
                end
            end                         % Add current flow to object 'curr'.
            % Now, update for gifts to recipients with positive wealth:
            curr.gpUnc.val = curr.gpUnc.val + pol.gp(cIndPan).*(cak>0)*dt;
            curr.gkUnc.val = curr.gkUnc.val + pol.gk(cIndPan).*(cap>0)*dt;
  
        end                           
                                            
        % Finally, draw jumps due to savings, LTC, medical spending,
        % productivity shocks, and death.
        akDriftPan = pol.akDot(cIndPan)*dt; % Get drift in kid assets and
        akDriftPan(cDeadPan) = pol.akDotAln(cIndPan(cDeadPan))*dt;
        apDriftPan = pol.apDot(cIndPan)*dt; % parent assets for panel members 
                                    % at nearest neighbor:
                                    % (nPan,1,nCln) arrays.
        apDriftPan(cDeadPan,1,:) = 0; % No drift for dead parents
                                    % (across all clones)
        % Now, draw Brownian innovations from normal: nPan-by-1 vectors
        dWp = randn(nPan,1);        % Draw Brownian innovations: identical
        dWk = randn(nPan,1);        % across clones, thus nPan-by-1 vectors.
        akShockPan = sigmaA.*cak.*sqrtdt.*dWk;  % Now, convert to shocks:
        apShockPan = sigmaA.*cap.*sqrtdt.*dWp;  % Volatility depends on assets
                                    % and thus on clone: (nPan,1,nCln) arrays.
        % Now, medical expenditure draw for parents:
        medShock = (panJumpInd==medInd);% nPan-by-1 logical vector with
                                    % those hit by medical shocks.
        nMed = sum(medShock);
        medExpDraw = zeros(nPan,1); % Set up draw for med. expenditures: zero
                                    % by default. Common across clones:
                                    % nPan-by-1 vector.
        medExpDraw(medShock) =  exp(medMu+medSig*randn(nMed,1));
        % Used to have the following, but is slower:
      %  medExpDraw = (panJumpInd==medInd) .* exp(medMu+medSig*randn(nPan,1));
                                    % Get draw from log-normal for each
                                    % family (medical expenditures)...
        % Note: For dead parents, probability of this shock is zero.                                                                                 
        cak = max(cak + akDriftPan + akShockPan, 0); 
                                    % Get new child asset position. Leave
                                    % at zero if it went below. Let agents
                                    % go beyond aup on top of grid (but
                                    % will always use the aup-policies for
                                    % these): (nPan,1,nCln) array.
        % For parent, also need to subtract medical expenditures (note that
        % house-selling revenue was already added above):
        cap = max(cap + apDriftPan + apShockPan - medExpDraw, 0);
                                    % (nPan,1,nCln) array.
        % Now, apply LTC shocks (common across clones):
        enterLTC = (panJumpInd==basicInd | panJumpInd==skillInd) & cSInd==1;
                                    % Logical variable if household just
                                    % entered LTC: nPan-by-1 vector.
        if FinalRun                 % In final run, store:
            res.pan.age.EnterLtc(enterLTC) = store.tp(j);
                                    % Record time when entered LTC.
            res.pan.once.NwpLtc(enterLTC,:,:) = cap(enterLTC,:,:) + ch(enterLTC,:,:);
                                    % Record net worth upon entering LTC
                                    % for all clones (will be identical for
                                    % main and third clone, but leave at
        end                         % full size for consistency).
        
        % Now, record ownership of those who entered LTC:
        saleVal = ch(enterLTC,1,1); % Record house value of the main clone:
                                    % nEnterLTC-by-1 vector.
        res.pan.once.OwnLtc(enterLTC) = (saleVal>0);
                                    % Mark those who owned when LTC shock
                                    % hit.
        if nCln>1                   % If we want clones, first create the
           % number 3 (=indClnLtc), whom we force to sell the house now:
             cHInd(enterLTC,1,indClnLtc) = 1;    % Is renter now.
                ch(enterLTC,1,indClnLtc) = 0;    % Strip of house.
               cap(enterLTC,1,indClnLtc) = cap(enterLTC,1,1) + saleVal;
            res.pan.age.Liq(enterLTC,1,indClnLtc) = store.tp(j);
                                    % Record when liquidation happened.
        end                         % Add sales value to parent assets 
        cSInd( panJumpInd==basicInd ) = 2;  % Set index to 2 if basic shock hits,  
        cSInd( panJumpInd==skillInd ) = 3;  % set index to 3 if skilled shock hits. 
        % Finally, draw child productivity shocks (common across clones):
        ekProb = eTransMat( cEkInd, : );
                                    % Get probabilities of ending up in the
                                    % Nw different states after dt: 
                                    % nPan-by-Nw matrix.
        cEkInd =( GetDrawMultProbDist(ekProb') )';
                                    % Draw new indices.
        if j==1                     % In first dt: Record initial owners.
            res.pan.once.h65 = ch(:,1,1);% Set this housing stock for main clone, and
        end                         % nPan-by-1 logical vector.
                                    
        if FinalRun && ( (store.tp(j)>=prdPan) || (j==Nj) )
                                    % If we jumped out of the current 2-year
                                    % panel period or we have reached the 
                                    % last interval: store variables.
            % Record exact age at this interview:
            res.pan.agevec(1,ctIndPan) = store.tp(j);                         
            
            % 1. Record stocks: Take real numbers for asset variables
            % (stock variables) and cumulative numbers for the others (flow
            % variables): 
            res.pan.stocks.ak( :,ctIndPan,:) =      cak;
            res.pan.stocks.ap( :,ctIndPan,:) =      cap;
            res.pan.stocks.h(  :,ctIndPan,:) =       ch;
            res.pan.stocks.ind(:,ctIndPan,:) =  cIndPan;

            % 2. Flows: Write in the cumulative variables in structure
            % curr:
            for ifd=1:numel(fdFlows)
                iname = fdFlows{ifd};               % Read out field name.
                res.pan.flows.(iname).val(:,ctIndPan,:) = curr.(iname).val;
                                            % Write cumulative flow into
                                            % panel results.
                curr.(iname).val(:) = 0;    % Set counter in curr back to 0.
                % Example from old code:  BeqPan(:,ctIndPan,:) =  cBeqPan;
                % If contemporaneous clone was kept track of, also record
                if isfield(curr.(iname),'ccl')  % this variable and set
                    res.pan.flows.(iname).ccl(:,ctIndPan) = curr.(iname).ccl;
                    curr.(iname).ccl(:) = 0;% set counter to 0.
                end
                % Finally, store current policies for all clones:
                if ~(strcmp(iname,'gpUnc') || strcmp(iname,'gkUnc'))
                    res.pan.flows.(iname).polval(:,ctIndPan,:) = pol.(iname)(cIndPan);
                    res.pan.flows.(iname).polccl(:,ctIndPan  ) = pol.(iname)(cIndCcl);
                end
            end
                
            for ifd=1:numel(fdBinary)       % Loop over all binary fields.
                iname = fdBinary{ifd};      % Read out field name.
                % First, read in current value (or the one at death for
                % exit interviews):
                res.pan.binary.(iname).val( :,ctIndPan,:) = curr.(iname).val;
                curr.(iname).val(:) = false;% Set default state to false 
                                            % for next panel period.
                % Then, read in time spent in the respective state:
                res.pan.binary.(iname).time(:,ctIndPan,:) = curr.(iname).time;
                curr.(iname).time(:) = 0;   % Set counter in curr back to 0.
                % Finally, do the same for contemporaneous clone:
                res.pan.binary.(iname).ccl(:,ctIndPan,:) = curr.(iname).ccl;
                curr.(iname).ccl(:) = false;
            end
            % Prepare for the next interview period of 2 years:   
            ctIndPan = ctIndPan+1;  % Count up time index in panel.
            prdPan = tRet + ctIndPan*dtPan;
                                    % Set next panel period.
        end                       
    end                             % End loop over ages j.                      
    
    % Children inherit what's left:
    cak = cak + (~cDeadPan).*(cap + ch);
    if ~FinalRun                    % If this was a burn-in run, let children
        cap      = cak;             % become parents: Copy assets 
        cEpInd   = cEkInd;          % and productivity index.
    end

    apStartMom(:,jPan+1) = GetQuantProbDist(cak(:,1,1),qtl);
                                    % Record moments of parent asset
                                    % distribution when entering
                                    % retirement.
    fprintf('Quantiles when entering retirement: %1.0f (0.1), %1.0f (0.25), %1.0f (0.5), %1.0f (0.75), %1.0f (0.9)\n',...
            apStartMom(:,jPan+1) );
    jPan = jPan+1;                  % Count up OLG runs.
end
disp('Artificial panel completed.') % Give success message.

% Make figure to show convergence:
figure('Name','Convergence of artificial panel')
runs = 0:1:(BurnInPan+1);           % Vector with runs: 1-by-(BurnIn+2).
plot(runs',apStartMom','-x')        % Plot quantiles of starting assets.
                                    % apStartMom: nQtl-by-(BurnInd+2).
legg = cell(size(qtl));             % Construct legend: Q10, Q25 etc.
for i=1:numel(qtl)
   legg{i}= sprintf('Q%1.0f',100*qtl(i));
end
legend(legg)    
xlabel('Quantile')                  % NOTE: WE NOW HAVE QUANTILES ALONG THE HORIZONTAL AXIS???
xticks(runs)                        % WHEN BURNIN=3 THIS HAPPENS BUT NOT WHEN BURNIN=2
ylabel('new retirees'' assets')
   

% Append other results to res...:
res.Nj       =          Nj;        
res.store.tp =    store.tp;        
res.store.dt =    store.dt;      

% Flows: All flows (in terms of .val) match exactly in the two approaches,
out = res;                          % Finally, hand over results structure.

end