% MAIN PROGRAM %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Solves the quantitative model presented in Section 4 and the
% counterfactual experiments in Section 7. 

% 'SSG_MotherShip' sets up all inputs required for the computational 
% algorithm (parameters, asset grids, earnings process, etc.) and groups 
% them into two structures named 'par' and 'opt'. These structures are then
% handed to the function 'SSG_SolveModel' which solves the model economy 
% and creates an artifical panel of model households. The function 
% 'SSG_GetMoments' uses the artificial panel to calculate many statistics 
% of interest and presents them in tables and graphs. 

% There are two main 'modes': In mode 'baseline' the model solution of the 
% baseline economy is calculated. In mode 'counterf' together with setting 
% 'CountFl' to a specific scenario (e.g. 'noKid') the  solution of a 
% counterfactual economy is calculated. In either mode, 'SSG_MotherShip'
% calls the function 'SSG_SolveModel' which returns a structure containing 
% the results (the structure is named 'res' for the baseline economy and 
% 'countRes'for a counterfactual.) 'SSG_GetMoments' is used for both modes.

% NOTES: - In both modes an artificial panel is calculated by default. 
%        - The economic behavior of "owners' clones-forced-to-rent" is 
%          calculated by default only in the baseline setting.
%        - Wealth equivalent variations (WEV) are calculated by default in
%          all scenarios.
%        - Figure 8 (NWT: Own vs Rent), Figure 10 (NWT: Community vs. 
%          nursing home), Figure 11 (NWT: Parents vs. childless), and 
%          Figure 12 (Timing of inter-generational transfers) uses STATA to
%          plot the model data generated here. 

clear; clc; close all;                  % Create blank sheet and set path.
addpath('HelperFct')  
s = RandStream('mt19937ar','Seed',0);	% Fix seed or state of random-number
RandStream.setGlobalStream(s);          % generator: Will use this to draw from
                                        % wage distribution.

                                        
%% Basic settings 
mode    = 'baseline'; % Which run-mode:
                      % 'baseline': Run baseline model.
                      % 'counterf': Do counterfactual specified in variable
                      %             'CountFl'.
% Counterfactuals:                      
CountFl = ''; % Which counterfactual to run. Default:'' Otherwise:
%  COUNTERFACTUAL FEATURES
       % Children:      'noKid'     Childless agents.
       %                            These agents have no children, but we
       %                            start them as having parents before age
       %                            65. We use the value-function-guess-
       %                            finding code to solve for alone
       %                            agents' policies when old. Then, only
       %                            need one OLG iteration (i.e. we do
       %                            backward iteration for the no-kids
       %                            generation) and start off the no-kids
       %                            generation from the ergodic baseline
       %                            distribution ("no-kids" is basically an
       %                            MIT shock for the dynasty).
       % LTC:           'Sweden'    Cost of private NH (q, i.e. basic care 
       %                            services) is covered by government;
       %                            agents can still do FHC but have to pay 
       %                            the differential qFHC-q. MA is also
       %                            available, but should not be taken in
       %                            equilibrium. Financed by lump-sum tax
       %                            on young.
       % Housing:       'LiqHous'   Liquid housing: Means that there is
       %                            ONLY RENTING available.
       %                'omega1'    Still have different house sizes, but 
       %                            NO WARM GLOW from owning. Houses may 
       %                            still be bought for tax advantage or
       %                            strategic considerations.
% COMBINATIONS OF COUNTERFACTUAL FEATURES
       %              'noKidSwed'   No kids and full FC insurance.
       %           'noKidLiqHous'   Childless agents when only renting is
       %                            available.
       %            'LiqHousSwed'   Sweden when only renting is available.
       %         'noKidLiqHousSwed' No kids, only renting, full FC insurance.
       %                'Om1Swed'   Sweden without utility benefit of owning.
% COMBINATIONS PRESENTED ONLY IN APPENDIX
       %               'noKidOm1'   Childless agents and housing warm glow
       %                            is turned off.
       %           'noKidOm1Swed'   No kids, no utility benefit of owning, 
       %                            full FC insurance.

bargainMode = 'xout';   % The  baragining modes differ in how they assign
                        % bargaining power for DISABLED HOME OWNERS.
                        % 'xout':   Bargaining power depends on selling
                        % (default) decision under outside option. Kid gets
                        %           power if parent sells (default option).
                        % 'adhoc':  Parents have power, but kids have power
                        %           when parent is broke.
                        % 'parent': Parent always gets power.
                        % 'kid':    Kid always gets power. 
                        
bargainModeRent = 'parent'; % The modes differ how they assign bargaining
                        % power for DISABLED RENTERS.
                        % 'parent': Parent has all power and makes take-it-
                        % (default) or-leave-it offer. 
                        % 'kid':    Alternative for robustness check. All
                        %           power with the kid.
TaxCpl = true;          % If capital income is taxed; turn this off to remove
                        % the tax advantage of housing. Default: true.                      
SCmustSell = false;     % If we force skilled-LTC cases into nursing home.
                        % Default: false (can opt for FHC).
Stars = true;           % If there is a star prod. state. Default: true. 
nPan  = 2.5*10^5;       % Number of individuals in simulated panel. Set 0 if 
                        % no panel simulation is wanted. Default: 2.5*10^5.   
BurnInPan=4;            % How many OLG runs are done for burn-in of the 
                        % artificial panel. Default: 4.
% Displaying results during loop:                       
tpStop      = 0;        % Stop code at a this age of parent for debugging.
                        % Default: 0 (makes full run).
DisplayEvery= 10;       % Display every ... iterations in value function loop
                        % Default: 10.
if strcmp(mode,'baseline')
    CloneWars = true;   % Do cloning in artificial panel only in baseline.
else
    CloneWars = false;
end
 
% Baseline parameters 
r     = 0.02;           % Interest rate. Default: 0.02.
g     = 2.00;           % Risk aversion. Default: 2.
xi    = 0.81;           % Weight on consumption (versus housing).
                        % Default: 0.81. 
altrk = 0.11;           % Measure of altruism (not alpha!) for kid: The 
                        % ratio of consumption of recipient to donor that 
                        % the donor would choose when households have same
                        % size. 
deltaHouse  = 0.017;    % Depreciation rate of housing. Default: 0.017.


% Calibration parameters
rho   = 0.0418;        % Discount rate 
altrp = 0.65;          % Measure of altruism (not alpha!) for parent                    
omega = 1.53;         % Warm glow from owning house 
Cma   = 5.65;          % Consumption floor in Medicaid.      
fracLtcBasic = 0.74;   % Fraction of those entering disability that need 
                       % basic care services, rest needs skilled.
                       % (i.e. xi=0.26=1-0.74).
alphap= altrp.^(xi*(g-1)+1); % Obtain parent's altruism parameter...
alphak= altrk.^(xi*(g-1)+1); % ... and the child's.
valNoConsFeasible=0.0000001*Cma; % Set small consumption value when dynasty income 
                                 % is negative and the only choice for the parent
                                 % will be Medicaid. Default: 0.0000001*Cma. 
% Set up grid:
% Financial wealth:
N         = 21;         % Number of grid points for wealth grid.
                        % New default: 21.
aup       = 4000;       % Top grid point for wealth. 
linGrid   = 0;          % If to use a linear grid (1) or log-type grid (0).
                        % Default: 0.
% Now, turn this around: Set first grid point a1 where we want it to have.
a1 = 100;                       % Default: 100.
if a1>aup/(N-1)                 % If a1 is chosen so high that linear grid
    linGrid=true;               % fits in, do this.
end
if linGrid               
    ddaa = aup/(N-1);           % Create function handle that returns index
    ClosestAssetInd = @(a) min( 1+round(a/ddaa), N);
                                % of closest asset grid point for given a.
else                            % Otherwise: Find Abs0 so that a1=100 is true.
    dlog = @(A) (log(aup-A)-log(-A)) /(N-1);        % Distance of log grid for given A=Abs0.
    findAbs0 = @(A) log(-A) + dlog(A) - log(a1-A);  % Numerically solve equation to
    Abs0 = fzero(findAbs0,[-10*aup,-10]);           % find A=Abs0 that makes the second  
                                % grid point fall on a1.
    ddlog= dlog(Abs0);          % Distance on log-grid: scalar.
    llAbs0= log(-Abs0);         % Create function handle that returns
    ClosestAssetInd = @(a) min( 1+round((log(a-Abs0)-llAbs0)/ddlog), N );                  
end                             % index of closest asset grid point to given 
                                % a in log-sense (matters little since grid
                                % is locally almost linear).

% Housing wealth:
hmin = 50;                  % Set minimal...           (default: 50)
hmax = 2000;                % ..and maximal house size (default: 2000)
Nhous= 7;                   % Number of houses (except renting). Default: 7.
lhmin= log(hmin);           % Create log grid for housing:
logh = linspace(lhmin,log(hmax),Nhous);
ddlh = logh(2)-logh(1);
houseSizes = exp(logh);     % Make grid vector.
Nh = numel(houseSizes)+1;   % Number of housing states: Nhous+1 (renting!).
% Create function handle that returns index of closest housing grid point
% (in log-sense) or 1 if renter.
ClosestHouseInd = @(h) max( 1, min( 2 + round( (log(h)-lhmin)/ddlh ), Nh) );
                            % For renter, h=0 gives max(1,-Inf)=1 --> OK.

% Labor productivity:     
NwReg = 3;                  % Number of regular (non-star) grid points.
                            % Usually 3 or 5. Default: 3.
Nw = NwReg+Stars;           % Number of grid points for prod. is regular 
                            % number plus one star state in case there is
                            % one.
% Labor productivity hazard (also: earnings process fit)
betasqFixed  = 1.2;         % Default: 1.2.
lambdasqFixed= 1.2;         % Default: 1.2.
lambdasqIdx  = round(lambdasqFixed/0.05);
starFrac     = 0.075;       % Fraction of stars we want. Default: 0.075.
dstar        = 1.25;        % Default: 1.25.
hitStarFrac  = @(thetasq) StarFrac(betasqFixed,lambdasqFixed,thetasq) - starFrac;
thetasq      = fzero(hitStarFrac,[0.01,5]);
thetasqIdx   = round(thetasq/0.05);
outProdProcess = ProdProcess(lambdasqIdx,thetasqIdx,betasqFixed,dstar); 
ergDistEpsSq = outProdProcess.pi;
epsvecSq     = outProdProcess.epsvec;
epsHaz1D     = outProdProcess.HazMat;
mu           = outProdProcess.mu;
delta        = outProdProcess.delta;
f            = outProdProcess.f;
fprintf('Prod. process input parameters:\nlambda~ = %1.2f, theta~ = %1.2f, beta~ = %1.2f, dstar = %1.2f, starFrac=%1.2f\n',[lambdasqFixed,thetasq,betasqFixed,dstar,starFrac])
fprintf('Prod. process output parameters:\n mu=%0.2e, delta=%1.2f, f=%1.3f\n',[mu,delta,f])
disp('Ergodic distribution:'), display(ergDistEpsSq')
epsvec       = epsvecSq - epsvecSq'*ergDistEpsSq;

%% Tuning the algorithm
% Iterations and precision:
maxIterVF = 3;          % Maximal OLG iterations for value function.
                        % Results do not change perceptibly any more after
                        % 2 iterations for any parameterization, so can go
                        % down to 2 here. Default: 3.
tolOLG     = 0.0001;    % Log-distance of consumption to determine convergence
                        % of the OLG loop. 0.01 means 1% difference.
maxIterTax  = 2;        % Maximal iterations to find tax increment in 
                        % counterfactual. Default: 2, converges rapidly.                        
tolTax     = 10^(-5);   % Distance to determine convergence in loop to find
                        % tax residual in counterfactuals with budget.
                        % balance. 0.01 means a 1-percentage-point difference.
                        % Default: 10^(-5).
maxIterDens = 10;       % Maximal OLG iterations for density. Default: 10.                        
tolQ       = 10^(-3);   % Tolerance (in terms of ,000$; bisection method
                        % will yield result for transfer Q* that is
                        % tolQ-close). Default: 10^(-3).                                 
StoreEvery = 0.5;       % Could re-introduce this if we run into memory 
                        % problems: store policies only every ... years.
                        % Would have to re-program in FullModel_Solve.m.
                        % Store every iteration if below 1.                      
% Changing choice sets for agents, turn off shocks:
BKwp          = true;   % If there are WP transfers in BK. Default: true.
FHCoption     = true;   % If there is an option to take FHC for home owners.
                        % Default: true.
rentersChoose = true;   % If renters can choose apartment size freely. 
                        % (Is set to true automatically below in liquid-housing
                        % counterfactual.) Default: true.
MedShocks   = 1;        % If 0, turn off medical shocks. Default: 1.
nMaxMed     = 3;        % Number of grid points to approximate medical shocks. 
                        % Default: 3.

% Extrapolation:                       
ExtrapHouse = true;     % If to extrapolate value functions when house-selling
                        % makes the agent jump out of the wealth grid.
                        % Default: true.
ExtrapFrac  = 0.75;     % When we do extrapolation off the grid (house-
                        % selling, bequests), find a point at ExtrapFrac*aup
                        % inside wealth grid where to "anchor" our estimate
                        % of consumption-function slopes. Default: 0.75.
ForwCrossDer= true;     % If to use the forward quotient to calculate marginal
                        % value of the other player's assets (in
                        % Consumption2.m). Setting true makes WP transfers
                        % happen less often with non-linear grid, since 
                        % concavity of value function matters then.
                        % Works better than centered. Default: true.
BackwOwnDer = false;    % If to use backward quotient to calculate marginal 
                        % value of own assets (in Consumption2.m). Note that
                        % this derivative is used for both getting transfers
                        % and for getting consumption. Did not work as well
                        % as using centered. Default: false.
reflect     = true;     % Reflect on grid boundaries: reflecting is more
                        % stable. Default: true.
                        
% Smoothing:   
%   - general: 
sigmaA      = 0.05;     % Iid uncertainty in return to wealth. Default: 0.05. 
kpa         = 0.02;     % kappa: Standard deviation of shocks to cMA (for 
                        % convexification of MA-decision) -- tried to take
                        % this out, but ran into trouble. Default: 0.02.
Khaz        = 1;        % K: Factor at which to increase death hazard at 
                        % age 95 in order to make dying less sudden. We
                        % multiply all death hazards at age 94 by this
                        % number and leave death hazards until age 93 as in the
                        % data. We vary j death hazard smoothly on vary
                        % for ages on [93,95] according to Khaz^(j-93).                        
                        % Default: 1.                     
%   - smoothing transfers:
wgtUnc      = 0.99;     % Pseudo-bargaining weight when players unconstrained:
                        % Is 1 in true model, but go slightly away in order
                        % to avoid false zeros. Default: 0.99.
QmaxFact = 2;           % Factor: How many times income (labor plus asset
                        % minus fixed spending) can maximal transfer be?
                        % Used to have QmaxFact=4. With stars and large
                        % asset grid, QmaxFact=2 works better. Default: 2.
taubar = 0.05;          % Parameterizes approximation to heavy-side step
                        % function that we use to make transfers continuous. 
pbar = 0.00001;         % Parameterizes approximation to heavy-side step function
                        % that we use to make transfers continuous when one
                        % of the transfer motives turns positive. We allow
                        % for the full transfer if p=tau_p/(tau_p+tau_k) is
                        % between pbar and 1-pbar. Default: pbar=0.00001.
                        
% Drawing the panel:                        
dtPan      = 2;         % Time interval for simulated panel: aggregate 
                        % variables over dtPan years. Default: 2 (as in HRS).               


%% Policy parameters
tauSS      = 0.124;     % Social-Security tax rate. Default: 0.124.
sf         = 0;         % Formal care subsidy: 0 in baseline, 
                        % changes in the counterfactuals (Sweden).
tauEst     = 0;         % Estate tax. Default: 0.
tauLTC     = 0;         % Tax rate needed for LTC policy: 0 in baseline, 
                        % changes in the counterfactuals.

%% Baseline demography
tRet       = 65;        % Retirement age. At this age, start to have fixed 
                        % soc.-sec benefit, and the death and LTC hazards 
                        % kick in.                             Default: 65.
dGen       = 30;        % Age difference between generations. Set this to 1  
                        % or 2 for test runs to shorten loops. Default: 30.
tDth       = tRet+dGen; % Die with certainty at this age.      Default: 95.
tOLG       = tRet-dGen; % Enter the overlapping stage at this age.Def.: 35.

%% "Technical parameters": Taken from data
qMA2010        = 48.06*(1.05)^6; % average Medicaid reimbursement rate
fracCareNHcost = 0.548;  % Percentage of NH costs due to care.                      
                         % Called phi in Table 3 (Calibration) in paper.
pbc  = fracCareNHcost*qMA2010; % price of basic care services     
FHCcost= 38.38;         % Calculated based on the median number of care hours (210) 
                        % in our data times the 0.5*(BLS+Metlife) hourly rate of home
                        % health aide. Default=38.38. 
FHCrat = FHCcost/pbc;   % How much more formal home care costs (in terms of 
                        % basic care services) than a nursing home. Is 
                        % greater than one due to returns to scale.
SCrat = 1.00;           % Factor by which price of skilled care is more 
                        % expensive than basic care. Default: 1.
phi     = @(n) 1+0.7*(n-1)-0.2*max(n-2,0);    
                        % Adult equivalence scale from OECD, function of
                        % number of household members n. Default:
                        %  1+0.7*(n-1)-0.2*max(n-2,0)
% Gender-wage gap:                        
beta0Data = 2;          % Baseline: Male earns twice female's wage,
                        % contributes beta=2/3 to household income.
                        % Default: 2.
mincer = @(j) 9.84126 + .0646223*(j-20)  -.0010749*(j-20).^2;
% Log-wage of an individual as function of age j. Mincer regression for an 
% individual: estimates from 2010 Census data (IPUMS). Efficiency units are 
% estimated in terms of dollars in the data. 
tauOldRich = 0.1;       % Ad-hoc tax rate on the rich old's SS income: Make 
                        % this payable to all above the median
                        % productivity. Default: 0.1.
thrSS      = [0.2, 1.25, 2.46];
                        % Thresholds in terms of average household income
                        % to determine social-security benefits.
rplRatSS   = [0.9, 0.33, 0.15];
                        % Marginal replacement rates for social-security
                        % benefits in the categories.

% In order to obtain gross household labor income we multiply an 
% indididual's labor earnings by 1+beta0: marginal + infra-marginal worker
% Labor force participation:
MalLFP = 1.0;                % Assume that all men work. Default: 1.
FemLFP = MalLFP/beta0Data;   % Female labor-force participation 
                             % would be this. Default: MalLFP/beta0Data;                                 
HHeffUnits  = @(j,eps) (MalLFP+FemLFP)*exp( mincer(j) + eps ) / 1000;
                             % Efficiency units of labor of an
                             % age-j, productivity-eps household.
                             % Multiply by number of working , divide
                             % by 1000 to get it in $K of year 2010.
                                    
% Initial distribution of assets:
grossLbrInc = HHeffUnits(35,epsvec);
TaxLbrInc   = (1-tauSS)*grossLbrInc;
netLbrInc   =( 1 - tax(TaxLbrInc,0) ) .* TaxLbrInc; % Net labor income
initWlthFactor = [0 0.5 1 1.5]';    % Set initial wealth for children as
                                    % multiple of yearly income. Default:
                                    % [0 0.5 1 1.5]' gives reasonable net
                                    % worth levels as we find in the SCF.
startA = initWlthFactor.*netLbrInc;    
% ergodic fractions:  31%     38%     23%    8%
%            startA:  0       22      86     307
% For initial density guess: With the following data, create a reasonable
% initial guess.
varLogAss45   = 1.1413; % Variance of log wealth at age 45 from data.
                        % Default: 1.1413.
ratAssEarn45  = 3;      % Ratio of wealth to earnings at age 45 in data
                        % (in levels, not logs!). Default: 3.
condVarAss45  = 1.044;  % Variance of log wealth conditional on earnings at
                        % age 45. Default: 1.044.
betaAssEarn45 = 0.4;    % Elasticity of wealth with respect to earnings.
                        % Default: 0.4.

% The vector mapEpsToEduc tells us for each of the Nw productivity states 
% to which of the four education categories in the data we map them (do
% this by roughly matching frequencies with the ergodic distribution).
% ****** Ergodic distribution ***********
%    0.3125    0.3750    0.2344    0.0781
% ***************************************
%   Education categories:
%           0: No high school (omitted category in
%              regressions, coded 1 in Stata, includes
%              previous 2s).
%           1: High school (coded 3 in Stata).
%           2: Some college (coded 4 in Stata).
%           3: College (coded 5 in Stata).
%   Productivity states: (default Nw=4)
%           First grid point:  Less than high-school (19% in our data).
%           Second grid point: High school (may have SOME college; 58% in data).
%           Third grid point:  College (23% in data).
%           Fourth grid point: Make stars college, too.
if Nw==3
    mapEpsToEduc = [0, 1, 3];          % For three grid points.                   
elseif Nw==4
    mapEpsToEduc = [1, 1, 2, 2]; 
elseif Nw==5                     
    mapEpsToEduc = [0, 1, 1, 2, 3];
elseif Nw==6
    mapEpsToEduc = [0, 1, 1, 2, 3, 3];
else
    fprintf('Have to specify mapping of productivity states to education levels for %1.0f prod. states!\n',Nw)
    keyboard
end
           
                        
% Now, read in coefficients from logits in the HRS data: 
                  % high sch.  some coll.   college     age
coeffLambdaFem = [-1.0948289  -1.6480283  -1.4372283  .03366679 ...
                   .02401442   .04122472    .038168   .00208306   -2.2810568];
                  % age*h.s.   age*some c.  age*coll.   age^2    constant
                        % Coefficients of logit of disability on
                        % observables --> estimates fraction of disabled in
                        % the alive population for given characteristics.
                 % high sch.  some coll.   college     age       
coeffPiHlthFem = [-.83241673  -.82313442  -.98370948  .04568197 ...
                  .03476895    .02754114   .03452614  .00082835   -3.1451009];
                  % age*h.s.   age*some c.  age*coll.   age^2    constant
                        % Coefficients of logit for death hazard of healthy
                        % (non-disabled) women. Left-hand-side variables
                        % are the same as for coeffLambdaFem. 
                  % high sch.  some coll.   college     age       
coeffPiSickFem = [ 0           0            0         .00644476 ...
                   0           0            0         .00121397   -1.1130199];
                  % age*h.s.   age*some c.  age*coll.   age^2    constant
                        % Coefficients of logit for death hazard of
                        % disabled women on same LHS variables as above
                        % --> Threw out some in selected model.
% Same for males now:
                  % high sch.  some coll.   college     age 
coeffLambdaMale = [-1.052385   -1.0090777   -1.5099989  .06231446  ...
                    .03720323    .01201594    .03186618   .0006974  -2.4520252];
                    % age*h.s.   age*some c.  age*coll.   age^2    constant
                        % Coefficients of logit of disability on
                        % observables --> estimates fraction of disabled in
                        % the alive population for given characteristics.
coeffPiHlthMale = [-.36721596  -.27066356   -.37782969  .05122931 ...
                     .02228407   .00117198   -.00989503  .00138806  -2.7685909];
coeffPiSickMale = [.24372047    .40266115    .35724119   .0684551  ...
                    0            0           0            0         -1.2358087];
% Pack all coeffisients into an array:              
coeffArr = cat(3, [coeffLambdaMale; coeffPiHlthMale; coeffPiSickMale] , ...
                  [coeffLambdaFem;  coeffPiHlthFem;  coeffPiSickFem ]  );
                        % 1st dim: Type of model (lambda, piH, piS).
                        % 2nd dim: Regressors in model.
                        % 3rd dim: Gender.
% Coefficients of linear prob. model for entering into nursing home for the
% hh stand-in (i.e. the surviving spouse):
                  % high sch.  some coll.   college     age       
coeffNHentry =   [-.0048235   -.0210752     -.0195835  .0058313  ...
                   .0007664   -.0005635     -.0007378  .000297   .0494922];
                  % age*h.s.   age*some c.  age*coll.   age^2    constant                        
 
% Hazard rate of medical shock for INDIVIDUAL (regression on disabled dummy):                        
indMedHazHlth = 0.1237489;   % Hazard of healthy (the regr. constant).
indMedHazSick = indMedHazHlth + 0.1803829; % Hazard of sick (plus 
                             % coefficient on disablity).
% For 2010:
medFixC = 0.6376;    % Fixed medical spending for people above 65 years 
                     % old is 637$ per year (from Stata estimation).  
                     % Default: 0.6376.
% Medical-cost shocks in data (in $ from 2010): 
% E[ln m] = 7.194371, StDev[ln m] = 1.510796
medMu = 7.194371 - log(1000);
% (need to divide by 1000 to get this in ,000$)
medSig = 1.510796;      % st.dev, also conditional. Default: 1.510796.
qnorm = norminv([0.5 0.75 0.9, 0.95, 0.99]); % Get these quantiles of the shock 
medShockQ = exp( medMu + qnorm*medSig );     % distribution to see if reasonable.
% Stata gives the following quantiles for those with permanent income in
% highest fifth:
% Q50: 1,318,  Q75: 3,215, Q90: 6,573, Q95: 12,157, Q99: 2,9143

%% Set up everything for main program: Hazards, initial density etc.

% Set up asset grid:
if linGrid==1
    avec = linspace(0,aup,N);           % Asset grid vector: 1-by-N.
    da   = aup/(N-1);                   % Mesh size of wealth grid. Scalar.
    avecExt = 0:da:aup+hmax;
    da   = diff(avec); % !!! need this to recylce code for non-linear grid
else                                    % Non-linear grid:
    loglb = log(0-Abs0);                % Lower bound for log grid.          
    logub = log(aup-Abs0);              % Upper bound for log grid.
    loggrid = linspace(loglb,logub,N);  % Create the log grid and then the
    avec = exp(loggrid)+Abs0;           % log-type asset grid, just the way 
                                        % GetLogGrid.m does it. 
    avec(1) = 0;                        % Set first grid point precisely to 
                                        % zero (can turn negative due to
                                        % computational error).
    logextup = log(aup+hmax-Abs0);      % Also create extended grid for house
                                        % selling. Need hightest assets
                                        % plus maximal house.
    dloga = loggrid(2)-loggrid(1);      % Get spacing of log-style grid. 
    logextgrid = loglb:dloga:(logextup+2*dloga); % Extend log grid to include
                                        % at least two grid points above
                                        % the maximal assets that can be
                                        % reached when selling house,
                                        % which are log(a_max+h_max).
    avecExt = exp(logextgrid)+Abs0;     % Create extended grid vector.
    avecExt(1) = 0;
    da   = diff(avec);                  % Distances: 1-by-(N-1) vector.
end
minda = min(da);                        % Get minimal distance.

% First, figure out what time increment dt should be:
maxInc = 50 + aup*r;                 % Set maximal income.
aDotMax    = (1+2*QmaxFact)*maxInc;  % Maximal drift we expect: gifts
                                     % can bet up to Qmax, and so does consumption.
maxSavHaz  =  (aup^2*sigmaA^2 + minda*aDotMax)/minda^2;
                        % Get maximal hazard rate of leaving a grid point we
                        % expect from savings: set max. cons. to 500
% Jumps for exogenous reasons: 
% - 0.31: Max. medical jump-hazard is 0.31
% - 0.35: Max. LTC hazard
% - 0.40: Max. death hazard
% - 0.05: Max. hazard of kid-productivity change
% --> max. hazard 1.1 of leaving grid point for exogenous reasons.
maxExogHaz = 1.1 + Stars*betasqFixed*0.1;
                        % Add star persistence if we have stars.
maxLeavHaz = maxExogHaz + 2*maxSavHaz;
                        % Add all up: Need twice the savings hazard since
                        % there are two agents.
dt = 1/ceil(maxLeavHaz);% Set dt so that the probability of staying at grid 
                        % point is safely above 0. Take ceiling so that a
                        % a cycle exactly ends after one year.
beta = beta0Data/(1+beta0Data); 
                        % Fraction of household income made by
                        % inframarginal worker (male): Get from gender-wage
                        % gap in the data.
  

% Set housing grid:
h0   = 0.1;  % SET A SMALL NUMBER THAT IS EASY TO SPOT 
rent = h0; % If renters choose, set small number here for computational reasons
% Now, will allow houses of any size (need not fall on wealth grid points):
hvec = [h0; houseSizes(:)];    % Nh-by-1 vector with house sizes.

% Create grids. State space is N x N x 2 x Nw x Nw:
% dim 1: ak, wealth of kid
% dim 2: ap, wealth of parent
% dim 3: old are healthy or need LTC
% dim 4: w, wage (or later pension) of young
% dim 5: P, pension (or wage) of old
% dim 6: h, housing of old
Ns = 3;                                 % 3 health states (death is not 
                                        % coded as state).
stDim = [N,N,Ns,Nw,Nw,Nh];              % Dimension vector of state space.
% 6-dim array for all states:
akgrid = reshape(  avec, [N,1,1,1,1,1]);
apgrid = reshape(  avec, [1,N,1,1,1,1]);
dak    = reshape(  da, [N-1,1,1,1,1,1]);
dap    = reshape(  da, [1,N-1,1,1,1,1]);
epsk   = reshape(epsvec,[1,1,1,Nw,1,1]);% Young's log-productivity (dim.4).
epsp   = reshape(epsvec,[1,1,1,1,Nw,1]);% Old's log-productivity (dim.5).


if Nh>1
     hgrid  = reshape(  hvec,[1,1,1,1,1,Nh]);
elseif Nh==1
     hgrid  = h0;
     hvec   = h0;
     if  Nh>1 && hvec(2)<avec(2)                  % % Housing grid (dim.6).
%         If there is housing: Give warning if smallest house too small.
         fprintf('WARNING: Smallest house (%1.0fK) sits below 2nd asset grid point (%1.0fK)!\n',[hvec(2),avec(2)])
     end
 end

Mextrap = find(avec>=ExtrapFrac*aup,1); % Find closest asset grid point to
                                        % ExtrapFrac * aup and get index
                                        % Mextrap at which we anchor the
                                        % slopes for consumption functions.
Mextrap = min(Mextrap,length(avec)-2);
VarAp  = (sigmaA*(apgrid)).^2;          % Variance in asset accumulation
VarAk  = (sigmaA*(akgrid)).^2;          % for parents and kids: Does not 
                                        % change over loop, so put into
                                        % 'par' structure. 1-by-N and N-by-
                                        % 1 vectors, respectively.
                                        
% Now, get joint productivity distribution of parent and kid when kid is born
% into its economic life. 
StartDist = diag(ergDistEpsSq); % Start them off perfectly correlated,
                                % parents (and children) are distributed
                                % according to the ergodic distribution.
MargDistEpsPar= sum(StartDist,1); 
                          % Starting distribution of parents is ergodic
                          % distribution as well.
MatchProb = eye(Nw);    % Have parent and child state fully correlated: 
                        % Ensures we start off children at ergodic
                        % distribution and need perfect correlation to get
                        % inter-generational correlation at LTC age right.
% Wages and productivity:
% Zero profits in the consumption-good and private-care sectors imply 
%  q = w = Ay 
Ay      = 1;                    % Productivity in good sector: normlized.
Af      = (fracCareNHcost*qMA2010)^(-1);
yMA     = (1-fracCareNHcost)*qMA2010;                            
                                % Consumption-goods input into Medicaid
                                % nursing home from year 2010. 

% Obtain hazard functions for LTC and death (for healthy and sick):
hazard = LTChazard2(coeffArr,0,dt); % Obtain coefficients for LTC and death.
                                    % Don't make figures.
polySigmaLTC = hazard.polySigmaLTC; % Hazards; read out from output
polyDelta    = hazard.polyDelta;  
polyDeltaLTC = hazard.polyDeltaLTC;     
polyMenAlv   = hazard.polyMenAlv;    
initLtcDat   = hazard.initLtcDat;
Smod         = hazard.S;            % Fraction of LTC cases by age and educ.

 
% Create jump matrix due to medical shocks:
% Even with small grid of 20K, the probability of receiving shock larger
% than 50K (i.e. jump three grid points or more) would be only 0.1%,
% conditional on making a draw (only 16% of sick make draw per year). So
% just use jumps of one or two grid points down to discretize medical
% shocks. 
p2dn = 1-logncdf(1.5*minda,medMu,medSig); 
                                    % Probability of going two grid points
                                    % down: Take empirical probability of
                                    % paying more than 1.5 times the grid
                                    % size.
meanMedShk = exp(medMu+0.5*medSig^2);
                                    % Calculate mean of log-normal
p1dn = (meanMedShk-p2dn*2*minda)/minda;   % Set prob. of going one grid point down 
                                    % to match mean of medical shock.
medProb = zeros(N,N);               % Set up transition matrix due to medical 
                                    % shocks. Off the diagonal, it contains
                                    % the transition probability
                                    % conditional on a medical. On the
                                    % diagonal, it contains the negative of
                                    % the probability for leaving the
                                    % current state --> need it in this
                                    % form to form transition matrices.
                                    % When already at lowest grid point, 
                                    % stay there for sure: only zeros in
                                    % first row. 
medProb(2,1:2) = [p1dn+p2dn, -p1dn-p2dn];
                                    % Can only go one down from 2nd grid
                                    % point, then loop over all others 
for k=3:N                           % where there are two jump prob's.
    medProb(k,k-2:k) = [p2dn, p1dn, -p1dn-p2dn];
end
medProb = sparse(medProb);          % Make matrix sparse.

medGovtSpdGrd = [p1dn*minda + p2dn*2*minda; ...
                 p2dn*minda];          % Government spending for the lowest 
                                    % two grid points of parent wealth when
                                    % hit by med. shock: pay everything for
                                    % those who are broke, and only part of
                                    % spending for those hit by hard shock
                                    % when on 2nd grid point.


% Construct initial wealth density: Used only for initial guess, does not
% matter what we put here. 
woMean = log(HHeffUnits(50,0));  
woVec  = woMean + epsvec;           % Log-wage at age 50 for all productivities.
varLogW =  0.5716;
MeanLogAss = log(ratAssEarn45) + woMean + 0.5*(varLogW-varLogAss45);
                                        % Unconditonal mean of log wealth
                                        % in data. Need to adjust by
                                        % variances since ratAssEarn45 is
                                        % in levels, not logs.
condLogMean   = MeanLogAss + betaAssEarn45*(woVec-woMean);
                                        % Mean of log-wealth, conditional
                                        % on the wage.                                       
condLogVar    = condVarAss45*ones(length(epsvec),1);
                                        % Conditional variance: constant
aa    = [0, (avec(1:N-1)+avec(2:N))/2, 2*avec(N) ];                                        
nInit = zeros(N,1,Nw);                  % Initial distribution over assets 
                                        % productivity: Set up array.
                                        % (Have sickness dimension on 2:
                                        % nobody sick in beginning).
for i = 1:Nw                            % Loop over all productivity levels:
    iCdf = logncdf(aa,condLogMean(i),sqrt(condLogVar(i))); 
                                        % Get cond. cdf of assets for this prod.
    iCdf(end) = 1;                      % Put all super-rich into top grid point.
    nInit(:,1,i)= MargDistEpsPar(i)*diff(iCdf);
                                        % Take difference to get cond. pdf.
end                                     % and multiply by marginal to get
                                        % joint distribution.

HHgrossLbrInc = HHeffUnits(35:64,epsvec);
                                        % Get household income by age and
                                        % prod. state: Nw-by-30 matrix.
nHH = MargDistEpsPar'.*ones(1,30)/30;   % Get fraction in each bin:
                                        % Nw-by-30 matrix. Distribution
                                        % over age uniform since kids don't
                                        % die.
figure('Name','Household income distribution')
subplot(1,2,1)                          % Now, make bar plot for frequency
bar(HHgrossLbrInc(:),nHH(:))            % distribution.
xlabel('HH gross income (K$, yearly)'), ylabel('frequency')
subplot(1,2,2)                          % Then get cdf and plot.
[cdfHH,incOrd,nHHOrd] = CdfProbDist(HHgrossLbrInc,nHH);
plot(incOrd,cdfHH);  xlabel('HH gross income (K$, yearly)'), ylabel('cdf')
% Finally, get some other statistics of interest and put in figure:
meanInc = sum(incOrd(:).*nHHOrd(:));    % Get mean.                                        
Ginc = GiniProbDist(incOrd,nHHOrd,1);   % Get Gini.
qtl = [0.1,0.25,0.5,0.75,0.9,0.95,0.99];% Quantiles we want to compute.
[qInc,shInc] = GetQuantProbDist(HHgrossLbrInc(:),qtl,nHH(:));
                                        % Get quantiles and income shares
                                        % of households above this
                                        % quantile.
% Put text boxes into the cdf plot with statistics:                                        
xtxt = 200;                             % x-value where to put text.
ytxt = 0.5;                             % y-value where to start.
dytxt= 0.05;                            % Distance between y-values.
text(xtxt,ytxt+dytxt,sprintf('mean=%3.0fK$',meanInc));  % Print mean,
text(xtxt,ytxt,sprintf('Gini=%1.3f',Ginc));             % Gini,
text(xtxt,ytxt-dytxt,'Quantiles Q and income share>=Q:')% and then the 
for i=1:numel(qtl)                      % quantiles and income shares.
    itxt = sprintf('Q%1.0f: %3.0fK$, %4.1f%%',[100*qtl(i),qInc(i),100*shInc(i)]);
    text(xtxt,ytxt-(i+1)*dytxt,itxt);
end
close all;

avHHeps = zeros(size(epsvec));          % Get average labor earnings over age 
for i=1:length(epsvec)                  % for each productivity category
                                        % to calculate tax thresholds.
    avHHeps(i) = mean(HHeffUnits(20:64,epsvec(i))); % Here, we take individuals 
end             % from age 20 on as we have in the real economy. Otherwise, average
% is unrealistically high                                        
AvHH  = MargDistEpsPar*avHHeps;         % Get average economy-wide labor earnings 
                                        % multiplying with the initial productivity
                                        % distribution (OK since ergodic).
ss = AvHH*thrSS;                        % Multiply social-security thresholds  
                                        % with average labor earnings.
SS = @(x)           (x<ss(1)).*( rplRatSS(1).* x  ) + ...
       + (ss(1)<x).*(x<ss(2)).*( rplRatSS(1)*ss(1) + rplRatSS(2)*( x   -ss(1))) + ...
       + (ss(2)<x).*(x<ss(3)).*( rplRatSS(1)*ss(1) + rplRatSS(2)*(ss(2)-ss(1)) + rplRatSS(3)*( x   -ss(2))) + ...
       + (ss(3)<x)           .*( rplRatSS(1)*ss(1) + rplRatSS(2)*(ss(2)-ss(1)) + rplRatSS(3)*(ss(3)-ss(2)));
                        % SS benefits rule, each income category is in one
                        % line (checked that it gives the same as in folder Code/)                       
Pgross = SS(avHHeps);   % Gross pension is the SS-benefit rule applied to 
                        % the mean household income a household would have
                        % had if stuck in one productivity category
                        % forever.
indMed = ceil(Nw/2);    % Get index for median productivity.
Pvec   = Pgross;        % Set SS income to gross income by default,...
Pvec(indMed+1:end) = (1-tauOldRich)*Pgross(indMed+1:end);
                        % ... but take out the ad-hoc tax for the rich.
% Create 6dim-arrays for pension payments:
Pk = reshape(Pvec,[1,1,1,Nw,1,1]);      % 6D array for kid's pension.
Pp = reshape(Pvec,[1,1,1,1,Nw,1]);      % 6D array for parent's pension.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Determine equilibrium prices: 
[w, q, qma, qfhc, qsc] = EqmPrices(Ay,Af,yMA,FHCrat,SCrat);
% Create (1,Ns,1,1,1,Nh) array of price of formal care:
qnet = zeros(1,1,Ns,1,1,Nh);            % Set up price array: zeroes for 
                                        % the healthy.
qnet(1,1,2,1,1,1   ) = q   -sf;         % Basic NH care.                                        
qnet(1,1,2,1,1,2:Nh) = qfhc-sf;         % Formal home care.
qnet(1,1,3,1,1,1   ) = qsc -sf;         % Skilled NH care (only first housing
                                        % housing dimension will be used, 
                                        % assume NH is necessary for these)
qnet(1,1,3,1,1,2:Nh) = qfhc-sf;         % Home owners with skilled care                                        
                                        % can also purchase FHC
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Pack structure with model parameters (everything that does not change on
% each loop iteration).
par.alphap= alphap; par.alphak= alphak; par.altrp = altrp; par.altrk = altrk;
par.omega= omega;par.xi   = xi;  % par.eta  = eta;  
par.r     = r;      par.rho   = rho;    par.g    = g;    par.Cma  = Cma;
par.FHCrat= FHCrat; par.valNoConsFeasible = valNoConsFeasible;
par.sigmaA= sigmaA; par.kpa  = kpa;      %par.bargW = bargW;   
par.beta  = beta;   
par.tauEst= tauEst; par.deltaHouse = deltaHouse; % par.sigmaHouse = sigmaHouse;
par.QmaxFact = QmaxFact;  
par.taubar= taubar; par.pbar  = pbar; par.tauOldRich = tauOldRich;
% Care costs & technology:
par.Ay    = Ay;     par.Af    = Af;     par.yMA  = yMA;   
% Pack in grid-related parameters:
par.linGrid=linGrid;
par.N      = N;     par.Nw    = Nw;     par.Nh   = Nh;  par.Ns = Ns;  % grid size
par.ClosestAssetInd = ClosestAssetInd;
par.aup    = aup;   par.avec  = avec;   par.avecExt = avecExt;  % par.da   = da;      % wealth 
par.apgrid = apgrid;par.akgrid= akgrid; par.dap  = dap; par.dak = dak;
%par.sgrid= sgrid;   % sickness 
par.VarAp  = VarAp; par.VarAk = VarAk;  par.varLogW=varLogW;  
par.epsvec = epsvec;par.epsk  = epsk;   par.epsp = epsp;    % income
par.Pvec   = Pvec;  par.Pk    = Pk;     par.Pp   = Pp;      % pensions 
par.Pgross = Pgross;
par.hvec   = hvec;  par.h0    = h0;     %par.hbar = hbar;    % housing
par.ClosestHouseInd = ClosestHouseInd;
par.hgrid  = hgrid; par.epsHaz1D = epsHaz1D; % par.HouseTransCost = HouseTransCost;
par.rent   = rent;  % par.aIndHouse = aIndHouse;
par.dGen   = dGen;  par.tDth  = tDth;   par.tRet = tRet;    % life cycle  
par.tOLG   = tOLG;  par.dt    = dt;
par.nPan   = nPan;  par.dtPan = dtPan;  par.BurnInPan = BurnInPan;  % simulated panel
par.q      =    q;  par.w     = w;      par.qma  = qma;  par.qnet = qnet; 
par.TaxCpl = TaxCpl; 

% Then fill in transition matrices, functions for life-cycle hazards, taxes
% etc. 
par.outProdProcess = outProdProcess; 
par.nInit     =nInit;      par.phi      =phi;       %par.corrW  = corrW; 
par.HHeffUnits=HHeffUnits; par.tauSS    =tauSS;     par.tauLTC = tauLTC;   
par.nMaxMed   =nMaxMed;      
par.MatchProb =MatchProb;  par.startA   =startA;  par.MargDistEps=MargDistEpsPar;
par.polySigmaLTC = polySigmaLTC; par.mapEpsToEduc = mapEpsToEduc; % par.kidEducFr    = kidEducFr;       
par.polyDelta    = polyDelta;       par.polyDeltaLTC = polyDeltaLTC;
par.polyMenAlv   = polyMenAlv;      par.fracLtcBasic = fracLtcBasic;     % par.polyMenLTC   = polyMenLTC;  par.fracMenIC    = fracMenIC;  
par.Khaz = Khaz;                    par.Smod = Smod;  
par.medProb      = medProb;         par.medGovtSpdGrd= medGovtSpdGrd;
par.indMedHazHlth= indMedHazHlth;   par.indMedHazSick= indMedHazSick;
par.medFixC      = medFixC;  par.medMu = medMu;  par.medSig = medSig;
par.coeffNHentry = coeffNHentry;    par.ErgDistEps = ergDistEpsSq;

par.InfC = true;                % By default: Have IC option (is removed in one counterfactual).
% Data moments:
par.initLtcDat        = initLtcDat;
par.rentersChoose = rentersChoose;  par.BKwp        =        BKwp;      
par.ExtrapHouse   =   ExtrapHouse;  par.FHCoption   =   FHCoption;                     
par.wgtUnc        =        wgtUnc;  par.bargainMode = bargainMode;
par.bargainModeRent = bargainModeRent; par.SCmustSell = SCmustSell;

% Now, pack structure 'opt' with the options for the algorithm:
opt.dt         = dt;
opt.tolQ       = tolQ;          % opt.dtMax       = dtMax;      
opt.MedShocks  = MedShocks;   opt.Mextrap     = Mextrap;
opt.tpStop     = tpStop;      opt.CountFl     = CountFl; 
opt.tolOLG     = tolOLG;      opt.reflect     = reflect;           
opt.ForwCrossDer=ForwCrossDer;opt.BackwOwnDer = BackwOwnDer;
opt.maxIterVF  = maxIterVF;   opt.maxIterDens = maxIterDens;
opt.maxIterTax = maxIterTax;  opt.DisplayEvery= DisplayEvery;
opt.Stars      = Stars;       opt.CloneWars   = CloneWars;

%% Baseline option
if strcmp(mode,'baseline')                  % If run-mode is baseline:
    opt.CountFl = '';
    opt.noKid   = false;                    % Agents have kids.                                  % Otherwise, solve model.
    tic                                 % Start time.
    fprintf('RUNNING BASELINE MODEL:\n')% Give message.
    DisplaySettings(par,opt);           % Display parameter settings.
    res = SSG_SolveModel(par,opt); % MAIN PROGRAM: Solve model.
    % save('resBaseline.mat', '-v7.3') % save entire workspace once after
    % calculating the baseline economy
    disp('-----------------------------------------------------------------')
    % load('resBaseline.mat')  % load baseline economy (if saved before) 
                               % if needed 
    fprintf('SHOWING BASELINE RESULTS:\n')  % Show results.
    mom = SSG_GetMoments(res,par,opt,1);       % Get moments from baseline
    % save('momBaseline','mom') % save the results of the baseline economy
    % once; some elements are inputs to the counterfactuals. 
    disp('-----------------------------------------------------------------')
    fprintf('DONE.\n')                      % Give message and
    toc                                     % show elapsed time.
end

%% Counterfactual option

opt.noKid = false;              % By default, agents have kids.

if strcmp(mode,'counterf')      % If we are in counterfactual mode:
    tic                         % Start to measure time.
    disp('-----------------------------------------------------------------')
    % Obtain tax increment needed to finance universal LTC coverage for
    % Sweden counterfactuals:
    load('momBaseline','mom')
    dtauSS = dTaxSweden(par,mom.care.all.MA); % Use model MA rate
    clear mom
    if strcmp(CountFl,'noKid')  % No-children counterfactual: Do finite-
        % horizon backward iteration.
        disp('Counterfactual: No children')
        load('resBaseline','res')
        par.apInit = res.pan.once.Nwp65(:,1,1);   % Append initial states of panel members
        par.EpInit = res.pan.once.EpInd65; % as initial distribution: nPan-by-1 vectors.
        clear res
        par.BurnInPan = 1;       % Only have one burn-in run: Parents of the
        % childless and childless as kids. Final
        % run is then childless as parents.
        opt.maxIterVF   = 1;     % Need only one value function iteration:
        % Childless have parents themselves, but foresee rationally that
        % they don't have kids in old age. Thus:
        opt.maxIterDens = 1;     % Need only one density loop.
        opt.noKid       = true;  % No-kids marker is true.
        countMom        =  'noKid'; %#ok<NASGU>  % Suppress Matlab message
    end                                         % that not used.
    
    if strcmp(CountFl,'Sweden')  % Sweden counterfactual:
        disp('Counterfactual: Sweden (full FC insurance)')
        par.tauSS = par.tauSS + dtauSS; % Increment SS contribution to pay
        sf = zeros(size(qnet));         % for better care.
        sf(1,1,2:Ns,1,1,1:Nh)=q;
        par.sf   = sf;               % Set formal-care subsidy to price of
        par.qnet = par.qnet - sf;    % basic care services. FHC still has to
        countMom = 'Sweden'; %#ok<NASGU>
    end                             % be paid partly, i.e. qfhs-sf>0.


    if strcmp(CountFl,'LiqHous')    % Liquid-housing counterfactual:
        disp('Counterfactual: Liquid housing')
        par.qnet   = zeros(1,1,Ns,1,1,1); 
        par.qnet(1,1,2:3,1,1,1)=q;    
        par.hgrid  = h0;
        par.hvec   = h0;
        par.Nh     = 1;                     % Only have renters now.
        rentersChoose = true;
        countMom  = 'LiqHous'; %#ok<NASGU>
    end

    if strcmp(CountFl,'omega1')  % For no-utility-from-owning counterfactual:
        disp('Counterfactual: No owning utility')
        par.omega      = 1;      % Set omega to zero.
        countMom       = 'omega1'; %#ok<NASGU>
    end

    if strcmp(CountFl,'noKidSwed')  % Sweden without kids counterfactual.
       disp('Counterfactual: Sweden without kids')
        % Sweden:
        par.tauSS = par.tauSS + dtauSS; % Increment SS contribution to pay
        sf = zeros(size(qnet));         % for better care.
        sf(1,1,2:Ns,1,1,1:Nh)=q;
        par.sf   = sf;
        par.qnet = par.qnet - sf;
        % noKid:
        load('resBaseline','res')
        par.apInit = res.pan.once.Nwp65(:,1,1);   % Append initial states of panel members
        par.EpInit = res.pan.once.EpInd65; % as initial distribution: nPan-by-1 vectors.
        clear res
        par.BurnInPan   = 1;
        opt.maxIterVF   = 1;        % Need only one value function iteration.
        opt.maxIterDens = 1;        % Need only one density loop.
        opt.noKid       = true;     % no-kids marker is true.
        countMom        = 'noKidSwed'; %#ok<NASGU>
    end
    
    if strcmp(CountFl,'noKidLiqHous')    % Childless and liquid-housing counterfactual:
        disp('Counterfactual: No children, renting only')
        % noKid:
        load('resBaseline','res')
        par.apInit = res.pan.once.Nwp65(:,1,1);  
        par.EpInit = res.pan.once.EpInd65; 
        clear res
        par.BurnInPan = 1; 
        opt.maxIterVF   = 1;     
        opt.maxIterDens = 1;    
        opt.noKid       = true; 
        % LiqHous:
        par.qnet   = zeros(1,1,Ns,1,1,1); 
        par.qnet(1,1,2:3,1,1,1)=q;    
        par.hgrid  = h0;
        par.hvec   = h0;
        par.Nh     = 1;                     
        rentersChoose = true;
        countMom  = 'noKidLiqHous'; %#ok<NASGU>
    end


    if strcmp(CountFl,'LiqHousSwed')    % Liquid-housing in Sweden counterfactual:
        disp('Counterfactual: Sweden with renting only')
        % Sweden:
        par.tauSS = par.tauSS + dtauSS; % Increment SS contribution to pay
                                        % for better care.
        % LiqHous:
        par.qnet   = zeros(1,1,Ns,1,1,1);
        par.qnet(1,1,2:3,1,1,1)=q-q; % Subsidy equals price of care
        par.hgrid  = h0;
        par.hvec   = h0;
        par.Nh     = 1;
        rentersChoose = true;
        countMom  = 'LiqHousSwed'; %#ok<NASGU>
    end


    if strcmp(CountFl,'noKidLiqHousSwed')    % Childless and liquid-housing counterfactual:
        disp('Counterfactual: Sweden without children and renting only')
        % Sweden:
        par.tauSS = par.tauSS + dtauSS; % Increment SS contribution to pay
        % for better care.
        % noKid:
        load('resBaseline','res')
        par.apInit = res.pan.once.Nwp65(:,1,1);
        par.EpInit = res.pan.once.EpInd65;
        clear res
        par.BurnInPan   = 1;
        opt.maxIterVF   = 1;
        opt.maxIterDens = 1;
        opt.noKid       = true;
        % LiqHous:
        par.qnet   = zeros(1,1,Ns,1,1,1);
        par.qnet(1,1,2:3,1,1,1)=q-q; % Subsidy equals price of care
        par.hgrid  = h0;
        par.hvec   = h0;
        par.Nh     = 1;
        rentersChoose = true;
        countMom  = 'noKidLiqHousSwed'; %#ok<NASGU>
    end

    if strcmp(CountFl,'Om1Swed')  % Sweden without attachment to home.
        disp('Counterfactual: Sweden without owning utility')
        % Sweden:
        par.tauSS = par.tauSS + dtauSS; % Increment SS contribution to pay
        sf = zeros(size(qnet));         % for better care.
        sf(1,1,2:Ns,1,1,1:Nh)=q;
        par.sf   = sf;
        par.qnet = par.qnet - sf;
        % No owning glow:
        par.omega = 1;          % No utility from owning.
        countMom  = 'Om1Swed';  %#ok<NASGU>
    end
    
    if strcmp(CountFl,'noKidOm1')% No children and no housing warm glow:
        disp('Counterfactual: No children and no owning utility')
        % noKid:
        load('resBaseline','res')
        par.apInit = res.pan.once.Nwp65(:,1,1);   % Append initial states of panel members
        par.EpInit = res.pan.once.EpInd65; % as initial distribution: nPan-by-1 vectors.
        clear res
        par.BurnInPan   = 1; 
        opt.maxIterVF   = 1;      % Need only one value function iteration.
        opt.maxIterDens = 1;      % Need only one density loop.
        opt.noKid       = true;   % Set no-kids marker.
        % No owning glow:
        par.omega       = 1;      % No utility from owning.
        countMom        = 'noKidOm1'; %#ok<NASGU>
    end
        
    if strcmp(CountFl,'noKidOm1Swed')  % Sweden without kids counterfactual.
        disp('Counterfactual: Sweden without children and no owning utility')
        % Sweden:
        par.tauSS = par.tauSS + dtauSS; % Increment SS contribution to pay
        sf = zeros(size(qnet));         % for better care.
        sf(1,1,2:Ns,1,1,1:Nh)=q;
        par.sf   = sf;
        par.qnet = par.qnet - sf;
        % noKid:
        load('resBaseline','res')
        par.apInit = res.pan.once.Nwp65(:,1,1);   % Append initial states of panel members
        par.EpInit = res.pan.once.EpInd65; % as initial distribution: nPan-by-1 vectors.
        clear res
        par.BurnInPan   = 1;
        opt.maxIterVF   = 1;        % Need only one value function iteration.
        opt.maxIterDens = 1;        % Need only one density loop.
        opt.noKid       = true;     % no-kids marker is true.
        % No owning glow:
        par.omega       = 1;        % No utility from owning.
        countMom        = 'noKidOm1Swed'; %#ok<NASGU>
    end
    
    DisplaySettings(par,opt);           % Display parameter settings.
    
    countRes = SSG_SolveModel(par,opt);
    
    disp('-----------------------------------------------------------------')
    fprintf('SHOWING COUNTERFACTUAL RESULTS:\n')
    if strcmp(CountFl,'noKid')
        load('momBaseline','mom');  % Load in moments from baseline calibration.
        countRes.NtwTrajPanBaseline = mom.NtwTrajPan;
        % Take net-worth trajectories from baseline
        % results into countRes to plot them against
        % no-kid trajectories in SSG_GetMoments.m.
    end
    countMom = SSG_GetMoments(countRes,par,opt,1);
    disp('-----------------------------------------------------------------')
    

% % Optional: Save the results that are produced in SSG_GetMoments if needed
%     fprintf('SAVING COUNTERFACTUAL RESULTS:\n')
%     if strcmp(CountFl,'omega1')
%         save('countMom_omega1.mat','countMom')
%     end
%     if strcmp(CountFl,'LiqHous')
%         save('countMom_LiqHous.mat','countMom')
%     end
%     if strcmp(CountFl,'noKid')
%         save('countMom_noKid.mat','countMom')
%     end
%     if strcmp(CountFl,'noKidLiqHous')
%         save('countMom_noKidLiqHous.mat','countMom')
%     end
%     if strcmp(CountFl,'Sweden')
%         save('countMom_Sweden.mat','countMom')
%     end
%     if strcmp(CountFl,'Om1Swed')
%         save('countMom_Om1Swed.mat','countMom')
%     end
%     if strcmp(CountFl,'LiqHousSwed')
%         save('countMom_LiqHousSwed.mat','countMom')
%     end
%     if strcmp(CountFl,'noKidSwed')
%         save('countMom_noKidSwed.mat','countMom')
%     end
%     if strcmp(CountFl,'noKidLiqHousSwed')
%         save('countMom_noKidLiqHousSwed.mat','countMom')
%     end
%     if strcmp(CountFl,'noKidOm1Swed')
%         save('countMom_noKidOm1Swed.mat','countMom')
%     end
%     if strcmp(CountFl,'noKidOm1')
%         save('countMom_noKidOm1.mat','countMom')
%     end
%     disp('DONE.')
end

