function out = SSG_GetMoments(res,par,opt,showTable)
% Calculate moments of interest for a given model economy (baseline or a 
% counterfactual scenario).

% SEARCH FOR TABLE OR FIGURE TO FIND THE CORRESPONDING ONE TO THE PAPER.
% THIS PROGRAM ALSO GENERATES MANY OTHER TABLES AND FIGURES THAT DID NOT
% MAKE IT INTO THE PAPER BUT ARE INFORMATIVE AND SO WE LEFT THEM IN. 

% Inputs:
% res:      STRUCTURE. Contains equilibrium results, mainly results from a
%           simulated panel (see description in SSG_SolveModel.m).
% par:      STRUCTURE. Parameters (see SSG_MotherShip.m).
% opt:      STRUCTURE. Contains options (see SSG_MotherShip.m).
% showTable:Logical scalar. Show tables and figures if true. If false, only
%           return output structure silently.
% 
% Output:   STRUCTURE out=mom with different substructures, containing all
% statistics computed; search for structure 'mom' in this file to see how
% outputs are defined etc. Also, output is printed out in tables and
% figures if showTable=true.

% Line style for comparison with kids vs. no kids, NH vs. CR, own vs. rent:
LinStylYesNo = {'+-k','+--k'};                % Mark the yes-group with solid line,
% the other group with dashed line.

% Set exclusion thresholds for wealth that we want to look at:
qExcl = [0.95, 0.97, 0.99];         % Above which quantile to exclude.
indExclThr = 1;                     % The first index (Q95) is the one 
                                    % we take.
                                    
noKid = opt.noKid;                  % Read out if we are in one of the no-
                                    % kids counterfactuals.
% Read out from input structur 'par' (model parameters):
% Read out grid-related parameters: 
N      = par.N;      Nw     = par.Nw;     Nh   = par.Nh;   Ns = par.Ns;   
hvec   = par.hvec;  tRet   = par.tRet; dGen   = par.dGen;  
deltaHouse = par.deltaHouse;
r      = par.r;  
% indHlth= 1;
indBC  = 2;                     % Indeces of basic care and skilled care
indSC  = 3;                     % in dim. 3: Define these as variables so we
                                % we can search for them when making
                                % changes.

[nPan, TPan,nCln] = size(res.pan.stocks.ind);   % Get size of panel: how many panel
                                        % members, how long panel is, and
                                        % how many clones there are (should
                                        % be 1 or 3).
if nCln>1                               % If there are clones: read out
    indCln65 = res.pan.indCln65;        % indeces of clone at 65 (2) and
    indClnLtc= res.pan.indClnLtc;       % clone at LTC (3).
end
res.pan.kidage = res.pan.agevec-dGen;   % 1-by-TPan vector with child age:
                                        % [37,39,...,65].
prdPan  = dGen / TPan;                  % Duration of one interview period:
                                        % 30/15 = 2.
deadPan = (res.pan.agevec>=res.pan.age.Died);                                      
res.pan.stocks.dead = deadPan;          % Create nPan-by-TPan matrix if 
                                        % panel member dead at each wave.
panAlive = ~res.pan.stocks.dead;        % Create nPan-by-TPan matrix if 
                                        % alive: will be used a lot.
% all(res.pan.stocks.dead(:)==deadPan(:)) % checked, works.
% Create (nPan,TPan,nCln) array that indexes alive main clones:
if nCln==1
    AliveMainParent = panAlive;         % nPan-by-TPan logical matrix if alive.
else                                    % If there are clones:
    AliveMainParent = cat(3,panAlive,false(nPan,TPan,nCln-1));
end                                     % all false for clones, mark only alive 
                                        % main households.

exitIntw = sum(panAlive,2) + 1;         % nPan-by-1 vector: At which wave
                                        % the exit interview happens.                                        
WavesUntilExit = exitIntw - (1:TPan);   % Calculate how many waves until 
                                        % exit interview each panel
                                        % member has in each wave:
                                        % nPan-by-TPan matrix.    
% BeqPanSm = sum(res.pan.once.Beq,2);           % Create (nPan,1,nCln) 'small' 
                                        % array with bequests.
% all(res.pan.once.Beq(:)==BeqPanSm(:)) % checked, works.                                        
hPan = res.pan.stocks.h;                % Read out housing panel variable.
                                        % Will be used a lot.
                                        % (nPan,TPan,nCln) array.
Own65 = (res.pan.once.h65>0);           % nPan-by-1 vector if main clone 
                                        % owns at age 65.
Rent65= ~Own65;                         % Same for renting dummy.

% Get some ever-variables we will need:
EverLtc = ~isnan(res.pan.age.EnterLtc); % nPan-by-1: if ever entered LTC.
EverLiq = Own65 & ~isnan(res.pan.age.Liq);
                                        % (nPan,1,nCln): if ever liquidated house.

NtwpPan3D = hPan + res.pan.stocks.ap;   % Get net worth of parent for all 
                                        % clones: (nPan,TPan,nCln) array.
% deadPan3D = deadPan(:,:,ones(1,nCln));
                                        % Extend dead dummy to (nPan,TPan,
                                        % nCln) array -- is common across
                                        % clones.
NtwpPan3D = ~deadPan.*NtwpPan3D + deadPan.*res.pan.once.Beq;                                       
% NtwpPan3D(deadPan3D) = res.pan.once.Beq(deadPan3D);
                                        % For those who died, write in
                                        % the bequest.

% Create vector with exclusion thresholds:
thrExcl = [ GetQuantProbDist(NtwpPan3D(AliveMainParent),qExcl ), Inf];
                                        % 1-by-4 vector with thresholds:
                                        % [Q95,Q97,Q99,Inf].
AlivParBelwThr =   AliveMainParent & (NtwpPan3D <thrExcl(indExclThr)); 
                                        % Create (nPan,TPan,nCln) logical
                                        % array that marks alive parents
                                        % whose wealth falls below the
                                        % exclusion threshold.

MainClnIndPan = res.pan.stocks.ind(:,:,1); % nPan-by-TPan matrix with linear
                                        % state indeces of main clone.
ArrIndPan = Lin2ArrInd(MainClnIndPan(:),[N,N,Ns,Nw,Nw,Nh]);
                                        % nPan*TPan-by-6 vector with array
                                        % indeces (6 for each
                                        % hh-time-observation) of the
                                        % panel.
kidIndDead =  Lin2ArrInd(MainClnIndPan(deadPan(:)),[N,Nw]); 
                                        % Get nDead-by-2 matrix with the
                                        % kid indeces for assets and prod.
                                        % on the alone grid.
ArrIndPan(deadPan(:),[1,4]) = kidIndDead;                           
                                        % Write these into columns 1 and 4
                                        % of our array when parent dead.
ArrIndPan(deadPan(:),[2,3,5,6]) = NaN;
                                        % Write NaNs into parent states
                                        % when dead.
% Now, obtain nPan-by-TPan matrices with indeces in the discrete dimensions
% (health and prod. common across clones, HIndPan is only valid for main
% clone).
SIndPan   = reshape(ArrIndPan(:,3),[nPan,TPan]);    % health
EkIndPan  = reshape(ArrIndPan(:,4),[nPan,TPan]);    % kid prod.
EpIndPan  = reshape(ArrIndPan(:,5),[nPan,TPan]);    % parent prod.
% diffEp = diff(EpIndPan,1,2);          % Checked: is consistent.
% all(diffEp(~isnan(diffEp))==0)
% EpInd1 = EpIndPan(:,1);
% all(EpInd1(~isnan(EpInd1))==res.pan.once.EpInd65(~isnan(EpInd1)))
HIndPan   = reshape(ArrIndPan(:,6),[nPan,TPan]);    % housing

% NOT IN PAPER
if showTable
    % Make overview histograms over the discrete states:
    figure('Name','Distribution over discrete states'),    
    subplot(2,2,1), histogram( SIndPan(:)), xlabel('health index'    )
    subplot(2,2,2), histogram( HIndPan(:)), xlabel('house index'     )
    subplot(2,2,3), histogram(EkIndPan(:)), xlabel('kid prod. index' )
    subplot(2,2,4), histogram(EpIndPan(:)), xlabel('par. prod. index')
end

% Show how child productivity evolves over time: Stars only come in slowly!
SaugIndPan = SIndPan;                   % Augmented nPan-by-TPan matrix with 
SaugIndPan(deadPan) = Ns+1;             % health states; create fourth state: dead.

SCountByAge  = zeros(Ns+1,TPan);        % Create z-by-TPan matrices to count
EkCountByAge = zeros(Nw  ,TPan);        % how many panel members fall into
HCountByAge  = zeros(Nh  ,TPan);        % the z different states.
for iq=1:TPan                            % Loop over panel periods
    EkCountByAge(:,iq) = histcounts(   EkIndPan(:,iq), 0.5:1:(Nw+0.5) )';
     SCountByAge(:,iq) = histcounts( SaugIndPan(:,iq), 0.5:1:(Ns+1.5) )';
     HCountByAge(:,iq) = histcounts(    HIndPan(:,iq), 0.5:1:(Nh+0.5) )';
end                                     % Count households in each bin.
mom.byAge.EkFrac = EkCountByAge./sum(EkCountByAge,1); % Divide to get frac-
mom.byAge.SFrac  =  SCountByAge./sum( SCountByAge,1); % tions, append to 
mom.byAge.HFrac  =  HCountByAge./sum( HCountByAge,1); % output structure.

if showTable  % NOT IN PAPER
    figure('Name','Evolution of discrete states by age')
    nRow=3; nCol=1;                         % Structure of subplots.
    subplot(nRow,nCol,1)                    % Plot child's prod. evolution
    plot(res.pan.agevec,mom.byAge.EkFrac);              % by age
    legg = cell(1,Nw);                      % Create legend.
    for i=1:Nw, legg{i} = sprintf('Prod. %1.0f',i);  end
    legend(legg)                            % More editing.
    xlabel('age'), ylabel('fraction of children')

    subplot(nRow,nCol,2)                    % Now health state.
    plot(res.pan.agevec,mom.byAge.SFrac);
    legg = {'healthy','basic LTC','skilled LTC','dead'};
    legend(legg);
    xlabel('age'), ylabel('fraction of parents')

    subplot(nRow,nCol,3)                    % Finally, parent housing state.
    plot(res.pan.agevec,mom.byAge.HFrac);
    legg = cell(1,Nh);
    legg{1}='rent';
    for i=2:Nh, legg{i} = sprintf('House %1.0fK',hvec(i));  end
    legend(legg);
    xlabel('age'), ylabel('fraction of alive parents')
end



% Artifical panel
qtl = [0.1; 0.25; 0.5; 0.75; 0.9];      % Vector with quantiles.
nQtl= numel(qtl);                       % Number of quantiles wanted: 5.
nAgeCat = TPan;                         % Take 15 categories now from the
                                        % interview structure.
nwQuantByAge = zeros(nQtl,nAgeCat,2);   % Array for quantiles (5-by-15-by-2):
                                        % Dim.1: The five quantiles (0.1,
                                        %        0.25, 0.5, 0.75, 0.9)
                                        % Dim.2: The six age categories for
                                        %        parents
                                        % Dim.3: Generation (1 for kids, 2
                                        %        for parents)                                        
aQuantByAge = zeros(nQtl,nAgeCat,2);    % The same for financial wealth.
ownRateByAge= zeros(1   ,nAgeCat  );    % Vector for ownership rate.
MARateByAge = zeros(1   ,nAgeCat  );    % Vector for MA rate.
NHRateByAge = zeros(1   ,nAgeCat  );    % Vector for NH rate.
FHCRateByAge= zeros(1   ,nAgeCat  );    % Vector for FHC rate.
ICRateByAge = zeros(1   ,nAgeCat  );    % Vector for IC rate.

NpanAlive= sum(AliveMainParent(:));     % Scalar: # of alive observations.
IC = res.pan.binary.IC.val;             % (nPan,TPan,nCln) arrays with care
NH = res.pan.binary.NH.val;             % decisions of alive households.
MA = res.pan.binary.MA.val;
PPNH = NH & ~MA;                        % Private-payer (PP) NH are nursing-
FHC= res.pan.binary.FHC.val;            % home residents that are not MA.
% all(FHC==)                            % Checked here, all fine.

own= (hPan>0);                          % Indicator variable whether own:
                                        %(nPan,TPan,nCln) logical array.
numOwnersOver65 = sum(own(AliveMainParent));
numOwnersAge65  = sum(Own65);
numOwners65Plus = numOwnersOver65 + numOwnersAge65;

numAliveOver65  = NpanAlive;
numAliveAge65   = size(Own65,1);
numAlive65Plus  = numAliveOver65 + numAliveAge65;

ownRate65Plus = 100 * (numOwners65Plus/numAlive65Plus);
mom.ownRate   = ownRate65Plus; 

                                        % Fraction of owners among alive.
if showTable                                        
    fprintf('Overall homeownership rate among parents: %1.1f%%.\n',mom.ownRate);
end

nrLTC = sum( SaugIndPan(:)==indBC | SaugIndPan(:)==indSC );
mom.fracLTC = nrLTC /NpanAlive;         % Get fraction in LTC.

fdNam = {'IC','FHC','PPNH','MA'};       % Cells with dummies for different
fdDum = { IC , FHC , PPNH , MA };       % care forms and the names.
for ifd=1:numel(fdNam)                  % Loop over care forms, count cases
    ifdNam = fdNam{ifd};
    care.all.(ifdNam).n    = sum( fdDum{ifd}(AliveMainParent) );
    care.all.(ifdNam).frac = care.all.(ifdNam).n / nrLTC;
end                                     % and calculate fraction in care form.

% No, split up into groups by asset ownership:
grpNam = {'OwnFin','OnlyOwn','Rent'};   % Name of group and dummy:
grpDum = { hPan>0  & res.pan.stocks.ap>=0.1,  ...   % own both,
           hPan>0  & res.pan.stocks.ap< 0.1,  ...   % only house, 
           hPan==0                                };% pool renters in one.

for igr=1:numel(grpNam)             % Loop over all groups: only own, 
    igrpNam = grpNam{igr};          % only fin. wealth etc.
    for ifd=1:numel(fdNam)          % Loop over care forms: IC etc.
        ifdNam = fdNam{ifd};        % Read out field name.
        iiDum = grpDum{igr} & fdDum{ifd}; % Create dummy for those with 
                                    % care form in group (e.g. IC and
                                    % only own).      
        care.(igrpNam).(ifdNam).n    = sum(iiDum(AliveMainParent));
                                    % Count cases.
        care.(igrpNam).(ifdNam).frac = care.(igrpNam).(ifdNam).n / nrLTC;
    end                             % Calculate fraction of LTC cases in
end                                 % this cell.
% Now, sum over both owner groups and get statistics for 
grpNam{4} = 'Own';
for ifd=1:numel(fdNam)              % Calculate same statistics for all owners:  
    care.Own.(fdNam{ifd}).n    = care.OwnFin.(fdNam{ifd}).n + care.OnlyOwn.(fdNam{ifd}).n;
    care.Own.(fdNam{ifd}).frac = care.Own.(fdNam{ifd}).n / nrLTC;
end
mom.care = care;                    % Append results to output structure.

if showTable
    fprintf('%1.1f%% of parents are healthy \n', 100*(1-mom.fracLTC) ) ;
    disp('--------------------------------------------------------------')                     
    disp('--------------------------------------------------------------')
    disp('TABLE 6: LTC arrangements')
    fprintf('group        \t    IC \t   FHC \t PP-NH \t    MA \n')
    disp('--------------------------------------------------------------')
    for igr=1:numel(grpNam)         % Loop over all groups: only own, rent etc.
        fprintf('%12s:',grpNam{igr});   % Print row header.
        runsum = 0;                 % Initiate sum over all forms.
        for ifd=1:numel(fdNam)      % Loop over care forms.
            fprintf('\t %4.1f%%',100*care.(grpNam{igr}).(fdNam{ifd}).frac);
            runsum = runsum + care.(grpNam{igr}).(fdNam{ifd}).frac;
        end
        fprintf(' | %5.1f%%\n',100*runsum); % Print row sum, go to next row.
        care.(grpNam{igr}).tot.n    = runsum*nrLTC;   % Store total numbers for this group.
        care.(grpNam{igr}).tot.frac = runsum;
    end                             % Finally, show overall fractions:
    disp('--------------------------------------------------------------')
    fprintf('all disabled:\t %4.1f%%\t %4.1f%%\t %4.1f%%\t %4.1f%% | 100.0%%\n', ...
         100*[  care.all.IC.frac, care.all.FHC.frac,care.all.PPNH.frac, care.all.MA.frac ]);
    disp('--------------------------------------------------------------')
    fprintf('data        :\t 48.7%% \t 8.4%% \t 15.5%% \t 27.5%% | 100.0%%\n')
    % Finally, show fractions in IC by own / rent
    fprintf('IC fractions: %1.1f%% for owners, %1.1f%% for renters.\n',...
             100*[care.Own.IC.n/care.Own.tot.n,care.Rent.IC.n/care.Rent.tot.n] );
    disp('--------------------------------------------------------------')
end


% Calculate how many parents immediately liquidated (don't consider
% clones):
if par.Nh>1                             % Unless in liquid-housing counterfactual:
    RentCouldOwnFrac = sum(   (res.pan.once.Nwp65(:,1,1)>=par.hvec(2)) ...
                            & (res.pan.once.h65         < 0.001      ))/par.nPan; 
    if showTable                        % Show as result.
        fprintf('%1.1f%% rent although they could own at age 65 ("liquidate at age 65.0")\n',...
                 100*RentCouldOwnFrac );
    end
end

for iq=1:nAgeCat                         % Loop over all age categories:
    % For kids, take net-worth over all (both with alive and with dead
    % parents):
    ajQuant = GetQuantProbDist(res.pan.stocks.ak(:,iq,1),qtl);
    nwQuantByAge(:,iq,1) = ajQuant;      % For kids, financial assets equal  
     aQuantByAge(:,iq,1) = ajQuant;      % net worth.
    % For parents, only take alive parents:
    jParAlive = panAlive(:,iq);          % nPan-by-1 logical vector if alive.
    nq = 5;                         % Take quintiles:
    nwQuant = linspace(0,1,nq+1);
    nwBins = zeros(size(nwQuant));
    if any(jParAlive)                   % If some still alive:
        apj = res.pan.stocks.ap( jParAlive,iq,1);% Read out financial assets of alive
        hj  = hPan(jParAlive,iq,1);      % parents, also housing assets.
        MAj = MA( jParAlive,iq,1);       % Only for main clone, thus:
        NHj = NH( jParAlive,iq,1);       % njAlive-by-1 vectors.
        FHCj= FHC(jParAlive,iq,1);
        ICj = IC( jParAlive,iq,1);
        nwQuantByAge(:,iq,2)= GetQuantProbDist(apj+hj,qtl);
        
        nwBins(2:end-1) = GetQuantProbDist(apj+hj,nwQuant(2:end-1)); % quintiles within each age group
        nwBins(1  ) = -1;               % Set lower limit negative to be save,                                
        nwBins(end) = Inf;              % set upper limit to infinity.
        for quint=1:5
            isinQuint                    = (apj+hj)>=nwBins(quint) &  (apj+hj)<nwBins(quint+1);
            ownRateByNtwQuintByAge(quint,iq) = sum(sum(isinQuint(:,:,1).*(hj(:,:,1)>0),2))/sum(sum(isinQuint(:,:,1),2)); %#ok
        end
        aQuantByAge( :,iq,2)= GetQuantProbDist(apj   ,qtl);
        njParAlive          = sum(jParAlive);    % Number of alive parents.
        MARateByAge( iq)    = sum( MAj>0)/njParAlive;
        NHRateByAge( iq)    = sum( NHj>0)/njParAlive;
        FHCRateByAge(iq)    = sum(FHCj>0)/njParAlive;
        ICRateByAge( iq)    = sum( ICj>0)/njParAlive;
        ownRateByAge( iq)   = sum( hj>0)/njParAlive;
                                        % Ownership rate out of those
                                        % alive.
    else                                % If none alive:
        nwQuantByAge(:,iq,2) = NaN;      % Record NaNs
         aQuantByAge(:,iq,2) = NaN;
    end
end   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIGURE 9: Homeownership in retirement (by wealth quintile) %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
figure, 
subplot(3,2,1), hold on, plot(res.pan.agevec(1:end-1),ownRateByNtwQuintByAge(1,:),'-','LineWidth',2),...
                         plot(67.5:5:92.5,[0.269, 0.240, 0.215, 0.112, 0.045, 0.013],'--k','LineWidth',2),...
                         title('first quintile'), legend('model','data');
subplot(3,2,2), hold on, plot(res.pan.agevec(1:end-1),ownRateByNtwQuintByAge(2,:),'-','LineWidth',2),...
                         plot(67.5:5:92.5,[0.895, 0.875, 0.846, 0.725, 0.555, 0.282],'--k','LineWidth',2),...
                          title('second quintile'), legend('model','data');
subplot(3,2,3), hold on, plot(res.pan.agevec(1:end-1),ownRateByNtwQuintByAge(3,:),'-','LineWidth',2),... 
                         plot(67.5:5:92.5,[0.949, 0.949, 0.931, 0.873, 0.786, 0.682],'--k','LineWidth',2),...
                         title('third quintile'), legend('model','data');
subplot(3,2,4), hold on, plot(res.pan.agevec(1:end-1),ownRateByNtwQuintByAge(4,:),'-','LineWidth',2),...
                         plot(67.5:5:92.5,[0.970, 0.960, 0.952, 0.921, 0.852, 0.775],'--k','LineWidth',2),...
                         title('fourth quintile'), legend('model','data');
subplot(3,2,5), hold on, plot(res.pan.agevec(1:end-1),ownRateByNtwQuintByAge(5,:),'-','LineWidth',2),...
                         plot(67.5:5:92.5,[0.980, 0.969, 0.963, 0.930, 0.870, 0.724],'--k','LineWidth',2),...
                         title('fifth quintile'), legend('model','data');



% Cross-sectional homeownership rate by net worth quintiles 
nq = 5;                         % Take quintiles:
nwQuant = linspace(0,1,nq+1);   % Cut-offs for quintiles: [0,0.2,...,0.8,1]:
                                % 1-by-6 vector.
nwBins = zeros(size(nwQuant));     % Vector with net-worth quintiles (i.e. bins 
                                % bins for net worth): 1-by-6.  
nwBins(2:end-1) = GetQuantProbDist(NtwpPan3D(AliveMainParent),nwQuant(2:end-1));
                                % Calculate quantiles [0.2,0.4,0.6,0.8].
nwBins(1  ) = -1;               % Set lower limit negative to be save,                                
nwBins(end) = Inf;              % set upper limit to infinity.
for iq=1:5
    isinQuint               = AliveMainParent & (NtwpPan3D>=nwBins(iq) &  NtwpPan3D<nwBins(iq+1));
    mom.owners.ntwQuint(iq) = sum(sum(isinQuint(:,:,1).*(hPan(:,:,1)>0),2))/sum(sum(isinQuint(:,:,1),2));
end


mom.byAge.nwQuant = nwQuantByAge;
mom.byAge.aQuant  =  aQuantByAge;
mom.byAge.ownRate = ownRateByAge;        % Append results to output 
mom.byAge.MARate  =  MARateByAge;        % structure.
mom.byAge.NHRate  =  NHRateByAge;
mom.byAge.FHCRate = FHCRateByAge;
mom.byAge.ICRate  =  ICRateByAge;

%% Obtain distribution of assets 65-69 and bequest distribution:
n            = nPan;
% Treat age 65 separately from regular interviews that start at age 67 (at 
% age 65 some variables cannot be measured, e.g. child variables, but we use
% age 65 information to calculate statistics that pertain to parents wealth
% for ages 65 to 69). 
totWlth      = res.pan.once.Nwp65(:,1,1); % total wealth at age 65
houWlth      = res.pan.once.h65(:,1,1);   % housing wealth at age 65, recorded after first selling decision, which is how it should be
finWlth      = totWlth - houWlth;         % financial wealth at age 65
idxBrok      = totWlth==0;
idxOwn       = houWlth>0;                 % minimum housing wealth equals 50K
idxRent      = houWlth==0;
housShr      = houWlth(idxOwn)./totWlth(idxOwn); % housing wealth as fraction of total wealth for owners
Wlth.totWlth{1}     = totWlth; % total wealth at age 65
Wlth.houWlth{1}     = houWlth; % housing wealth at age 65
Wlth.finWlth{1}     = finWlth; % financial wealth at age 65
Wlth.totWlthOwn{1}  = totWlth(idxOwn);  % total wealth of owners at age 65
Wlth.totWlthRent{1} = totWlth(idxRent); % total wealth of renters at age 65
Wlth.idxBrok{1}     = idxBrok; % broke indicator at age 65
Wlth.housShr{1}     = housShr; % housing share of owners at age 65
Wlth.samplSz{1}     = n;       % sample size at age 65
for iw=1:14 % loop over ages 67-93 which corresponds to 14 interviews
    n       = sum(panAlive(:,iw)); % read out sample size
    finWlth = res.pan.stocks.ap(panAlive(:,iw),iw,1); % financial wealth
    houWlth =              hPan(panAlive(:,iw),iw,1); % housing wealth
    totWlth = finWlth + houWlth;
    idxOwn  = houWlth>0;
    idxRent = houWlth==0;
    idxBrok = totWlth==0;
    housShr = houWlth(idxOwn)./totWlth(idxOwn); 
    Wlth.samplSz{1+iw} = n;
    Wlth.finWlth{1+iw} = finWlth;
    Wlth.houWlth{1+iw} = houWlth;
    Wlth.totWlth{1+iw} = totWlth;
    Wlth.totWlthOwn{1+iw}  = totWlth(idxOwn);
    Wlth.totWlthRent{1+iw} = totWlth(idxRent);
    Wlth.idxBrok{1+iw} = idxBrok;
    Wlth.housShr{1+iw} = housShr;
end
mom.ntwp65.quant    = GetQuantProbDist([Wlth.totWlth{1}; Wlth.totWlth{2}; Wlth.totWlth{3}],[0.1 0.25 0.5 0.75 0.9 0.95]); % net worth ages 65-69
mom.ntwp65.quantFin = GetQuantProbDist([Wlth.finWlth{1}; Wlth.finWlth{2}; Wlth.finWlth{3}],[0.1 0.25 0.5 0.75 0.9 0.95]); % financial worth ages 65-69
mom.ntwp65.quantH   = GetQuantProbDist([Wlth.houWlth{1}; Wlth.houWlth{2}; Wlth.houWlth{3}],[0.1 0.25 0.5 0.75 0.9 0.95]); % house value ages 65-69
if Nh>1
mom.ntwp65.quantOwn = GetQuantProbDist([Wlth.totWlthOwn{1}; Wlth.totWlthOwn{2}; Wlth.totWlthOwn{3}],[0.1 0.25 0.5 0.75 0.9 0.95]);  % total wealth of  owners ages 65-69
mom.ntwp65.housShr  = GetQuantProbDist([Wlth.housShr{1};    Wlth.housShr{2};    Wlth.housShr{3}],   [0.1 0.25 0.5 0.75 0.9 0.95]);  % percentage wealth in housing for owners ages 65-69
else
    mom.ntwp65.quantOwn = [];
    mom.ntwp65.housShr  = [];
end
mom.ntwp65.quantRent= GetQuantProbDist([Wlth.totWlthRent{1}; Wlth.totWlthRent{2};Wlth.totWlthRent{3}],[0.1 0.25 0.5 0.75 0.9 0.95]); % total wealth of renters ages 65-69

% Finally, get quantiles of distributions and append to output mom:
if ~isfield(res.pan.once,'FBeq')        % If financial bequests were not        
    res.pan.once.FBeq = res.pan.once.Beq - res.pan.once.HBeq;
end                                     % recorded, create field.

beqQtl = [0.25 0.5 0.75 0.9 0.95];               % Quantiles we want for bequest 
mom.beq.quant      = beqQtl;                % distribution: record.
mom.beq.all.quant  = GetQuantProbDist( res.pan.once.Beq( :,:,1),beqQtl); % total bequest distribution
mom.beq.fin.quant  = GetQuantProbDist(res.pan.once.FBeq( :,:,1),beqQtl); % financial bequest distribution
mom.beq.h.quant    = GetQuantProbDist(res.pan.once.HBeq( :,:,1),beqQtl); % housing  bequest distribution
% Calculate how many leave negligible bequest in different categories.
neglThr = 25;                           % Negligibility threshold: 25K.                    
mom.beq.all.frac0  = (sum( res.pan.once.Beq( :,:,1)<neglThr))/nPan;
mom.beq.fin.frac0  = (sum(res.pan.once.FBeq( :,:,1)<neglThr))/nPan;
mom.beq.h.frac0    = (sum(res.pan.once.HBeq( :,:,1)<neglThr))/nPan;

% Divide population into three groups and compare bequests:                                        
grDum = { IC, FHC, PPNH, MA, ~EverLtc };% Different care form and if 
                                        % healthy at death.
grLbl = {'IC','FHC','PPNH','MA','hlth'};% Labels for groups.
grMrk = cell(1,numel(grDum));           % Set up cell for group markers.
exitIntwMain = false(nPan,TPan,nCln);   % Array that marks exit interviews
exitIntwMain(:,:,1) = ((1:TPan)==exitIntw); % of main clone.

for igr=1:numel(grDum)                  % Loop over groups.
    iFdNam = [grLbl{igr},'atDth'];      % Put together field name.
    grMrk{igr} = any( grDum{igr} & exitIntwMain, 2);
                                        % Create marker if exit interview
                                        % of puts panel member into group
                                        % i: (nPan,1,nCln) logical array.
    ni = sum(grMrk{igr} (:));           % Get group size...
    mom.beq.(iFdNam).n     = ni;        % and record.
    if ni<50                            % If group is very small, put NaNs:
        mom.beq.(iFdNam).quant = nan(size(beqQtl));
        mom.beq.(iFdNam).frac0 = nan;
    else                                % If somebody in group:
        mom.beq.(iFdNam).quant = GetQuantProbDist(res.pan.once.Beq(grMrk{igr}),beqQtl);
                                        % Get bequest quantiles and...
        mom.beq.(iFdNam).frac0 = sum(res.pan.once.Beq(grMrk{igr})<neglThr) / ni;
    end                                 % ...fraction of negligible bequests.
end                                     
                                       
% Checked: looks good.
% sum(grMrk{1}(:,1,1) & grMrk{2}(:,1,1))  % Groups don't overlap
% sum(grMrk{1}(:,1,1) & grMrk{3}(:,1,1))
% sum(grMrk{3}(:,1,1) & grMrk{5}(:,1,1))

if showTable
    disp('---------------------------------------------------------------')
    disp('TABLE 4  : Net worth distribution at the start of retirement') 
    disp('TABLE F.2:                "               (when xi=0.90)    ') 
    disp('                 p10       p25      p50     p75     p90      p95 ')
    disp('-----------------------------------------------------------------')
    fprintf('housing:      %4.0fK    %4.0fK     %4.0fK    %4.0fK    %4.0fK    %4.0fK \n', ...
                               mom.ntwp65.quantH  )
    fprintf('non-housing:  %4.0fK    %4.0fK     %4.0fK    %4.0fK    %4.0fK    %4.0fK \n', ...
                               mom.ntwp65.quantFin)
    fprintf('net worth:    %4.0fK    %4.0fK     %4.0fK    %4.0fK    %4.0fK    %4.0fK \n', ...
                               mom.ntwp65.quant   )
    fprintf('wlth. in hous.:  %2.0f%%     %2.0f%%      %2.0f%%      %2.0f%%    %2.0f%%    %2.0f%% \n', ...
                           100*   mom.ntwp65.housShr )
    disp('---------------------------------------------------------------')
    
    disp('---------------------------------------------------------------')
    disp('TABLE 5  : Net worth distribution by homeownership at the start of retirement') 
    disp('TABLE F.3:                "               (when xi=0.90)                     ') 
    disp('                 p10       p25      p50     p75     p90      p95 ')
    disp('-----------------------------------------------------------------')
    fprintf('owners:      %4.0fK    %4.0fK     %4.0fK    %4.0fK    %4.0fK    %4.0fK \n', ...
                               mom.ntwp65.quantOwn )
    fprintf('renters:    %4.0fK    %4.0fK     %4.0fK    %4.0fK    %4.0fK    %4.0fK \n', ...
                              mom.ntwp65.quantRent)
    disp('---------------------------------------------------------------')

    disp('---------------------------------------------------------------')
    disp('TABLE 7  : Bequest distribution')
    disp('TABLE 8  : Bequest distribution by asset class') 
    disp('TABLE F.1:                "     (when xi=0.90)') 
    disp('          non-negligible   p25      p50      p75      p90      p95      ')
    disp('---------------------------------------------------------------')
    fprintf('total:          %2.0f%%    %4.0fK    % 4.0fK    %4.0fK    %4.0fK     %3.0fK \n', ...
                        [100-100*mom.beq.all.frac0, mom.beq.all.quant         ] )
    fprintf('financial:      %2.0f%%   %4.0fK     %4.0fK   %4.0fK    %4.0fK     %3.0fK \n', ...
                        [100-100*mom.beq.fin.frac0, mom.beq.fin.quant      ] )
    fprintf('housing:        %2.0f%%   %4.0fK     %4.0fK   %4.0fK    %4.0fK     %3.0fK \n', ...
                        [100-100*  mom.beq.h.frac0,   mom.beq.h.quant           ] )
    disp('---------------------------------------------------------------')
    disp('"non-negligible" means >15K.')
    disp('---------------------------------------------------------------')
       
end


%% Graphs on portfolio composition: NOT IN PAPER
% First, create dummies that indicate those who have:
onlyH = res.pan.stocks.ap <neglThr & res.pan.stocks.h >0;   % only housing wealth,
onlyF = res.pan.stocks.ap>=neglThr & res.pan.stocks.h==0;   % only fin. wealth,
ntwp0  = res.pan.stocks.ap <neglThr & res.pan.stocks.h==0;  % none of either.
bothHF= res.pan.stocks.ap>=neglThr & res.pan.stocks.h >0;   % both types.
% Now, we go over three groups for whom we want to characterize portfolios
% by age: 
grLbl =  { 'all', 'hlth', 'Ltc'};   % Group labels, and     
grHlth = { AliveMainParent, ...     % dummies that mark groups: 
           AliveMainParent & SaugIndPan==1, ... % (nPan,TPan,nCln) arrays.
           AliveMainParent & (SaugIndPan==indBC | SaugIndPan==indSC) };
               
for igr=1:numel(grHlth)             % Loop over all health groups.
    niByAge = sum(grHlth{igr}(:,:,1),1);
                                    % Number of alive parents in group i by
                                    % panel wave: 1-by-TPan vector.
    % Collect results in structure 'portf':
    % First, calculate fraction among group that has:
    portf.onlyH.(grLbl{igr}) = sum( onlyH .*grHlth{igr}, 1) ./ niByAge; % only housing wealth,
    portf.onlyF.(grLbl{igr}) = sum( onlyF .*grHlth{igr}, 1) ./ niByAge; % only fin. wealth,
    portf.ntwp0.(grLbl{igr}) = sum( ntwp0 .*grHlth{igr}, 1) ./ niByAge; % no wealth at all,
    portf.bothHF.(grLbl{igr})= sum( bothHF.*grHlth{igr}, 1) ./ niByAge; % both types.
    % Outputs are (1,TPan,nCln) arrays.
    % Then, get mean wealth in each category:
    portf.meanH.(grLbl{igr}) = sum( res.pan.stocks.h .*grHlth{igr}, 1) ./ niByAge;    % housing,
    portf.meanF.(grLbl{igr}) = sum( res.pan.stocks.ap.*grHlth{igr}, 1) ./ niByAge;    % financial
    portf.meanHF.(grLbl{igr})= portf.meanH.(grLbl{igr}) + portf.meanF.(grLbl{igr});   % sum of both.
    % Finally, get housing share (of group as total) by division:
    portf.fracH.(grLbl{igr}) = portf.meanH.(grLbl{igr})./ ...
                (portf.meanH.(grLbl{igr}) + portf.meanF.(grLbl{igr}) );
end

mom.byAge.portf = portf;            % Append results to output sructure.

if showTable
    figure('Name','Portfolio composition'); % Open figure.
    grTitle = {'all','healthy','LTC'};      % Titles for groups: show left.
    nRow = numel(grHlth);           % Each group in a row.
    nCol = 3;                       % Three plots per row.
    for igr=1:nRow                  % Loop over health groups.
        igrNam = grLbl{igr};        % Read out group name.
        subplot(nRow,nCol,(igr-1)*nCol+1)   % First subplot in this row:
        plot(res.pan.agevec,portf.onlyH.( igrNam)(1,:,1), '-b',...
             res.pan.agevec,portf.onlyF.( igrNam)(1,:,1),'--k',...
             res.pan.agevec,portf.ntwp0.( igrNam)(1,:,1),'-.r',...
             res.pan.agevec,portf.bothHF.(igrNam)(1,:,1),'.-g'    )
                                    % Plot fraction in each group (for the 
                                    % main clone).
        if igr==1   , legend('only housing','only financial','none','both h&f'), end
                                    % Put legend only for 'all'.
        if igr==1   , title('fraction of parents')                  , end
                                    % Put title on top of column.
        if igr==nRow, xlabel('age')                                 , end
                                    % Label x-axis only in last row.
                      ylabel(['group: ',grTitle{igr}])
                                    % Put group name to the left.
         
        subplot(nRow,nCol,(igr-1)*nCol+2)   % Second subplot in this row:
        plot(res.pan.agevec,portf.meanH.( igrNam)(1,:,1), '-b',...
             res.pan.agevec,portf.meanF.( igrNam)(1,:,1),'--k',....
             res.pan.agevec,portf.meanHF.(igrNam)(1,:,1),'.-g'    );
                                    % Show means in each category.
        if igr==1   , legend('housing','financial','net worth (sum)'), end
        if igr==1   , title('mean wealth')                          , end
        if igr==nRow, xlabel('age')                                 , end
                                    % Labeling as before.

        subplot(nRow,nCol,(igr-1)*nCol+3)   % Third subplot in this row:
        plot(res.pan.agevec,portf.fracH.(igrNam)(1,:,1),'-b') % housing share.
        if igr==1   , title('housing share')                        , end
        if igr==nRow, xlabel('age')                                 , end
    end
end

%% ZOOM IN ON OWNERS' BEHAVIOR:
% Create three dummies for owner groups we want to look at (only main
% clones, will then look at contemporaneous clones of the main clone):
hlthOwners = false(nPan,TPan,nCln);
hlthOwners(:,:,1) = (SaugIndPan==1) & hPan(:,:,1)>0.001;
                                    % Main clones who are healthy and own
                                    % -- need this later, do also if Nh=1.
if par.Nh>1                         % Don't do this when only renting possible.
% Following two groups:
LtcOwners = false(nPan,TPan,nCln);
LtcOwners(:,:,1)  =  (SaugIndPan==2 | SaugIndPan==3) & (hPan(:,:,1)>0.001);
                                    % LTC owners (main clones in either of
                                    % the two LTC states).
IcOwners = LtcOwners & IC;          % Owners who get IC.
% Calculate FHC expenditures of panel members:
priceNHbasic = par.qnet(1,1,indBC,1,1,1);
priceNHskill = par.qnet(1,1,indSC,1,1,1);
priceNH2D    = (SIndPan==indBC)*priceNHbasic + (SIndPan==indSC)*priceNHskill;
                                    % nPan-by-TPan matrix with PP-NH price
                                    % for panel members.
qFHCbasic = par.qnet(1,1,indBC,1,1,2);  % Read out FHC price in basic...
qFHCskill = par.qnet(1,1,indSC,1,1,2);  % ...and skilled care.
priceFHC2D = (SIndPan==indBC)*qFHCbasic + (SIndPan==indSC)*qFHCskill;
                                    % Create (nPan,TPan) matrix with FHC
                                    % price for each panel individual.
                                    % Depends only on LTC state, thus
                                    % independent of which clone it is.
priceFHC3D  = priceFHC2D(:,:,ones(1,nCln));
                                    % Create (nPan,TPan,nCln) array with 
                                    % the same information, will need this 
                                    % for indexing strategy. 
                                    
% Create list of variables we will loop over.
dumList = {'hlth','ltc','ic' };     % Cell with short field names.
dumVar  = {hlthOwners,LtcOwners, IcOwners};
                                    % Cell with the dummy variables that 
                                    % mark these sets.

for iq=1:numel(dumList)              % Loop over three dummies.
    dum = dumList{iq};               % Read out dummy: 'hlth','ltc' or 'ic'.
    % Now, calculate conditional means of variables ep, Qp, IC, Htm:
    owners.(dum).ep.val = mean(   res.pan.flows.ep.val(dumVar{iq}) ...
                             + (r+deltaHouse)*hPan(dumVar{iq}) ...
       + res.pan.binary.FHC.val(dumVar{iq}).*priceFHC3D(dumVar{iq})    );
    % Expenditure for owners: Consumption plus housing cost (foregone 
    % interest plus depreciation).
    % For LTC owners: consumption + housing depreciation + FHC costs.
    % % costs.
    owners.(dum).Q.val    = mean(res.pan.flows.Qp.val(   dumVar{iq})+res.pan.flows.Qk.val(   dumVar{iq}) );
    % Have to add both parts of Q to get Qstar here: Q-transfers can go
    % both ways for IC-owners!
    owners.(dum).IC.val    = mean(res.pan.binary.IC.val(  dumVar{iq}) );
    owners.(dum).Htm.val   = mean(res.pan.binary.Htm.val(dumVar{iq}) );
    % Now, calculate the same statistics for the contemporaneous
    % clones: field suffix 'ccl. Note that these only have size
    % nPan-by-TPan since they are only defined for the main clone.
    % Thus, have to use only the first page of dumVar{j} for indexing:
    jdumCcl = dumVar{iq}(:,:,1); % Only 
    % Obtain nPan-by-Tpan matrix if contemporaneous clone is in PP-NH:
    PPNHccl = res.pan.binary.NH.ccl & ~res.pan.binary.MA.ccl;
    owners.(dum).ep.ccl = mean(   res.pan.flows.ep.ccl(jdumCcl) ...
                         + PPNHccl(jdumCcl).*priceNH2D(jdumCcl)    );
    owners.(dum).Q.ccl    = mean(res.pan.flows.Qp.ccl(  jdumCcl));
% NOTE: Do not have to add Qk for clones: Since clones always rent,
% Q-transfers can only go parent-to-child.
    owners.(dum).IC.ccl    = mean(res.pan.binary.IC.ccl( jdumCcl) );
    owners.(dum).Htm.ccl   = mean(res.pan.binary.Htm.ccl(jdumCcl) ); 
    owners.(dum).n        =  sum( dumVar{iq}(:) );   % Sample size:
                                    % Same for main clone and contemp.
end                                 % clone within a group.
mom.owners = owners;                % Append results to output structure.
% Now, show results of this exercise in table:
if showTable
    % Divide panel expenditures and transfers by 2 since these are over 2
    % years:
    
       disp('---------------------------------------------------------------')
       disp('TABLE 11: Behavioral effects of home-owning')
    fprintf('               h-to-m     exp''re     exch.tr.     IC       N  \n') 
       disp('---------------------------------------------------------------')
    % Create cells with row headers that describe the different groups in 
    dumLong    = {'healthy       :',...     % table: first for the main clone,
                  'disabled      :',...     % ...
                  'receiving IC  :'    };
    dumLongCcl = {'healthy clones:',...     % ... then for contemporaneous
                  'disab. clones :',...     % clones.
                  'IC clones     :'    };
    for iq=1:numel(dumList)                  % Loop over three groups.
        dum = dumList{iq};                   % Read out short dummy name.
        fprintf(dumLong{iq});                % Row header for main clone.
        fprintf('%5.1f%%     %5.1fK    %5.1fK    %5.1f%%    %5.0f\n', ...
                 [ 100*owners.(dum).Htm.val; owners.(dum).ep.val/2; ...
                      owners.(dum).Q.val/2; 100*owners.(dum).IC.val; owners.(dum).n ] );
                                            % Print outcomes of main clone.
        fprintf(dumLongCcl{iq});             % Now, the same for contemporaneous
                                            % clone.
        fprintf('%5.1f%%     %5.1fK     %5.1fK    %5.1f%%         \n', ...
                 [ 100*owners.(dum).Htm.ccl; owners.(dum).ep.ccl/2; ...
                       owners.(dum).Q.ccl/2; 100*owners.(dum).IC.ccl  ] );
    end                  
    disp('---------------------------------------------------------------')
    disp('NOTE: ''exp''re'' means expenditure; defined as spending on ')
    disp('      consumption + housing (rent for renters, depreciation ')
    disp('      plus foregone interest for owners) + spending on care ')
    disp('      (NH and FHC). ''exch.tr.'' are exchange-motivated ')
    disp('      transfers (Q). ''h-to-m'' are hand-to-mouth, defined ')
    disp('      as expenditure equals income.) Clones are copies made')
    disp('      made of agents contemporaneously.')
    disp('---------------------------------------------------------------')
end

end

%% Compare life-time moments between clones: 
% Create restricted owning and renting dummies at age 65, excluding those
% who are beyond net-worth threshold at age 65 (check threshold only once
% to have a balanced panel, even if households drop below threshold later,
% don't include them):
incl65 = (res.pan.once.Nwp65(:,1,1)<thrExcl(indExclThr));
                                        % Create dummy variable if
                                        % household included according to
                                        % age-65 net worth: nPan-by-1.
% Also, create the same dummies and restrict sample by age-65 net worth
% (do this once here and not in functions to save computation time):                                        
Own65restr  =  Own65 & incl65;          % Restrict owning and renting 
Rent65restr = Rent65 & incl65;          % dummies to those included.

% Function that throws back for owners versus renters (at age 65).
    function out = OutcomesOwnRent65(xx,restr)
        % Inputs:   
        % xx:       (nPan,1,nCln) array with some variable for an
        %           individual.
        % restr:    Logical scalar. Restrict sample by variable incl65 if
        %           true.
        % Output:  1-by-3 vector with means over initial owners, their
        %          clones, and initial renters.
        if restr                            % With restriction:
            out = [ mean(xx( Own65restr,1,       1)),...  % for initial owners,
                    mean(xx( Own65restr,1,indCln65)),...  % for their clones,
                    mean(xx(Rent65restr,1,       1))    ];    
        else                                % Without restriction:
            out = [ mean(xx( Own65     ,1,       1)),...  % for initial owners,
                    mean(xx( Own65     ,1,indCln65)),...  % for their clones,
                    mean(xx(Rent65     ,1,       1))    ];  
        end
    end

% Among those who enter LTC at some point, split into those who own when
% becoming disabled and those who rent:
Own65Ltc  = EverLtc &  res.pan.once.OwnLtc & incl65; % nPan-by-1 vector for owners
Rent65Ltc = EverLtc & ~res.pan.once.OwnLtc & incl65; % and renters.
Own65LtcRestr =  Own65Ltc & incl65;
Rent65LtcRestr= Rent65Ltc & incl65;

% Functions that throws back outcomes grouped by these variabels
    function out = OutcomesOwnRentLTC(xx,restr)
        if restr                         % Restricted case:
            out = [ mean( xx( Own65LtcRestr,1,       1) ), ...   % mean for owners,
                    mean( xx( Own65LtcRestr,1,indCln65) ), ...   % for clones of owners take at 65
                    mean( xx(Rent65LtcRestr,1,       1) )     ]; % for renters. 
        else
            out = [ mean( xx( Own65Ltc,1,       1) ), ...   % mean for owners,
                    mean( xx( Own65Ltc,1,indCln65) ), ...   % for clones of owners take at 65
                    mean( xx(Rent65Ltc,1,       1) )     ]; % for renters.
        end
    end

% Finally, split up owners at LTC into those who keep, their clones, and
% those who sell:
KeepLtc   =  Own65Ltc &  res.pan.once.KeepLtc;
SellLtc   =  Own65Ltc & ~res.pan.once.KeepLtc;
KeepLtcRestr = KeepLtc & incl65;
SellLtcRestr = SellLtc & incl65;

% Functions that throws back outcomes grouped by these variabels
    function out = OutcomesKeepSellLTC(xx,restr)
        if restr                        % Restricted case:
            out = [ mean( xx(KeepLtcRestr,1,        1) ), ...   % mean for keepers,
                    mean( xx(KeepLtcRestr,1,indClnLtc) ), ...   % for keepers' clones,
                    mean( xx(SellLtcRestr,1,        1) )     ]; % and for sellers.            
        else                            % Unrestricted case:
            out = [ mean( xx(KeepLtc,1,        1) ), ...   % mean for keepers,
                    mean( xx(KeepLtc,1,indClnLtc) ), ...   % for keepers' clones,
                    mean( xx(SellLtc,1,        1) )     ]; % and for sellers.
        end
    end


if nCln>1 && par.Nh>1                   % Only do this if clones were crea-
                                        % ted and there are houses.
r = par.r;                              % Read out interest rate.
 
mom.OwnRent65.ntw65.own  = mean(res.pan.once.Nwp65( Own65restr));     % Mean net worth of owners at 65.
mom.OwnRent65.ntw65.rent = mean(res.pan.once.Nwp65(Rent65restr));     % and of renters.
% 

DiscNtwLtc = exp(-r*(res.pan.age.EnterLtc - tRet)) ...
             .* res.pan.once.NwpLtc;    % Discount from LTC entry to 65:
                                        % nPan-by-1 vector.
mom.OwnRent65.DiscNtwLtc = [ mean(DiscNtwLtc( Own65restr & EverLtc,1,       1)),...  % for initial owners,
                   mean(DiscNtwLtc( Own65restr & EverLtc,1,indCln65)),...  % for their clones,
                   mean(DiscNtwLtc(Rent65restr & EverLtc,1,       1))    ];   

% Expected discounted bequest:              
DiscDth = exp(-r*(res.pan.age.Died - tRet));
                                        % nPan-by-1 variable with
                                        % discounting to death.
mom.OwnRent65.DiscBeq = OutcomesOwnRent65(DiscDth.*res.pan.once.Beq,1);
tPanMid = (res.pan.agevec - prdPan/2);
DiscPan       = exp(-r*(tPanMid-tRet)); % 1-by-TPan discounting vector for
                                        % flows: take middle of interval 
                                        % over which flows occurred.
% Exp. disc. exchange transfers:                               
% restrict for dollar amounts:
mom.OwnRent65.DiscQp  = OutcomesOwnRent65( sum(DiscPan.*res.pan.flows.Qp.val,2), 1 );
mom.OwnRent65.DiscGp  = OutcomesOwnRent65( sum(DiscPan.*res.pan.flows.gp.val,2), 1 );
% Don't restrict sample for the following non-dollar variables:
mom.OwnRent65.TimeIC  = OutcomesOwnRent65( sum(  res.pan.binary.IC.time     ,2), 0 );
mom.OwnRent65.TimeHtm = OutcomesOwnRent65( sum(  res.pan.binary.Htm.time    ,2), 0 );
mom.OwnRent65.EverNH65= OutcomesOwnRent65(       res.pan.binary.NH.ever        , 0 );
mom.OwnRent65.EverMA65= OutcomesOwnRent65(       res.pan.binary.MA.ever        , 0 );
mom.OwnRent65.EverLiq = OutcomesOwnRent65(             EverLiq                 , 0 );
% % Quick check if for the dead gifts are really 0: works for all.
%  sum(    res.pan.flows.Qp.val(WavesUntilExit(:,:,ones(1,3))<0))
% sum(res.pan.binary.IC.time (WavesUntilExit(:,:,ones(1,3))<0))
% % Statistics that are independent of cloning:
mom.OwnRent65.LifeExp.own  = mean( res.pan.age.Died( Own65) - tRet );
mom.OwnRent65.LifeExp.rent = mean( res.pan.age.Died(Rent65) - tRet );
mom.OwnRent65.EverLtc.own  = mean(          EverLtc( Own65)        );
mom.OwnRent65.EverLtc.rent = mean(          EverLtc(Rent65)        );
mom.OwnRent65.PctOwn65     = mean(                   Own65         );


if showTable
    fprintf('----------------------------------------------------------------\n')
    disp('TABLE 10: Effects of owning at age 65 on expected future outcomes')
    fprintf('AT AGE 65:\n')
    fprintf('Variable             owners    clones   renters\n')
    fprintf('----------------------------------------------------------------\n')
    fprintf('Net worth:           %5.1fK    %5.1fK    %5.1fK\n', [mom.OwnRent65.ntw65.own, mom.OwnRent65.ntw65.own, mom.OwnRent65.ntw65.rent] );
    fprintf('Exp. disc. net                                 \n')
    fprintf('worth upon LTC  :    %5.1fK    %5.1fK    %5.1fK\n', mom.OwnRent65.DiscNtwLtc );
    fprintf('Exp.disc.bequest:    %5.1fK    %5.1fK    %5.1fK\n', mom.OwnRent65.DiscBeq    );
    fprintf('Exp.disc.exch.IV:    %5.1fK    %5.1fK    %5.1fK\n', mom.OwnRent65.DiscQp     );
    fprintf('Exp.disc.altr.IV:    %5.1fK    %5.1fK    %5.1fK\n', mom.OwnRent65.DiscGp     );
    fprintf('Life expectancy :    %5.2fy    %5.2fy    %5.2fy\n', [mom.OwnRent65.LifeExp.own,mom.OwnRent65.LifeExp.own,mom.OwnRent65.LifeExp.rent] ); 
    fprintf('Exp.time in IC  :    %5.2fy    %5.2fy    %5.2fy\n', mom.OwnRent65.TimeIC);
    fprintf('Exp.time h-to-m :    %5.2fy    %5.2fy    %5.2fy\n', mom.OwnRent65.TimeHtm); 
    fprintf('Prob. ever LTC  :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*[mom.OwnRent65.EverLtc.own,mom.OwnRent65.EverLtc.own,mom.OwnRent65.EverLtc.rent]);
    % fprintf('Pr. ever non-IC :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*res65vec(EverNicF));
    fprintf('Prob. ever NH   :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*mom.OwnRent65.EverNH65 ); 
    fprintf('Prob. ever MA   :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*mom.OwnRent65.EverMA65     ); 
    fprintf('Prob. ever rent :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*[mom.OwnRent65.EverLiq(1),1,1] ); %[EverLiq,1,1]); 
    fprintf('(%% of agents)        (%4.1f%%)             (%4.1f%%)\n',100*[mom.OwnRent65.PctOwn65,1-mom.OwnRent65.PctOwn65]);
    fprintf('(# of panel ind.):   (%4.0f)             (%4.0f)\n',[sum(Own65),sum(Rent65)]);
       disp('----------------------------------------------------------------')
       disp('Note: "clones" here are copies of owner at age 65. Means of mo-')
       disp('netary variables (IVs, net worth, bequests) are calculated for the')
       disp('sample of parents whose net worth at age 65 is below Q95 of the ')
       disp('net-worth distribution.')
       disp('Exp. disc. net worth upon LTC is conditional on ever entering LTC.')
       disp('----------------------------------------------------------------')
end


prdLtc = ceil( (res.pan.age.EnterLtc-tRet)/prdPan );    % Get panel
                                        % period in which individual
                                        % entered LTC.

mom.OwnRentLtc.NtwLtc = OutcomesOwnRentLTC(res.pan.once.NwpLtc,1);
DiscDthToLtc = exp(-r*(res.pan.age.Died-res.pan.age.EnterLtc));
                                        % nPan-by-1 vector: Discounting
                                        % from death to LTC entry.
mom.OwnRentLtc.DiscBeq= OutcomesOwnRentLTC(DiscDthToLtc.*res.pan.once.Beq,1);

% Create variable to discount flows to panel wave at which individual became 
% disabled: Take middle of interview period (+0.5).
DiscToLtcPan = exp( -r*( par.dtPan*( (1:TPan) + 0.5 - prdLtc ) )  );
DiscToLtcPan( (1:TPan) < prdLtc ) = 0;  % Don't count periods before LTC.
% Create moments on discounted flows (use restricted sample since dollar
% amount):
mom.OwnRentLtc.DiscQp = OutcomesOwnRentLTC( sum( DiscToLtcPan.*res.pan.flows.Qp.val, 2), 1 );
mom.OwnRentLtc.DiscGp = OutcomesOwnRentLTC( sum( DiscToLtcPan.*res.pan.flows.gp.val, 2), 1 );
% Moments on time in different care forms (don't restrict sample):
mom.OwnRentLtc.TimeIC = OutcomesOwnRentLTC( sum(     res.pan.binary.IC.time        , 2), 0 );
% Create time hand-to-mouth since wave when LTC hit:
% Time spent in LTC per panel period: (nPan,TPan) matrix.
TimeLtcPan = max( res.pan.agevec - res.pan.age.EnterLtc, 0 );
% Now, get time spent hand-to-mouth in LTC: In period in which LTC hit,
% take maximally time in LTC;
TimeHtmLtc = min( res.pan.binary.Htm.time, TimeLtcPan); % .* ((1:TPan) >= prdLtc );
mom.OwnRentLtc.TimeHtm= OutcomesOwnRentLTC( sum(            TimeHtmLtc            , 2), 0 );
% "ever"-moments:
mom.OwnRentLtc.EverNH = OutcomesOwnRentLTC(          res.pan.binary.NH.ever           , 0  );
mom.OwnRentLtc.EverMA = OutcomesOwnRentLTC(          res.pan.binary.MA.ever           , 0  );
% See if ever left IC (don't restrict sample)
TimeFC = res.pan.binary.FHC.time + res.pan.binary.NH.time;
                                        % Time in formal care
EverFC = ( sum(TimeFC,2) >0);                    % If ever in formal care.                    
mom.OwnRentLtc.EverFC = OutcomesOwnRentLTC(               EverFC                      , 0  );
mom.OwnRentLtc.EverLiq= OutcomesOwnRentLTC(               EverLiq                     , 0  );

mom.OwnRentLtc.LifeExp.own = mean(  res.pan.age.Died(     Own65Ltc)...
                                  - res.pan.age.EnterLtc( Own65Ltc)    );
mom.OwnRentLtc.LifeExp.rent= mean(  res.pan.age.Died(    Rent65Ltc)...
                                  - res.pan.age.EnterLtc(Rent65Ltc)    );
mom.OwnRentLtc.Nown = sum( Own65Ltc);
mom.OwnRentLtc.Nrent= sum(Rent65Ltc);

% NOT IN PAPER
if showTable 
    fprintf('AT ENTRY INTO DISABILITY:\n')
    fprintf('Variable             owners    clones   renters\n')
    fprintf('----------------------------------------------------------------\n')
    fprintf('Net worth       :    %5.1fK    %5.1fK    %5.1fK\n'   ,    mom.OwnRentLtc.NtwLtc ); 
    fprintf('Exp.disc.bequest:    %5.1fK    %5.1fK    %5.1fK\n'   ,    mom.OwnRentLtc.DiscBeq);
    fprintf('Exp.disc.exch.IV:    %5.1fK    %5.1fK    %5.1fK\n'   ,    mom.OwnRentLtc.DiscQp );
    fprintf('Exp.disc.altr.IV:    %5.1fK    %5.1fK    %5.1fK\n'   ,    mom.OwnRentLtc.DiscGp );
    fprintf('Life expectancy :    %5.2fy    %5.2fy    %5.2fy\n'   , [mom.OwnRentLtc.LifeExp.own, mom.OwnRentLtc.LifeExp.own, mom.OwnRentLtc.LifeExp.rent] );
    fprintf('Exp.time in IC  :    %5.2fy    %5.2fy    %5.2fy\n'   ,    mom.OwnRentLtc.TimeIC );
    fprintf('Exp.time h-to-m :    %5.2fy    %5.2fy    %5.2fy\n'   ,    mom.OwnRentLtc.TimeHtm);
    fprintf('Prob. ever FC   :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*mom.OwnRentLtc.EverFC );
    fprintf('Prob. ever NH   :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*mom.OwnRentLtc.EverNH ); 
    fprintf('Prob. ever MA   :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*mom.OwnRentLtc.EverMA ); 
    fprintf('Prob. ever rent :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*[mom.OwnRentLtc.EverLiq(1),1,1] ); 
    fprintf('(%% of panel ind):    (%4.1f%%)             (%4.1f%%)\n',100*[mom.OwnRentLtc.Nown,mom.OwnRentLtc.Nrent]/(mom.OwnRentLtc.Nown+mom.OwnRentLtc.Nrent) );
    fprintf('(# of panel ind):  (%6.0f )           (%6.0f )\n',[mom.OwnRentLtc.Nown,mom.OwnRentLtc.Nrent]);
       disp('----------------------------------------------------------------')
       disp('Note: clones" here are copies of owner at age 65. Means of mo-')
       disp('netary variables (IVs, net worth, bequests) are calculated for the')
       disp('sample of parents whose net worth at age 65 is below Q95 of the ')
       disp('net-worth distribution.')
       disp('----------------------------------------------------------------')
end


% Calculate similar moments to before:
mom.KeepSellLtc.NtwLtc = OutcomesKeepSellLTC(res.pan.once.NwpLtc, 1 );

mom.KeepSellLtc.DiscBeq= OutcomesKeepSellLTC(DiscDthToLtc.*res.pan.once.Beq, 1);

% Create moments on discounted flows (restrict sample):
mom.KeepSellLtc.DiscQp = OutcomesKeepSellLTC( sum( DiscToLtcPan.*res.pan.flows.Qp.val, 2), 1 );
mom.KeepSellLtc.DiscGp = OutcomesKeepSellLTC( sum( DiscToLtcPan.*res.pan.flows.gp.val, 2), 1 );
% Moments on time in IC and hand-to-mouth (don't restrict sample):
mom.KeepSellLtc.TimeIC = OutcomesKeepSellLTC( sum(     res.pan.binary.IC.time        , 2), 0 );
mom.KeepSellLtc.TimeHtm= OutcomesKeepSellLTC( sum(             TimeHtmLtc            , 2), 0 );
% "ever"-moments:
mom.KeepSellLtc.EverNH = OutcomesKeepSellLTC(          res.pan.binary.NH.ever            , 0 );
mom.KeepSellLtc.EverMA = OutcomesKeepSellLTC(          res.pan.binary.MA.ever            , 0 );
mom.KeepSellLtc.EverFC = OutcomesKeepSellLTC(               EverFC                       , 0 );
mom.KeepSellLtc.EverLiq= OutcomesKeepSellLTC(               EverLiq                      , 0 );
% Life expectancy:
mom.KeepSellLtc.LifeExp.keep= mean(  res.pan.age.Died(    KeepLtc)...
                                   - res.pan.age.EnterLtc(KeepLtc)    );
mom.KeepSellLtc.LifeExp.sell= mean(  res.pan.age.Died(    SellLtc)...
                                   - res.pan.age.EnterLtc(SellLtc)    );
% Number of observations:                               
mom.KeepSellLtc.Nkeep = sum(KeepLtc);
mom.KeepSellLtc.Nsell= sum(SellLtc);
% NOT IN PAPER
if showTable
    fprintf('OWNERS AT ENTRY INTO DISABILITY:\n')
    fprintf('Variable            keepers    clones   sellers\n')
    fprintf('----------------------------------------------------------------\n')
    fprintf('Net worth       :    %5.1fK    %5.1fK    %5.1fK\n', mom.KeepSellLtc.NtwLtc );
    fprintf('Exp.disc.bequest:    %5.1fK    %5.1fK    %5.1fK\n', mom.KeepSellLtc.DiscBeq);
    fprintf('Exp.disc.exch.IV:    %5.1fK    %5.1fK    %5.1fK\n', mom.KeepSellLtc.DiscQp );
    fprintf('Exp.disc.altr.IV:    %5.1fK    %5.1fK    %5.1fK\n', mom.KeepSellLtc.DiscGp );
    fprintf('Life expectancy :    %5.2fy    %5.2fy    %5.2fy\n', [mom.KeepSellLtc.LifeExp.keep,mom.KeepSellLtc.LifeExp.keep,mom.KeepSellLtc.LifeExp.sell]); 
    fprintf('Exp.time in IC  :    %5.2fy    %5.2fy    %5.2fy\n', mom.KeepSellLtc.TimeIC );
    fprintf('Exp.time h-to-m :    %5.2fy    %5.2fy    %5.2fy\n', mom.KeepSellLtc.TimeHtm);
    %fprintf('Prob. ever LTC  :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*mom.KeepSellLtc.TimeHtm);
    fprintf('Pr. ever FC     :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*mom.KeepSellLtc.EverFC);
    fprintf('Prob. ever NH   :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*mom.KeepSellLtc.EverNH); 
    fprintf('Prob. ever MA   :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*mom.KeepSellLtc.EverMA); 
    fprintf('Prob. ever rent :    %5.1f%%    %5.1f%%    %5.1f%%\n',100*[mom.KeepSellLtc.EverLiq(1),1,1]); 
    fprintf('(# of panel ind):   (%6.0f)            (%6.0f)\n',[mom.KeepSellLtc.Nkeep,mom.KeepSellLtc.Nsell]);
    fprintf('(%% of panel ind):    (%4.1f%%)             (%4.1f%%)\n',100*[mom.KeepSellLtc.Nkeep,mom.KeepSellLtc.Nsell]/(mom.KeepSellLtc.Nkeep+mom.KeepSellLtc.Nsell));
       disp('----------------------------------------------------------------')
       disp('Note: "clones" here are copies of keepers at entry into LTC. Means of mo-')
       disp('netary variables (IVs, net worth, bequests) are calculated for the')
       disp('sample of parents whose net worth at age 65 is below Q95 of the ')
       disp('net-worth distribution.')
       disp('----------------------------------------------------------------')
end
        
end

%% 3. Panel statistics: Wealth trajectories
% First, do sanity check and see if wealth distribution in panel behaves 
% as in population.
nWavBefExit = 1;                        % Set here how many waves before         
                                        % exit we check owning and care
                                        % status.
nWav = 4;                               % Set number of waves we want to 
                                        % see before death. 4 is what we
                                        % have in our data work.
                                        
% If we are in one of the no-kids counterfactuals: Only plot the no-kids 
% quantiles against the baseline results that we had saved.
if noKid        
    Nc = 1;                     % Just have one characteristic: kids versus no kids.
    GrNames = {'with kids','no kids'};  % Group names: 1-by-2 cell.
    GrCalc  = [false      , true    ];  % 1-by-2 matrix: If we have to compute
                                % moment for a group. Don't do it for those
                                % with kids (will load in), but have to
                                % calculate for 'no kids'
    markGrs = false(nPan,TPan,Nc,2);
    markGrs(:,:,:,2) = true;    % Mark all in no-kids case.

% In baseline (and all counterfactuals that are NOT the no-kid kind):
else 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%    
% FIGURE 7:  Net-worth trajectories: All                        %
% FIGURE 8:  Net-worth trajectories: Own vs. Rent               %
% FIGURE 10: Net-worth trajectories: Community vs. nursing home %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% NOTE: Figures 8 and 10 only include the model-generated data but not the
% empirical counterpart. The exaxt figure verions from the paper are 
% produced in STATA using the data generated her with the empirical data

    % Names for groups:
    Nc = 3;                                 % Have 3 characteristica:
    % 1.: kids or no kids
    % 2.: own or rent
    % 3.: NH or CR.
    GrNames = cell(Nc,2);                   % Have cell with 6 groups: 
                                            % Each characteristic has 2
                                            % groups.
    GrNames{1,1} = 'all (model)';           GrNames{1,2} = 'all (data)';
    GrNames{2,1} = 'own';                   GrNames{2,2} = 'rent';
    GrNames{3,1} = 'nursing-home resident'; GrNames{3,2} = 'community resident'; 
    
    GrCalc = true(Nc,2);                    % Nc-by-2 logical matrix: If we
    GrCalc(1,2) = false;                    % have to calculate moments for 
                                            % this group. False only for
                                            % the no-kid group, which we
                                            % will do later. 

    markGrs = false(nPan,TPan,Nc,2);        % (nPan,TPan,Nc,2) array that marks 
                                            % the cases we want to consider in
                                            % each group.
    markGrs(:,:,1,1) = true;                % 1. All.
    WavBefDth  = (WavesUntilExit==nWavBefExit);  % Mark if this is the wave
                                                 % that is the number of waves
                                                 % before death that we
                                                 % want: nPan-by-TPan
                                                 % logical matrix. 
    % Only look at main clone here:
    OwnBefExit = any( WavBefDth & (hPan(:,:,1) > 0.001  ) , 2 );
                                            % nPan-by-1 vector: If individual
                                            % owned in wave before exit.
    RentBefExit= any( WavBefDth & (hPan(:,:,1) < 0.001 )  , 2 );
                                            % Marke those wh rented in wave
                                            % before exit: nPan-by-1 vector
    NHBefExit  = any( WavBefDth & (NH(:,:,1)> 0.5   ) , 2 );   % Same for NH residency
    markGrs(:,:,2,1) =   OwnBefExit(:,ones(1,TPan));  % (2,1) Owners before exit.
    markGrs(:,:,2,2) =  RentBefExit(:,ones(1,TPan));  % (2,2) Rent before exit.
    markGrs(:,:,3,1) =    NHBefExit(:,ones(1,TPan));  % (3,1) NH residents.
    markGrs(:,:,3,2) =  ~ NHBefExit(:,ones(1,TPan));  % (3,2) Community residents.

end

NtwpPan = NtwpPan3D(:,:,1);             % Create nPan-by-TPan matrix with 
                                        % net worth for the main clone.
NtwTrajPan = zeros(5,nWav+1,Nc,2);      % (5,nWav+1,Nc,2) array for trajectory   
                                        % of net worth before death:
                                        % dim 1: 5 percentiles
                                        % dim 2: exit interview, one
                                        %        wave before death, up to 
                                        %        nWav waves before death.  
                                        % dim 3: characteristic (e.g.
                                        %        own or not)
                                        % dim 4: yes or no for
                                        %        characteristic.
for g=1:Nc                              % Loop over all three characteristica. 
    for y=1:2                           % Loop over yes and no.
        if GrCalc(g,y)                  % If we have to calculate for this group: 
                                        % Now, calculate moments for
                                        % all waves before death unless
                                        % we are in no-kids group.
            for w=0:1:nWav              % Loop over waves before death:
                gwSample = ~res.pan.once.Surv95 & markGrs(:,:,g,y) & (WavesUntilExit==w);
                                        % nPan-by-Tpan matrix: Mark cases
                                        % of current group g and current
                                        % wave w. Exclude those who
                                        % survived up to age 95 since this
                                        % is not as in the data.
                wNtwp = NtwpPan(gwSample);      
                                        % Get column vector with net worth
                                        % of panel members in this wave:
                                        % length Nw (depends on sample).
                if isempty(wNtwp)       % In some counterfactuals, some    
                    NtwTrajPan(:,w+1,g,y) = NaN;  % categories will be empty, so set 
                                        % the quantiles to NaN.
                else                    % If there are cases to consider;           
                    qw   = GetQuantProbDist(wNtwp,[0.1, 0.25, 0.5, 0.75, 0.9]); 
                    NtwTrajPan(:,w+1,g,y) = qw;% Get quantiles and record them.                    
                end                   
            end
            
        else                    % If group is not calculated:
            if g==1 && y==2     % In characteristic 1, no category, we write in the moments from the data
                NtwTrajPan(:,:,1,2) = ...       
                    [  0,  0,  0,  0,  0; ...   % Q10 in data (go backward in time)
                       0,  1,  2,  6, 14; ...   % Q25
                      24, 53, 80, 90, 98;...    % Q50
                     200,228,258,282,288;...    % Q75
                     543,594,625,651,646    ];  % Q90
                                        
            else                % Otherwise, we are in a no-kid counterfactual:              % 
                                % Load in baseline moments WITH kids: all.
                if isfield(res,'NtwTrajPanBaseline') % (if field exists),
                    NtwTrajPan(:,:,1,1) = res.NtwTrajPanBaseline(:,:,1,1);               
                else            % otherwise put NaN: not plotted.
                    NtwTrajPan(:,:,1,1) = NaN;
                end
                
            end
            
        end            
    end
end
mom.NtwTrajPan = NtwTrajPan;        % Append results to output structure.

% Now, loop over characteristics and plot:
TrajLineWidths = [0.6 1.2 1.8 2.4 3.0];
if showTable
    for g=1:Nc                      % Loop over all characteristica...
        gNtwTraj = NtwTrajPan(:,:,g,:); % Read out all moments for this characteristic.    
        yMax = 1.1*max(gNtwTraj(~isnan(gNtwTraj)));
        %yMax  = 800;
                                    % Get a uniform upper bound for plotting.

        figure;                     % Create new figure for each characteristic.
        hold on
        for q=1:5                   % Loop over the five quantiles and ...
            for y=1:2               % Loop over yes and no for the characteristic.
                plot(nWav:-1:0,NtwTrajPan(q,:,g,y)',LinStylYesNo{y},...
                     'color','b','LineWidth',TrajLineWidths(q))
            end                     % ...plot with the respective style in blue.
        end
        grid on
        ylim([0,yMax])
        legend({ GrNames{g,1}, GrNames{g,2} }); % Legend with group names
        xticks(0:1:nWav),  xticklabels(nWav:-1:0)
        ylabel('net worth')
        xlabel('Waves until exit interview')
    end
end

if noKid==0
if Nh>1
hPanMain = hPan(:,:,1);                 % Create nPan-by-TPan matrices with 
apPanMain= res.pan.stocks.ap(:,:,1);    % housing wealth and financial
                                        % wealth of main clone.
% Now, write bequests into the waves where parent is dead:                                        
hPanMain   = ~deadPan.*hPanMain + deadPan.*res.pan.once.HBeq(:,1,1);                                        
FinBeqMain = res.pan.once.Beq(:,1,1) - res.pan.once.HBeq(:,1,1);
                                        % Create nPan-by-1 vector with
                                        % financial bequest.
apPanMain    = ~deadPan.*apPanMain + deadPan.*FinBeqMain;
totwPanMain  = apPanMain + hPanMain;                                                                             

hTrajPan = zeros(4,nWav+1);             % Set up (4,nWav+1) matrices for 
apTrajPan= zeros(4,nWav+1);             % trajectories of the 4 quantiles
                                        % over the last waves, for housing
                                        % and financial wealth.   
totwTrajPan    = zeros(4,nWav+1); 
totwTrajPanOwn = zeros(4,nWav+1);
hTrajPanOwn    = zeros(4,nWav+1);
apTrajPanOwn   = zeros(4,nWav+1);
% Now, plot quantiles of housing and financial wealth separately for 
for w=0:1:nWav                      % Loop over waves before death:
    wSample    = ~res.pan.once.Surv95 & (WavesUntilExit==w);
    wSampleOwn = ~res.pan.once.Surv95 & markGrs(:,:,2,1) & (WavesUntilExit==w);
                            % nPan-by-Tpan matrix: Mark cases of current                           
                            % wave w. Exclude those who
                            % survived up to age 95 since this
                            % is not as in the data.
    totwTrajPan( :,w+1)    = GetQuantProbDist( totwPanMain(wSample),   [0.25, 0.5, 0.75, 0.9]); 
    hTrajPan( :,w+1)       = GetQuantProbDist( hPanMain(wSample),      [0.25, 0.5, 0.75, 0.9]); 
    apTrajPan(:,w+1)       = GetQuantProbDist(apPanMain(wSample),      [0.25, 0.5, 0.75, 0.9]);
    totwTrajPanOwn( :,w+1) = GetQuantProbDist( totwPanMain(wSampleOwn),[0.25, 0.5, 0.75, 0.9]); 
    hTrajPanOwn( :,w+1)    = GetQuantProbDist( hPanMain(wSampleOwn),   [0.25, 0.5, 0.75, 0.9]); 
    apTrajPanOwn(:,w+1)    = GetQuantProbDist(apPanMain(wSampleOwn),   [0.25, 0.5, 0.75, 0.9]);
end

% NOT IN PAPER
% Now, make figure with these:
figure;                     % Create new figure for each characteristic.
for q=1:4                   % Loop over the five quantiles and ...
    % Plot for housing wealth:
    hold on
    subplot(1,3,1),plot(nWav:-1:0, totwTrajPan(q,:)',LinStylYesNo{1},'color','b','LineWidth',2)
   ylim([0,800]),xticks(0:1:nWav), xticklabels(nWav:-1:0),ylabel('net worth'),xlabel('Waves until exit interview')
    grid on
    hold on
    subplot(1,3,2),plot(nWav:-1:0, hTrajPan(q,:)',LinStylYesNo{1},'color','b','LineWidth',2)
    ylim([0,800]),xticks(0:1:nWav), xticklabels(nWav:-1:0),ylabel('housing wealth'),xlabel('Waves until exit interview')
    grid on
    hold on
    subplot(1,3,3),plot(nWav:-1:0,apTrajPan(q,:)',LinStylYesNo{2},'color','b','LineWidth',2)
    ylim([0,800]),xticks(0:1:nWav),  xticklabels(nWav:-1:0),ylabel('financial wealth'), xlabel('Waves until exit interview')

    grid on
end

% NOT IN PAPER
figure;                     % Create new figure for each characteristic.
for q=1:4                   % Loop over the five quantiles and ...
    % Plot for housing wealth:
    hold on
    subplot(1,3,1),plot(nWav:-1:0, totwTrajPanOwn(q,:)',LinStylYesNo{1},'color','b','LineWidth',2)
    ylim([0,900]),xticks(0:1:nWav), xticklabels(nWav:-1:0),ylabel('net worth'),xlabel('Waves until exit interview')
    grid on
    hold on
    subplot(1,3,2),plot(nWav:-1:0, hTrajPanOwn(q,:)',LinStylYesNo{1},'color','b','LineWidth',2)
    ylim([0,900]),xticks(0:1:nWav), xticklabels(nWav:-1:0),ylabel('housing wealth'),xlabel('Waves until exit interview')
    grid on
    hold on
    subplot(1,3,3),plot(nWav:-1:0,apTrajPanOwn(q,:)',LinStylYesNo{2},'color','b','LineWidth',2)
    ylim([0,900]),xticks(0:1:nWav),  xticklabels(nWav:-1:0),ylabel('financial wealth'), xlabel('Waves until exit interview')

    grid on
end

end
end

%% HOME OWNERSHIP
if par.Nh>1
% Plot home-ownership rates by 5-year age bins, center at age bin:                                        
if showTable
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % FIGURE 9: Homeownership in retirement %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    figure;
    plot(res.pan.agevec(1:end-1),ownRateByAge(1:end-1),'-b','LineWidth',2);
    hold on
    if noKid==0                        % Regular economy with children:
        % Data are for HRS bins 65-69, 70-74 etc. 
        plot(67.5:5:92.5,[0.813, 0.799, 0.782, 0.713, 0.622, 0.497],'--k','LineWidth',2);
        ylabel('home ownership rate (parents)')
    else                               % No-kid counterfactual:
        plot(67.5:5:92.5,[0.725, 0.680, 0.641, 0.654, 0.591, 0.491],'--k','LineWidth',2);
        ylabel('home ownership rate (childless)')
    end
    legend('model','data')              % Do some editing of figure.
    grid on
    ylim([0 1])
    xlim([tRet,tRet+dGen])
    xlabel('age')           
end

end

%% Decomposition of housing liquidations
% Take four "life phases":
% 1. 65 to death for those always healthy
% 2. 65 to LTC onset (for those who get disabled at some point).
% 3. LTC onset (for those who get disabled at some point)
% 4. LTC onset to death  (for those who get disabled at some point).
% For 3. and 4., also split them up in basic-care and skilled-care groups.
if par.Nh>1                             % Do this only if houses exist.
EverLiqMn = EverLiq(:,1,1);            % If main clone ever liquidated:
                                       % nPan-by-1 vector.
nrAllLiq = sum(EverLiqMn);             % Total number of liquidations.

% Now, count panel individuals that could have liquidated in a given
% group/life phase ("NrAll") and number of liquidations that really
% happened ("NrLiq"). Will later calculate liquidation probabilities and
% share of total liquidations accounted for in loop. 

% Those who were never hit by LTC:
mom.DecompLiq.hlth.NrAll = sum(Own65 & ~EverLtc            );
mom.DecompLiq.hlth.NrLiq = sum(Own65 & ~EverLtc & EverLiqMn);  

% Those who liquidated before LTC onset:
mom.DecompLiq.BefLtc.NrAll = sum(Own65 & EverLtc);
mom.DecompLiq.BefLtc.NrLiq = sum(Own65 & EverLtc & ~res.pan.once.OwnLtc);

% Those who liquidate exactly at LTC onset:
EverSC = any(SaugIndPan==indSC,2);      % Create nPan-by-1 vector if ever
                                        % in skilled care.
% Those who get basic-care shock and liquidate:                                        
mom.DecompLiq.AtLtcBasic.NrAll = sum(res.pan.once.OwnLtc & ~EverSC);
mom.DecompLiq.AtLtcBasic.NrLiq = sum(res.pan.once.OwnLtc & ~EverSC & ~res.pan.once.KeepLtc);
% Those who get skilled-scare shock and liquidate:
mom.DecompLiq.AtLtcSkill.NrAll = sum(res.pan.once.OwnLtc &  EverSC);
mom.DecompLiq.AtLtcSkill.NrLiq = sum(res.pan.once.OwnLtc &  EverSC & ~res.pan.once.KeepLtc);
% Now, bring the two together to "at LTC onset":
mom.DecompLiq.AtLtc.NrAll = mom.DecompLiq.AtLtcBasic.NrAll + mom.DecompLiq.AtLtcSkill.NrAll;
mom.DecompLiq.AtLtc.NrLiq = mom.DecompLiq.AtLtcBasic.NrLiq + mom.DecompLiq.AtLtcSkill.NrLiq;

% Those who liquidated after LTC onset:
% First, those in basic care:
mom.DecompLiq.AfterLtcBasic.NrAll = sum(EverLtc & ~EverSC & res.pan.once.KeepLtc             );
mom.DecompLiq.AfterLtcBasic.NrLiq = sum(EverLtc & ~EverSC & res.pan.once.KeepLtc & EverLiqMn );
% Then, those in skilled care:
mom.DecompLiq.AfterLtcSkill.NrAll = sum(EverLtc &  EverSC & res.pan.once.KeepLtc             );
mom.DecompLiq.AfterLtcSkill.NrLiq = sum(EverLtc &  EverSC & res.pan.once.KeepLtc & EverLiqMn );
% Finally, add them together:
mom.DecompLiq.AfterLtc.NrAll = mom.DecompLiq.AfterLtcBasic.NrAll + mom.DecompLiq.AfterLtcSkill.NrAll;
mom.DecompLiq.AfterLtc.NrLiq = mom.DecompLiq.AfterLtcBasic.NrLiq + mom.DecompLiq.AfterLtcSkill.NrLiq;

% Now, loop over all fields and calculate the fraction of liquidations
% accounted for ("frac") and the liquidation probability in this phase
% ("prob"):
fdNam = fieldnames(mom.DecompLiq);
for fd=1:numel(fdNam)
    mom.DecompLiq.(fdNam{fd}).frac = mom.DecompLiq.(fdNam{fd}).NrLiq / nrAllLiq;
    mom.DecompLiq.(fdNam{fd}).prob = mom.DecompLiq.(fdNam{fd}).NrLiq / mom.DecompLiq.(fdNam{fd}).NrAll;
end

% NOT IN PAPER
if showTable
    fprintf('----------------------------------------------------------------\n')
    fprintf('DECOMPOSING LIQUIDATIONS\n')
    fprintf('Life phase        #in group    #liq.   liq.prob.    %% of all liq.\n')
    fprintf('----------------------------------------------------------------\n')
    fprintf('Always healthy (Group 1):\n')
    fprintf('65-death        :   %5.0f     %5.0f      %4.1f%%         %4.1f%%\n', ...
                [mom.DecompLiq.hlth.NrAll, mom.DecompLiq.hlth.NrLiq, ...
                    100*mom.DecompLiq.hlth.prob, 100*mom.DecompLiq.hlth.frac  ]  );
    fprintf('----------------------------------------------------------------\n')
    fprintf('Disabled at some point (Group 2):\n')
    fprintf('before LTC onset:   %5.0f     %5.0f      %4.1f%%         %4.1f%%\n', ...
                [mom.DecompLiq.BefLtc.NrAll, mom.DecompLiq.BefLtc.NrLiq, ...
                    100*mom.DecompLiq.BefLtc.prob, 100*mom.DecompLiq.BefLtc.frac  ]  );
    fprintf('at LTC onset    :   %5.0f     %5.0f      %4.1f%%         %4.1f%%\n', ...
                [mom.DecompLiq.AtLtc.NrAll, mom.DecompLiq.AtLtc.NrLiq, ...
                    100*mom.DecompLiq.AtLtc.prob, 100*mom.DecompLiq.AtLtc.frac  ]  );            
    fprintf(' [of which basic  : %5.0f     %5.0f      %4.1f%%         %4.1f%%]\n', ...
                [mom.DecompLiq.AtLtcBasic.NrAll, mom.DecompLiq.AtLtcBasic.NrLiq, ...
                    100*mom.DecompLiq.AtLtcBasic.prob, 100*mom.DecompLiq.AtLtcBasic.frac  ]  );
    fprintf(' [of which skilled: %5.0f     %5.0f      %4.1f%%         %4.1f%%]\n', ...
                [mom.DecompLiq.AtLtcSkill.NrAll, mom.DecompLiq.AtLtcSkill.NrLiq, ...
                    100*mom.DecompLiq.AtLtcSkill.prob, 100*mom.DecompLiq.AtLtcSkill.frac  ]  );            
    %fprintf('    at LTC onset:               %5.1f%%                    %5.1f%%\n', ...
    %                         100*[     fracLiqAtLtc,       massLiqAtLtc/massAllLiq ]     );  
    fprintf('after LTC onset :   %5.0f     %5.0f      %4.1f%%         %4.1f%%\n', ...
                [mom.DecompLiq.AfterLtc.NrAll, mom.DecompLiq.AfterLtc.NrLiq, ...
                    100*mom.DecompLiq.AfterLtc.prob, 100*mom.DecompLiq.AfterLtc.frac  ]  );
    fprintf(' [of which basic:   %5.0f     %5.0f      %4.1f%%         %4.1f%%]\n', ...
                [mom.DecompLiq.AfterLtcBasic.NrAll, mom.DecompLiq.AfterLtcBasic.NrLiq, ...
                    100*mom.DecompLiq.AfterLtcBasic.prob, 100*mom.DecompLiq.AfterLtcBasic.frac  ]  );
    fprintf(' [of which skill:   %5.0f     %5.0f      %4.1f%%         %4.1f%%]\n', ...
                [mom.DecompLiq.AfterLtcSkill.NrAll, mom.DecompLiq.AfterLtcSkill.NrLiq, ...
                    100*mom.DecompLiq.AfterLtcSkill.prob, 100*mom.DecompLiq.AfterLtcSkill.frac  ]  );            
    fprintf('----------------------------------------------------------------\n')
    nOwn65 = sum(Own65);
    fprintf('Notes: %4.1f%% (N=%5.0f of %5.0f) of owners at 65 liquidate\n',...
        [100*nrAllLiq/nOwn65, nrAllLiq, nOwn65]);

    fprintf('       home at some point.\n')
    fprintf('----------------------------------------------------------------\n')
end

end


%% Transfers:

if ~strcmp(opt.CountFl,'noKid')     % Only do this in economy with kids.
% nPanAlive = sum(panAlive(:));   % number of observations in panel (parent
                                    % not dead)

% First, will construct sub-structures res.pan.flows.ivp and res.pan.flows.ivk
% with total inter-vivos
% transfers (gifts plus exchange), measured in different ways and for
% different clones:
MTprefix = {'','pol'};              % Will loop over two measurement types:
                                    % i) two-year cumulative, as in HRS.
                                    %    Found in fields 'val' and 'ccl'.
                                    % ii)"policy" type: current flow right
                                    %    at the interview. Found in fields
                                    %    'polval' and polccl'.
CloneSuffix = {'val','ccl'};        % Suffix for clone type:
                                    % 'val': clones main, 65 and LTC.
                                    % 'ccl': contemp. clone.
for mt=1:numel(MTprefix)           % Loop over all four combinations of 
    for cs=1:numel(CloneSuffix)    % measurement type and clone suffix.
        fdNam = [MTprefix{mt},CloneSuffix{cs}];
                                    % Put together field name: 'val','ccl',
                                    % 'polval', or 'polccl'.
        res.pan.flows.ivp.(fdNam) = res.pan.flows.gp.(fdNam) + res.pan.flows.Qp.(fdNam);
                                    % Add parent transfers to one variable.
        res.pan.flows.ivk.(fdNam) = res.pan.flows.gk.(fdNam) - res.pan.flows.Qk.(fdNam);
                                    % For kid, have to take minus
                                    % Q-transfer since these have negative
                                    % sign!
        res.pan.flows.ivnet.(fdNam) =  res.pan.flows.ivp.(fdNam) ...
                                     - res.pan.flows.ivk.(fdNam)     ;
                                    % Net inter-vivos transfers.
        res.pan.flows.ivabs.(fdNam) =  res.pan.flows.ivp.(fdNam) ...
                                     + res.pan.flows.ivk.(fdNam)     ;
                                    % Sum of inter-vivos transfers.
        res.pan.flows.Q.(fdNam) = res.pan.flows.Qp.(fdNam) + res.pan.flows.Qk.(fdNam);
                                    % Net Q-transfers over two years: get
                                    % Q.
        res.pan.flows.Qabs.(fdNam) = res.pan.flows.Qp.(fdNam) - res.pan.flows.Qk.(fdNam);
    end                             % Add absolute values to get total                      
end                                 % flow
                                    
% Calculate in general how big transfers of different types are:
res.pan.flows.GUnc.val =  res.pan.flows.gpUnc.val + res.pan.flows.gkUnc.val;
res.pan.flows.gpCst.val = res.pan.flows.gp.val - res.pan.flows.gpUnc.val;
res.pan.flows.gkCst.val = res.pan.flows.gk.val - res.pan.flows.gkUnc.val;
% Now, add the two to each other.
res.pan.flows.GCst.val  = res.pan.flows.gpCst.val + res.pan.flows.gkCst.val;

% Cell with names of transfers we want to show statistics for:
fdNam = {'GCst','GUnc','Qabs','ivabs'}; % (all are summed over both agents)

for ifd=1:numel(fdNam)                  % Loop over the fields.              
    % Get percentage of positive transfers and conditional mean...
    [mom.(fdNam{ifd}).condmean,~, mom.(fdNam{ifd}).FracPos] = ...
         GetConditionalMeanPanel(res.pan.flows.(fdNam{ifd}).val,AliveMainParent,true);
    mom.(fdNam{ifd}).mean = mom.(fdNam{ifd}).condmean .* mom.(fdNam{ifd}).FracPos;
end % ... and calculate grand mean (including zeros).

% NOT IN PAPER
if showTable                            % Show results:
    % Fill cell with row headers for the table:
    rowHeader = {'altruistic (to broke)      :', ...
                 'wealth-pool. (to non-broke):', ...
                 'exchange-motivated         :', ...
                 'total                      :'      };
       disp('---------------------------------------------------------------')
       disp('---------------------------------------------------------------')
       disp('FLOW TRANSFERS BY TYPE (BI-YEARLY, BOTH DIRECTIONS):')
    fprintf('                             %%>0   cond.mean    mean    %% of \n')
    fprintf('                                    (>/<0):             total \n')
    for ifd=1:numel(fdNam)              % Loop over all fields to show.
        if ifd==numel(fdNam)            % Put line before 'total' entry.
            disp('---------------------------------------------------------------')
        end
        fprintf('%s ',rowHeader{ifd})   % Print row header, then moments:
        fprintf('%4.1f%%   %3.0f K$   %4.1f K$   %4.1f%%\n', ...
                [100*mom.(fdNam{ifd}).FracPos, mom.(fdNam{ifd}).condmean, ...
                 mom.(fdNam{ifd}).mean, 100*mom.(fdNam{ifd}).mean/mom.ivabs.mean ] );
    end                                        
    disp('----------------------------------------------------------------')
end


%% B. Zoom in on different transfer types:
% Will loop over different transfer types:
% I. all inter-vivos (gifts+exchange). 
%   Will loop over 3 variables (=directions) of inter-vivos transfers:
%   parent-to-child, child-to-parent, net parent-to-child.
ivNam = {    'ivp'        ,    'ivk'        ,    'ivnet'    };
% Labels of rows: say which direction the transfers go, IV is arleady in
% header.
ivLbl = {'parent-to-child','child-to-parent','net p-to-ch'};
% II. Only gifts. Will look only at two:
gNam  = {     'gp'        ,     'gk'                        };
gLbl  = gNam;

% List labels for all transfer types to create table headers:
lblTransferTypes = {'INTER-VIVOS TRANSFERS (GIFTS+EXCHANGE)'    ,...
                    'GIFTS (ALTRUISTIC)'                        ,...
                    'ALTRUISTIC COMPONENT OF EXCHANGE TRANSFERS'    };
% Assemble cells (with sub-cells) that go across transfer types:
TTvarNames       = { ivNam, gNam }; % Variable names, ...
TTvarLbls        = { ivLbl, gLbl }; % ..variable labels.
if showTable
    TTprint      = {  true,false }; % If to print out results for each 
else                                % each type. Only do so for total
    TTprint      = { false,false }; % IVTs. Don't print at all if no 
end                                 % table is wanted.

% Will loop over different measurement types: 
% a) cumulative flows over the 2-year interview period
%    --> use this to compare to HRS, i.e. to match calibration targets
% b) flow (=policy) at current interview
%    --> use these to understand model properties. Flows from a) are often
%        "contaminated" by past states, especially at the end of life.
% Table headers for the two types of transfers:
HeaderMeasType = {'CUMULATIVE 2-YEAR TRANSFER FLOWS, BI-YEARLY',...
                  'TRANSFER FLOW AT INTERVIEW, YEARLY'  };
% MTprefix = {'','pol'};           %   Recall here measurement types.

% Now, define 6 or more groups (=subsamples in panel) over which to loop:
LTC = (SaugIndPan==indBC) | (SaugIndPan==indSC); % Create dummy for LTC.

grpDum  = {own & LTC,   ~own & LTC,  ~own & IC, own & IC, ...
           PPNH | FHC,  ~own & PPNH,     FHC  ,  ...   % , MA: checked MA here, looks OK.
           SaugIndPan==1, hlthOwners, SaugIndPan==1 & ~own,...
           AliveMainParent                                     };
      % Will automatically condition on alive main clone in loop again!      
grpNam = { 'LTCown'  ,  'LTCrent' ,  'ICrent' , 'ICown', ...
          'PPall'   ,   'PPrent'   ,    'FHC', ...  % , 'MA' 
          'hlthAll' ,   'hlthOwn'   ,   'hlthRent'   ,...
          'all'                                           };     


% NOT IN PAPER
if showTable, disp('---------------------------------------------------------------'), end
for tt=1:numel(TTvarNames)              % Loop over different transfer types
                                        % (all IV, gifts...).
    ttVarNam = TTvarNames{tt};          % Read out variable names, e.g.: 
                                        % {'ivp','ivk','ivnet'}.
    ttVarLbls= TTvarLbls{tt};           % Same for labels.
    for mt=1:numel(HeaderMeasType)      % Loop over two measurement types.
        % Construct field names we will look at for this transfer type:
        valFdNam = [MTprefix{mt},'val'];   % Either 'val' or 'polval'.
        if TTprint{tt}
            disp('---------------------------------------------------------------')
            % Print table header:
            fprintf('%s\n',lblTransferTypes{tt})    % Print transfer type.
            fprintf('(%s)\n',HeaderMeasType{mt})    % Print measurement type.
            fprintf('                                 %%     mean (cond''l   overall\n')
            fprintf('                              non-zero:   on >/<0)      mean  \n')
        end
        for iGrp=1:numel(grpDum)
            iGrpNam = grpNam{iGrp};             % Read out short group name.
            for vv=1:numel(ttVarNam)            % Loop over all transfer variables
                                                % of this transfer type.
                vvNam = ttVarNam{vv};           % Short variable name: 'ivp', 'ivk', ivnet'.
                [cm,n,fp] = GetConditionalMeanPanel( res.pan.flows.(vvNam).(valFdNam) ,...
                                          AliveMainParent & grpDum{iGrp},...
                                          true);
                                                % Get conditional mean (on
                                                % non-zero), number of
                                                % observations, and fraction
                                                % non-zero.
                if fp==0                        % Compute overall mean:
                    om = 0;                     % 0 if nobody gives.
                else                            % Otherwise:
                    om = cm*fp;                 % Compute by multiplying.
                end
                
                % Now, store results in structure mom:
                if strcmp(MTprefix{mt},'pol')  % If we are calculating current 
                    mom.(vvNam).(iGrpNam).pol.condmean = cm; % flows/policies
                    mom.(vvNam).(iGrpNam).pol.n        =  n; 
                    mom.(vvNam).(iGrpNam).pol.fracpos  = fp;
                    mom.(vvNam).(iGrpNam).pol.mean     = om;
                                                % add marker 'pol'.
                else                            % For the moments we want to 
                    mom.(vvNam).(iGrpNam).condmean = cm; % take to the HRS,
                    mom.(vvNam).(iGrpNam).n        =  n; % write in the bi-yearly
                    mom.(vvNam).(iGrpNam).fracpos  = fp; % cumulative flows
                    mom.(vvNam).(iGrpNam).mean     = om; 
                % Finally, for HRS moments also count which percentage is
                % above 500$ and the grand mean taking only observations
                % above this threshold:
                    sum500 = sum( res.pan.flows.(vvNam).(valFdNam)(AliveMainParent & grpDum{iGrp})       );
                    nGr500 = sum( res.pan.flows.(vvNam).(valFdNam)(AliveMainParent & grpDum{iGrp}) > 0.5 );
                    mom.(vvNam).(iGrpNam).mean500 = sum500/n;
                    mom.(vvNam).(iGrpNam).frac500 = nGr500/n;
                end                             % without the 'pol'-marker.
                
                if TTprint{tt}                  % If results for this transfer
                                                % are to be printed out:
                    if vv==1                    % Print group name only 
                        fprintf('%8s:  ',grpNam{iGrp}),   % the first time.
                    else                        % Otherwise, leave space.
                        fprintf('%8s   ',[]          )    
                    end
                    fprintf('%15s \t',ttVarLbls{vv})
                                                % Print direction:
                                                % parent-to-child etc.
                    fprintf('%4.1f%%    %5.1f K$   %5.1f K$\n', ...
                            [100*fp ,        cm ,      om    ]      );
                                                % Show results for current
                                                % group.
                end                             % Print IV type in table.
                
            end                         % End loop over 3 inter-vivos directions.
        end                             % End loop over groups (6 or more).
        if TTprint{tt}, disp('---------------------------------------------------------------'), end
    end                                 % End loop over 2 measurement types.
    if TTprint{tt}, disp('---------------------------------------------------------------'), end
end                                     % End loop over 2 transfer types.
if showTable, disp('---------------------------------------------------------------'), end


% Used to have results here on altruistic component of IV transfers:
% QPaltrComp, QKaltrComp. However, moved away from this: It is not clear
% what this altruistic component is supposed to mean. Rather stick with
% just separating Q and g, as done now:

% NOT IN PAPER
if showTable
    disp('---------------------------------------------------------------')
    disp('GIFT COMPONENT OF INTER-VIVOS TRANSFERS')
    disp('(gifts / all transfers)')
    disp('                    2-year     current   mean total         ')
    disp('                  cumulative     flow     IVTs (2yr)    N   ')
end
for iGrp=1:numel(grpDum)            % Do this for all sub-groups:
    iGrpNam = grpNam{iGrp};         % Read out group name.
    % First for parents:
    mom.ivpFracG.(iGrpNam)    = mom.gp.(iGrpNam).mean     / mom.ivp.(iGrpNam).mean;
    % COULD ALSO DO FOR CURRENT FLOW/POLICY, BUT LEAVE OUT FOR NOW:                        
    mom.ivpFracGPol.(iGrpNam) = mom.gp.(iGrpNam).pol.mean / mom.ivp.(iGrpNam).pol.mean         ;


    % Then for children:
    mom.ivkFracG.(iGrpNam)    = mom.gk.(iGrpNam).mean     / mom.ivk.(iGrpNam).mean    ;
    mom.ivkFracGPol.(iGrpNam) = mom.gk.(iGrpNam).pol.mean / mom.ivk.(iGrpNam).pol.mean;
    % Has to be minus for child-Q since these are negative numbers.

% NOT IN PAPER
    if showTable
        fprintf('%8s:  ',iGrpNam)       % Print group name.

        fprintf('p-to-k:     %3.0f%%       %3.0f%%    %5.1f K   %6.0f\n',...
                [100*mom.ivpFracG.(iGrpNam),100*mom.ivpFracGPol.(iGrpNam),...
                     mom.ivp.(iGrpNam).mean, mom.ivp.(iGrpNam).n] );
        fprintf('%8s:  ',[])            % Don't put group name this time.
        fprintf('k-to-p:     %3.0f%%       %3.0f%%    %5.1f K   %6.0f\n',...
                [100*mom.ivkFracG.(iGrpNam), 100*mom.ivkFracGPol.(iGrpNam),...
                        mom.ivk.(iGrpNam).mean, mom.ivk.(iGrpNam).n ] );
    end
end
if showTable                        % Last table line.
    disp('---------------------------------------------------------------')
end
% Example: PPall, child-to-parent:  7K altr.comp., 7.5K gifts, 17.3K all.
%                   --> 75%. makes sense
%          ICrent, par-to-child:    11.8 altr.comp., 1.0 gifts,    12.2 all 
%                   --> 105%: makes no sense. 

% ZOOM IN ON A SET OF KID TRANSFERS WE WANT TO CHECK:
dumVar = { AliveMainParent & FHC, AliveMainParent & PPNH         , ...
           AliveMainParent & (FHC | PPNH) , ...
           AliveMainParent & IC & own , AliveMainParent & SaugIndPan>1 & own , ...           
           AliveMainParent & SaugIndPan==1     }; 
dumFdNam={ 'FHC','PPrent','PPall','ICown' ,'LTCown' , 'hlthAll'};  
                                % Field names in output structure
                                % associated with the variable.
dumLbl = { 'FHC', 'PP-NH','PC'   ,'IC own','LTC own', 'healthy'};  
                                % Labels for printing.
qq  = 0.95;                   % Which quantile of kid asset distribution 
                                % to take as cut-off.
akQthr = GetQuantProbDist(res.pan.stocks.ak(:,:,1),qq);
                                % Compute this quantile.
apQthr = GetQuantProbDist(res.pan.stocks.ak(AliveMainParent)...
                          +res.pan.stocks.h(AliveMainParent)   ,qq);                                 
EkInd3D = EkIndPan(:,:,ones(1,nCln));
                                % Create 3D version of kid prod. for
                                % indexing: (nPan,TPan,nCln) array.
thr = sort([500,akQthr,1000]);  % Thresholds we'd like to consider: 500K,
                                % Q95 of kid wealth (654K), 1m:
                                % 1-by-nThr vector.
%fprintf('akQ50   akQ75  akQ90  %%ak>500K  %%ak>1m   ')
eki = [3,4];                    % Kid prod. indeces we want to count values
                                % for: take the highest two. 1-by-Ne
% NOT IN PAPER                  % vector.
if showTable                    % Print table header.                            
    disp('-------------------------------------------------------------------------')
    disp('Comparing observations with positive IV transfer by kid')
    disp('(bi-yearly cumulative flows)')
    fprintf('               N   #pos')                                
    fprintf(' ak>%4.0fK',thr)                               
    fprintf(' epsk=%1.0f',eki) 
    fprintf(' h>%3.0fK\n',apQthr)
end
for vv=1:numel(dumVar) 
    N = sum(dumVar{vv}(:));        % Total # observations.
    vvIvkPos = dumVar{vv} & res.pan.flows.ivk.val>0;
                                % Dummy if positive IVK within this group.
    vvIvkPan = res.pan.flows.ivk.val(vvIvkPos);
                                % Read out vector with these positive IVKs.
                                % nIV-by-1 vector.
    nPos = numel(vvIvkPan);                   % # positive IVK observations.
    vvPctAbvThr = 100*sum(res.pan.stocks.ak(vvIvkPos)>thr,1)/nPos;
                                % Get 1-by-nThr vector with percentage of
                                % ak-values associated with positive IVTs
                                % are above each threshold value.
    vvPctEk = 100*sum(EkInd3D(vvIvkPos)==eki,1)/nPos;
                                % Get 1-by-nE vector with fraction of
                                % positive IVT observations among which kid
                                % prod. equals levels 3 and 4.
    vvPctHih = 100*sum(res.pan.stocks.h(vvIvkPos)>apQthr,1)/nPos;
    if showTable                % Print table row.
        fprintf('%7s:  ',dumLbl{vv})        
        fprintf('%6.0f %6.0f',[N,nPos])
        fprintf('     %3.0f%%',vvPctAbvThr);
        fprintf('   %3.0f%%',vvPctEk);   
        fprintf('   %3.0f%%\n',vvPctHih)
    end
end
if showTable
    fprintf('Note: %3.0fK is Q%2.0f of the child net-worth distribution and\n' ,[akQthr,100*qq]);
    fprintf('      %3.0fK is Q%2.0f of the parent net-worth distribution in the model.\n',[apQthr,100*qq]);
    disp('-------------------------------------------------------------------------')
end
% Search for calibration target for kid altruism: See how sensitive
% different kid-IVT means are to excluding top observations. Answer: None
% is too sensitive.
mainClone = false(nPan,TPan,nCln);      % Create logical array that marks 
mainClone(:,:,1) = true;                % main clone.

% NOT IN PAPER
if showTable                                   
disp('-------------------------------------------------------------------------')
disp('IV-TRANSFER MEANS (BI-YEARLY, CUM.) BY GROUP AND EXCLUSION CRITERION')
disp('                      exclusion threshold:')
fprintf('donor : par.status:  ')
for i=1:numel(qExcl)                    % thresholds:
    fprintf('   Q%2.0f',100*qExcl(i))
end
    fprintf('  none  \n')

% Create variable that only has transfer flows above 500$:    
res.pan.flows.ivk.val500 = (res.pan.flows.ivk.val>0.5).*res.pan.flows.ivk.val;
% kidIVmeansByThrd = zeros(numel(dumVar),numel(thrExcl));
                                        % Set up matrix to store results.
for vv=1:numel(dumVar)                  % Loop over different variables/groups.
    mom.ivk.(dumFdNam{vv}).MeanByExclThr = zeros(1,numel(thrExcl));
                                        % Set up field in output structure
                                        % with vector containing the means
                                        % by exclusion threshold.
    if vv==1                            % For first group, print that this
        fprintf('kid   :  ')            % is the kid as donor.
    else                                % From then on, just leave blank.
        fprintf('         ')
    end
    fprintf('%9s:  ',dumLbl{vv})        % Print group label.
    for tt=1:numel(thrExcl)             % Loop over exclusion thresholds.
      %  tvInd = dumVar{vv} & NtwpPan3D<thrExcl(tt);  % This would be the 
      % restriction only on the parent.
        tvInd = dumVar{vv} & res.pan.stocks.ak<thrExcl(tt) & ...
                NtwpPan3D<thrExcl(tt) ; % Create dummy for group and both agent's
                                        % net worth being below threshold.
      %  mvvtt = GetConditionalMeanPanel(res.pan.flows.ivk.val,tvInd);
                                        % Get conditional mean for the
                                        % threshold, store in output 
        % This is the code excluding transfers below 500 -- basically
        % doesn't matter for the grand means:
        mvvtt = GetConditionalMeanPanel(res.pan.flows.ivk.val500,tvInd);
        mom.ivk.(dumFdNam{vv}).MeanByExclThr(1,tt) = mvvtt;
        fprintf('  %4.1f',mvvtt)        % structure and print out result.
                                        % Calculate donditional mean, print
    end                                 % and store in matrix.
    fprintf('\n')                       % New line for next group. 
end
% Now, do the same for healthy parents' IVTs. First, set up field in
% structure to store vector with means for all exlusion thresholds:
mom.ivp.hlthAll.MeanByExclThr = zeros(1,numel(thrExcl));
fprintf('parent:  %9s:  ','healthy') 
hlth = AliveMainParent & SaugIndPan==1;
for tt=1:numel(thrExcl)             % Loop over exclusion thresholds.  
    ptind = hlth & res.pan.stocks.ak<thrExcl(tt) & ...
            NtwpPan3D<thrExcl(tt) ; % Create dummy for healthy parent and
                                    %  both agent's net worth being below threshold.
    mom.ivp.hlthAll.MeanByExclThr(1,tt) = GetConditionalMeanPanel(res.pan.flows.ivp.val,ptind);
    fprintf('  %4.1f',mom.ivp.hlthAll.MeanByExclThr(1,tt))
end                                 % calculate donditional mean, print
fprintf('\n')                       % and store in vector.

disp('Note: Means are conditional on group (FHC etc.) and on both parent and ')
disp('      child net worth being below quantile Q.. of model''s parent net-')
disp('      worth distribution.')
disp('-------------------------------------------------------------------------')
end

end



%% LIFE-TIME INTER-VIVOS TRANSFERS

if ~strcmp(opt.CountFl,'noKid')
% Again, use the 1-by-4 vector with exlusion values. Will loop over these
% and compute numbers separately to check robustness:
nExcl = numel(thrExcl);             % Number of exclusion thresholds.
%dtPan = par.dtPan;                  % Read out panel period: 2 years.

for tt=1:nExcl                      % Loop over all thresholds.
    ttE = thrExcl(tt);              % Read out exlusion threshold.
    % Read out vectors with panel observations of main clone that are alive
    % and are below the threshold:
    % First, create (nPan,TPan,nCln) dummy for observations of (alive) main
    % clone with net worth below the threshold:
    ttParIn =   AliveMainParent & (NtwpPan3D        <ttE) ; 
    ttSampleSize = sum(ttParIn(:));           % Sample size excluding top 5%
    ttNtwp   =         NtwpPan3D(ttParIn);    % Vectors with net worth of all
    ttNtwk   = res.pan.stocks.ak(ttParIn);    % parents and kids in sample.
    ttNtwall = [ttNtwp;ttNtwk];  
    tot.sum.ntwp(tt)  = sum(ttNtwp);
    tot.sum.ntwk(tt)  = sum(ttNtwk);
    tot.sum.ntwall(tt)= sum(ttNtwall);
    tot.ntwp(  tt)    = mean(ttNtwp);          % Avg. net worth of alive parents,
    tot.ntwk(  tt)    = mean(ttNtwk);          % of kids, and 
    tot.ntwall(tt)    = mean([ttNtwp;ttNtwk]); % total in economy.
    % Now, obtain yearly transfer measures:
    ttBeqIn =  mainClone(:,1,:) & (res.pan.once.Beq<ttE);
                                    % Bequests in sample if below threshold.
    tot.sum.Beq(1,tt)    = sum(res.pan.once.Beq(ttBeqIn));
    % Bequest flow per interview period:
    tot.BeqPerIntw(1,tt) = tot.sum.Beq(1,tt) / ttSampleSize;
    % Now, get average bequest by dividing by number of panel members:
    tot.BeqAvg(    1,tt) = tot.sum.Beq(1,tt) / nPan;
    % --> Since life expectancy is 19 at age 65, there should be about 10
    %     (~19/2+eps) interviews per panel member. Thus, BeqAvg ~
    %     10*BeqPerIntw. 
    % Now, get transfer flows per interview period:
    tot.gpPerIntw( 1,tt) = mean( res.pan.flows.gp.val(ttParIn) );
    tot.QpPerIntw( 1,tt) = mean( res.pan.flows.Qp.val(ttParIn) );
    tot.ivpPerIntw(1,tt) = mean( res.pan.flows.ivp.val(ttParIn));
    tot.ivkPerIntw(1,tt) = mean( res.pan.flows.ivk.val(ttParIn));
    % Also calculate sums to check below:
    tot.sum.ivp(1,tt) = sum( res.pan.flows.ivp.val(ttParIn));
    tot.sum.ivk(1,tt) = sum( res.pan.flows.ivk.val(ttParIn));
    tot.sum.gp(1,tt)  = sum( res.pan.flows.gp.val(ttParIn));
end
tot.ivpBeqRat = tot.ivpPerIntw ./ tot.BeqPerIntw;
% Note: It doesn't matter here if we take transfers per INTERVIEW, per
% PERSON-YEAR (including dead parents) etc. as long as we divide by the
% same number for both IV-transfers and bequests. 
% Do check with sums: Should be equal with threshold inf, but can vary for
% other thresholds: Works.
% checkIvpBeq = tot.sum.ivp ./ tot.sum.Beq;

tot.BeqNtwRat    = (tot.sum.Beq/2) ./ tot.sum.ntwall ;
tot.BeqNtwpRat   = (tot.sum.Beq/2) ./ tot.sum.ntwp ; % Also, divide by total net worth of parents
tot.BeqNtwkRat   = (tot.sum.Beq/2) ./ tot.sum.ntwk ; % and by total net worth by kids
tot.ivpNtwRat    = (tot.sum.ivp/2) ./ tot.sum.ntwall;
tot.ivkNtwRat    = (tot.sum.ivk/2) ./ tot.sum.ntwk;
tot.gpNtwRat     = (tot.sum.gp/2)  ./ tot.sum.ntwall;

SumGpByWav = sum(res.pan.flows.gp.val(:,:,1).*AlivParBelwThr(:,:,1),1);
% 1-by-15 vector: Total parent gifts
% in panel by wave for alive main clone below
% the wealth threshold.
SumIvpByWav= sum(res.pan.flows.ivp.val(:,:,1).*AlivParBelwThr(:,:,1),1);
% Same for total parent inter-vivos.

% Now, get average ages at which different transfers are given:
ageVec = res.pan.agevec;        % Ages at which panel members are interviewed:; 1-by-15.
% Finally, find average age at which different transfer types are
% given: Sum over age
avgAgeGp = sum(ageVec.*SumGpByWav )/sum(SumGpByWav );
avgAgeIvp= sum(ageVec.*SumIvpByWav)/sum(SumIvpByWav);

% Create logical marker for main clones whose bequest falls below
% exclusion threshold: (nPan,1,1) array.
BeqBlwThr = res.pan.once.Beq(:,1,1)<thrExcl(indExclThr);
BB = res.pan.once.Beq(BeqBlwThr,1,1);% Create NBB-by-1 vector with
% those bequests.
avgAgeBeq= sum(res.pan.age.Died(BeqBlwThr).*BB)/sum(BB);

% NUMBERS CITED WITHIN PAPER
       disp('---------------------------------------------------------------')
       disp('AGE AT WHICH AVERAGE TRANSFER $ IS GIVEN BY PARENTS OVER 65:')
       disp('---------------------------------------------------------------')
       disp('Transfer type:                    |        Model       Data    ')
       disp('---------------------------------------------------------------')
    fprintf('Inter-vivos transfer by parent    |        %5.1f       75.5\n',avgAgeIvp)
    fprintf('         (only altruistic gifts)  |        %5.1f        -- \n',avgAgeGp )
    fprintf('Bequests                          |        %5.1f       83.7\n',avgAgeBeq)
       disp('---------------------------------------------------------------')

if showTable
   disp('---------------------------------------------------------------')
   disp('COMPARING TO GALE & SCHOLZ (CROSS SECTION):') 
fprintf('          |Model with exclusion threshold: |    data\n') 
fprintf('          |')
for i=1:numel(qExcl)                    % thresholds:
    fprintf('     Q%2.0f',100*qExcl(i))
end
fprintf('    none  \n')
disp('---------------------------------------------------------------')
% Set up cells with all statistics we want to show and labels for them:
fprintf('Yearly transfer flows as a ratio out of total wealth:\n')
fdTot = {'BeqNtwRat','ivpNtwRat','gpNtwRat','ivkNtwRat'};
fdLbl = {'bequests' ,'IV parent','g parent','IV kid'   };
fdDat = [   1.06    ,    0.32   ,   NaN     ,   0.03    ];   % statistic from data.
for tt=1:numel(fdTot)                       % Loop over the statistics.
    fprintf('%9s |',fdLbl{tt})              % Print row label.
    fprintf('%7.2f%%',100*tot.(fdTot{tt}))% Print the 4 model statistics
    fprintf('|%7.2f%%\n',fdDat(tt))         % (adjusted) and data counterpart,
end                                         % which is set apart by "|".
disp('---------------------------------------------------------------')
disp('Ratio IV(parent)-to-bequest:')
fprintf(' ivp/beq. |')                      % Print row label.
fprintf('%7.2f%%',100*tot.ivpBeqRat)     % Print the 4 model statistics
fprintf('|%7.2f%%\n',33.3)                  % Ratio in data is precisely
                                            % one-third in G&S.
disp('---------------------------------------------------------------')
fprintf('Net worth per person in economy:\n')
fdTot = {'ntwall','ntwp','ntwk'};           % Set up cell with the different
fdLbl = {'all' ,'parents','kids'};          % net worth measures.
for tt=1:numel(fdTot)                       % Loop over the statistics.
    fprintf('%9s |',fdLbl{tt})              % Print row label.
    fprintf('%6.0fK$',tot.(fdTot{tt}))  % Print the 4 model statistics.
    fprintf('\n')         
end  
disp('---------------------------------------------------------------')
fprintf('Average bequest in the economy:\n')
fprintf('     beq. |')
fprintf('%6.0fK$',tot.BeqAvg);
fprintf('\n')
disp('---------------------------------------------------------------')
disp('---------------------------------------------------------------')
    % NOTE: Gale and Scholz: Aggregate net worth in SCF = 11976. 
    %       Bequests (105) + trusts (14.2) + life insurance (7.8) = 127.
    %       Bequest-to-wealth ratio:                       127/11976=1.06%
    %       Support given to children and grandchildren: 37.74/11976=0.32%
    %       Support given to parents and grandparents:    3.44/11976=0.03%

    % In the SCF, transfers are only recorded if they are above $3,000 
    % implying significant under-reporting. College expenses are excluded but
    % would account for another 0.29% to the IVT-to-wealth ratio.
    
end

end


%% Finally, make figure of inter-vivos transfers by age:
if ~strcmp(opt.CountFl,'noKid') % Only do this in economy with kids.
    
CoreIntw = 1:TPan < exitIntw;   % Mark core interviews by logical variable:
                                % nPan-by-TPan matrix.
nCore = sum(CoreIntw,1);        % Get number of core interviews by wave:
                                % 1-by-TPan vector.
% Calculate average IVT by parent by age: 
% Pulled the following to top of file:
SumIvpPan = sum(AlivParBelwThr(:,:,1).*res.pan.flows.ivp.val(:,:,1).*CoreIntw,1);

% Now, only consider if transfer amount higher than 500:
SumIvpPanOver500 = sum(AlivParBelwThr(:,:,1).*res.pan.flows.ivp.val(:,:,1).*CoreIntw.*(res.pan.flows.ivp.val(:,:,1)>=0.5),1);
% Finally, do the same for altruistic IVTs by parent:
SumGpPan = sum(AlivParBelwThr(:,:,1).*res.pan.flows.gp.val(:,:,1).*CoreIntw,1);

mom.byAge.ivp        = SumIvpPan./nCore;       % Divide by number of 
mom.byAge.ivpOver500 = SumIvpPanOver500./nCore;% observations, append to
mom.byAge.gp         = SumGpPan./nCore;        % output structure.

% NOT IN PAPER
if showTable
    figure;                         % Make figure.
    hold on
    plot(res.pan.agevec,mom.byAge.ivp       , '-b', 'LineWidth',2 );
    plot(res.pan.agevec,mom.byAge.ivpOver500,'-xb', 'LineWidth',2 );
    plot(res.pan.agevec,mom.byAge.gp        ,'-.k', 'LineWidth',2 );
    xlabel('age at interview')
    ylabel('transfers by parent in past 2 years (''000$)')
    legend('average IVT','average IVT (>500$)','average altruistic IVT')
end

% NOT IN PAPER
% Make figure on inter-generational earnings correlation:
IntergEarnCorr = zeros(1,TPan);     % Vector for time evolution.
for t=1:TPan                        % Loop over all waves.
    tAlive = panAlive(:,t);     % nPan-by-1 vector if alive.                 
    tKidEarn = par.epsvec( EkIndPan(tAlive,t) );    % Get kid earnings and    
    tParEarn = par.epsvec( EpIndPan(tAlive,t) );    % parent earnings (as 
                                    % multiple of their means) whenever 
                                    % parents are alive.
    cc = corrcoef(tKidEarn,tParEarn);   % GEt 2-by-2 matrix with corr.coeff.      
    IntergEarnCorr(t) = cc(1,2);    % and read into vector.
end
IntergEarnCorrTheor = IntergEarnCorr(1).^(1:TPan);
                                    % Calculate what we would get by just
                                    % geometrically estimating ahead as for
                                    % AR(1).
if isfield(res,'EpInd65')           % If parent indeces at 65 were stored...                                       
    % Finally, get total intergenerational earnings correlation in this model:
    KidEps = par.epsvec( EkIndPan(:,:,1) );     % Kid epsilon: nPan-by-TPan 
    KidEarn = par.HHeffUnits(res.pan.kidage,KidEps );  % Get kid earnings.                                 
    ParEarn = par.epsvec( res.EpInd65 ).*ones(1,TPan);    % Parent eps: nPan-by-TPan  
    cc = corrcoef(KidEarn(AliveMainParent),ParEarn(AliveMainParent));                       
    IntergEarnCorrTotal = cc(1,2);      
end
if showTable
    figure;                             % Plot the two against each other.
    plot(res.pan.kidage,IntergEarnCorr,'-b',res.pan.kidage,IntergEarnCorrTheor,'--k');
    xlabel('kid age t')
    ylabel('correlation kid earnings at t and parent earnings (at 65)')
    legend('panel correlation','compounded 2-year autocorrelation')
    if exist('IntergEarnCorrTotal','var')   % Also put overall coefficient if 
        text(40,0.7,sprintf('overall correlation: %1.3f',IntergEarnCorrTotal))
    end                                     % we have it.
end

mom.IntergEarnCorr = IntergEarnCorr;        % Append results to output 
                                            % structure.
end



%% Wealth equivalent variation (WEV): statistics.
if par.Nh>1
qtl = [0.1, 0.25, 0.5, 0.75, 0.9];
if noKid                        % Which field captures the willingness to
    wevFd = 'WEVp65';           % pay of the decision maker for housing:
else                            % parent's WEV in no-kid economy, dynastic
    wevFd = 'WEVdyn65NoRedistrb';   
end
buyHm = res.pan.once.h65>0;         % Set WEV=0 for households that don't buy.
res.pan.once.(wevFd)(res.pan.once.(wevFd)>0 & ~buyHm)=0;

% Obtain mean and quantiles of this WEV at age 65:
% In line with what we do otherwise, exclude parents with net worth above
% the exclusion threshold to calculate mean:
BlwThr = res.pan.once.Nwp65(:,1,1) < thrExcl(indExclThr);   % nPan-by-1 vector.
% Then, calculate mean excluding these:
mom.wev65.mean = mean(res.pan.once.(wevFd)(BlwThr));
% 
mom.wev65.quant= GetQuantProbDist(res.pan.once.(wevFd),qtl);


% Now, get means conditional on parent productivity:
mom.wev65.byEp.mean = zeros(Nw,1);
for iw=1:Nw
    mom.wev65.byEp.mean(iw) = ...
       mean( res.pan.once.(wevFd)(BlwThr & EpIndPan(:,1)==iw) );   
end

% Now, for each quintile of net-worth distribution, get average WEV:
nq = 5;                         % Take quintiles:
nwQuant = linspace(0,1,nq+1);   % Cut-offs for quintiles: [0,0.2,...,0.8,1]:
                                % 1-by-6 vector.
nwBins = zeros(size(nwQuant));     % Vector with net-worth quintiles (i.e. bins 
                                % bins for net worth): 1-by-6.  
nwBins(2:end-1) = GetQuantProbDist(res.pan.once.Nwp65,nwQuant(2:end-1));
                                % Calculate quantiles [0.2,0.4,0.6,0.8].
nwBins(1  ) = -1;                  % Set lower limit negative to be save,                                
nwBins(end) = Inf;                 % set upper limit to infinity.
mom.wev65.byNwp.mean = zeros(nq,1);
for iq=1:nq 
    isinQuint = res.pan.once.Nwp65>=nwBins(iq) & res.pan.once.Nwp65<nwBins(iq+1);
                                % Logical marker for the quintile:
                                % nPan-by-1 vector.
    mom.wev65.byNwp.mean(iq) = mean( res.pan.once.(wevFd)(isinQuint(:,1,1)) );
end                             % Get conditional mean in quintile.

% Could calculate here also a percentage, for example:
% Fraction of house value (of the chosen house) a household is willing to
% give up to have access to the housing technology.
WEVpctHous                            = res.pan.once.(wevFd) ./ res.pan.once.h65;
WEVpctHous( res.pan.once.(wevFd)==0 ) = 0;  % Assign zeros if WEV is zero.
WEVpctHous( res.pan.once.h65==0 )     = 0;  % Assign zeros if hoousing is zero.
mom.wev65.pctHousMean                 = mean(WEVpctHous(BlwThr));
mom.wev65.pctHousQuant                = GetQuantProbDist(WEVpctHous,qtl);
 
if showTable
    
    disp('---------------------------------------------------------------')

    disp('TABLE 12: Value of housing technology, measured by dynasty wealth equivalent variation (WEV) at age 65') 
    disp('TABLE F.4:                            "                               (when xi=0.90)                 ')
    disp('---------------------------------------------------------------')
    disp('               mean  |  p10      p25       p50       p75     p90      ')
    fprintf(' WEV:        %4.0fK    %4.0fK    %4.0fK     %4.0fK     %4.0fK    %4.0fK \n', ...
                    [mom.wev65.mean  mom.wev65.quant ]                               )
    fprintf(' WEV/House:  %2.1f%%       %2.1f%%      %2.1f%%     %2.1f%%    %2.1f%%   %2.1f%% \n', ...
            100* [mom.wev65.pctHousMean  mom.wev65.pctHousQuant ]                               ) 
    disp('---------------------------------------------------------------')
    fprintf('by prod.:           %4.0fK    %4.0fK     %4.0fK     %4.0fK \n', ...
                                 mom.wev65.byEp.mean'                              );
    fprintf('by parent:          %4.0fK    %4.0fK     %4.0fK     %4.0fK    %4.0fK\n ', ...
                                 mom.wev65.byNwp.mean' );
    disp('net-worth quantile')                        
    disp('---------------------------------------------------------------')
    disp('Note: For means, observations with parent net worth above Q95 of ')
    disp('      parent net-worth are excluded (however not in last line).')
    disp('---------------------------------------------------------------')
end
end


if showTable 
    
   disp('-----------------------------------------------------------------------')
   disp('TABLE 3: Calibration: Calibration targets ')
   disp('Moment                                         Data          Model ')        
   disp('-----------------------------------------------------------------------')
fprintf('Median household wealth 65-70                   206.0K      %4.0fK$\n' , mom.ntwp65.quant(3)      );   
fprintf('Home ownership 65+                              75.2%%        %4.1f%%\n',     mom.ownRate      ); 
fprintf('Informal care                                   48.7%%        %4.1f%%\n', 100*mom.care.all.IC.frac       ); 
fprintf('Medicaid uptake rate                            27.5%%        %4.1f%%\n', 100*mom.care.all.MA.frac       ); 
if ~strcmp(opt.CountFl,'noKid')     % Only show this in economy with kids.
    fprintf('Mean inter-vivos transfers: p-to-c              3,356K       %4.0f$ \n', ...
             1000*mom.ivp.hlthAll.MeanByExclThr(indExclThr)); % trimmed. 
    fprintf('among healthy (including zeros, excl. top)\n');
end
   disp('-----------------------------------------------------------------------')
fprintf('Note: Transfers are bi-yearly, excluding households with wealth above\n')
fprintf('      exclusion threshold of Q%2.0f of the wealth distribution (%1.0f K$).\n',...
                [100*qExcl(indExclThr), thrExcl(indExclThr)]  );
   disp('-----------------------------------------------------------------------')

% NOT IN PAPER   
disp('INITIAL WEALTH AND BEQUESTS')
disp('Moment                           Data       Model   Mod.>500$')     
disp('-----------------------------------------------------------------------------')
disp('                                   Total       Total     Fin       Hous      ')
fprintf('Q10 wealth at 65-70                2K       %4.0fK     %4.0fK     %4.0fK\n',[mom.ntwp65.quant(1),  mom.ntwp65.quantFin(1), mom.ntwp65.quantH(1)]);
fprintf('Q25 wealth at 65-70               54K       %4.0fK     %4.0fK     %4.0fK\n',[mom.ntwp65.quant(2),  mom.ntwp65.quantFin(2), mom.ntwp65.quantH(2)]);
fprintf('Q50 wealth at 65-70              206K       %4.0fK     %4.0fK     %4.0fK\n',[mom.ntwp65.quant(3),  mom.ntwp65.quantFin(3), mom.ntwp65.quantH(3)]);
fprintf('Q75 wealth at 65-70              553K       %4.0fK     %4.0fK     %4.0fK\n',[mom.ntwp65.quant(4),  mom.ntwp65.quantFin(4), mom.ntwp65.quantH(4)]); 
fprintf('Q90 wealth at 65-70             1229K       %4.0fK     %4.0fK     %4.0fK\n',[mom.ntwp65.quant(5),  mom.ntwp65.quantFin(5), mom.ntwp65.quantH(5)]);
fprintf('Q95 wealth at 65-70             1966K       %4.0fK     %4.0fK     %4.0fK\n',[mom.ntwp65.quant(6),  mom.ntwp65.quantFin(6), mom.ntwp65.quantH(6)]);
fprintf('Q25 bequests                       0K       %4.0fK     %4.0fK     %4.0fK\n',[mom.beq.all.quant(1), mom.beq.fin.quant(1),   mom.beq.h.quant(1)]); 
fprintf('Q50 bequests                      22K       %4.0fK     %4.0fK     %4.0fK\n',[mom.beq.all.quant(2), mom.beq.fin.quant(2),   mom.beq.h.quant(2)]); 
fprintf('Q75 bequests                     198K       %4.0fK     %4.0fK     %4.0fK\n',[mom.beq.all.quant(3), mom.beq.fin.quant(3),   mom.beq.h.quant(3)]); 
fprintf('Q90 bequests                     521K       %4.0fK     %4.0fK     %4.0fK\n',[mom.beq.all.quant(4), mom.beq.fin.quant(4),   mom.beq.h.quant(4)]); 
fprintf('Q95 bequests                     806K       %4.0fK     %4.0fK     %4.0fK\n',[mom.beq.all.quant(5), mom.beq.fin.quant(5),   mom.beq.h.quant(5)]);  
disp('-----------------------------------------------------------------------------')

outProdProcess=par.outProdProcess;


disp('---------------------------------------------------------------')
disp('TABLE D.1: Earnings process fit')
   disp('Moment                         Data           Model')     
   disp('---------------------------------------------------')
fprintf('Bottom Q10                      1.6%%        %5.1f%%\n', outProdProcess.ModelBottom102040(1));  
fprintf('Bottom Q20                      5.3%%        %5.1f%%\n', outProdProcess.ModelBottom102040(2));  
fprintf('Bottom Q40                      16.2%%       %5.1f%%\n', outProdProcess.ModelBottom102040(3));  
fprintf('Top Q40                         67.6%%       %5.1f%%\n', outProdProcess.ModelTop402010(1));  
fprintf('Top Q20                         44.5%%       %5.1f%%\n', outProdProcess.ModelTop402010(2));  
fprintf('Top Q10                         28.5%%       %5.1f%%\n', outProdProcess.ModelTop402010(3));  
fprintf('Gini coeff.                     37.4%%       %5.1f%%\n', 100*outProdProcess.ModelGini); 
fprintf('Mean error                                   %5.1f\n', outProdProcess.OECDerror);  
disp('---------------------------------------------------')

end
    

out=mom;
end