function X = spkron(A,B)
% Fast computation of Kronecker product for sparse matrices.
% HOWEVER: In the following tests kron.m is slightly faster:
% clear
% nA = 100;  % or 150, 200 (don't do 1000 on laptop)
% nB = nA;
% A = sprand(nA,nA,10/nA);
% B = sprand(nB,nB,10/nB);
% tic, XX = spkron(A,B); toc
% tic, X = kron(A,B); toc
% tic, Z = kronmult({A,B})
% 
% Inputs:
% A:        sparse matrix (n1A-by-n2A) with kA non-zero elements.
% B:        sparse matrix (n1B-by-n2B) with kB non-zero elements.
% 
% Output:
% X:        sparse matrix (n1A*n1B-by-n2A*n2B) with kA*kB non-zero
%           elements.

[n1A, n2A] = size(A);                   % Read out size of input matrices
[n1B, n2B] = size(B);
n1X = n1A*n1B;                          % Get size of output matrix
n2X = n2A*n2B;

[ind1A, ind2A, valA] = find(A);         % Read out indeces and values of the
[ind1B, ind2B, valB] = find(B);         % non-zero elements of the inputs
kA = length(valA);                      % ('find' is fast for this),
kB = length(valB);                      % and number of non-zero elements

% Now, create kB-by-kA matrices that contain the indices and values of the
% output matrix X:
ind1X = (ind1A(:,ones(kB,1)) - 1) * n1B  +  ( ind1B(:,ones(kA,1)) )';
                                        % Extend the column vector ind1A
                                        % rightward kB times (and multiply),
                                        % for ind1B convert to row vector
                                        % and copy kA times downward, and
                                        % add up to get indeces for output.
ind2X = (ind2A(:,ones(kB,1)) - 1) * n2B  +  ( ind2B(:,ones(kA,1)) )';
                                        % Same for indeces in dim. 2
valX  = valA * valB' ;                  % Take outer product of A and B's
                                        % values to get entries of X.

X = sparse(ind1X(:),ind2X(:),valX(:),n1X,n2X);   
                                        % create sparse output matrix
