function [xp, xk] = StatFBgift(IncP,IncK,Ap,Ak,thetap,thetak,alphap,alphak,...
                               valNoConsFeasible,tol,dims)
% Returns the dictator gift for parent (xp) and child (xk) in a static 
% altruism framework with incomes IncP & IncK, marginal-utility shifters Ap
% & Ak, curvature on felicity thetap & thetak and altruism parameters
% alphap & alphak. This gift may be negative, i.e. the agent wants to take
% away from the other. The function returns NaN if total dynasty income is
% negative, i.e. IncP+IncK<=0, which can occur with negative IncP (which we
% have for nursing homes).
%
% Inputs:
% IncP:         Array conformable with size dims. Parent flow income, may
%               negative.
% IncK:         Array conformable with size dims. Child flow income, may be
%               negative.
% Ap:           Array conformable with size dims. Parent's marginal-utility
%               shifter.
% Ak:           Array conformable with size dims. Child's marginal-utility
%               shifter.
% thetap:       Array conformable with size dims. Parent's curvature in
%               felicity.
% thetap:       Array conformable with size dims. Child's curvature in
%               felicity.
% alphap:       Array conformable with size dims. Parent's altruism. 
% alphak:       Array conformable with size dims. Child's altruism. 
% valNoConsFeasible:
%               Scalar. Small number that 

% dims:         Row vector, optional. Dimensionality of the output. Must
%               equal the dimensionality spanned by inputs IncP to alphak.

% Outputs:
% xp:           Array of size dims. Parent's dictator gift. NaN is handed
%               over if total dynasty income is negative.
% xk:           Array of size dims. Child's dictator gift. NaN is handed
%               over if total dynasty income is negative.

if nargin<11                % If dimensionality is not handed over:                   
    dimTest = IncP+IncK+Ap+Ak+thetap+thetak+alphap+alphak; 
    dims     = size(dimTest);
end                         % Add up all relevant inputs to get size that 
                            % outputs will have.

yp = IncP.*ones(dims);      % Extend income variables to full dimension 
yk = IncK.*ones(dims);      % for indexing.

NegDynInc = (yp+yk<=0);     % Logical variable that tells us if dynasty as
                            % a whole has negative income (occurs when 
                            % dynasty cannot afford NH and will end up in
                            % Medicaid).                 
yp(NegDynInc) = -yk(NegDynInc) + 3*valNoConsFeasible;
                            % Make dynasty income slightly positive to
                            % avoid complex numbers in the loop below.
                            % Will later set dictator gifts for these cases
                            % to NaN. It is inexpensive to do this since 
                            % total number of calculations is low.
dx = (yp+yk-valNoConsFeasible)/2;                 
                            % Range of possible net transfers is from 
                            % yp to -yk, thus range is yp+yk. Divide by 2
                            % to get initial interval size. Take away a
                            % small number in order not to get Infs and
                            % complex numbers due to imprecions. Note that
                            % this dx is valid for both the parent and the
                            % child transfer (same interval!).
xp = (yp-yk)/2;             % Get starting guess for parent: middle between 
                            % maximum transfer (yp) and minimal one (-yk).
xk = -xp;                   % Starting guess for parent is middle between
                            % maximum (yk) and minimal one (-yp), which
                            % gives us exactly the negative of xp.
% Now, do vectorized bisection method:
while any(abs(dx(:))>tol)   % Until none of the dx's exceeds required tolerance:
    dx   = dx/2;            % Half the interval size.
    FOCk =  -Ak.*(yk-xk).^(-thetak) + alphak.*Ap.*(yp+xk).^(-thetap);
                            % Evaluate kid's static gift FOC at current
                            % guess.
    xk   = xk + sign(FOCk).*dx;
                            % FOCs are decreasing in gift, thus must add dx
                            % if FOC positive at midpoint and subtract it
                            % if FOC negative.
    FOCp =  -Ap.*(yp-xp).^(-thetap) + alphap.*Ak.*(yk+xp).^(-thetak);
    xp   = xp + sign(FOCp).*dx;   
end                         % Then, do the same for the parent.

xp(NegDynInc) = NaN;        % Finally, in cases with negative dynasty
xk(NegDynInc) = NaN;        %  income, set function output to NaN.
