function [upHaz, dnHaz] = SavingsHaz(adot,var,da,kid,reflect)
% Given drift and variance of the law of motion for assets, gives back the
% hazard rates of jumping up and down on a grid of mesh size 'da'. Uses the
% upwind principle. 
%
% Inputs:
% adot:     (N,N,2,Nw,Nw,Nh) array. Drift of assets.
% var:      Array conformable with adot (either a row or a column vector). 
%           variances of assets.
% da:       (N-1)-by-1 vector if kid=1 and 1-by-(N-1) vector if kid=0.
%           Distances of asset grid (may be non-linear).
% kid:      Logical scalar. 1 if this is law of motion for kid, 0 if it's the
%           the parent's.
% reflect:  Logical scalar. 1 if agents should be reflected back on top of grid.
%
% Outputs:
% upHaz:    (N,N,2,Nw,Nw,Nh) array. Upward hazards.
% dnHaz:    (N,N,2,Nw,Nw,Nh) array. Downward hazards. 
if kid
    daup = [da; da(end)];       % N-by-1 vector with forward distance to 
                                % on a^k-grid (repeat last da on top).
    dadn = [da(1); da];         % N-by-1 vector with distance of next 
                                % grid point looking backward/down. Repeat
                                % first distance on bottom (does not matter
                                % what we put here).
else                            % If this is for parent, have row vectors.
    daup = [da da(end)];
    dadn = [da(1), da];
end
dam  = (daup+dadn)/2;           % N-by-1 vector with average distance
                                % up and distance down.
                                % First, calculate upward and downward jump
                                % hazards in wealth due to savings and shocks
% Centered differencing suggests the following coefficients:
% Drift of Markov chain should equal adot, which gives us:
% [ cup * daup -  cdn * dadn  ]*dt  =  adot * dt,
% where cup and cdn are the up- and downward coefficients we are looking
% for, daup and dadn are the distances to the next grid point looking up
% and down. Now, note that dt drops out and that we can draw daup out of
% the brackets:
% [ cup - cdn * dadn/daup ] = adot / daup .
% Method 1:
% Take prob. mass out of cdn and put it into cup. This gives us the restri-
% ction cup = c = -cdn. Solving for c yields then:
% c = adot / [ ( 1+dadn/daup) * daup ]                      (C-M1)
% So for the linear grid, we get 0.5*adot/da, as it should be.
% Method 2: Create half of drift by downward movements and other half by 
% upward movements: 
%  cup * daup = 0.5*adot,                                   (C-M2)
% -cdn * dadn = 0.5*adot,
% which again gives us 0.5*adot/da for the linear grid, as it should be.
% The first-order conditions with respect to consumption expenditure e
% in the discrete game would be:
% u'(e)*dt + dt*[d cup / de] *(Vup-V)  +  dt*[d cdn/ d e] *(Vdn-V)  =  0.
% Now, eliminate dt. Also, note that adot = ... - e. Now, use (C-M2) and
% re-arrange to get the FOC for method 2, which is just to average up- and
% downward difference quotients: 
% u'(e) = 0.5 (Vup-V)/ daup + 0.5*(V-Vdn)/dadn              (FOC-M2 )
% Using method 1, we find analogously that
% u'(e) = (Vup-V) / [ daup * (1+dadn/daup) ]  + (V-Vdn) / [daup *
% (1+dadn/daup) ], 
% which we can arrange to have
% u'(e) = [ (Vup-V)/daup ] * daup/(daup+dadn)  +  
%       = [ (V-Vdn)/dadn ] * dadn/(daup+dadn)    ,  (FOC-M1)
% which tells us to weigh the upward and downward quotients differently. 

% Method 2:
% caiup =  0.5*adot./daup;
% caidn = -0.5*adot./dadn;
%Old code: This is consistent with the upwind principle.
caiup = max( adot,0) ./ daup;
caidn = max(-adot,0) ./ dadn;
csiup = 0.5*var ./ (daup.*dam);
csidn = 0.5*var ./ (dadn.*dam);
                                % Calculate coefficients for drift and
                                % volatility terms in dimension i.                                 
upHaz = caiup + csiup;         % Sum the two coefficients to get hazards
dnHaz = caidn + csidn;         % of jumping up or down in a^k.

% if any(upHaz(:)<0) || any(dnHaz(:)<0)
%     keyboard                  % Methods 1 and 2 give negative hazards
% end                           % all the time, would need dt=10^(-5) or
%                               % so... not practical!

%upHaz = (var/2 + da.*max( adot,0)) / da^2;  % Hazard of going up.
%dnHaz = (var/2 + da.*max(-adot,0)) / da^2;  % Hazard of going down.

if kid
    dnHaz(1,:,:,:,:,:) = 0;                 % Don't let kid's asset go down
    if reflect                              % at lowest grid point, and
        upHaz(end,:,:,:,:,:) = 0;           % neither up at highest when 
    end                                     % reflect is true.
else                                        
    dnHaz(:,1,:,:,:,:) = 0;                 % Same for parent.
    if reflect   
        upHaz(:,end,:,:,:,:) = 0;
    end
end
