function out = RatePoly2(t,epsInd,poly,dth,par)
% Obtains rates (hazard rates or probabilities) for age t and productivity
% index epsInd. 'poly' gives polynomial coefficients for the hazard/
% probability that we are interested in (e.g.: hazard of entering LTC, 
% hazard of death, fraction of men alive in household, fraction of alive
% men that is in LTC).
%
% Inputs:
% t:        array of size 'siz' or scalar. age of agents.
% epsInd:   array of size 'siz' or scalar. productivity indeces of agents.
% poly:     6-by-4 matrix. polynomial coefficients for approximating
%           the hazards/probability as a function of age and education.
%           dim.1 (row): coefficients belonging to each polynomial
%                        (highest-order in first row,constant in last row)
%           dim.2 (col): education levels:
%                        - 1: high-school dropout
%                        - 2: high school
%                        - 3: come college
%                        - 4: college
% dth:      logical scalar. 1 if a death hazard is wanted, 0 otherwise.
% par:      STRUCTURE. contains model parameters.
% 
% Output:   array of size 'siz'.

tRet         = par.tRet;                % Read out retirement age from par,
mapEpsToEduc = par.mapEpsToEduc;        % assignment vector (Nw-by-1) of
Khaz         = par.Khaz;                % productivity levels to education
tDth         = par.tDth;                % groups, and the adjustment factor
                                        % for death hazards at ages 94 to
                                        % 95.
                                        
                                        
haz    = 0*epsInd + 0*t;                % Create empty output matrix of same 
                                        % size as the covariates.
siz = size(haz);                        % Dimension of output (and input).
if isscalar(t)                          % If t is a scalar:
    t = t*ones(siz);                    % blow up t to full size
end 
if isscalar(epsInd)                     % Same for the productivity indeces
    epsInd = epsInd*ones(siz);
end

educ = mapEpsToEduc(epsInd);            % Get education levels pertaining to
                                        % each productivity in epsInd:
                                        % array of size 'siz'.
for iEduc=0:3                           % Loop over all 4 education levels:
   isEduci      = (educ==iEduc);        % Get logical index for data points 
                                        % with specific education level.
   haz(isEduci) = ( t(isEduci)>=tRet ).*polyval(poly(:,iEduc+1),t(isEduci));
end                                     % Evaluate the polynomial for the
                                        % specific education level at ages 
                                        % in vector t, and write them into 
                                        % hazard matrix.
if dth                                  % If a death hazard is calculated:
    haz = haz.*( 1 + Khaz*max(0,t-tDth-2) );% Adjust the death hazard by the   
end                                     % factor Khaz at age 94 and then 
                                        % smoothly bring it down to the 
                                        % original hazard at 93.
out = haz;