function [QbarLunc,QbarUunc] = QBounds(tauk,taup,Qmin,Qmax,pbar,taubar,BKwp)
% Creates bounds QbarLunc and QbarUunc that we impose on transfers Q in
% regions where neither agent is broke. The bounds are continuous in the
% transfer motives tauk and taup and do not exceed the minima and maxima
% specified in Qmin and Qmax. If there are positive WP-transfers (which 
% occurs in three of the four quadrants defined by tauk,taup), the bounds  
% are equal to each other and give the WP transfer. In the fourth quadrant
% (both agents want to hold on to their wealth), the bounds give a large
% interval on which exchange-motivated transfers will be searched for if 
% BKwp=false, but they return 0 if BKwp=true since this is the WP gift in
% this case.
% See script TransferBounds_demo.m, which shows how this function works.
%
% Inputs:
% tauk:     Array of size SIZ. Kid's transfer motive.
% taup:     Array of size SIZ. Parent's transfer motive.
% Qmin:     Array of size that is compatible with SIZ. Minimal transfer.
% Qmax:     Array of size that is compatible with SIZ. Maximal transfer.
% pbar:     Scalar. 
% taubar:   Scalar.
% BKwp:     Logical scalar. If true, the function computes gifts inside the 
%           state space ("BK wealth pooling"), which are zero when both
%           transfer motives are negative. If false, get bounds on transfers 
%           when both transfer motives are negative. The bounds coincide on
%           one number in the three quadrants where at least one transfer
%           motive is positive (true for both options).
%
% Outputs:
% QbarLunc: Array of size SIZ. Lower bound we impose on transfers.
% QbarHunc: Array of size SIZ. Upper bound we impose on transfers.

SIZ = size(tauk);           % Read out size of tauk and taup.

if numel(Qmin)<numel(tauk)  % If Qmin is not conformable with tau_k and tau_p:
    QQmin = Qmin.*ones(SIZ);% Blow it up to full size.
else                        % Otherwise keep as is.
    QQmin = Qmin;
end
if numel(Qmax)<numel(tauk)  % Do the same for Qmax.
    QQmax = Qmax.*ones(SIZ);
else
    QQmax = Qmax;
end

p = taup./(taup+tauk);      % Get p: Ratio of the two transfer motives.

PPsiU = PsiU(p,pbar);       % Get Psi_U(p): Makes upper bound continuous 
                            % in South-West and North-East quadrants when
                            % going along lines on which sum tauk+taup is
                            % constant. 
PPsiL = PsiL(p,pbar);       % Get Psi_L(p): Similar for lower bound.

% Start with South-West quadrant (both transfer motives negative):
% These are almost all the cases, so calculate this for all:
if BKwp==true               % If we want to get gifts within state space:
    QbarLunc = zeros(SIZ);  % These are zero in the South-West quadrant.
    QbarUunc = zeros(SIZ);
else                        % If we want bounds for exchange-motivated transfers:
    LoBd = phi(abs(taup+tauk),taubar).*QQmin;   % Lower bound for Q on iso-(taup+tauk) lines.
    UpBd = phi(abs(taup+tauk),taubar).*QQmax;   % Upper bound for Q on iso-(taup+tauk) lines.

    QbarLunc = (1-PPsiL).*LoBd + PPsiL.*UpBd;    % Get lower and upper bound as 
    QbarUunc = (1-PPsiU).*LoBd + PPsiU.*UpBd;    % convex combination of upper and 
                                                % lower bounds, weights given by
end                                             % the Psi-functions.

% Now, do the other quadrants (in terms of tau_p and tau_k): 
% First, North-East quadrant: both want to give.
QNE  = (taup>=0) & (tauk>=0);           % Mark these cases.
% if any(QNE(:))                        % Checked: These cases do indeed
%                                       %occur.
%     fprintf('Both transfer motives positive: %1.0f cases\n',sum(QNE(:)) );
% end
phiNE= phi(taup(QNE)+tauk(QNE),taubar); % Get phi of sum of transfer motives:
                                          % This gets us continuity when
                                          % going towards to origin.
QfixNE = (1-PPsiU(QNE)).*phiNE.*QQmax(QNE) + PPsiL(QNE).*phiNE.*QQmin(QNE);  
                                          % Set to zero when both transfer
                                          % motives are large and positive,
                                          % but make continuous at axes.
QbarUunc(QNE) = QfixNE;                   % Write solution into output arrays.
QbarLunc(QNE) = QfixNE;

% Second, do South-East quadrant: parent wants to give, kid wants to receive:
QSE  = (taup>=0) & (tauk< 0);             % Mark these cases.
QfixSE = phi(taup(QSE)-tauk(QSE),taubar).*QQmax(QSE);
QbarUunc(QSE) = QfixSE;                   % Set transfer as a fraction of Qmax,
QbarLunc(QSE) = QfixSE;                   % letting the fraction go to zero
                                          % as we go close to the origin.

% Now, do North-West quadrant: Kid wants to give, parent wants to receive.
QNW  = (taup< 0) & (tauk>=0);             % Similar to SE-quadrant.
QfixNW = phi(tauk(QNW)-taup(QNW),taubar).*QQmin(QNW);
QbarUunc(QNW) = QfixNW;
QbarLunc(QNW) = QfixNW;                               
