function out = ProdProcess(lambdaIdx,thetaIdx,betasq,dstar)
% Returns the hazard matrix, efficiency units and their ergodic distribution, 
% and statistics on the earnings process (Gini, Bottom Q10, etc.)

% OECD disposable household income:
% bottom 10%:  1,6
% bottom 20%:  5,3
% bottom 40%: 16,2
%    top 40%: 67,6
%    top 20%: 44,5
%    top 10%: 28,5
% Gini: 0.374
addpath('HelperFct')  
% betasq   = 2;                   % beta~: This is fixed for now at 2.
tauSS  = 0.124; % Social-Security tax rate. Default: 0.124.

startAge = 20;                  % At which age we start to consider households
                                % for the purpose of calculating income
                                % shares etc.
nAge = 65-startAge;
earners= 1.5;                   % How many earners per hh. Checked, does 
                                % not matter for earnings shares and Gini.
ebar = [-1; 0; 1; 1+dstar];     % Base vector that defines the distances 
                                % between the epsilons. 
varTarget = 0.52;  % 0.5716;   % Target for variance of Mincer residual.
acTarget  = 0.9725;             % Target for autocorrelation. In KK autocorrelation
                                % is given by 0.985.
lambdasqVec =(0.05:0.05:2)';
thetasqVec  =lambdasqVec';
Nl   = length(lambdasqVec);
Nt   = Nl;
qtl  = [0.1, 0.2, 0.4, 0.6, 0.8, 0.9]; 
nQtl = numel(qtl);              % For which quantiles we want to see income
                                % shares.

muMat    = zeros(Nl,Nt);        % Set up matrices to store the coefficients
deltaMat = zeros(Nl,Nt);        % we obtain for (mu,delta,f) conditional
fMat     = zeros(Nl,Nt);        % on (lambda~,theta~).
epsMat   =  cell(Nl,Nt);        % Store epsilon grids we get.
piMat    =  cell(Nl,Nt);        % Also store ergodic distribution.
SharesMat= zeros(Nl,Nt,nQtl);   % Store all nQtl income shares.
GiniMat  = zeros(Nl,Nt);        % Store Gini coefficients


% Calculate after-tax income shares
% Mincer regression (Census 2010):
mincer     = @(j) 9.84126 + .0646223*(j-20)  -.0010749*(j-20).^2;
HHeffUnits = @(j,eps) earners*exp( mincer(j) + eps ) / 1000;

for i=1:Nl                      % Loop over all values for lambda~...
   for j=1:Nt                   % ... and theta~.
        [muij,deltaij,fMat(i,j),epsvecij,piij,~] = ...
            GetMuDeltaF(lambdasqVec(i),thetasqVec(j),betasq,ebar,varTarget,acTarget);
                                % Obtain the values for (mu,delta,f) 
        muMat(   i,j) =    muij;% Record the values we get.
        deltaMat(i,j) = deltaij;
        epsMat{  i,j} =epsvecij;
        piMat{   i,j} =    piij;
        grossLbrIncij = HHeffUnits(startAge:64,epsvecij);
        TaxLbrInc     = (1-tauSS)*grossLbrIncij;
        xijAll        =( 1 - tax(TaxLbrInc,0) ) .* TaxLbrInc; % Net labor income
        [~,sharesij,~] = GetQuantProbDist(xijAll,qtl,piij.*ones(1,nAge)/nAge);
        SharesMat(i,j,:) = reshape(sharesij,[1,1,nQtl]);
        GiniMat(i,j) = GiniProbDist(xijAll,piij.*ones(1,nAge)/nAge);  % Record Gini.
   end
end


% Now, get results for our reference values and print out:
lambdasqRef       = lambdasqVec(lambdaIdx);
thetasqRef        = thetasqVec(thetaIdx);
ModelGini         = GiniMat(lambdaIdx,thetaIdx);
ModelBottom102040 = 100*squeeze(SharesMat(lambdaIdx,thetaIdx,1:3))';
DataBottom102040  = [1.6 5.3 16.2];  % OECD
ModelTop402010    = 100*(1-squeeze(SharesMat(lambdaIdx,thetaIdx,4:end))');
ModelTop2010      = 100*(1-squeeze(SharesMat(lambdaIdx,thetaIdx,5:end))');
DataTop402010     = [67.6 44.5 28.5]; % OECD
KKTop2010         = [63.5 47.0];      % Kinderman & Krueger
disp('Model bottom 10/20/40:'    ), disp(ModelBottom102040)
disp('Data  bottom 10/20/40:'    ), disp(DataBottom102040)
disp('Model top    40/20/10:'    ), disp(ModelTop402010)
disp('Data  top    40/20/10:'    ), disp(DataTop402010)
disp('Model top    20/10:'       ), disp(ModelTop2010)
% disp('K&K   top    20/10:'       ), disp(KKTop2010)
disp('Model Gini:'               ), disp(ModelGini)
disp('Data  Gini:'               ), disp(0.374)
OECDerror = sum((ModelBottom102040-DataBottom102040).^2+(ModelTop402010-DataTop402010).^2);
disp('OECDerror:'               ), disp(OECDerror)
%KKerror    = sum((ModelTop2010-KKTop2010 ).^2);
% ilRef = round(Nl/2);                 % Could also take indeces in the middle.
% itRef = round(Nt/2);
% lambdasqRef = lambdasqVec(ilRef);
% thetasqRef  =  thetasqVec(itRef);
[mu,delta,f,epsvec,pi,HazMat] = GetMuDeltaF(lambdasqRef,thetasqRef,betasq,ebar,varTarget,acTarget);

out.epsvec=epsvec;
out.pi    =pi;
out.HazMat=HazMat;
out.mu    =mu;
out.delta =delta;
out.f     =f;
out.OECDerror         = OECDerror;
out.ModelGini         = ModelGini;
out.ModelBottom102040 = ModelBottom102040;
out.ModelTop402010    = ModelTop402010;    

    function [mu,delta,f,epsvec,pi,HazMat] = GetMuDeltaF(lambdasq,thetasq,betasq,ebar,varTarget,acTarget)

% Fixing these three parameters, get the transition matrix that is
% normalized by flow velocity f:
HazMatSq = [ -lambdasq, lambdasq,         0       ,     0   ; ...
                1     ,    -2   ,         1       ,     0   ; ...
                0     , lambdasq, -lambdasq-thetasq, thetasq; ...
                0     ,  betasq ,         0        , -betasq      ];
% Solve for ergodic distribution:
pi    = zeros(4,1);            
pi(2) = 1/( 1 + 1/lambdasq  + (1+thetasq/betasq)/(lambdasq+thetasq) );  
pi(1) = pi(2)/lambdasq;
pi(3) = pi(2)/(lambdasq+thetasq);
pi(4) = pi(3)*thetasq/betasq;

% Check: works.
% HazMatSq'*pi                % inflows=outflows
% sum(pi)                     % normalization: sum up to 1.

mufct = @(delta) -delta.*(ebar'*pi);      % Mean parameter mu we need given
                                            % delta.
                                            
hitVar = @(delta) ( ( mufct(delta)+delta.*ebar' ).^2 )*pi - varTarget; 
                                        % Unconditional variance of epsilon
                                        % for a given delta.
% Plotted: works.                                       
% deltaVec = (0.1:0.1:3)';
% plot(deltaVec,hitVar(deltaVec),'-b',[deltaVec(1),deltaVec(end)],[0,0],'-k');
delta = fzero(hitVar,[0.01,5]);
mu    = mufct(delta);

epsvec = mu + delta*ebar;
% Checks:
% epsvec'*pi                      % Get mean zero
% (epsvec'.^2)*pi                 % and desired variance.
% TrMat1yrApprox = @(f) eye(Nw) + f.*HazMatSq;
%                                     % First-order accurate 1-year
%                                     % transition matrix.
TrMat1yr = @(f) expm(f*HazMatSq);   % Precise continuous-time transition
                                    % matrix (use matrix exponential)
                                    
CondlMean = @(f) TrMat1yr(f)*epsvec;
                                    % Given flow velocity f (scalar),
                                    % returns 4-by-1 vector with
                                    % conditional expectation
                                    % E_t[eps_{t+1}|ept_t].

hitAutoCorr = @(f) sum( pi.*epsvec.*CondlMean(f) )/varTarget - acTarget;

% Checked here: nicely monotone.
% fvec = 0.01:0.01:0.4;
% acf  = zeros(size(fvec));
% for i=1:numel(fvec)
%     acf(i) = hitAutoCorr(fvec(i));
% end
% plot(fvec,acf)
f = fzero(hitAutoCorr,[0.001,1]);   % Find f that nails autocorrelation coefficient.

HazMat = f.*HazMatSq;               % Finally, get the implied hazard matrix.

    end
                                    % First-order accurate 1-year
                                    % transition matrix.

end
                          



