function pi = PoissonErgodicDist(hazMat)
% Obtains ergodic distribution of a Poisson process with hazard
% (transition) matrix hazMat by solving a linear system of equations.
%
% Inputs:
% hazMat:       N-by-N matrix with Poisson hazards for transitions.
%               hazMat(i,j)>=0 gives hazard rate of transitioning from state
%               i to state j. On diagonal, have to put the negative of the
%               leaving rate from each state, i.e. each row of hazMat must 
%               sum up to 0.              
% Output:
% pi:           N-by-1 vector with the ergodic probabilities of being in
%               each state.

[N,~] = size(hazMat);               % Read out number of states.
% Create system of equations HH*pi = bb. The first N-1 equations are
% inflow=outflow for states 1 to N-1 (note that the Nth inflow=outflow
% equation is implied by the other N-1; another way to see that there is
% superfluous information in hazMat is that we only need to know the first
% N-1 columns in order to infer the Nth column, since all rows must sum up
% to 1). The last equation says that the probabilities in pi sum up to 1.
HH = [ hazMat(1:N,1:N-1)'; ...
       ones(1,N)               ];
bb = [zeros(N-1,1); 1 ];
pi = HH \ bb;                       % Solve the system of equations.

% Check here what happens if we simulate:                                % horizon for convergence.

% TrMat1 = expm(hazMat);              % Obtain transition probabilities over 
%                                     % 1 unit of time (e.g. one year) in the
%                                     % true continuous-time Poisson model by
%                                     % taking the matrix exponential.
% % nChop = 1000;                     % Check here: Discretizing the matrix,
% % dt    = 1/nChop;                  % the results converge to TrMat1 as we 
% % TT  = (eye(N)+hazMat*dt);         % increase the chopping number, nChop.
% % TrMat1b  = TT^nChop;              % TrMat1b is basically the same as
% %                                   % TrMat1. 
% ErgMat = TrMat1^T;                  % Should obtain a matrix in which
%                                     % columns are constant (i.e. prob.
%                                     % of being in a state does not
%                                     % depend on initial conditions).
% pi2 = ErgMat(1,:)';                 % Read out first row: often does not sum up
%                                     % to 1 since we gain/lose prob. mass
%                                     % over the iterations! Otherwise, get
%                                     % the same as when solving system of
%                                     % equations...
                                    