function out = PatchToArray(x,ArraySize,dim)
% Patches the column vector x to create array of size ArraySize. If dim is
% handed over, the x-vector varies in dimension dim. Note that with the new
% Matlab array extension, often it will be easier to just use 
% (1:10)'.*ones(1,100,100) instead of PatchToArray((1:10)',[10,100,100],1).
%  
% Input:
% x:            N-by-1 column vector.
% ArraySize:    1-by-nDim row vector: Desired size of output array.
% dim:          Scalar (optional). Dimension in which x vector will lie. So
%               it must be fulfilled that length(x) = ArraySize(dim).
%               Default: Dimension 1.
%
% Output:
% Array of size ArraySize.
%
% Examples:
% - "xx = PatchToArray([1;2], [2 3])"  returns a matrix
%   [1 1 1;
%   [2 2 2]
% - "PatchToArray(a,[10,2,3])" is equivalent to "a(:,ones(1,2),ones(1,3))"
% - "AA = PatchToArray(A,[N,2,N,3],[2,4])" applied to a 2-by-3 matrix A 
%   delivers the same as the commands (and computes it the same way)
%   "Aa = reshape(A,[N,2,N,4]); AA = Aa(ones(1,N),:,ones(1,N),:)"


if nargin<3                         % Put x in first dimension if not 
    dim = 1;                        % specified
end

nDim = length(ArraySize);           % Read out number of dimensions for
                                    % output array.
                                    
ExtDim = 1:nDim;                    % Set up vector that gives the dimen- 
ExtDim(dim) = [];                   % sions which have to be extended.

xxSiz = ArraySize;                  % xxSiz: dimensionality of the helper
xxSiz(ExtDim) = 1;                  % array xx. Add dummy dimensions of 
yy = reshape(x,xxSiz);              % length 1 to x.

IsExtDim = false(1,nDim);           % Logical vector: tells us if dimension
IsExtDim(ExtDim) = true;            % at index i is an extended dimension.

indOut = cell(1,nDim);              % Create a cell for indexing.
for i=1:nDim                        % Loop over all dimensions.
    if IsExtDim(i)                  % If the dimension is to be extended:
        indOut{i} = ones(1,ArraySize(i));   
                                    % Put [1,1,1,1,...1].
    else                            % If the dimension is kept:
        indOut{i} = 1:ArraySize(i); % Put [1,2,...N].
    end
end

out = yy(indOut{:});                % Index xx by indOut to obtain the 
                                    % result.
                                     