function [linInd,akInd,apInd] = PanelIndices(cak,cap,SInd,EkInd,EpInd,HInd,...
                                     ClosestAssetInd,akDot,apDot,cDeadPan)
% The main output are the linear indeces on the 6D or 2D grid (depending if 
% parent is alive) that the panel members are on. The function assigns
% continuous asset positions to the nearest neighbor, but corrects when
% close to the origin for both agents. Why do we do this? There are
% many policies that are discontinous at zero assets:
% 1. Wait until ap=0 to claim Medicaid.
% 2. Wait until ap=0 to sell house.
% 3. Other agent waits until a=0 to give gifts. 
% --> In all these situations, it is better to approximate the policy for 
%     asset values in range a in [0,da/2] by policies on the second grid 
%     point (da) instead of the policies at a=0. ZERO IS A SPECIAL POINT!
%     Note: In 1.,2.,3. we always have a negative drift. Also, 1.,2.,3. can
%     only occur to parents in the OLG economy, thus condition our
%     adjustment on drift and parent being alive.
%
% Inputs:
% cak, cap:         nPan-by-1 vectors with current child and parent assets.
% SInd, EkInd, EpInd, HInd:
%                   nPan-by-1 vectors with current indeces for panel members 
%                   in the other dimensions.
% ClosestAssetInd:  Function handle that assigns closest asset grid point
%                   without correction for drift.
% akDot, apDot:     (N,N,Ns,Nw,Nw,Nh) arrays with child's and parent's
%                   asset drift on the 6D state space.
% cDeadPan:         nPan-by-1 logical vector if parent dead.
%
% Outputs:
% linInd:           nPan-by-1 vector with integers (main output). Linearized
%                   index on the 6D grid (for those with alive parents) or
%                   on the 2D grid (for alone kids, i.e. dead parents).
% akInd, apInd:     nPan-by-1 vectors with integers, giving the index of the
%                   parent grid point that panel member is assigned to
%                   (bonus output).

[N,~,Ns,Nw,~,~] = size(akDot);          % Read out size of grid.
[~,~,nCln] = size(cap);                 % Get number of clones: 1 or 4.
akRawInd = ClosestAssetInd(cak);        % Get "raw" indeces for ak and ap:
apRawInd = ClosestAssetInd(cap);        % Closest grid point, i.e. 0 for 
                                        % point da/4. (nPan,1,nCln) array.
linRawInd= (HInd-1)*N*N*Ns*Nw*Nw + (EpInd-1)*N*N*Ns*Nw + (EkInd-1)*N*N*Ns ...
          +(SInd-1)*N*N        +(apRawInd-1)*N         + akRawInd;
                                        % Get the raw linear indeces on the
                                        % 6D grid: Multiply index in the
                                        % corresponding dimension by block
                                        % size (see also function
                                        % Arr2LinInd.m):
                                        % (nPan,1,nCln) array.
% Checked here for nCln=1: Works, is identical to what we had:                                        
% linRawInd2 = Arr2LinInd([akRawInd,apRawInd,SInd,EkInd,EpInd,HInd],[N,N,Ns,Nw,Nw,Nh]);
%  all(linRawInd==linRawInd2)
                                        % Get the raw linear indeces on the
                                        % 6D grid.                                         
% Initialize function outputs with raw values (will later correct wrong entries),
% all (nPan,1,nCln) arrays:                             
akInd  = akRawInd;                      % Child assets,                                   
apInd  = apRawInd;                      % parent assets,
linInd = linRawInd;                     % and linear index on 6D/2D grid.

% Now, have three types of corrections:
% 1. have to change to 2D indeces for those with dead parents:
cDeadPan3D = cDeadPan & true(1,1,nCln);  % Extend dead dummy to 3D array.
EkInd3D = EkInd.*ones(1,1,nCln);         % Same for kid productivity.
linInd(cDeadPan3D) = (EkInd3D(cDeadPan3D)-1)*N + akRawInd(cDeadPan3D);
                                        % For alone kid, get index on 2D
                                        % alone grid: Insert a (nDead,1,nCln) 
                                        % array here.
% 2. Adjust child assets:                                         
akChkInd = ~cDeadPan & akRawInd==1 & cak>0;
                                        % Indeces we have to check: parent
                                        % alive, we assigned grid point to
                                        % a^k=0 but child assets positive.
                                        % (nPan,1,nCln) array.
upkLinInd= linRawInd(akChkInd)+1;       % Get linear index going one grid 
                                        % point up from 0 in child assets
                                        % for those we have to check:
                                        % nkCheck-by-1 vector.
kCorrChk = akDot(upkLinInd)<0;          % nkCheck-by-1 logical vector that 
                                        % says if drift negative above ak=0 
                                        % for the members we check.
akCorrInd= 1 + kCorrChk;                % nkCheck-by-1 vector with corrected
                                        % a^k-index: Is 2 if drift
                                        % negative, stays at 1 otherwise.                                        
 akInd(akChkInd) = akCorrInd;           % Write updated indeces into output.
linInd(akChkInd) = linInd(akChkInd)+kCorrChk;
                                        % Update also linear index:
                                        % Add 1 if we have to make
                                        % correction.
                                        
% 3. Adjust parent assets:
apChkInd = ~cDeadPan & apRawInd==1 & cap>0;
uppLinInd= linRawInd(apChkInd)+N;       % Since parent assets are on 2nd dim.,
                                        % have to add N to linear index now
                                        % to go one grid point up: 
                                        % npCheck-by-1 vector.
pCorrChk = apDot(uppLinInd)<0;          % npCheck-by-1 logical vector that 
                                        % tells us if correction has to be 
                                        % made for ap-index.
apCorrInd= 1 + pCorrChk;                % Corrected ap-index: npCheck-by-1
apInd(apChkInd) = apCorrInd;            % Update output for ap-index
linInd(apChkInd)= linInd(apChkInd)+pCorrChk*N;
