function out = NetIncPar(ap,Pp,nnp,hp,par)
% Returns net income of a parent household of age j with assets ap,
% pension Pp, household size nnp, subtracting fixed medical expenditures 
% and expenditures for housing depreciation of owners. Rent is subtracted 
% if renters are made to rent a fixed house size, otherwise no rent is 
% subtracted since rent is then part of Stage-4 expenditures.
% 
% Inputs: Arrays of arbitrary dimension that are conformable. Give example
% here for OLG case (6 dimensions), but may also have 4 dimensions if kid 
% is alone.
% ap:     Array, e.g. size (1,N,1,1,1,1 ). Parent household's assets.
% Pp:     Array, e.g. (1,1,1,1,Nw,1). Parent household's pension benefits
%         at age 65 with 2 persons net of taxes (is written into structure
%         par in MotherShip).
% nnp:    Array, e.g. (1,1,2,1,Nw,1). Size of parent household.
% hp:     Array, e.g. (1,1,1,1,1,Nh). Parent's housing wealth.
% par:    Structure, containing parameters.
%
% Output: Array, e.g. of size (1,N,2,1,Nw,Nh) in 6D case.

% Read out parameters from par:
r         = par.r;                      % Interest rate.
w         = par.w;                      % Wage rate.
medFixC   = par.medFixC;                % Fixed medical spending per person 
                                        % above 65 years.
tauLTC    = par.tauLTC;                 % Tax rate for LTC policy.
deltaHouse= par.deltaHouse;             % Depreciation of house.
rent      = par.rent;                   % Rent paid by renters.
rentersChoose = par.rentersChoose;      % If renters can choose house size.
TaxCpl    = par.TaxCpl;                 % If capital income is taxed.

pension   = w*Pp.*nnp/2;                % Pension (social-security benefit)
                                        % for household is fraction of what
                                        % it got at age 65 with two
                                        % household members. Need to adjust
                                        % for wage growth in Ay>1
                                        % counterfactual (otherwise w=1).
                                        % 6D case: (1,1,2,1,Nw,1) array.
grCplInc  = r*ap;                       % Get gross capital income.
                                        % In 6D: (1,Nw,1,1,1,1) array.

netCplInc = (1 - TaxCpl.*tax(grCplInc,tauLTC) ).*grCplInc;
                                        % Tax only capital income for the
                                        % old, not pensions, if this is
                                        % specified in TaxCpl. Gives, e.g.
                                        % (1,Nw,1,1,1,1) array.

medFixExp   = medFixC*nnp;              % Medical (fixed) expenditures, for 
                                        % all household members. 
penalty     = 0;                        % could add a penalty on depreciation for owners                    
HouseExp    = (penalty+deltaHouse)*hp;        % Expenditures for housing depreciation,
                                        % e.g. (1,1,1,1,1,Nh) array.
if rentersChoose                        % If we let renters choose house 
    HouseExp(1) = 0;                    % size, then rent is in e4 (expenditures
                                        % in Stage 4).
else                                    % If renters don't choose house size:
    HouseExp(1) = rent;                 % Write in rent for renters.
end
out = netCplInc + pension - medFixExp - HouseExp;  
                                        % Put all together: Expands to
                                        % (1,N,2,1,Nw,Nh) array in 6D.
                                        