function out = MakeSmoother(sigmaA,nIter,par)
% Function gives back a smoother matrix Nsts-by-Nsts that gives a local
% average. Create it by a stochastic process. Both assets are made to vary
% with volatility sigmaA*A, where A=ak+ap are total assets of the dynasty.
% Also, we let the kid's productivity vary with the true (calibrated)
% processs. No transitions due to health or housing.

% Given drift and variance of the law of motion for assets, gives back the
% hazard rates of jumping up and down on a grid of mesh size 'da'. Uses the
% upwind principle. 
%
% Inputs:
% sigmaA:   Scalar. St.Dev. of dynasty assets.
% nIter:    Scalar. How many iterations to run of the process.
% par:      Structure with info on grid.
% kid:      Logical scalar. 1 if this is law of motion for kid, 0 if it's the
%           the parent's.
% reflect:  Logical scalar. 1 if agents should be reflected back on top of grid.
%
% Output:   N-by-N matrix with kernel that smooths around the current grid
%           point. 

avec = par.avec;    dap = par.dap;    N    = par.N;
                                    % Read out grid parameters.
minda= min(dap);                    % Grid distance (minimal one for non-linear
                                    % grid).                                  
Amat = avec + avec';                % Get total assets of family.

aHaz = (sigmaA*Amat).^2 / (2*minda^2); % Hazard of leaving grid point in asset 
                                    % direction.
% Create hazards for kid's assets:                                    
aDnHazKid      = aHaz;              % Going down...
aDnHazKid(1,:) = 0;                 % ... is not possible at lowest grid point,
aUpHazKid      = aHaz;              % going up...
aUpHazKid(N,:) = 0;                 % is not possible at the highest grid point.
% Same for parent:
aDnHazPar      = aHaz;
aDnHazPar(:,1) = 0;
aUpHazPar      = aHaz;
aUpHazPar(:,N) = 0;
% Now, get hazard of leaving the current grid point:
aLeaveHaz = aDnHazKid + aUpHazKid + aDnHazPar + aUpHazPar;

dt = 1 / max(aLeaveHaz(:));         % Set dt such that the maximal probability 
                                    % of leaving a grid point is 1/2.
% NOTE: SHOULD THIS THEN NOT BE 0.5/max(aLeaveHaz(:))???                                   
% Construct N-by-N sparse matrices with transition probabilities due to:
fromInd=1:N^2;

toInd = fromInd-1;                  % going down in ak,
aDnKidMat = dt*sparse(fromInd(  2:end  ),toInd(  2:end  ),aDnHazKid(  2:end  ),N^2,N^2);

toInd = fromInd+1;                  % going up in ak,
aUpKidMat = dt*sparse(fromInd(  1:end-1),toInd(  1:end-1),aUpHazKid(  1:end-1),N^2,N^2);

toInd = fromInd-N;                  % going down in ap,
aDnParMat = dt*sparse(fromInd(N+1:end  ),toInd(N+1:end  ),aDnHazPar(N+1:end  ),N^2,N^2);

toInd = fromInd+N;                  % going up in ap.
aUpParMat = dt*sparse(fromInd(  1:end-N),toInd(  1:end-N),aUpHazPar(  1:end-N),N^2,N^2);

% Get probability of staying as 1 minus the probability of leaving:
aStayMat = speye(N^2) - dt*sparse(fromInd,fromInd,aLeaveHaz(:),N^2,N^2);
% Sum up the five matrices to get an N-by-N sparse matrix with transition
% probabilities over dt:
aMat = aDnKidMat + aUpKidMat + aDnParMat + aUpParMat + aStayMat;                                  

out = aMat^nIter;                   % Let the process run for nIter times,
                                    % i.e. get the transition probabilities
                                    % over a time interval of length nIter*dt.
% Pi = full(out)  % Visualize sparse matrix as a full matrix.                                     
%% % Check here what the kernel looks like:                                    
% % k = round(N*N/3 + N/2);             % A point in the middle
% % k = N;                              % A point on the end of the x-axis
% k = N*(N-1)+1;                      % A point at the end of the y-axis.
% % k = N^2;                            % right upper corner
% aa = reshape( out(k,:), [N,N] );   % Take some row k (the transition 
%                                     % probabilities out of that state) and
%                                     % bring to matrix form.
% surf(aa);                           % Plot kernel for this point.                                      

