function out = LTChazard2(coeffArr,makePlot,dj)
% Function returns polynomial coefficients to approximate the hazards of
% LTC and of death, as well as the mass of LTC widows and the mass of
% healthy households 65-68 age years old given estimated logits in the data
% (lambda, piH, piS, piMen) and given initial stocks in the data (H0, S0,
% MenAlv0). Have checked that this gives exactly the same results as
% LTChazard.m in folder Code/.
%
% Inputs:
% coeffArray:       3-by-9-by-2 array with a bunch of coefficients from
%                   logistic regressions of y-variables in dim.1 on the
%                   x-variable in dim. 2 for sample of sex in dim. 3.
%                   dim.1 (row): which independent variable? 
%                       - 1: lambda (fraction of disabled)
%                       - 2: pi_H   (death hazard of non-disabled)
%                       - 3: pi_S   (death hazard of disabled)
%                   dim. 2 (column): which regressor? (omittted education
%                   category: high-school dropout)
%                       - 1: dummy for high school 
%                       - 2: dummy for some college
%                       - 3: dummy for college
%                       - 4: age
%                       - 5: (high school)*age
%                       - 6: (some college)*age
%                       - 7: (college)*age
%                       - 8: age^2
%                       - 9 constant 
%                   dim. 3 (page): which gender?
%                       - 1: male
%                       - 2: female
% makePlot:         true or false. Make plots if true. Default: false.
% dj:               Scalar, optional. Time increment to use in solving
%                   ODEs. Default is 0.05.
%
% Outputs:
% out:           STRUCTURE with a vector in each field, which gives
%                   polynomial coeffisients to approximate various hazards
%                   as functions of age (by education group)
%    polySigmaLTC   6-by-4 matrix. polynomial coefficients for approximating
%                   the hazard of healthy women to become an LTC case as a
%                   function of age. 
%                   dim.1 (row): coefficients belonging to each polynomial
%                                (highest-order in first row, constant in
%                                last row)
%                   dim.2 (col): education levels:
%                                - 1: high-school dropout
%                                - 2: high school
%                                - 3: some college
%                                - 4: college
%    polyDelta      6-by-4 matrix. coefficients for polynomial that
%                   approximates death hazard of healthy women; dimensions
%                   as for polySigmaLTC
%    polyDeltaLTC   6-by-4 matrix. approximates death hazard of LTC women
%    polyMenAlv     6-by-4 matrix. approximates the fraction of men alive 
%                   in healthy households.
%    polyMenLTC     6-by-4 matrix. approximates fraction of male that has
%                   LTC need (out of men that are alive in a household with
%                   a healthy woman -- this is NOT the total measure of
%                   such husbands).
%    initLtcDat     4-by-2 matrix. Initial fraction of disabled by
%                   dim.1: education category (indices above)
%                   dim.2: gender (1: male, 2:female)
%    H              Nj-by-4-by-2 array. stock of healthy women in the
%                   model. Dimensions:
%                   dim.1: age        
%                   dim.2: education (see indeces above)
%                   dim.3: gender (page 1 is male, page 2 is female) 
%    S              Nj-by-4-by-2 array. fraction of sick by age.
%

if nargin<2                 % Default is to make no plot
    makePlot=false;
end
if nargin<3                 % Default is 0.05 for time increment.
    dj = 0.05;              % Recall: time unit is one year. dj is increment 
end                         % used in the approximations to the
                            % differential equations.
Ne = 4;                     % Number of education levels by which we 
                            % differentiate 
                            
lambdaInd = 1;              % create variables for the indeces of the
piHInd    = 2;              % different models in the array 'coeffArr' for
piSInd    = 3;              % better readability of the code

maleInd   = 1;              % same for gender: index for males, 
femInd    = 2;              % and for females



logist = @(x) 1./(1+exp(-x));   % set up logistic function

logistreg = @(j,educ,coeff) ...
   logist( coeff(1)*(educ==1)    + coeff(2)*(educ==2)    + coeff(3)*(educ==3) + coeff(4)*j + ...
           coeff(5)*(educ==1).*j + coeff(6)*(educ==2).*j + coeff(7)*(educ==3).*j + ...
           coeff(8)*j.^2         + coeff(9) ) ;
                            % predicted probability for age j and education
                            % (0,1, 2 or 3), given the 9-element coefficient
                            % vector from the logistic regression.


%% Calculate LTC and death hazards for women

ageVec = 0:dj:30;           % let age of women run from 65 to 95
Nj     = length(ageVec);    % length of age vector
% Set up arrays for stocks and hazards. Dimensions:
% dim.1: age        (Nj points)
% dim.2: education  (Ne=4 states)
% dim.3: gender     (2 states: page 1 is male, page 2 is female)

deltaStore    = zeros(Nj,Ne,2); % storage arrays for the hazards we are 
deltaLTCStore = zeros(Nj,Ne,2); % interested in
sigmaLTCStore = zeros(Nj,Ne,2);

Dmod = zeros(Nj,Ne,2);      % Create storage arrays for stocks in model
Hmod = zeros(Nj,Ne,2);      % economy. D for dead, H for healthy, and S for 
Smod = zeros(Nj,Ne,2);      % sick. In the model, all individuals start out
Hmod(1,:,:) = 1;            % healthy at age 65 by assumption.

Ddat = zeros(Nj,Ne,2);      % arrays for stocks in the data. Difference to 
Hdat = zeros(Nj,Ne,2);      % model stocks: some people start off sick and 
Sdat = zeros(Nj,Ne,2);      % and some healthy (but none dead). 

for iEdu=0:(Ne-1)           % Loop over all education levels and both sexes:
    for kSex=1:2            % Set initial conditions in data.
        Sik = logistreg(0,iEdu,coeffArr(lambdaInd,:,kSex));
        Sdat(1,iEdu+1,kSex) = Sik;     % Get fraction of disabled for this group      
        Hdat(1,iEdu+1,kSex) = 1-Sik;   % at age 65 of woman (this means age
    end                     % age 68 for men), and put it into array. 
end

% We will now determine the hazard rates sigmaLTC(j,educ) as a function of age
% and education that an individual goes from healthy to sick. We impose
% that there are no transitions back from LTC to healthy, and choose the
% rate sigmaLTC such that it replicates lambda, i.e. the fraction of LTC
% cases in the healthy population in the data. As the other hazards, this
% is based on the assumption that hazards for a given (age,gender,educ)
% cell are invariant over the HRS waves.

% Triple loop to determine hazards (matrix logarithm takes most of time -- 
% have to loop this anyway, so no harm in triple loop here).
for jAge=1:Nj                       % Go forward in time to solve 
    j = ageVec(jAge);               % Read out age
    for iEdu=0:(Ne-1)               % Solve forward the ODE for each educ.
        for kSex=1:2                % and each gender (separately!)
            Hjik = Hdat(jAge,iEdu+1,kSex);       % Differential equations
            Sjik = Sdat(jAge,iEdu+1,kSex);       % Read out current stocks.
            Djik = Ddat(jAge,iEdu+1,kSex);
            piH        = logistreg( j  , iEdu, coeffArr(   piHInd,:,kSex) );
            piS        = logistreg( j  , iEdu, coeffArr(   piSInd,:,kSex) );
            
                                    % get death hazards
            lambdaLead2= logistreg( j+2, iEdu, coeffArr(lambdaInd,:,kSex) );
                                    % get lambda(j+2,.), i.e. the
                                    % proportion of LTC cases among the
                                    % living two years ahead
            phi = (lambdaLead2*(Sjik - piS*Sjik + Hjik - piH*Hjik ) - (1-piS)*Sjik ) / Hjik;
                                    % Obtain phi(j,.), the 2-year hazard of
                                    % becoming sick for the healthy at j.
            P2yr = [1-piH-phi, 0    , 0;  % transition matrix over 2
                    phi      , 1-piS, 0;  % years in data
                    piH      , piS  , 1];
            logP2yr  = logm(P2yr);          % take matrix logarithm, and read off
            delta    = logP2yr(3,1) / 2;    % continuous-time hazard rates
            deltaLTC = logP2yr(3,2) / 2;    % (time unit: one year, thus have to
            sigmaLTC = logP2yr(2,1) / 2;    % divide by 2 since data are bi-yearly)
            % Update the stocks using the hazards:
            hazMat = [ -delta-sigmaLTC,     0      ,   0  ;
                          sigmaLTC    ,   -deltaLTC,   0  ;
                          delta       ,    deltaLTC,   0   ];
            if jAge<Nj              % all but last iteration:
                XdatNew = (eye(3)+dj*hazMat) * [Hjik; Sjik; Djik];
                Hdat(jAge+1,iEdu+1,kSex) = XdatNew(1);  % map forward model 
                Sdat(jAge+1,iEdu+1,kSex) = XdatNew(2);  % stocks
                Ddat(jAge+1,iEdu+1,kSex) = XdatNew(3);
                
                XmodOld = [ Hmod(jAge,iEdu+1,kSex) ; ...
                            Smod(jAge,iEdu+1,kSex) ; ...
                            Dmod(jAge,iEdu+1,kSex)     ];
                XmodNew = (eye(3)+dj*hazMat) * XmodOld;
                Hmod(jAge+1,iEdu+1,kSex) = XmodNew(1); 
                Smod(jAge+1,iEdu+1,kSex) = XmodNew(2); 
                Dmod(jAge+1,iEdu+1,kSex) = XmodNew(3); 
            end                             % keep track of stock of dead to check
               deltaStore(jAge,iEdu+1,kSex) = delta   ;
            deltaLTCStore(jAge,iEdu+1,kSex) = deltaLTC;    % store the hazards
            sigmaLTCStore(jAge,iEdu+1,kSex) = sigmaLTC;
        end
    end
end

% Now, approximate hazard functions using monomial basis:
jgrid = [65 68 70 80 90 95];        % pick coarse grid for age j
nPol  = length(jgrid);              % number of grid points is the order of
                                    % the approximating polynomial.
jInd   = floor(1+(jgrid - 65)/dj);  % Get indeces of grid on the age grid.
polySigmaLTC = zeros(nPol,Ne);      % Set up matrices for the polynomial
polyDelta    = zeros(nPol,Ne);      % coefficients.
polyDeltaLTC = zeros(nPol,Ne);
for iEdu=0:(Ne-1)                   % For women: loop over education levels:
    sigmai    = sigmaLTCStore(jInd(:),iEdu+1,2);
                                    % read off LTC hazard on coarse grid 
                                    % (at jInd) for women of the education
                                    % level in questions.
    deltai    = deltaStore(jInd(:),iEdu+1,2);    % Do the same for the death hazard
    deltaLTCi = deltaLTCStore(jInd(:),iEdu+1,2); % and the death hazard of LTC cases

   polySigmaLTC(:,iEdu+1) = vander(jgrid)\sigmai; 
                                    % get polynomial coefficients for the
                                    % monomial basis using the Vandermonde  matrix.
    polyDelta(:,iEdu+1) = vander(jgrid)\deltai; % hazard of the healthy,
    polyDeltaLTC(:,iEdu+1)= vander(jgrid)\deltaLTCi;
end

                                    
%% Plot stocks and hazard in data and model                                    
% PLOT STOCKS FOR DATA:

if makePlot                             % If graphs are asked for:
    % Create (1,4,2) arrays: income group in dim.2, gender in dim.3 (male
    % comes first).
    LifeExp = sum( Hdat+Sdat, 1) * dj;  % Get life expectancy at 65
    LTCexp  = sum(    Sdat , 1)  * dj;  % Expected years in LTC at 65

    LifeExpMod = sum(Hmod+Smod,1) * dj; % Life expectancy in model
    LTCExpMod  = sum(  Smod   ,1) * dj; % Exp. years in LTC in model

    GenderName={'Males','Females'};
    GroupNames={'high-school dropout','high school','some college','college'};
                                        % names for educ. groups in plot
    GroupStyle = {'-b', '--b', '--r', '-r'};    
                                        % line styles for the group

    figure('Name','Stocks in data');
    for kSex=1:2                        % For both sexes:
        j0 = 65 + (kSex==1)*3;          % Get starting age for gender
        subplot(2,2,kSex)
        hold on                         % Make NYT graph
        legTxt = cell(Ne,1);            % Create cell for legend text
        for iEdu=0:(Ne-1)               % Plot LTC proportion by age
            plot(j0+ageVec,100*Sdat(:,iEdu+1,kSex),GroupStyle{iEdu+1},'LineWidth',2)
            legTxt{iEdu+1} = [GroupNames{iEdu+1}, sprintf(': %1.2f exp. LTC yrs',LTCexp(1,iEdu+1,kSex)) ];
        end                             % Show exp. years in LTC in legend
        xlim([j0, j0+30])
        ylim([0 12])                    % Label axes etc.
        title(GenderName{kSex})
        legend(legTxt)
        ylabel(sprintf('in LTC (out of 100 persons alive at 65)'))
        xlabel('age')

        subplot(2,2,2+kSex)             % Show fraction alive by age
        hold on
        for iEdu=0:(Ne-1)
            plot(j0+ageVec,100*(1-Ddat(:,iEdu+1,kSex)),GroupStyle{iEdu+1},'LineWidth',2)
            legTxt{iEdu+1} = [GroupNames{iEdu+1}, sprintf(': %1.2f life exp.',LifeExp(1,iEdu+1,kSex)) ] ;
        end                             % Life expectancy in legend
        xlim([j0, j0+30])
        ylim([0 100])
        title(GenderName{kSex})
        ylabel('alive (out of 100 individuals alive at 65)')
        xlabel('age')
        legend(legTxt)
    end


    % PLOT STOCKS FOR OUR MAADEL
    figure('Name','Stocks in model');
    for kSex=1:2                        % For both sexes:
        j0 = 65 + (kSex==1)*3;          % Starting age for gender
        subplot(2,2,kSex)
        hold on                         % Make NYT graph
        legTxt = cell(Ne,1);
        for iEdu=0:(Ne-1)               % Show LTC proportion by age
            plot(j0+ageVec,100*Smod(:,iEdu+1,kSex),GroupStyle{iEdu+1},'LineWidth',2)
            legTxt{iEdu+1} = [GroupNames{iEdu+1}, sprintf(': %1.2f exp. LTC yrs',LTCExpMod(1,iEdu+1,kSex)) ];
        end                             % exp. years in LTC in legend
        xlim([j0, j0+30])
        ylim([0 12])
        title(GenderName{kSex})
        legend(legTxt)
        ylabel(sprintf('in LTC (out of 100 individuals)'))
        xlabel('age')

        subplot(2,2,2+kSex)             % Show fraction alive by age
        hold on
        for iEdu=0:(Ne-1)
            plot(j0+ageVec,100*(1-Dmod(:,iEdu+1,kSex)),GroupStyle{iEdu+1},'LineWidth',2)
            legTxt{iEdu+1} = [GroupNames{iEdu+1}, sprintf(': %1.2f life exp.',LifeExpMod(1,iEdu+1,kSex)) ] ;
        end                             % Life expectancy in legend
        xlim([j0, j0+30])
        ylim([0 100])
        title(GenderName{kSex})
        ylabel('alive (out of 100 individuals)')
        xlabel('age')
        legend(legTxt)
    end


    % PLOT HAZARDS (BOTH MODEL AND DATA -- IDENTICAL BY ASSUMPTION)
    figure('Name','Hazards (both model and data)');
    for kSex=1:2                        % For both sexes:
        j0 = 65 + (kSex==1)*3;          % Get starting age for gender
        
        subplot(3,2,kSex)               % First plot in row: 
        hold on                        
        for iEdu=0:(Ne-1)               % LTC hazard by age
            plot(j0+ageVec,100*sigmaLTCStore(:,iEdu+1,kSex),GroupStyle{iEdu+1},'LineWidth',2)
        end                             
        xlim([j0, j0+30])
        ylim([0 35])
        title(GenderName{kSex})
        ylabel(sprintf('Yearly LTC hazard in %%'))
        xlabel('age')                   % Give legend for groups in first graph:
        if kSex==1, legend(GroupNames), end
        
        subplot(3,2,2+kSex)             % Death hazard in LTC
        hold on
        for iEdu=0:(Ne-1)
            plot(j0+ageVec,100*(deltaLTCStore(:,iEdu+1,kSex)),GroupStyle{iEdu+1},'LineWidth',2)
        end                            
        xlim([j0, j0+30])
        ylim([0 80])
        title(GenderName{kSex})
        ylabel('Yearly death hazard in LTC')
        xlabel('age')                   % For women: say there is only one profile
        if kSex==2
            text(j0+5,60,'(LR test rejects group-specific hazards)')
        end
        
        subplot(3,2,4+kSex)             % Death hazard when healthy:
        hold on
        for iEdu=0:(Ne-1)
            plot(j0+ageVec,100*(deltaStore(:,iEdu+1,kSex)),GroupStyle{iEdu+1},'LineWidth',2)
        end                            
        xlim([j0, j0+30])
        ylim([0 80])
        title(GenderName{kSex})
        ylabel('Yearly death hazard when healthy')
        xlabel('age')                  
        
    end

end



%% Calculate polynomial coefficients for men (men alive, and LTC %)
menAliv = ( Hmod(:,:,maleInd) + Smod(:,:,maleInd) ) ./ Hmod(:,:,femInd);
                                    % Obtain household size on age grid 
                                    % (by educ. categories) as follows:
                                    % Choose the number of males paired
                                    % with healthy women households such that
                                    % the total number of men is the same 
                                    % as the fraction implied by the
                                    % hazards from the data (Again, assume
                                    % that all men in model start out
                                    % healthy and then follow the estimated
                                    % process, in order to be consistent
                                    % with the methodology for women. Also,
                                    % recall that LTC and dead women have
                                    % zero husbands by assumption).
menLTCfrac = Smod(:,:,maleInd) ./ ( Hmod(:,:,femInd).*menAliv  );  
                                    % Then, set the fraction men with LTC
                                    % needs (among all men in healthy-women
                                    % households) such that it matches the
                                    % number of sick men in each
                                    % age-education cell.
polyMenAlv = zeros(nPol,Ne);        % Set up empty arrays for polynomials  
polyMenLTC = zeros(nPol,Ne);        % for men. 
for iEdu=0:(Ne-1)                   % For all education categories:
    menAlivi = menAliv(jInd(:),iEdu+1);             % Get polynomial coeff.
    polyMenAlv(:,iEdu+1) = vander(jgrid)\menAlivi;  % for men alive,...
    menLTCi  = menLTCfrac(jInd(:),iEdu+1);
    polyMenLTC(:,iEdu+1)  = vander(jgrid)\menLTCi;  % and for men in LTC.
end


%% Plot results for men
if makePlot
    figure('Name','Men: survival and fraction in LTC');
    for iEdu=0:(Ne-1)                   % Plot fraction of men alive and fraction
        subplot(1,2,1)                  % of men in LTC for all education categories
        hold on
        plot(ageVec,menAliv(:,iEdu+1),GroupStyle{iEdu+1},'LineWidth',2)
        ylabel('men alive')
        xlabel('age')

        subplot(1,2,2)
        hold on
        plot(ageVec,menLTCfrac(:,iEdu+1),GroupStyle{iEdu+1},'LineWidth',2)
        ylabel('fraction of men in LTC')
        xlabel('age')
    end
    legend(GroupNames)                  % Legend only in second graph.
end
                                  


%% Return results
out.polySigmaLTC = polySigmaLTC;    
out.polyDelta    = polyDelta;       out.polyDeltaLTC = polyDeltaLTC;  
out.polyMenAlv   = polyMenAlv;      out.polyMenLTC   = polyMenLTC;
out.initLtcDat   = squeeze( Sdat(1,:,:) );
out.H            = Hmod;            out.S            = Smod;
%out.dj           = dj;                  % give back MODEL results for stocks

