function HazMat = KKTransMat(HazMatReg,starIn,starPers)
% Constructs hazard matrix for productivities a la Kindermann & Krueger
% (KK). Assume a standard process, usually AR(1)-type, the regular income
% states, i.e. states 1,...,Nw-1; the hazard transition matrix for this
% process is handed over as HazMatReg. The function then adds one star
% state that is reached from state Nw-1 with hazard starIn and then left
% towards the central income state (index Nw/2) with hazard starPers and
% returns the Nw-by-Nw hazard transition matrix of the full process.
%
% Input:
% HazMatReg:        (Nw-1)-by-(Nw-1) matrix with hazard transition matrix
%                   between regular states, where Nw is the total number of
%                   income states (including the star state).
% starIn:           Scalar. Hazard at which highest regular state (Nw-1) is
%                   left towards star state. 
% starPers:         Scalar. Hazard at which star state is left towards the 
%                   middle regular state (Nw/2).
%
% Output:           Nw-by-Nw hazard matrix for changes between income
%                   states:
%                   Row index: which state we come from
%                   Col index: which state we go to 
%                   --> Rows sum to 1.


[NwReg,~] = size(HazMatReg);            % Get number of regular and
Nw = NwReg+1;                           % number of total states.

toStar = zeros(Nw-1,Nw-1);              % Construct matrix with leaving             
toStar(Nw-1,Nw-1) = starIn;             % hazard from the regular states:
                                        % Only leave the highest regular
                                        % state with hazard starIn.
fromStar= zeros(1,Nw-1);                % Now, construct final row of matrix:
FallStarInd = round(Nw/2);              % Go only to the central point on
fromStar(FallStarInd) = starPers;       % the regular grid with hazard
                                        % starPers.
HazMat = [ HazMatReg-toStar  ,  [zeros(Nw-2,1); starIn        ] ; ...
                fromStar     ,        -starPers               ] ;
                                        % Finally, assemble the output
                                        % matrix.
