function out = JumpHazardsAln(llAln,par,opt)
% Calculates hazards of leaving the current grid point in the Markov-chain
% approximation method for the young in the case in which the young are
% alone on the world ('aln'). If the first input is of dimension (N,3,Nw,Nh), 
% i.e. we have k=3, then indeces 2 and 3 of dim.2 are treated as basic LTC 
% and skilled LTC (this occurs when kids are old, i.e. when creating the
% value-function guess and the no-kids counterfactual. If first input is
% (N,1,Nw,Nh), we have k=1 (in the main loop) and kids are only healthy. 
% 
% Inputs: F
% STRUCTURE llAln with variables that change on each loop iteration, has
% fields:
%   tk:           Scalar. kid's age
%   akDotAln:     (N,k,Nw,Nh) array with kid's drift in wealth.
%   VarAkAln:     (N,k,Nw,Nh) array with variance of kid's wealth.
%   epskIndAln:   (N,k,Nw,Nh) array with productivity indeces of kid.
% STRUCTURE par (with parameters that don't change each loop iteration), 
% with fields:
%   N:            scalar. number of grid points for assets.
%   Nw:           scalar. number of grid points for wages.
%   tRet:         scalar. retirement age
%   avec:         1-by-N vector. grid vector for wealth
%   da:           scalar. grid mesh size for assets.
%   epsvec:       1-by-Nw vector. grid vector for productivity
%   polySigmaLTC: 6-by-4 matrix. In each column, has 6 polynomial coefficients
%                 to calculate hazard of becoming LTC case for each of the
%                 4 education categories.
%   polyDelta:    6-by-4 matrix. Coefficients for hazard of dying when
%                 healthy, structure as polySigmaLTC.
%   polyDeltaLTC: 6-by-4 matrix. Coefficients for hazard of dying when in
%                 LTC, structure as polySigmaLTC.
%   epsHaz1D:     Nw square matrix. Hazard matrix for income 
%                 change on the vectorized state space.
%   medHaz:       Function handle. gets hazard rate of medical-expense
%                 shock by sickness, income state of kid, income state of
%                 parent and housing state.
%   medMu:        Function handle. gets mean of medical shock distribution.
%   sigMu:        Function handle. gets st.dev. of medical shock dist.
% STRUCTURE opt with fields:
%   reflect:      Logical scalar. Reflect back at boundaries (if true), use 
%                 extrapolated values at upper boundaries otherwise.
%   NeglProb:     Scalar. probabilities smaller than this are set to zero
%
% Output:
% STRUCTURE h with hazards in the Markov-chain method: jumping up (upAk) and
% down in wealth (dnAk), dying (death) and leaving the current grid
% point (leave). 
% upAk, dnAk:   (N,k,Nw,Nh) arrays with probalities of going up or down
%               in wealth. 
% JumpMat:      Nsts-by-Nsts sparse matrix, where Nsts = N*k*Nw*Nh is the
%               number of grid points of the vectorized state. Contains the
%               hazards of transitioning on vectorized state space. Has
%               ZEROS on the diagonal, NOT the negative of the hazard of
%               leaving the state (as the inputs have) for calculations
%               later on.
% medJump:      Nsts-by-Nsts sparse matrix with jumps due to
%               medical-expenditure risk.
% LtcBasicJump: Nsts-by-Nsts sparse matrix with jumps due to becoming a
%               basic LTC case.
% LtcSkillJump: Nsts-by-Nsts sparse matrix with jumps due to becoming an
%               LTC case requiring skilled care.
% death:        (N,k,Nw,Nh) array with probalities of death
% leave:        (N,k,Nw,Nh) array with hazard of leaving the current grid
%               point.

% Read out structure llAln:
tk         = llAln.tk;          akDotAln = llAln.akDotAln; 
epskIndAln = llAln.epskIndAln;  VarAkAln = llAln.VarAkAln; 
nnkAlnArr  = llAln.nnkAlnArr;
% Read out structure 'par':
dak        = par.dak;            polySigmaLTC = par.polySigmaLTC;
polyDelta  = par.polyDelta;      polyDeltaLTC = par.polyDeltaLTC;
fracLtcBasic = par.fracLtcBasic;
epsHaz1D   = par.epsHaz1D;            medProb      = par.medProb;
indMedHazHlth = par.indMedHazHlth;   indMedHazSick = par.indMedHazSick;
                                
reflect = opt.reflect;          % Read out structure 'opt'. 

[N,k,Nw,Nh] = size(akDotAln);   % Read out size of input 
Nsts     = N*k*Nw*Nh;           % Determine total number of states

dakup = [dak; dak(end)];        % N-by-1 vector with forward distance to 
                                % on a^k-grid (repeat last da on top).
dakdn = [dak(1); dak];          % N-by-1 vector with distance of next 
                                % grid point looking backward/down. Repeat
                                % first distance on bottom (does not matter
                                % what we put here).
dakm  = (dakup+dakdn)/2;        % N-by-1 vector with average distance
                                % up and distance down.

                                % First, calculate upward and downward jump
                                % hazards in wealth due to savings and shocks

% The following is consistent with up-wind differencing:                                
caiup = max( akDotAln,0) ./ dakup;
caidn = max(-akDotAln,0) ./ dakdn;
% Tried the following, which is consistent with centered differencing (but
% can lead to negative probabilities): Does not work, leads to complex
% numbers.
% caiup =  0.5*akDotAln ./ dakup;
% caidn = -0.5*akDotAln ./ dakdn;

csiup = 0.5*VarAkAln ./ (dakup.*dakm);
csidn = 0.5*VarAkAln ./ (dakdn.*dakm);
                                % Calculate coefficients for drift and
                                % volatility terms in dimension i.                                 
h.upAk = caiup + csiup;         % Sum the two coefficients to get hazards
h.dnAk = caidn + csidn;         % of jumping up or down in a^k.
% h.dnAk = (VarAkAln/2 + da.*max(-akDotAln,0)) / da^2;
% h.dnAk(1,:,:,:) = 0;            % OLD CODE.

if reflect                      % If reflection is chosen: set hazard
    h.upAk(end,:,:,:) = 0;      % of jumping out of grid to zero.
end

% Now, turn to jumps in state space that are due to exogenous reasons:
% med.spending, income and sickness --> Create large transition matrix.
% row: state we come from, column: state we go to

% Income risk:
if k<2                          % Transition hazard matrix for income shocks:
    incJump = TransMatLarge([N,1,Nw,Nh],3,epsHaz1D);
else                            % Read out from parameters if there is risk,                      
    incJump = sparse(Nsts,Nsts);
end                             % All-zero if retired

% Jumps from becoming an LTC case:
if k<2                           % When there is no disability risk:
    LTCjump = sparse(Nsts,Nsts); % Zero jump matrix.
else                             % When there are disabled dimensions: 
    hazLTC = zeros(N,k,Nw,Nh);   % Set up array with hazard of going into LTC.
                                 % (both kinds). Only go there from healthy
                                 % state:
    hazLTC(:,1,:,:) = ones(N,1,Nw,Nh) .* RatePoly2(tk,epskIndAln,polySigmaLTC,0,par);
                                 % Fill hazard rates for healthy state.
    % Create Nsts-by-Nsts sparse matrix with jumps:
    indFrom = (1:(Nsts-N))';     % Indeces on vectorized state space from which 
                                 % we jump (exclude last block of N so that we
                                 % do not exceed Nsts in the indeces 'indTo'
    indTo   = indFrom + N;       % Indeces to which we jump
    LTCjump =   sparse(indFrom,indTo  ,hazLTC(1:end-N),Nsts,Nsts) ...
              - sparse(indFrom,indFrom,hazLTC(1:end-N),Nsts,Nsts);
end                              % Create LTC transition matrix: put positive
                                 % hazard N*N right of diagonal (get sick),
                                 % negative hazard on the diagonal
                                 % Tried with Kronecker products, but this way
                                 % is a lot faster.
% For now, assume that fixed fraction of people ends up in basic care when
% transiting to LTC:
h.LtcBasicJump =    fracLtcBasic *LTCjump;
h.LtcSkillJump = (1-fracLtcBasic)*LTCjump;
                                % Have to modify matrix LtcSkillJump if we
                                % also want jumps from basic to skilled
                                % LTC.

% Jumps in wealth due to medical shocks:                              
if k<2                           % If kid is not retired yet:                            
    medJump = sparse(Nsts,Nsts); % No medical shocks (all-zero matrix)
else                             % If kid is retired:
    % First, create k-by-Nw matrix of medical-shock hazards by sickness
    % state and income level (they differ because the number of persons in
    % a household differs by income and the individual hazards differ by
    % sickness):
    nnkHlth1D = reshape( nnkAlnArr(1,1,:,1), [1,Nw]);
                                 % Obtain 1-by-Nw vector with number of
                                 % healthy individuals in household by income. 
    medHaz2D = [nnkHlth1D *indMedHazHlth; ...       % healthy,
                ones(1,Nw)*indMedHazSick; ...       % basic LTC, 
                ones(1,Nw)*indMedHazSick     ];     % skilled LTC (assume same as basic for now)
                                 % Take in individual medical hazards to
                                 % get the k-by-Nw med.-hazard matrix.
    medHazMat= sparse(1:k*Nw,1:k*Nw,medHaz2D(:));
                                 % Put these hazards on the diagonal of a
                                 % square k*Nw sparse matrix,
    mmed    = spkron( speye(Nh), medHazMat);
                                 % Kronecker-multiply by Nh-by-Nh identity
                                 % matrix to include housing dimension:
                                 % have k*Nw*Nh sparse matrix
    medJump = spkron( mmed, medProb);
                                 % and do the Kronecker product with the
    %clear medHazMat medHaz2D ...% med.-shock transition probability matrix
    %      nnkHlth2D mmed        % to obtain the N*k*Nw*Nh square hazard
                                 % matrix on the vectorized state space.
                                 % Don't use 'kron.m' here -- slow!
                                 % Don't clear variables... slows down.
    
% The following is for the case where the medical shock hazard depends on
% income, age etc.: Do not delete because we may have to go back to it...
%     ss  = zeros(1,2,Nw);     % create arrays for sickness and productivity
%     ss(:,2,:) = 1;
%     ee = PatchToArray(epsvec',[1,2,Nw],3);
%     hhaz = medHaz(tk,ss,ee); % get hazard rate of medical shock, its (log)
%     mmu  =  medMu(tk,ss,ee); % mean and (log) st.dev. by sickness and 
%     ssig = medSig(tk,ss,ee); % productivity
%     aa   = [avec(1)-da/2, avec+da/2]'; 
%                              % N+1 column vector with points half-way
%                              % between wealth grid points               
%     aaa  = PatchToArray(aa,[1,2,Nw,N+1],4);
%                              % Replicate to have it for each sickness and 
%                              % pension level.
%     mmm  =  mmu(:,:,:,ones(1,N+1));
%     sss  = ssig(:,:,:,ones(1,N+1));
%                              % extend mu and sig to same dimension as aaa to...
%     medCcdf = logncdf(aaa,mmm,sss);
%                              % ...evaluate medical cdf on aaa-grid for each
%                              % grid point
%     medCcdf(:,:,:,end) = 1;  % set to 1 at last grid point
%     medC0p = ( medCcdf > 1-NeglProb );                        
%     medCcdf(medC0p) = 1;     % set very small probabilities to zero
%     nMax = max( sum( reshape(~medC0p,[1*2*Nw,N+1]), 2) );
%                              % get maximal number of grid points needed for
%                              % medical-shock approximation.
%     pp = diff(medCcdf,1,4);  % get probability of jumping to grid points
%                              % (1st difference in 4th dimension)
%     hh = pp.*hhaz(:,:,:,ones(1,N));% multiply by hazard of medical shock to get
%                              % hazard rate of jumping to grid points
%     % Fill array of dim. (N,2,Nw,N) which contains the probabilies of drop-
%     % ping down 1,2,... wealth grid points (indeces 2,3,... in 4th dimension) 
%     % and the probability of leaving the current grid point (on 1st index 
%     % of 4th dimension) for each grid point (indexed as usual on the first 
%     % three dimensions).
%     hhh = zeros(N,2,Nw,N);                      % no chance to drop down from
%                                                 % first wealth grid point,
%                                                 % thus not in loop.
%     for i=2:nMax-1                              % loop over all wealth grid points:
%        hhh(i,:,:,1) = -sum(hh(1,:,:,2:end),4);  % diagonal: minus prob. of leaving
%        hhh(i,:,:,i) =  sum(hh(1,:,:,i:end),4);  % hazard of going to zero wealth
%        if i>2                                   % grid point 3 and above
%            hhh(i,:,:,2:i-1) = hh(1,:,:,2:i-1);  % going to other positive wealth
%        end                                      % levels
%     end
%                                                 % For wealth levels high
%     hhRest = hh(ones(1,N-nMax+1),:,:,2:nMax);   % enough we can't go down to 
%     hhh(nMax:N,:,:,2:nMax) = hhRest;            % zero wealth: can treat    
%     hhh(nMax:N,:,:,1     ) = -sum(hhRest,4);    % them all the same.
%     
%     medJump = sparse(Nsts,Nsts);                % set up transition matrix 
%                                                 % on vectorized state space
%     for j=0:nMax-1                              % loop over number of points 
%                                                 % we jump down
%        indFrom = (1+j:Nsts)';                   % indeces in vectorized 
%                                                 % state space from which
%                                                 % the jump occurs (have to
%                                                 % cut out first j --> do
%                                                 % not have indeces below)
%        indTo   = indFrom - j;                   % index of vectorized state 
%                                                 % space to which jump
%                                                 % occurs. 
%        haz     = hhh(:,:,:,j+1);                % read out hazards of jumping j
%                                                 % down (hazard of leaving
%                                                 % for j=0)
%        medJump = medJump + sparse(indFrom,indTo,haz(indFrom),Nsts,Nsts);
%                                                 % write these hazards into
%                                                 % the correct positions of
%     end                                         % the matrix (by adding a
%                                                 % sparse matrix).
        
end

h.medJump = medJump;
% Add all jump hazards (due to exogenous reasons, not savings) to one matrix:
JumpMat = incJump + LTCjump + medJump;

indDiag = 1:(Nsts+1):(Nsts^2);
JumpLeaveHaz = (-1)*reshape( full(JumpMat(indDiag)), [N,k,Nw,Nh] );
                              % Read hazard of leaving current state 
                              % off the diagonal, and turn into array (this
                              % is faster than calling spdiags.m).
JumpMat(indDiag) = 0;         % Set diagonal to zero to hand over the jump
h.JumpMat = JumpMat;          % matrix --> the negative values on the diagonal
                              % are part of p_middle, which will be
                              % calculated from h.leave, see below.
                              % Indexing the diagonal is faster than
                              % spdiags.m.
% Hazard of death:
if k<2
    h.death = 0;
    %h.death = zeros(N,1,Nw,Nh);% Zero if not retired yet.
else                          % If retired, fill in hazards:
    h.death = zeros(1,k,Nw,1);
    h.death(:,1,:,:) = RatePoly2(tk,epskIndAln,polyDelta   ,1,par); % healthy,
    h.death(:,2,:,:) = RatePoly2(tk,epskIndAln,polyDeltaLTC,1,par); % LTC basic,
    h.death(:,3,:,:) = RatePoly2(tk,epskIndAln,polyDeltaLTC,1,par); % LTC skilled
end                             % (set both LTC states equal for now).

h.leave = h.upAk + h.dnAk + h.death + JumpLeaveHaz;
                              % Sum all hazards to leave current state.

if k<2                        % For case where kid is young, also create a 
                              % matrix to map forward the density:
    indFrom = (1:(Nsts-1))';  
    indTo   = indFrom + 1;    % Create matrix that represents upward jumps
    haz     = h.upAk;         % in kid's wealth.
    haz(N,:,:,:) = 0;         % Don't let jump up at top grid point
    upAkMat =   sparse(indFrom,indTo  ,haz(indFrom),Nsts,Nsts) ... 
              - sparse(indFrom,indFrom,haz(indFrom),Nsts,Nsts); 
          
    indFrom = (2:Nsts)';      % Now the same for downward jumps     
    indTo   = indFrom - 1;  
    haz     = h.dnAk;
    dnAkMat =   sparse(indFrom,indTo  ,haz(indFrom),Nsts,Nsts) ... 
              - sparse(indFrom,indFrom,haz(indFrom),Nsts,Nsts);
    akMat = upAkMat + dnAkMat;
                              % Hand over the density mapping matrix:
    h.DensMat = akMat + incJump;
end
                              
                              
out = h;                      % return results
