function [hazAkp, haz345, maxLeavHaz] = JumpHazards2(ll,par,opt)
% Returns the hazard rates of leaving a grid point towards neighboring grid 
% points in the Markov-chain approximation method for the large grid (dim.
% N,N,Ns,Nw,Nw,Nh), on which both parent and kid are alive. Does so only for
% the hazards that vary in age (the kid's productivity shocks do not, for
% example).
% 
% Inputs:
% STRUCTURE 'll' (changes on each loop iteration) with fields:
%   tp:           Scalar. age of parents.
%   apDot, akDot: (N,N,Ns,Nw,Nw,Nh) arrays. Drifts of wealth for parents
%                 and kids.
%   nnpArr:       (1,1,Ns, 1,Nw, 1) array. Number of persons in parent
%                 household.
% STRUCTURE 'par' with parameters (do not change over loop) with fields:
%   N:            Scalar. Number of grid points for assets.
%   Ns:           Scalar. Number of grid points for health.
%   Nw:           Scalar. Number of grid points for wages.
%   Nh:           Scalar. Number of grid points for housing.
%   da:           Scalar. Grid mesh size for assets.
%   tRet:         Scalar. Retirement age
%   VarAk:        (N,1,1,1,1,1) array. Variance of kid's assets.
%   VarAp:        (1,N,1,1,1,1) array. Variance of parent's assets.
%   polySigmaLTC: 6-by-4 matrix. In each column, has 6 polynomial coefficients
%                 to calculate hazard of becoming LTC case for each of the
%                 4 education categories.
%   polyDelta:    6-by-4 matrix. Coefficients for hazard of dying when
%                 healthy, structure as polySigmaLTC.
%   polyDeltaLTC: 6-by-4 matrix. Coefficients for hazard of dying when in
%                 LTC, structure as polySigmaLTC.
%   indMedHazHlth:Scalar. hazard of healthy INDIVIDUAL to get medical-
%                 spending shock   
%   indMedHazSick:Scalar. hazard of sick INDIVIDUAL to get medical-spending
%                 shock.
% STRUCTURE opt with field:
%   reflect:      Logical scalar. Reflect back at boundaries (if true), use 
%                 extrapolated values at upper boundaries otherwise.
%
% Output:
% hazAkp:       (N,N,Ns,Nw,Nw,Nh,4) array. First 6 dimension are the usual
%               state space, dimension 7 tells us which hazard is meant:
%               Dim. y: index 1: akUp, hazard that kid's assets go up.
%                       index 2: akDn, kid's assets down.
%                       index 3: apUp, parent's assets up.
%                       index 4: apDn, parent's assets down.
% haz345:       Structure with fields containing hazard rates:
%                       Med:        medical shock to parent
%                       LtcBasic:   disabled, need basic care services
%                       LtcSkill:   disabled, need skilled care services
%                       Death:      death
%                       All hazards are (Ns,Nw,Nw) arrays. Dimensions:
%                       dim. 1: disability (healthy, LTC basic, LTC sk.)
%                       dim. 2: kid's productivity
%                       dim. 3: parent's productivity
%                       
% maxLeavHaz:   Scalar. Maximal hazard rate at which a grid point is left
%               (needed to calculate time increment dt of algorithm).

                                    % Read out structure 'll':
tp    = ll.tp;                      % Age of parents and kids,
apDot = ll.apDot;  akDot = ll.akDot;% drift in both assets,
nnpArr= ll.nnpArr; % menLTC= ll.menLTC;   
                                    % household size and sick husbands.
                                    
% Read out structure 'par':
N            = par.N;              Nw           = par.Nw; 
Nh           = par.Nh;             Ns           = par.Ns;
tRet         = par.tRet; dak = par.dak; dap = par.dap;
VarAp        = par.VarAp;          VarAk        = par.VarAk;
polySigmaLTC = par.polySigmaLTC;   polyDelta    = par.polyDelta;
polyDeltaLTC = par.polyDeltaLTC;   epsHaz1D     = par.epsHaz1D;
indMedHazHlth= par.indMedHazHlth;  indMedHazSick= par.indMedHazSick;
fracLtcBasic = par.fracLtcBasic;   

reflect = opt.reflect;              % Read out option 'reflect'.

% Set up variables for results:
hazAkp = zeros(N,N,Ns,Nw,Nw,Nh,4);   % Set up arrays for asset hazards.
haz345.Med      = zeros(Ns,Nw,Nw);   % Set up medical hazard.
haz345.LtcBasic = zeros(Ns,Nw,Nw);   % basic LTC state,
haz345.LtcSkill = zeros(Ns,Nw,Nw);   % skilled LTC state,
haz345.Death    = zeros(Ns,Nw,Nw);   % death.


%% Transition hazards due to savings and Brownian Motion:
% Kid's wealth: get upward and downward hazard:
[hazAkp(:,:,:,:,:,:,1), hazAkp(:,:,:,:,:,:,2)] = SavingsHaz(akDot,VarAk,dak,1,reflect);
% Same for parent's wealth:
[hazAkp(:,:,:,:,:,:,3), hazAkp(:,:,:,:,:,:,4)] = SavingsHaz(apDot,VarAp,dap,0,reflect);



%% Jumps in parent's wealth due to medical shocks:
% Only obtain the hazards in subspace of the dimensions 3,4,5. Hazard does
% not depend on kid's productivity (dim. 4), but leave this in to have
% contiguous dimensions.
if tp>=tRet
    nnpHlth3D = nnpArr(1,1,1,1,:,1) .* ones(1,1,1,Nw,1,1);
    nnpHlth3D = reshape( nnpHlth3D, [1,Nw,Nw] );
    % nnpHlth3D    = reshape( nnpArr(1,1,1,:,:,1), [1,Nw,Nw] );
                                    % Get (1,Nw,Nw) arrays for number of
                                    % members in healthy households in each
                                    % productivity combination
    haz345.Med(1,:,:) = nnpHlth3D*indMedHazHlth;
                                    % Calculate (1,Nw,Nw) array with hazard
                                    % rate of med. shock in these hh.
    haz345.Med(2:Ns,:,:) = indMedHazSick;
end                                 % For sick household, there is just one
                                    % woman with LTC. For now, give same 
                                    % hazard to skilled and basic LTC
                                    % state.


%% Transition matrix due to LTC risk:
% Again, get transition hazards only for dimensions 3 (sickness), 4 (kid's
% productivity) and 5 (parent's prod.) in order to save on memory. 
epsParInd3D = PatchToArray(1:Nw,[1,Nw,Nw],3);
                                    % Array with productivity indeces of
                                    % healthy parents in 3D, with parent
                                    % prod. on dimension 3.     
rr = RatePoly2(tp,epsParInd3D,polySigmaLTC,0,par);                                    
haz345.LtcBasic(1,:,:) =    fracLtcBasic * rr;
haz345.LtcSkill(1,:,:) = (1-fracLtcBasic)* rr;
haz345.LtcSkill(2,:,:) = 0;         % Allow for flow from basic to skilled 
                                    % LTC, but set to 0 for now. Don't
                                    % allow flows from skilled to basic,
                                    % though.                             

%% Death hazards:
% Again, only get death hazards for dimensions 3-5.
% Conditional on being healthy:
haz345.Death(1,:,:) = RatePoly2(tp,epsParInd3D,polyDelta   ,1,par);
% Conditional on requiring LTC:
rr = RatePoly2(tp,epsParInd3D,polyDeltaLTC,1,par);
haz345.Death(2,:,:) = rr;           % For now, give basic and skilled LTC
haz345.Death(3,:,:) = rr;           % individuals the same death hazard.

%% Calculate maximal hazard at which a grid point is left:
hazProdLeav = -full(diag(epsHaz1D));% Read off the hazards of leaving the 
                                    % kid's current productivity state from
                                    % the diagonal of the Nw-by-Nw
                                    % transition matrix.
hazLeav345 = haz345.Med + haz345.LtcBasic + haz345.LtcSkill + haz345.Death;
                                    % Get (3,Nw,Nw) array with sum of
                                    % hazards in haz345.
hazLeav = sum(hazAkp,7)                             ...
          + reshape(hazLeav345   , [1,1,Ns,Nw,Nw,1]) ...
          + reshape(hazProdLeav  , [1,1,1 ,Nw, 1,1])     ;     
                                    % Sum up hazards due to asset movements 
                                    % and due to medical shocks, LTC,
                                    % death, and child's productivity
maxLeavHaz = max(hazLeav(:));       % shocks, and take max over state space.

