function out = HazMatLTC345(hazLTC345,sToInd)
% Given LTC hazards 'hazLTC345', return sparse transition matrix on
% vectorized state space in dimensions 3 (sickness), 4 (kid's productivity),
% 5 (parent's prod.) for transitions that are due to the hazard of either
% entering basic LTC need (sToInd=2) or entering skilled LTC need
% (sToInd=3). Remark: Cannot treat all LTC transitions jointly in a 3-by-3
% transtion matrix (as is done for productivity) since hazards depend on
% productivities. Thus splitting the problem in two makes sense, so we stay
% in the realm of (Ns,Nw,Nw) arrays on the state space of dimensions 3,4,5.
%
% Inputs:
% hazLTC345:    (Ns,Nw,Nw) array, containing LTC hazards given sickness
%               state, kid's productivity and parent's productivity of
%               going into health state with index sToInd from each other
%               state.
% sToInd:       Scalar, integer. To which health state the transitions go. 
%               2 for basic LTC, 3 for skilled LTC.
%
% Output:       Ns*Nw*Nw square sparse matrix with at most 2 non-zero 
%               elements per row.

[Ns,Nw,~] = size(hazLTC345);        % Read out dimensions.
nSts345 = Ns*Nw*Nw;                 % Number of states in state space of 
                                    % dimensions 3-5.
indFrom = (1:nSts345)';             % Indeces on vectorized state space in
                                    % dimensions 3-5 from where we jump.
                                    % we jump (exclude last grid point so
                                    % that we do not nSts345 in 'indTo')
sIndUp = sToInd-(1:Ns)';            % How many indeces we go up in the health
                                    % dimension when jumping to index
                                    % sToInd: Ns-by-1 vector.
indUp  = sIndUp.*ones(1,Nw,Nw);     % Increase size to 3-5 state space:
                                    % (Ns,Nw,Nw) array.
indTo  = indFrom + indUp(:);        % Indeces on  3-5 state space to which 
                                    % we jump.
out =  sparse(indFrom,indTo  ,hazLTC345(indFrom),nSts345,nSts345) ...
     - sparse(indFrom,indFrom,hazLTC345(indFrom),nSts345,nSts345);
                                    % Put together resulting sparse matrix.
 