function G = GiniProbDist(x,n,ord)
% Calculates Gini coefficient for a random variable taking values in x and
% probabilities given in n.
% Use the following formula from Wikipedia:
% Gini = 1 -  sum_{i=1}^N( Prob_i * (S_{i-1}+S_i) )/ S_N ,
% where N is the number of realizations, Prob_i=n(i) is the probability of
% each realization, and where 
% S_i = sum_{j=1}^i Prob_j x_j, 
% where x_j=x(j) is observation j and where the observations are ordered
% ascendingly. Also, it is assumed that Prob_j>0 -- thus zero-prob.
% observations have to be discarded.
% 
% Inputs:
% x:        Array of any size. Realizations the random variable takes. Has 
%           to be an increasing vector if ord=true. 
% n:        Array of same size as n, optional. Probabilities of the 
%           realizations in x. If not handed over, equal probabilities are 
%           assumed.
% ord:      Logical scalar. If true, inputs are assumed to be ordered
%           and this step is left out.
% 
% Output:   Scalar. Gini coefficient of the distribution.


if nargin<2                         % If no probability distribution is 
    N = numel(x);
    n = ones(N,1)/N;                % handed over, assume that all realizations
end                                 % in x have the same probability.
if nargin<3                         % Default for 'ord' is false.
    ord = false;
end
if ~ord                             % Have to order inputs if not ordered.
    [x, ind] = sort(x(:));          % Sort values in x and vectorize.
    n = n(ind)/sum(n(:));           % Sort frequencies in the same way and 
                                    % normalize to 1.
else                                % Otherwise, just vectorize.
    x = x(:);
    n = n(:)/sum(n(:));
end

OKind = n>0;                        % Get indices with positive probability.
xx = x(OKind);                      % and kick out zero-prob. events.
nn = n(OKind);
S = cumsum(xx.*nn);                 % Get S from Wiki article: Cumulative
                                    % mean of observations up to i,
                                    % discarding zero-prob. events.
Ssum = S + [0; S(1:end-1)];         % Sum with S just below (and 0 at bottom).
G = 1 -  sum( nn.*Ssum )/S(end);    % Finally, apply the formula.
