function out = GetTransMatAR1(xi,sig2x,xvec,dt)
% Sets up a transition matrix for (mean-zero) Ornstein-Uhlenbeck (i.e.
% AR(1)-process in continuous time) with mean reversion coefficient xi
% and unconditional variance sig2x for use on a time grid of dt. Jumps are
% only possible to adjacent states, i.e. we use the Markov-chain-
% approximation method. For dt=0, the function returns hazard rates for a
% Poisson process with states xvec. 
% CAUTION: Should use this for large grids for xvec that rarely reach top
% and bottom grid points. Do not use for 3-point approximation, for
% example. Variance cannot be set correctly at bottom and top grid point in
% this approach.
% 
% Process:
% dx(t) = -xi*x(t)dt + sig_W dW(t) ,                                (dX)
% where the unconditional variance is
%   E[x(t)^2] = sig^2_W / (2*xi) = sig^2_x                          (var)

% Inputs: 
% xi:           Scalar. Mean-reversion coefficient, must be positive.
% sig2:         Scalar. Unconditional variance of process x(t), must be
%               positive.
% xvec:         n-by-1 vector with equally-spaced grid for x, centered at
%               0.
% dt:           Scalar. Time increment for time grid, must be positive to
%               get an AR1 probabilities. If dt=0, the function returns the
%               hazard rates for a Poisson process with states xvec
% 
% Output:
% n-by-n transition (or hazard) matrix for the process.

if nargin<4                     % Set time increment dt to 1 if not 
    dt = 1;                     % handed over
end
if nargin<3                     % Set grid vector to [-2,-1.5,...,2] if not
    xvec = -2:0.5:2;            % handed over
end
if nargin<2                     % Set standardized variance if not 
    sig2x = 1;                   % handed over
end

if dt>0                         % If probabilities are asked for, set 
    GetProbs=1;                 % marker to 1.
else                            % If hazards are asked for: Set marker to
    GetProbs=0; dt=1;           % 0 and dt=1.
end

sig2W = 2*xi*sig2x;             % From Eq. (var), get conditional variance
                                % implied by the unconditional variance. 

n = length(xvec);               % Get grid size for x.
out = zeros(n);                 % Set up transition matrix.
dx = xvec(2)-xvec(1);           % Difference between x-grid points.
p = sig2W*dt / (2*dx^2);        % p: Probability of goint one grid point up 
                                %    or down in absence of drift.
q = -xi*xvec*dt / (2*dx);       % q: Adjustment for drift.
px =  xi*xvec(end)*dt/dx;       % Transition probability for leaving the 
                                % top and bottom grid point (set this in
                                % order to get correct drift -- variance
                                % may be off)

if GetProbs==1 && (max(q(2:end-1))>p || px>1)
    disp('Warning: GRID FOR STATE VARIABLE IS TOO SMALL for given dt!')
end                             % Give warning if probabilies go negative!
                                % Hazards can be whatever number...

                                % Set probabilities at extreme grid points:
out(1,1)       = GetProbs - px; % for probalities, set to 1-px here, for 
out(end,end)   = GetProbs - px; % hazards put negative entries into matrix.
out(1,2) = px;  out(end,end-1) = px; 
if n>2                          % Set probabilities for leaving inner grid
   for i=2:n-1                  % points:
      out(i,i)   = GetProbs-2*p;% Probability of staying is 1-2p (hazard is
                                % -2p), ...
      out(i,i+1) = p+q(i);      % probability of going one up is p+q, 
      out(i,i-1) = p-q(i);      % ... probability of going one down is p-q.
   end
end
