function [qq,shares,fctLorenz] = GetQuantProbDist(x,qtl,p)
% Returns the quantiles qtl of a probability distribution p that is given
% as a discrete approximation over values x (e.g. from a discretized model,
% e.g. trinomial trees). Can also be used for empirical quantiles: Use uniform
% frequency p or omit the input.
%
% Inputs:
% x:    Array of any size. Values the random variable takes.
% qtl:  Vector (row or column), optional. The desired quantiles; default is 
%       [.1 .25 .5 .75 .9].
% p:    Array of same size as x, optional. Frequencies/probabilities, in same order
%       as x. Is normalized if it does not sum up to 1 (e.g. for
%       conditional distributions). Default: each x has the same weight
%       (use this for empirical quantiles from a data set).

%
% Outputs:
% qq:       1-by-numel(qtl) vector. Quantiles.
% shares:   1-by-numel(qtl) vector. Share of total of x accounted for by
%           observations below each quantile. For example, if x is income,
%           these are the shares of total income accounted for by
%           observations below each quantile value (these are points on the
%           Lorenz curve).
% fctLorenz:Function handle. Lorenz curve.

if nargin<3                             % If probabilities are not handed
    p = ones(size(x));                  % over: Give same weight to each
end                                     % realization.
if nargin<2 || isempty(qtl)             % Set default for quantiles if not
    qtl = [.1 .25 .5 .75 .9]';          % handed over by user.
end

x = x(:);                               % Vectorize inputs.
p = p(:);                               

[x, ind] = sort(x);                     % Sort values in x.
p = p(ind)/sum(p(:));                   % Sort frequencies in the same way
                                        % and normalize mass to 1.

NN = cumsum(p);                         % Get cdf.
OKind = [true; NN(2:end)>NN(1:end-1)];  % Get all indices where cdf is 
                                        % increasing (these indices need
                                        % not coincide with n==0 due to
                                        % computation error -- this way of
                                        % doing things is safer).                                        
                        
qq = interp1(NN(OKind),x(OKind),qtl);   % Interpolate inverted cdf at 
                                        % the quantiles.
if nargout>1
    qtl = reshape(qtl,[1,numel(qtl)]);  % Make qtl a row vector.
    tot = sum(x.*p);                    % Get total sum over all.
    shx = cumsum(x.*p)/tot;             % Get cumulative share up to point x:
                                        % These are the y-values of the
                                        % Lorenz curve. x-values of Lorenz
                                        % curves are the cdf values in NN.
    fctLorenz = @(x) interp1([0;NN(OKind)],[0;shx(OKind)],x);
                                        % Linearly interpolate between
                                        % points on the Lorenz curve:
                                        % function handle for Lorenz curve.
    shares = fctLorenz(qtl);            % Get shares for the quantiles the 
                                        % user wants.
%     NNbelow = [0; NN(1:end-1)];
%     % Now, integrate up the area left of the cdf above each quantile in 
%     % the vector qtl:
%     dF  = max(0, NN - max(NNbelow,qtl) );
%                                         % Get "density" to calculate each
%                                         % quantile: 0 if an observation
%                                         % falls below the quantile,
%                                         % probability of observation if
%                                         % safely above quantile. For the
%                                         % one observation that falls
%                                         % exactly on the quantile, assign a
%                                         % share of observations to the
%                                         % quantile such that we have
%                                         % exactly the fraction in qtl when
%                                         % calculating sum.
%     shares = sum( x.*dF , 1) / tot;     % Take sum of observations above 
end                                     % quantile and return it.
