function [mean,n,fracNZ] = GetConditionalMeanPanel(x,ind,nonzero)
% Calculates conditional mean of random variable x conditional on a dummy ind.
%
% Input:
% x:        Array of size (n1,...,nk) containing the values that the random
%           variable takes. May have singleton dimensions if random
%           variable does not vary in that dimension.
% ind:      Logical array of size [n1,...,nk] if observation should be
%           included.
% nonzero:  Logical scalar, default is false. If true, function computes
%           mean conditional on a variable being non-zero and returns
%           fraction of non-zeros as third output.
% 
% Outputs:
% mean:     Scalar. Mean of x conditional on observation being in subset 
%           given by ind. If nonzero=true, then mean is also conditional on
%           x being non-zero.
% n:        Scalar. Number of observations that are indexed by ind.
% fracNZ:   Scalar. Fraction of n observations that are non-zero, e.g.
%           percentage of households among healthy (given in ind) for which
%           inter-vivos transfer flows.

if nargin<3                             % Set default of nonzero to false.
    nonzero = false;
end

n = sum(ind(:));                        % Get total number of observations.
if nonzero                              % If we are to condition on non-zero:
    nz = ~(x==0);                       % Create logical array if non-zero.
    Nnz= sum(nz(:).*ind(:));            % Number of non-zero observations.
    fracNZ = Nnz/n;                     % Obtain fraction of non-zeros.
    mean = sum( x(:).*ind(:) ) / Nnz;   % Obtain mean conditional on ind and                                       % on being non-zero.
else                                    % If regular mean is asked for:
    mean = sum(x(:).*ind(:))/n;         % Obtain conditional mean among all 
    fracNZ = [];                        % with ind=true, leave third output
end                                     % empty.
