function out = GetConditionalMean(x,n,dimMarg)
% Calculates conditional mean of random variable x with density/frequency
% distribution n. The marginal variable(s) is/are the grid variables that
% are given by the indeces in the vector dimMarg. Returns the unconditional
% mean if dimMarg is empty (=[]) or omitted.
%
% Input:
% x:        Array of size (n1,...,nk) containing the values that the random
%           variable takes. May have singleton dimensions if random
%           variable does not vary in that dimension.
% n:        Array of size [n1,...,nk] with frequency/probability of random
%           variable (need not sum up to 1)
% dimMarg:  Optional. Scalar or row vector with indeces of dimensions
%           (smaller than k in length) that are taken as the marginal
%           variables. If omitted, the function returns the unconditional
%           mean.
% 
% Output:   Vector of length n(dimMarg) in case that dimMarg is a scalar,
%           array with number of dimensions given in dimMarg otherwise.
%           Returns a scalar (the unconditional mean) if dimMarg is omitted
%           or empty.
%
% Example:
% x  = randn(4,3,100);                  % Calculates mean along third dimen-
% n  = ones(4,3,100);                   % sion of x, i.e. returns:
% cm = GetConditionalMean(x,n,[1 2]);   % a 4-by-3 matrix, ...
% c1m= GetConditionalMean(x,n,1);       % ... or mean by first dim.
%                                       %     (1-by-4 vector
% m  = GetConditionalMean(x,n,[]);      % ... or the unconditional mean.

if nargin<3                             % Set marginal dimension empty if
    dimMarg = [];                       % user has not handed over.
end

siz = size(n);                          % Size of n and x.
dim = length(siz);                      % How many dimensions we have. 

if numel(x)<numel(n)                    % If x is not of the same size as
    xFull = x.*ones(siz);               % n (happens if x does not vary in
else                                    % some dimensions): bring to full 
    xFull = x;                          % size.
end                                     

if isempty(dimMarg)                     % If unconditional mean is asked for:
    nMarg = 1;                          % only one dimension for output
else                                    % otherwise, get 
    nMarg = siz(dimMarg);               % grid size for dimensions that stay
end

dimOut = 1:dim;                         % Get dimensions that are integrated
dimOut(dimMarg) = [];                   % out.
nOut = prod( siz(dimOut) );             % Number of grid points to integrate out

xx = permute(xFull,[dimOut, dimMarg]);  % Put dimensions to be integrated 
xx = reshape(xx,[nOut, nMarg]);         % out first and collapse them to one 
                                        % dimension.
nn = permute(n,[dimOut, dimMarg]);      % Reshape density in the same way.
nn = reshape(nn,[nOut, nMarg]);

xn  = sum(xx.*nn,1);                    % Product of x and n for each x-value.
nnn = sum(nn,1);                        % Sum up density to divide...
out = squeeze(xn./nnn);                 % ...and return, squeezing out 
                                        % singleton dimensions.
                                        