function ll = GetBargainSolSell(ll,scen,idxIn,dt,par,opt)
% Obtains bargaining solution and transfer for homeowners 
% (both disabled and healthy).

% Inputs:
% ll: STRUCTURE that contains solution of the game under the inside option 
%       given a transfer (Q=0). Must have the following fields:
%       VkOut, VpOut:   (N,N,1,Nw,Nw,Nhi) arrays. Value of outside option 
%                       for kid and parent. 
%       VkIn0, VpIn0:   (N,N,1,Nw,Nw,Nhi) arrays. Value of inside option
%                       with transfer Q=0.
%       gksb:           (N,1,1,Nw,Nw,Nhi) array. Kid's second-best transfer
%                       under inside option with Q=0.
%       gpsb:           (1,N,1,Nw,Nw,Nhi) array.
%       gkstatfb: 
%       gpstatfb: 
%       xOut:           Selling decision under outside option.
% idxIn:Row vector. Housing indices for which to get bargaining decision.
% scen: Scalar. Indicates which inside option is considered. These are:
%       1: bargain on house-selling only, for healthy and skilled care (Q<=0).
%       2: bargain on IC+keep for disabled owners (Q unrestricted).
% dt:   Scalar. The fixed time increment in the value-function-iteration
%       algorithm.
% STRUCTURE par, containing model parameters read out in the beginning.
% STRUCTURE opt, with options for algorithm read out in the beginning.
%
% Output: A modified input STRUCTURE ll with the following fields 
%         -- all are (N,N,1,Nw,Nw,Nhi) arrays, where Nhi is the number of housing
%         states given in idxIn, unless it is stated otherwise. 
% - xbarg:  Logical. Is 1 if bargaining is successful, 0 if outside option
%           is played.
% - QxIn: Equilibrium transfer Q_x.
% - ekIn:  Kid's expenditure under bargaining solution (i.e. given Q)
% - epIn:  Parent's expenditure under bargaining solution.
% - gkIn:  Kid's gifts under bargaining solution (given Q).
% - gpIn:  Parent's gifts under bargaining solution (given Q).
% - QPaltrComp: (1,N,1,Nw,Nw,Nhi) array. Altruistic component of Q given by
%            the parent. This is defined as the gift the parent would have
%            given under inside option (keep house) and a zero transfer,
%            Q=0.
% - QKaltrComp: (N,1,1,Nw,Nw,Nhi) array. Altruistic component of Q given by
%            child
% - akDotIn: Law of motion for ak under inside option.
% - apDotIn: Law of motion for ap under inside option.
% - VkIn:    Kid's updated value function under the inside
% option.l[kplp0o9;o-p=[
% - VpIn:    Parent's updated value function under the inside option.
%       --> VkIn and VkOut have to be combined outside this function with a
%           Smoother to get the value function under the bargaining
%           solution.

% Read out parameters:
N          = par.N;             Nw  = par.Nw;     Nhi    = numel(idxIn); 
hgrid      = par.hgrid;         r   = par.r;      wgtUnc = par.wgtUnc;    
deltaHouse = par.deltaHouse;  pbar  = par.pbar;   taubar = par.taubar;
alphak     = par.alphak;     alphap = par.alphap; h0   = par.h0;  
bargainMode= par.bargainMode;       % Which bargaining mode is chosen. 
% Read out options:
tolQ   = opt.tolQ;                  % Tolerance of bisection method. 

hhvec      = hgrid(1  ,1  ,1,   1,   1,idxIn); % Read out housing wealth.
if idxIn(1)==1                      % If the first state is renting:
    hhvec(1)=0;                     % no housing wealth there.
end


% Read out if parent would sell under outside option.   
xOut     =     ll.xOut(1:N,1:N,1,1:Nw,1:Nw,idxIn); 
VpOut    =    ll.VpOut(1:N,1:N,1,1:Nw,1:Nw,idxIn);  % Depends on scenario: if healthy,
VkOut    =    ll.VkOut(1:N,1:N,1,1:Nw,1:Nw,idxIn);  % then outside option is keep/sell
VpIn0    =       ll.Vp(1:N,1:N,1,1:Nw,1:Nw,idxIn);  % determined by parent alone. If  
VkIn0    =       ll.Vk(1:N,1:N,1,1:Nw,1:Nw,idxIn);  % disabled, outside option given
VakIn0   =      ll.Vak(1:N,1:N,1,1:Nw,1:Nw,idxIn);
VapIn0   =      ll.Vap(1:N,1:N,1,1:Nw,1:Nw,idxIn);
WakIn0   =      ll.Wak(1:N,1:N,1,1:Nw,1:Nw,idxIn);
WapIn0   =      ll.Wap(1:N,1:N,1,1:Nw,1:Nw,idxIn);

gpsb     =     ll.gpsb(1  ,1:N,1,1:Nw,1:Nw,idxIn); % by Scenario 0.
gksb     =     ll.gksb(1:N,1  ,1,1:Nw,1:Nw,idxIn);
gpstatfb = ll.gpstatfb(1  ,1  ,1,1:Nw,1:Nw,idxIn);
gkstatfb = ll.gkstatfb(1  ,1  ,1,1:Nw,1:Nw,idxIn);

ek       =       ll.ek(1:N,1:N,1,1:Nw,1:Nw,idxIn);  % Need consumption under inside option when Q=0 for kid
ep       =       ll.ep(1:N,1:N,1,1:Nw,1:Nw,idxIn);  % ... and for parent
epUnc    =    ll.epUnc(1:N,1:N,1,1:Nw,1:Nw,idxIn);  
ekUnc    =    ll.ekUnc(1:N,1:N,1,1:Nw,1:Nw,idxIn); 
% Set up gift arrays of full size (need this for our indexing strategy):
% These are the gifts given in equilibrium if Q=0 flows in bargaining
% stage.
gkIn              = zeros(N,N,1,Nw,Nw,Nhi);
gkIn(:,1,:,:,:,:) = ll.gk(:,1,:,:,:,idxIn); % child gives gifts when parent broke
gpIn              = zeros(N,N,1,Nw,Nw,Nhi);
gpIn(1,:,:,:,:,:) = ll.gp(1,:,:,:,:,idxIn); % parent gives gifts when child broke 

NetIncPar1= ll.IncP(1  ,1:N,1,1   ,1:Nw,idxIn);     % net income depends on house size
NetIncKid1= ll.IncK;
NetIncPar = NetIncPar1.*ones(N,N,1,Nw,Nw,Nhi);   % Bring these to full size
NetIncKid = NetIncKid1.*ones(N,N,1,Nw,Nw,Nhi);   % since we'll do logical indexing.
Ucp       = ll.Ucp(1:N,1:N,1,1:Nw,1:Nw,idxIn);   % Flow felicity  used in surplus function      
Uck       = ll.Uck(1:N,1:N,1,1:Nw,1:Nw,idxIn);     
nnk       = ll.nnk;
h1        = ll.h(:,:,:,:,:,idxIn).*ones(N,N,1,Nw,Nw,Nhi);       
nnpArr1   = ll.nnpArr.*ones(N,N,1,Nw,Nw,Nhi);

% The inside scenario determines which transfers are allowed. These
% exogenous bounds are as follows:
QmaxFact = par.QmaxFact;

if scen==1                          
   QbarU = zeros(N,N,1,Nw,Nw,Nhi);  % Zero in scenario 1, highest-possible
else                                % transfer in other scenarios.
   QbarU = QmaxFact*(NetIncPar+(r+deltaHouse)*hhvec);      
end                                 % (also take into account "income" from
% housing wealth here: parent would have to pay the rent (r+deltaHouse)*
% houseSize if they didn't have the house. 

QbarL = min(-QmaxFact*NetIncKid,1.1*NetIncPar); 
                                    % Lowest possible transfer. Make sure
                                    % here that for large houses and poor
                                    % kids, this transfer covers at least
                                    % the housing depreciation.


                      
if scen==1
    if strcmp(bargainMode,'xout') || strcmp(bargainMode,'kid')                                         
        KidPower =  true(N,N,1,Nw,Nw,Nhi); 
    elseif strcmp(bargainMode,'parent')    
        KidPower = false(N,N,1,Nw,Nw,Nhi); 
    end
else    % In scenario 2, read out bargain-mode:
    if strcmp(bargainMode,'xout')   % Default: Kid has bargaining power if
        KidPower = xOut;            % sells under outside option, otherwise
    elseif strcmp(bargainMode,'parent') 
        KidPower = false(N,N,1,Nw,Nw,Nhi);     
    elseif strcmp(bargainMode,'kid')        % 'kid': always with kid.
        KidPower =  true(N,N,1,Nw,Nw,Nhi);
    end
end                                                                   
                                   
                                    
pBr = false(N,N,1,Nw,Nw,Nhi);       % Indices where parent is broke.
pBr(:,1,:,:,:,:) = 1;
kBr = false(N,N,1,Nw,Nw,Nhi);       % Indices where kid is broke.
kBr(1,:,:,:,:,:) = 1;         
OneBroke = pBr|kBr;                 % States where at least one is broke.
NoneBroke= ~OneBroke;               % States where none is broke. 
ll.pBr = pBr;                       % Put into ll-structure to hand over to
ll.kBr = kBr;                       % surplus functions.                       
                                    
                                    
%%%%%%%%%%%%%%%%%%%%%%%%
% UNCONSTRAINED        %
%%%%%%%%%%%%%%%%%%%%%%%%
% Get child's transfer motive under inside option:%
mukIn0 = WapIn0-WakIn0;
%mukIn0(end,end,1,:,:) = mukIn0(end-1,end,1,:,:);
% Just to check: Get transfer motive from diagonal. 
% mukInDiag = ( VkIn0(1:N-1,2:N,:,:,:,:)-VkIn0(2:N,1:N-1,:,:,:,:) )/da;
% Get parent's transfer motive under inside option
mupIn0 = VakIn0-VapIn0;
%mupIn0(end,end,1,:,:) =mupIn0(end,end-1,1,:,:);
% Just to check: Get transfer motive from diagonal. 
% mupInDiag = ( VpIn0(2:N,1:N-1,:,:,:,:)-VpIn0(1:N-1,2:N,:,:,:,:) )/da;
% Mark wealth-pooling region:
taup  = 1 - VapIn0./VakIn0;     % Get tau_p and tau_k: Transfer measured 
tauk  = 1 - WakIn0./WapIn0;     % as wedge, i.e.: tau_p is the tax one can 
                                % impose on a transfer given by the parent
                                % (or subsidy, for negative tau_p) for the
                                % parent to be indifferent to transfer to
                                % the child.
% Call function to get bounds that we impose when none of the agents is broke:                                
[QbarLunc,QbarUunc] = QBounds(tauk,taup,QbarL,QbarU,pbar,taubar,0);

KidWP   = (mukIn0>=0);      % Points where kid is a wealth-pooler.
ParWP   = (mupIn0>=0);      % Points where parent is a wealth pooler.
WPregIn = KidWP | ParWP;    % We are in WP region (under the inside
                            % option) if either transfer motive is non-negative.
                            
Done   = false(N,N,1,Nw,Nw,Nhi);    % Marker if cases are done.       
xbarg  = false(N,N,1,Nw,Nw,Nhi);    % Default: outside option is played,
QxIn   = zeros(N,N,1,Nw,Nw,Nhi);    % with zero transfers
VkIn   = VkIn0;                     % Set value function under inside option
VpIn   = VpIn0;                     % to that at Q=0 as default


%% If house is kept under outside option (possible for healthy owners):
%  nothing to do. 
if scen==1 
    Done(~xOut) = true;             % If parent does not sell under outside 
end                                 % option, there is nothing to bargain
                                    % on. We are done with these cases.
                                    % Leave values for Q=0 for the inside
                                    % option (value functions, gifts,
                                    % QxIn).

%% Unconstrained cases
%%% Unconstrained cases in regular (non-WP) region: 
% Mark regular region (none of the players wants to give transfers):
Unc = ~Done & NoneBroke;            % Mark the unconstrained cases for which
                                    % we have to determine bargaining
                                    % outcome.
% This is the code when we interpret positive transfer motives as actually
% being WP:
Sk0   = VkIn0-VkOut;                % Kid surplus when Q=0
Sp0   = VpIn0-VpOut;                % Parent surplus when Q=0

% Try with surplus sharing on the interior:
QthrK =  Sk0./(mukIn0*dt);          % Get threshold for kid.
QthrP = -Sp0./(mupIn0*dt);          % Get threshold transfer for parent to 
                                    % do inside option from closed form in
                                    % paper (this formula is for
                                    % unconstrained transfers that can have
                                    % any sign). If VpOut>VpIn0 then parent 
                                    % needs compensation to hold house
                                    % which corresponds to Q<0.
                   % Otherwise, do what we did before.
Qeq = (  wgtUnc  *KidPower + (1-wgtUnc)*(1-KidPower)).*QthrP  + ...
      ((1-wgtUnc)*KidPower +   wgtUnc  *(1-KidPower)).*QthrK       ;
                      % wgtUnc: Bargaining weight on parent is 1 in true
                        % model, but go slightly lower to help stability.


% If sold under outside option: Go almost all the way to parent's threshold 
% (wgtUnc~<1), if not sold under outside option: Go almost all the way to
% kid's threshold.

% Now, impose our bounds (imposes a transfer when at least one of the
% transfer motives is positive: 
Qtilde = max(QbarLunc, min( QbarUunc, Qeq));

QxIn(Unc) = Qtilde(Unc);        % This is our candidate transfer for the 
                                % unconstrained cases.
SbarK  = Sk0 - Qtilde.*mukIn0*dt;       % Check both agents' surplus. We 
SbarP  = Sp0 + Qtilde.*mupIn0*dt;       % have a bargaining solution iff
UncBarg = Unc & (SbarK>=0) & (SbarP>=0);% both agree.                         
xbarg(UncBarg)    = true;          % Write in the bargaining solution.

% Set kid's and parent's value function under the inside option:
VkIn(Unc) = VkOut(Unc) + SbarK(Unc);
VpIn(Unc) = VpOut(Unc) + SbarP(Unc);

Done(Unc) = true;                   % We are done with the unconstrained cases now.                                    


                                    
%% %%%%%%%%%%%%%%%%%%%%%%
% AT LEAST 1 BROKE     %
%%%%%%%%%%%%%%%%%%%%%%%%
% Set up arrays for altruistic component of Q-transfer (set up full
% dimensionality to facilitate indexing, will shrink them later).
QPaltrComp = zeros(N,N,1,Nw,Nw,Nhi);
QKaltrComp = zeros(N,N,1,Nw,Nw,Nhi);
% First, obtain indifference thresholds from proposition in paper: 
% Lower bound:
QstarL                = QbarL;   % Set to lower bound by default.
QstarL(1,1,:,:,:,:)   = min( gpstatfb, gpsb(1,1,:,:,:,:) ); 
QstarL(1,2:N,:,:,:,:) = gpsb(1,2:N,:,:,:,:);
WPbound = WPregIn & pBr & ~kBr;         % Mark cases in which only parent is 
                                        % broke and we have a WP-region.
QstarL(WPbound) = NetIncPar(WPbound) - epUnc(WPbound);
                                        % Set QstarL for these cases.
% Upper indifference bound:                                        
QstarU                = QbarU;   % Set to upper bound by default.
QstarU(1,1,:,:,:,:)   = -min( gkstatfb, gksb(1,1,:,:,:,:) );
QstarU(2:N,1,:,:,:,:) = -gksb(2:N,1,:,:,:,:);
WPbound = WPregIn & kBr & ~pBr;         % Mark cases in which only kid is 
                                        % broke and we have a WP-region.
QstarU(WPbound) = ekUnc(WPbound) - NetIncKid(WPbound); 

% Now, impose the exogenous bounds on transfers:
Qlb = min( QbarU, max(QbarL, QstarL) ); % From equation in paper.
Qub = min( QbarU, max(QbarL, QstarU) ); % (N,N,1,Nw,Nw,Nhi) arrays.

% Case 0) If total dynasty income is negative (can happen when both are
% broke and house is very large and dynasty cannot afford housing-deprec.
% costs): Parent must sell.
case0 = pBr & kBr & (NetIncPar+NetIncKid<=0);
% xbarg(case0) = false;               % No bargaining solution; can keep 
Done(case0)  = true;                  % default values. Note that in this
                                      % case, decisions are never smoothed
                                      % since both are broke, thus outside
                                      % option played with Prob=1.

% Set up array for kid's reservation transfer in cases where one agent is
% broke:
nBroke = sum(OneBroke(:));
DoneOneBroke = Done(OneBroke);      % Create new variable to keep track of 
                                    % which broke cases are done. 
bargOneBroke = false(nBroke,1);    % As default, set that bargaining fails
QxInOneBroke = zeros(nBroke,1);     % with zero transfers.

gkOneBrokeQ0  = gkIn(OneBroke);     % Read out gifts that would occurr 
gpOneBrokeQ0  = gpIn(OneBroke);     % under Q=0: These will be the altruistic
                                    % component of Q if the bargaining
                                    % outcome is success.
gkOneBroke    = gkIn(OneBroke);     % Set up gift arrays in equilibrium after  
gpOneBroke    = gpIn(OneBroke);     % Q has flown: These will be set to zero
                                    % if bargaining is successful.
% Take values at zero transfers as default for one-broke cases:                                    
VkInOneBroke = VkIn0(OneBroke);  VkOutOneBroke = VkOut(OneBroke);
VpInOneBroke = VpIn0(OneBroke);  VpOutOneBroke = VpOut(OneBroke);
% If parent sells under outside option:
KidPowerOneBroke =  KidPower(OneBroke);

% Set up Nbroke-by-1 vectors of lower and upper bounds for cases where one
% is broke:
 QbarLOneBroke =  QbarL(OneBroke);  QbarUOneBroke =  QbarU(OneBroke);
QstarLOneBroke = QstarL(OneBroke); QstarUOneBroke = QstarU(OneBroke);
   QlbOneBroke =    Qlb(OneBroke);    QubOneBroke =    Qub(OneBroke);

% Get agents' surplus at upper and lower bounds of the interval we have to
% consider for the transfer Q:
SkQlb = zeros(nBroke,1);            % Set surplus to zero by default (this
SpQlb = zeros(nBroke,1);            % is what we will have for case 0). 
SkQub = zeros(nBroke,1);            % Then compute surplus for upper and 
SpQub = zeros(nBroke,1);            % lower bound.
[SkQlb(~DoneOneBroke),SpQlb(~DoneOneBroke)] = Surplus(QlbOneBroke(~DoneOneBroke),dt,ll,idxIn,OneBroke & ~Done,par);
[SkQub(~DoneOneBroke),SpQub(~DoneOneBroke)] = Surplus(QubOneBroke(~DoneOneBroke),dt,ll,idxIn,OneBroke & ~Done,par); 

% Case i) Intervals [QstarL,QstarU] and [Qlb,Qub] don't overlap.

% Case i,a) QstarL>=QbarU
% This occurs for healthy parents who would like to give gifts to their
% kids. Could also happen if disabled parents wants to give huge gifts (that 
% exceed the computational bound on transfers), but very unlikely.
iaCase = ~DoneOneBroke & (QstarLOneBroke>=QbarUOneBroke);     % Mark this case.
if any(iaCase(:))                           % If any cases to do:
    Skia = SkQub(iaCase);                   % Get both agents' surplus.
    Spia = SpQub(iaCase);
    bargia = (Skia>0) & (Spia>0);           % Have bargaining solution iff
                                            % both have positive surplus
                                            % at QbarU.
    bargOneBroke(iaCase) = bargia;          % Record this solution.
    QxInOneBroke(iaCase) = QbarUOneBroke(iaCase);      % Transfer in these cases is the upper bound. 
    % Value under inside option is outside option plus surplus:
    VkInOneBroke(iaCase) = VkOutOneBroke(iaCase) + Skia;
    VpInOneBroke(iaCase) = VpOutOneBroke(iaCase) + Spia;
    % Parent tops up QbarU with gift, if there is a positive difference:
      gpOneBroke(iaCase) = QstarLOneBroke(iaCase) - QbarUOneBroke(iaCase); 
      gkOneBroke(iaCase) = 0;               % Child does not give.
    DoneOneBroke(iaCase) = true;            % We are done with these cases
end

% Case i,b): QstarU<=QbarL
% Kids would like to give gifts that exceed the exogenous upper bound
% (a multiple of parent income) CHECKED: NEVER STOPPED HERE.
ibCase = ~DoneOneBroke & (QstarUOneBroke<=QbarLOneBroke);     % Mark this case.
if any(ibCase(:))
    Skib   = SkQlb(ibCase);         % Similar to case ib.
    Spib   = SpQlb(ibCase);
    bargib = (Skib>0) & (Spib>0);
    bargOneBroke(ibCase) = bargib;
    QxInOneBroke(ibCase) = QbarLOneBroke(ibCase);
    VkInOneBroke(ibCase) = VkOutOneBroke(ibCase) + Skib;
    VpInOneBroke(ibCase) = VpOutOneBroke(ibCase) + Spib;
      gkOneBroke(ibCase) = QbarLOneBroke(ibCase) - QstarUOneBroke(ibCase);
      gpOneBroke(ibCase) = 0;
    DoneOneBroke(ibCase) = true;    % No need to update consumption and
                                    % gift functions: they already contain
end                                 % what is played under Q=0.
                                                                      
gkOneBroke(~DoneOneBroke) = 0;      % In all remaining cases, gifts are 
gpOneBroke(~DoneOneBroke) = 0;      % zero; note that we chose the bounds 
                                    % Qlb and Qub so that exactly this
                                    % happens.
                                    
% Case ii) Our task now is to find the threshold transfer that makes the
% party without bargaining power (call it "the weak party") indifferent: 
% This is the parent in scenario 1 and kid in scenario 2.

% Case ii)a) First corner solution. If the weak party cannot be convinced
% of the inside option even at the most favorable transfer to them (the
% parent and Qlb in scenario 1, the kid and Qub in scenario 2), then:
% - There is no bargaining solution.
% - We write in the most favorable transfer to the weak party into the
%   transfer in order to preserve continuity under the inside option -- this 
%   is relevant only if decisions are smoothed. (In scen. 1: Note that 
% if the threshold transfer for the parent to keep is very close to the 
% lower bound for Q (Qlb, the value at which the parent would give back 
% transfers or we enter WP region), then we are setting this threshold; 
% so continuity leaves Qlb as the only reasonable option)
  
% Mark these cases:
aCase = ~DoneOneBroke & (  ( KidPowerOneBroke & (SpQlb<=0) ) | ...
                           (~KidPowerOneBroke & (SkQub<=0) )      ); 
if any(aCase)                       % If there are any such cases:                     
    QxInOneBroke(aCase) =   KidPowerOneBroke(aCase)  .*QlbOneBroke(aCase) + ...
                          (1-KidPowerOneBroke(aCase)).*QubOneBroke(aCase)      ;
                                    % For continuity in inside solutions,
                                    % write in lower bound if kid has power
                                    % and upper bound if parent has power.
    % Set value function under inside option:
    VkInOneBroke(aCase) =       VkOutOneBroke(aCase)                + ...
                             KidPowerOneBroke(aCase) .*SkQlb(aCase) + ...
                          (1-KidPowerOneBroke(aCase)).*SkQub(aCase)      ;
    VpInOneBroke(aCase) =       VpOutOneBroke(aCase)                + ...
                             KidPowerOneBroke(aCase) .*SpQlb(aCase) + ...
                          (1-KidPowerOneBroke(aCase)).*SpQub(aCase)      ;
    DoneOneBroke(aCase) = true;     % We are done with these cases.
end

% Case ii)b) The weaker party accepts even the least favorable transfer 
% (healthy owners: parents accept Qub, disabled owners: kids accept Qlb)
% --> the 2nd corner solution.

bCase = ~DoneOneBroke & (  ( KidPowerOneBroke & (SpQub>=0) ) | ...
                           (~KidPowerOneBroke & (SkQlb>=0) )       );
if any(bCase)
    bargOneBroke(bCase) = (  KidPowerOneBroke(bCase) & (SkQub(bCase)>=0) )  |  ...
                          ( ~KidPowerOneBroke(bCase) & (SpQlb(bCase)>=0) )        ;
    QxInOneBroke(bCase) =    KidPowerOneBroke(bCase) .*QubOneBroke(bCase) + ...
                          (1-KidPowerOneBroke(bCase)).*QlbOneBroke(bCase)         ;
    VkInOneBroke(bCase) = VkOutOneBroke(bCase) +  KidPowerOneBroke(bCase)  .*SkQub(bCase) ...
                                               +(1-KidPowerOneBroke(bCase)).*SkQlb(bCase) ;
    VpInOneBroke(bCase) = VpOutOneBroke(bCase) +  KidPowerOneBroke(bCase)  .*SpQub(bCase) ...
                                               +(1-KidPowerOneBroke(bCase)).*SpQlb(bCase) ;
    DoneOneBroke(bCase) = true;         % We are done these cases.
end



% Case ii)c) Interior case. Find root of parent surplus.
%      A) First try if we can find an intersection on the linear part of
%      the child's surplus function in closed form (for computational
%      efficiency) --> Omit this, let's see later if we have speed
%      problems.
%      B) Find root.

cCase = ~DoneOneBroke;              % These are all the remaining cases.
CCASE = false(N,N,1,Nw,Nw,Nhi);     % Create 6D-indices that mark this case
CCASE(OneBroke) = cCase;            % to hand over to Surplus.m.

if any(cCase(:))                    % If there are cases left to consider:
    % First, read out relevant variables for this case (do this once
    % outside the bisection loop to save time; variables will be used by
    % the nested function SurplusNest.m).
    kBrCC       =       kBr(CCASE);  pBrCC       =       pBr(CCASE);
    ekUncCC     =     ekUnc(CCASE);  epUncCC     =     epUnc(CCASE);
    ek0CC       =        ek(CCASE);  ep0CC       =        ep(CCASE);
    NetIncKidCC = NetIncKid(CCASE);  NetIncParCC = NetIncPar(CCASE); 
    Uck0CC      =       Uck(CCASE);  Ucp0CC      =       Ucp(CCASE);
    gkInCC      =      gkIn(CCASE);  gpInCC      =      gpIn(CCASE);
    WakIn0CC    =    WakIn0(CCASE);  WapIn0CC    =    WapIn0(CCASE);
    VakIn0CC    =    VakIn0(CCASE);  VapIn0CC    =    VapIn0(CCASE);
    mukIn0CC    =    mukIn0(CCASE);  mupIn0CC    =    mupIn0(CCASE);
    h1CC        =        h1(CCASE);  nnpArr1CC   =   nnpArr1(CCASE);
    VkIn0CC     =     VkIn0(CCASE);  VpIn0CC     =     VpIn0(CCASE);
    VkOutCC     =     VkOut(CCASE);  VpOutCC     =     VpOut(CCASE);     
    
    dx = 0.5*(QubOneBroke(cCase) - QlbOneBroke(cCase));
    x = QlbOneBroke(cCase) + dx;    % Find zero of the weak party's surplus
    
    cKidPower = KidPowerOneBroke(cCase);   % Get vector that says if kid has power.
    while any(abs(dx)>tolQ)         % by bisection method on [Tlb, Tub]
        dx = 0.5*dx;                % to find the reservation transfer:
        [SSkk,SSpp] = SurplusNest(x);   % Get surplus of both.
        
        x = x + dx.*( cKidPower.*sign(SSpp) - (1-cKidPower).*sign(SSkk) ); 
    end 
    
    % Finally, check strong party's surplus to see if it's a bargaining
    % solution:
     bargOneBroke(cCase) = ( cKidPower & (SSkk>0) )  |  (~cKidPower & SSpp>0 ) ;
    QxInOneBroke(cCase) = x;    
    VkInOneBroke(cCase)  = VkOutOneBroke(cCase) + SSkk;
    VpInOneBroke(cCase)  = VpOutOneBroke(cCase) + SSpp;
end                               

% For successful bargaining outcomes, read out gifts that would have been
% given anyway under Q=0: this is the altruistic component of Q.
QPaltrCompOneBroke =  gpOneBrokeQ0;
QKaltrCompOneBroke = -gkOneBrokeQ0;

% Now, write the solution for the one-broke cases into the larger arrays:
xbarg(OneBroke)      = bargOneBroke;    QxIn(OneBroke) = QxInOneBroke;
gkIn(OneBroke)       = gkOneBroke;      gpIn(OneBroke) = gpOneBroke;
QPaltrComp(OneBroke) = QPaltrCompOneBroke;
QKaltrComp(OneBroke) = QKaltrCompOneBroke;
VkIn(OneBroke) = VkInOneBroke;  VpIn(OneBroke) = VpInOneBroke;

% Calculate consumption expenditures under the transfer Q for the inside
% option:
ekIn = kBr.*min(ekUnc,NetIncKid+QxIn-gkIn+gpIn) + (1-kBr).*ekUnc;
epIn = pBr.*min(epUnc,NetIncPar-QxIn+gkIn-gpIn) + (1-pBr).*epUnc;
% Get law of motion including the transfer Q:
akDotIn = NetIncKid - ekIn - gkIn + gpIn + QxIn;
apDotIn = NetIncPar - epIn + gkIn - gpIn - QxIn;


% Add output arrays to input structure:                              
ll.xbarg = xbarg;   ll.QxIn= QxIn;
% Hand over equilibrium consumption under bargaining solution:
ll.ekIn= ekIn;  ll.epIn= epIn;
% For gifts, only hand over indices where recipient is broke (are 0
% otherwise):
ll.gkIn       =  gkIn;
ll.gpIn       =  gpIn;
% Also hand over altruistic component of Q (only take indices where
% recipient is broke, otherwise it is 0 anyway -- note that gifts cannot
% flow to non-broke owners in equilibrium.

qPaltrComp = min(QxIn(1,:,1,:,:,:),QPaltrComp(1,:,1,:,:,:));
QPaltrComp(1,:,1,:,:,:)=qPaltrComp;
ll.QPaltrComp= QPaltrComp;
qKaltrComp = max(QxIn(:,1,1,:,:,:),QKaltrComp(:,1,1,:,:,:));
QKaltrComp(:,1,1,:,:,:)= qKaltrComp;
ll.QKaltrComp=QKaltrComp;
ll.VkIn       = VkIn;       ll.VpIn    = VpIn;
ll.akDotIn    = akDotIn;    ll.apDotIn = apDotIn;
ll.mukIn0     = mukIn0;     ll.mupIn0  = mupIn0;

function [SkQ,SpQ] = SurplusNest(Q)
    % Surplus of kid and parent from inside over outside option for a 
    % transfer Q. This is the same function as Surplus.m. Created a nested
    % version since indexing multiple times during the bisection loop took
    % a lot of time. Q has to be in range [Qlb,Qub], which implies zero
    % gifts from both sides
    % Input:    Q, array of size s.ckUnc(ind) or scalar.
    % Output: two arrays of size of Q:
    %       SkQ:            Surplus of kid from inside over outside option
    %       SkP:            Surplus of parent.               
    ekQ  = kBrCC.*min(ekUncCC,NetIncKidCC+Q) + (1-kBrCC).*ekUncCC;
    epQ  = pBrCC.*min(epUncCC,NetIncParCC-Q) + (1-pBrCC).*epUncCC;
    UckQ = utilde( ekQ, h0, nnk      , par);
    UcpQ = utilde( epQ, h1CC, nnpArr1CC, par);
    duk = UckQ - Uck0CC; % difference of flow felicity with Q and with Q=0
    dup = UcpQ - Ucp0CC;
    dek = ekQ  -  ek0CC;  % difference of flow expenditures with Q and with Q=0
    dep = epQ  -  ep0CC;
    dgk = 0    - gkInCC;  % difference of flow gift with Q and with Q=0
    dgp = 0    - gpInCC;
    dVk =  (duk + alphak*dup - dek.*WakIn0CC - dep.*WapIn0CC ...
          +(dgk - dgp - Q).*mukIn0CC  )*dt;
    dVp =  (dup + alphap*duk - dep.*VapIn0CC - dek.*VakIn0CC ...
          -(dgk - dgp - Q).*mupIn0CC  )*dt;
    SkQ = (VkIn0CC - VkOutCC) + dVk;             
    SpQ = (VpIn0CC - VpOutCC) + dVp; 
        
end

end