function ll = GetBargainSolIC(ll,dt,par,opt)
% Obtains reservation transfer of kid to prefer IC, which is the equilibrium
% transfer if bargaining is successful (since all bargaining power is with
% the parent). When option opt.bargainModeRent is set on 'kid', then the
% bargaining power is with the kid and the reservation transfer of the
% parent is returned. This function is for disabled renters only, see
% GetBargainSolSell.m for home owners (both disabled and healthy).

% Inputs:
% ll:   STRUCTURE which contains solution of the game under the inside option 
%       option given zero transfers (Q=0). Must have following fields:
%       VkOut, VpOut:   (N,N,1,Nw,Nw,1) arrays. Value of outside option 
%                       for kid and parent. 
%       VkIn0, VpIn0:   (N,N,1,Nw,Nw,1) arrays. Value of inside option
%                       with transfer Q=0.
%       gksb:           (N,1,1,Nw,Nw,1) array. Kid's second-best transfer
%                       under inside option with Q=0.
%       gpsb:           (1,N,1,Nw,Nw,1) array.
%       gkstatfb:       Kid's static first-best transfer.
%       gpstatfb:       Parent's static first-best transfer.
% dt:   Scalar. The fixed time increment in the value-function-iteration
%       algorithm.
% STRUCTURE par, containing model parameters read out in beginning.
% STRUCTURE opt, containing fields read out in the beginning. 
%
% Output: A modified input structure ll with the following fields 
%         -- all are (N,N,1,Nw,Nw,1) arrays, unless otherwise stated.
% - ibarg: Logical. Is 1 if bargaining is successful, 0 if outside option
%          is played.
% - QiIn:  Equilibrium transfer Q_i.
% - ekIn:  Kid's expenditure under bargaining solution (i.e. given Q)
% - epIn:  Parent's expenditure under bargaining solution.
% - gkIn:  Kid's gifts under bargaining solution (given Q).
% - gpIn:  Parent's gifts under bargaining.
% - QPaltrComp: (1,N,1,Nw,Nw,Nhi) array. Altruistic component of Q given by
%          the parent. This is defined as the gift the parent would have
%          given anyway under the inside option and a zero transfer,
%          Q=0.
% - akDotIn: Law of motion for ak under inside option.
% - apDotIn: Law of motion for ap under inside option.
% - VkIn:  Kid's updated value function containing the bargaining solution
% - VpIn:  Parent's updated value function.
%       --> VkIn and VkOut have to be combined outside this function with a
%           Smoother to get the value function under the bargaining
%           solution.

% Read out parameters:
N  = par.N;   Nw = par.Nw;     QmaxFact = par.QmaxFact;     
wgtUnc = par.wgtUnc; pbar = par.pbar; taubar = par.taubar;
alphak     = par.alphak;     alphap = par.alphap; h0   = par.h0;

% Which bargaining mode:
if strcmp(par.bargainModeRent,'kid')
    ParPower = false;               % Give power to % kid if asked for,...
else                                 
    ParPower = true;                % default is that parent has power.
end

% Read out value functions:
VpOut     = ll.VpOut(1:N,1:N,1,1:Nw,1:Nw,1);    % Renter value conditional 
VkOut     = ll.VkOut(1:N,1:N,1,1:Nw,1:Nw,1);    % on FC.
VpIn0     =    ll.Vp(1:N,1:N,1,1:Nw,1:Nw,1);    % Renter value conditional
VkIn0     =    ll.Vk(1:N,1:N,1,1:Nw,1:Nw,1);    % on IC with Q=0
VakIn0    =   ll.Vak(1:N,1:N,1,1:Nw,1:Nw,1);
VapIn0    =   ll.Vap(1:N,1:N,1,1:Nw,1:Nw,1);
WakIn0    =   ll.Wak(1:N,1:N,1,1:Nw,1:Nw,1);
WapIn0    =   ll.Wap(1:N,1:N,1,1:Nw,1:Nw,1);
% Read out other inputs:
gpsb      = ll.gpsb(1,1:N,1,1:Nw,1:Nw,1);
gksb      = ll.gksb(1:N,1,1,1:Nw,1:Nw,1);
gpstatfb  = ll.gpstatfb(1,1,1,1:Nw,1:Nw,1);
gkstatfb  = ll.gkstatfb(1,1,1,1:Nw,1:Nw,1);

ek        = ll.ek(1:N,1:N,1,1:Nw,1:Nw,1);  % Need consumption under inside option when Q=0 for kid
ep        = ll.ep(1:N,1:N,1,1:Nw,1:Nw,1);  % ... and for parent
epUnc     = ll.epUnc(1:N,1:N,1,1:Nw,1:Nw,1); 
ekUnc     = ll.ekUnc(1:N,1:N,1,1:Nw,1:Nw,1);  
% Set up gift arrays of full size (need this for our indexing strategy):
gkIn              = zeros(N,N,1,Nw,Nw,1);      % and read in gifts that are
gkIn(:,1,:,:,:,:) = ll.gk(:,1,1,:,:,1);         % given in equilibrium 
gpIn              = zeros(N,N,1,Nw,Nw,1);       % if Q=0 in the bargaining
gpIn(1,:,:,:,:,:) = ll.gp(1,:,1,:,:,1);         % stage.

NetIncPar1 = ll.IncP(1,1:N,1,1,1:Nw,1); % net income depends on house size
NetIncKid1 = ll.IncK;
NetIncPar = NetIncPar1.*ones(N,N,1,Nw,Nw,1);   % Bring these to full size
NetIncKid = NetIncKid1.*ones(N,N,1,Nw,Nw,1);   % since we'll do logical indexing.
Ucp       = ll.Ucp(1:N,1:N,1,1:Nw,1:Nw,1);   % Flow felicity  used in surplus function      
Uck       = ll.Uck(1:N,1:N,1,1:Nw,1:Nw,1);     
nnk       = ll.nnk;
h1        = ll.h(:,:,:,:,:,1).*ones(N,N,1,Nw,Nw,1);       
nnpArr1   = ll.nnpArr.*ones(N,N,1,Nw,Nw,1);

QbarL = 0;       % Lower exogenous bound for Q is zero.
QbarU = QmaxFact*NetIncPar;    % No upper exogenous bound in model, but
                               % set computational upper bound. Only have
                               % renting parents, thus no adjustment
                               % necessary for housing wealth.                          
                                   
% Read out options:
tolQ   = opt.tolQ;                  % Tolerance of bisection method.
                                    

pBr = false(N,N,1,Nw,Nw,1);         % Indices where parent is broke
pBr(:,1,:,:,:,:) = 1;
kBr = false(N,N,1,Nw,Nw,1);         % Indices where kid is broke
kBr(1,:,:,:,:,:) = 1;         
OneBroke = pBr|kBr;                 % States where at least one is broke.
NoneBroke= ~OneBroke;               % States where none is broke. 
ll.pBr = pBr;                       % Put into ll-structure to hand over to
ll.kBr = kBr;                       % surplus functions.  
                                                                                                                             
%%%%%%%%%%%%%%%%%%%%%%%%
% UNCONSTRAINED        %
%%%%%%%%%%%%%%%%%%%%%%%%
% Get child's transfer motive under inside option:%
mukIn0 = WapIn0-WakIn0;
% Get parent's transfer motive under inside option
mupIn0 = VakIn0-VapIn0;
taup  = 1 - VapIn0./VakIn0;     % Get tau_p and tau_k: Transfer measured 
tauk  = 1 - WakIn0./WapIn0;     % as a tax wedge.
% Call function to get bounds that we impose when none of the agents is broke:                                
[QbarLunc,QbarUunc] = QBounds(tauk,taup,QbarL,QbarU,pbar,taubar,0);
% Mark wealth-pooling region:
KidWP = (mukIn0>=0);     % Points where kid is a wealth pooler.
ParWP = (mupIn0>=0);     % Points where parent is a wealth pooler.
WPregIn = ParWP | KidWP; % We are in WP region (under the inside
                         % option) if either transfer motive is non-negative.
Done   = false(N,N,1,Nw,Nw,1);    % Marker if cases are done.       
ibarg  = false(N,N,1,Nw,Nw,1);    % Default: outside option is played,
QiIn   = zeros(N,N,1,Nw,Nw,1);    % with zero transfers
VkIn   = VkIn0;                     % Set value function under inside option
VpIn   = VpIn0;                     % to that at Q=0 as default                               
                                    
%% Unconstrained cases
%%% Unconstrained cases in regular (non-WP) region:   
Unc = ~Done & NoneBroke;            % Mark unconstrained cases
% Reg  = ~Done & NoneBroke & ~KidWP;   % Mark regular cases: None broke, parent not
                                    % a wealth-pooler, and we are not done
                                    % yet.
Sk0  = VkIn0-VkOut;       % Kid surplus when Q=0
Sp0  = VpIn0-VpOut;       % Parent surplus when Q=0
QthrK= Sk0./(mukIn0*dt);  % Get threshold transfer for child to 
                          % do inside option from closed form 
                          % in paper (this formula is for
                          % unconstrained transfers that can have
                          % any sign). This is also the correct
                          % transfer when parent is wealth-pooler.
QthrP= -Sp0./(mupIn0*dt); % Parent's threshold.

if ParPower               % If parent has power, take kid's reservation transfer.
    Qeq =   wgtUnc  *QthrK + (1-wgtUnc)*QthrP;  
else                      % Otherwise, take parent's reservation transfer.
    Qeq = (1-wgtUnc)*QthrK +   wgtUnc  *QthrP;
end                       % wgtUnc: Bargaining weight on parent is 1 in true
                          % model, but go slightly lower to help stability.
% Now, impose our bounds (imposes a transfer when at least one of the
% transfer motives is positive: 
Qtilde = max(QbarLunc, min( QbarUunc, Qeq));    


% % Now, do cases in which at least one transfer motive is positive:                                    
QiIn(Unc)= Qtilde(Unc);     % Write in candidate transfer.
SbarP  = Sp0 + Qtilde.*mupIn0*dt;
SbarK  = Sk0 - Qtilde.*mukIn0*dt; % Also check surplus of kid
UncBarg = Unc & (SbarP>=0) & (SbarK>=0); % Get unconstrained bargaining solution
                                        % for transfers.
ibarg(UncBarg) = true;

% Set kid's and parent's value function under the inside option:                              
VpIn(Unc) = VpOut(Unc) + SbarP(Unc);           
VkIn(Unc) = VkOut(Unc) + SbarK(Unc);
 
Done(Unc) = true;                   % Mark regular cases as done.                                    

%% %%%%%%%%%%%%%%%%%%%%%%
% AT LEAST 1 BROKE     %
%%%%%%%%%%%%%%%%%%%%%%%%
% Set up arrays for altruistic component of Q-transfer (set up full
% dimensionality to facilitate indexing, will shrink them later).
QPaltrComp = zeros(N,N,1,Nw,Nw,1);
QKaltrComp = zeros(N,N,1,Nw,Nw,1);
% First, obtain indifference thresholds from proposition in paper: 
% Lower bound:
QstarL                = zeros(N,N,1,Nw,Nw,1);   % Set to zero by default.
QstarL(1,1,:,:,:,:)   = min( gpstatfb, gpsb(1,1,:,:,:,:) );
QstarL(1,2:N,:,:,:,:) = gpsb(1,2:N,:,:,:,:);
WPbound               = WPregIn & pBr & ~kBr;         
                                        % Mark cases in which only parent is 
                                        % broke and we have a WP-region.
QstarL(WPbound) = NetIncPar(WPbound) - epUnc(WPbound);
                                        % Set QstarL for these cases.
% Upper indifference bound:                                        
QstarU                 = QbarU;   % Set to upper bound by default.
QstarU(1,1,:,:,:,:)    = -min( gkstatfb, gksb(1,1,:,:,:,:) );
QstarU(2:N,1,:,:,:,:)  = -gksb(2:N,1,:,:,:,:);
WPbound                = WPregIn & kBr & ~pBr;         
                                        % Mark cases in which only kid is 
                                        % broke and we have a WP-region.
QstarU(WPbound) = ekUnc(WPbound) - NetIncKid(WPbound); 

% Now, impose the exogenous bounds on transfers:
Qlb = min( QbarU, max(QbarL, QstarL) ); % From equation in paper.
Qub = min( QbarU, max(QbarL, QstarU) ); % (N,N,1,Nw,Nw,1) arrays.
% Set up array for kid's reservation transfer in cases where one agent is
% broke:
nBroke        = sum(OneBroke(:));
DoneOneBroke  = Done(OneBroke);   % Keep track of which cases are done in
                                   % subset where one is broke.  
bargOneBroke  = false(nBroke,1);
QiInOneBroke  = zeros(nBroke,1);   % Nbroke-by-1 vector
gkOneBrokeQ0  = gkIn(OneBroke);   % Read out gifts that would occurr 
gpOneBrokeQ0  = gpIn(OneBroke);   % under Q=0: These will be the altruistic
                                    % component of Q if the bargaining
                                    % outcome is success.
gkOneBroke    = gkIn(OneBroke); % Set up gift arrays in equilibrium after  
gpOneBroke    = gpIn(OneBroke); % Q has flown: These will be set to zero
                                % if bargaining is successful.
% Take values at zero transfers as default for one-broke cases:                                    
VkInOneBroke = VkIn0(OneBroke);  VkOutOneBroke = VkOut(OneBroke);
VpInOneBroke = VpIn0(OneBroke);  VpOutOneBroke = VpOut(OneBroke);

% Set up Nbroke-by-1 vectors of lower and upper bounds for cases where one
% is broke:
QlbOneBroke = Qlb(OneBroke);        QubOneBroke = Qub(OneBroke);

% Case i) If a caregiving child of a renting parent wants to give gifts 
% anyway
iCase =  QstarU(OneBroke)<=QbarL; % Mark this case.
% Obtain the surplus for both agents:
Ski = VkInOneBroke(iCase) - VkOutOneBroke(iCase);
Spi = VpInOneBroke(iCase) - VpOutOneBroke(iCase);
bargi= (Ski>0) & (Spi>0);   % Mark cases in which both are better off
                            % under inside option and Q=0
bargOneBroke(iCase) = bargi;    % Write solution back into larger vector 

DoneOneBroke(iCase) = true;     % We are done with these cases

gkOneBroke(~DoneOneBroke) = 0;      % In all remaining cases, gifts are 
gpOneBroke(~DoneOneBroke) = 0;      % zero; note that we chose the bounds 
                                    % Qlb and Qub so that exactly this
                                    % happens.
                                    
                                    
% Case ii) Our task now is to find the lowest transfer from parent to kid
% that induces the kid to provide IC.

% Case ii)a) First corner solution. If kid cannot be moved by Qub, we 
% write in Qub to preserve continuity under the inside option (note that 
% if the threshold transfer for the kid to do IC is very close to the 
% upper bound for Q (Qub, the value at which the kid would give back 
% transfers or we enter WP region), then we are setting this threshold; 
% so continuity leaves Qub as the only reasonable option)
[SkQub,SpQub] = Surplus(QubOneBroke,dt,ll,1,OneBroke,par);
[SkQlb,SpQlb] = Surplus(QlbOneBroke,dt,ll,1,OneBroke,par);

if ParPower                     % Default: Parent has power.
    aCase = ~DoneOneBroke & (SkQub<=0); % Mark cases.
    % The upper bound is what happens under the inside option:
    QiInOneBroke(aCase) = QubOneBroke(aCase);
    % Set value function under inside option:
    VkInOneBroke(aCase) = VkOutOneBroke(aCase) + SkQub(aCase);
    VpInOneBroke(aCase) = VpOutOneBroke(aCase) + SpQub(aCase);
else                            % Alternative: Kid has power.
    aCase = ~DoneOneBroke & (SpQlb<=0); % Mark cases: Parent does not want
    % IC even at lowest transfer. 
    % The lower bound is what happens under the inside option:
    QiInOneBroke(aCase) = QlbOneBroke(aCase);
    % Set value function under inside option:
    VkInOneBroke(aCase) = VkOutOneBroke(aCase) + SkQlb(aCase);
    VpInOneBroke(aCase) = VpOutOneBroke(aCase) + SpQlb(aCase);
end

DoneOneBroke(aCase) = true;     % We are done with these cases.

% Case ii)b) Kid can already be moved by Qlb, the lowest possible
% transfer (2nd corner solution)
if ParPower                     % Default scenario: Parent has barg. power.
    bcase = ~DoneOneBroke & (SkQlb>=0);        % Then this is the kid's reservation transfer.
    bargOneBroke(bcase) = SpQlb(bcase)>0;      % We have a bargaining solution if parent's surplus
                                               % is positive.
    QiInOneBroke(bcase) = QlbOneBroke(bcase); % Set Qlb as transfer under inside option.
    % Set value functions under this option:
    VkInOneBroke(bcase) = VkOutOneBroke(bcase) + SkQlb(bcase);
    VpInOneBroke(bcase) = VpOutOneBroke(bcase) + SpQlb(bcase);
else                            % Alternative scenario: kid has power.
    bcase = ~DoneOneBroke & (SpQub>=0);  % Mark cases: Parent would even
                                % pay the highest-possible transfer.
    bargOneBroke(bcase) = SkQub(bcase)>0;% We have a bargaining solution  
                                         % if kid's surplus is positive.
    QiInOneBroke(bcase) = QubOneBroke(bcase); % Set Qub as transfer under inside option.
    % Set value functions under this option:
    VkInOneBroke(bcase) = VkOutOneBroke(bcase) + SkQub(bcase);
    VpInOneBroke(bcase) = VpOutOneBroke(bcase) + SpQub(bcase);
end
% gkOneBroke(bcase) = 0;        % Gifts are zero under this inside option
% gpOneBroke(bcase) = 0;        % by construction of Qlb and Qub.
DoneOneBroke(bcase) = true;

% Case ii)c) Interior case. Find root of kid surplus (parent surplus for
% robustness scenario with kid bargaining power).
%      A) First try if we can find an intersection on the linear part of
%      the parent's surplus function in closed form (for computational
%      efficiency) --> Omit this, let's see later if we have speed
%      problems.
%      B) Find root.

cCase = ~DoneOneBroke;              % These are all the remaining cases.
CCASE = false(N,N,1,Nw,Nw,1);       % Create 6D-indices that mark this case.
CCASE(OneBroke) = cCase;            % to give to Surplus.m.

if any(cCase(:))                   % If there are cases left to consider:
    % First, read out relevant variables for this case (do this once
    % outside the bisection loop to save time; variables will be used by
    % the nested function SurplusNest.m).
    kBrCC       =       kBr(CCASE);  pBrCC       =       pBr(CCASE);
    ekUncCC     =     ekUnc(CCASE);  epUncCC     =     epUnc(CCASE);
    ek0CC       =        ek(CCASE);  ep0CC       =        ep(CCASE);
    NetIncKidCC = NetIncKid(CCASE);  NetIncParCC = NetIncPar(CCASE); 
    Uck0CC      =       Uck(CCASE);  Ucp0CC      =       Ucp(CCASE);
    gkInCC      =      gkIn(CCASE);  gpInCC      =      gpIn(CCASE);
    WakIn0CC    =    WakIn0(CCASE);  WapIn0CC    =    WapIn0(CCASE);
    VakIn0CC    =    VakIn0(CCASE);  VapIn0CC    =    VapIn0(CCASE);
    mukIn0CC    =    mukIn0(CCASE);  mupIn0CC    =    mupIn0(CCASE);
    h1CC        =        h1(CCASE);  nnpArr1CC   =   nnpArr1(CCASE);
    VkIn0CC     =     VkIn0(CCASE);  VpIn0CC     =     VpIn0(CCASE);
    VkOutCC     =     VkOut(CCASE);  VpOutCC     =     VpOut(CCASE); 
    
    dx = 0.5*(QubOneBroke(cCase) - QlbOneBroke(cCase));
    x = QlbOneBroke(cCase) + dx;    % Find zero of the kid's surplus
    while any(abs(dx)>tolQ)         % by bisection method on [Tlb, Tub]
        dx = 0.5*dx;                % to find the reservation transfer
        [SSkk,SSpp] = SurplusNest(x); % of kid to provide IC.
        if ParPower                 % This is the standard scenario.
            x = x - sign(SSkk).*dx;     
        else                        % Alternative scenario where kid has
            x = x + sign(SSpp).*dx; % power: Find parent's reservation 
        end                         % transfer. 
    end                             
    if ParPower                     % This is a bargaining solution if      
        cBarg = (SSpp>0);           % the parent wants IC at this transfer.          
    else
        cBarg = (SSkk>0);           % (if kid wants IC in alternative).
    end
    bargOneBroke(cCase) = cBarg;    % If parent likes allocation at kid's 
                                    % threshold better than outside option,
                                    % we have a bargaining solution. 
    QiInOneBroke(cCase) = x;        % The root is our inside transfer.
    % gkOneBroke(cCase)  = 0;       % By construction of bounds, gifts are
    % gpOneBroke(cCase)  = 0;       % are again zero here. 
    % Update value functions for this case.
    VkInOneBroke(cCase)= VkOutOneBroke(cCase) + SSkk; 
    VpInOneBroke(cCase)= VpOutOneBroke(cCase) + SSpp;
end      

% For successful bargaining outcomes, read out gifts that would have been
% given anyway under Q=0: this is the altruistic component of Q.
QPaltrCompOneBroke =  gpOneBrokeQ0;
QKaltrCompOneBroke = -gkOneBrokeQ0;


% Now, write the solution for the one-broke cases into the larger arrays:
     ibarg(OneBroke) = bargOneBroke;    QiIn(OneBroke) = QiInOneBroke;
      gkIn(OneBroke) = gkOneBroke;      gpIn(OneBroke) = gpOneBroke;
QPaltrComp(OneBroke) = QPaltrCompOneBroke;
QKaltrComp(OneBroke) = QKaltrCompOneBroke;
      VkIn(OneBroke) = VkInOneBroke;    VpIn(OneBroke) = VpInOneBroke;

% Calculate consumption expenditures under the transfer Q for the inside
% option:
ekIn = kBr.*min(ekUnc,NetIncKid+QiIn-gkIn+gpIn) + (1-kBr).*ekUnc;
epIn = pBr.*min(epUnc,NetIncPar-QiIn+gkIn-gpIn) + (1-pBr).*epUnc;
% Get law of motion including the transfer Q:
akDotIn = NetIncKid - ekIn - gkIn + gpIn + QiIn;
apDotIn = NetIncPar - epIn + gkIn - gpIn - QiIn;
                                               
% Add output arrays to input structure:                              
ll.ibarg = ibarg;   ll.QiIn= QiIn;
% Hand over equilibrium consumption under bargaining solution:
ll.ekIn= ekIn;  ll.epIn= epIn;
% For gifts, now hand over full size:
ll.gkIn = gkIn;
ll.gpIn = gpIn;
% Also hand over altruistic component of Q (only take indices where
% recipient is broke, otherwise it is 0 anyway -- note that gifts cannot
% flow to non-broke owners in equilibrium.
qPaltrComp = min(QiIn(1,:,1,:,:),QPaltrComp(1,:,1,:,:));
QPaltrComp(1,:,1,:,:)=qPaltrComp;
ll.QPaltrComp= QPaltrComp;
qKaltrComp = max(QiIn(:,1,1,:,:),QKaltrComp(:,1,1,:,:));
QKaltrComp(:,1,1,:,:,:)= qKaltrComp;
ll.QKaltrComp=QKaltrComp;
ll.VkIn        = VkIn;       ll.VpIn    = VpIn;
ll.akDotIn     = akDotIn;    ll.apDotIn = apDotIn;
ll.mukIn0      = mukIn0;     ll.mupIn0  = mupIn0;

function [SkQ,SpQ] = SurplusNest(Q)
    % Surplus of kid and parent from inside over outside option for a 
    % transfer Q. This is the same function as Surplus.m. Created a nested
    % version since indexing multiple times during the bisection loop took
    % a lot of time. Q has to be in range [Qlb,Qub], which implies zero
    % gifts from both sides
    % Input:    Q, array of size s.ckUnc(ind) or scalar.
    % Output: two arrays of size of Q:
    %       SkQ:            Surplus of kid from inside over outside option
    %       SkP:            Surplus of parent.               
    ekQ  = kBrCC.*min(ekUncCC,NetIncKidCC+Q) + (1-kBrCC).*ekUncCC;
    epQ  = pBrCC.*min(epUncCC,NetIncParCC-Q) + (1-pBrCC).*epUncCC;
    UckQ = utilde( ekQ, h0, nnk      , par);
    UcpQ = utilde( epQ, h1CC, nnpArr1CC, par);
    duk = UckQ - Uck0CC; % difference of flow felicity with Q and with Q=0
    dup = UcpQ - Ucp0CC;
    dek = ekQ  -  ek0CC;  % difference of flow expenditures with Q and with Q=0
    dep = epQ  -  ep0CC;
    dgk = 0    - gkInCC;  % difference of flow gift with Q and with Q=0
    dgp = 0    - gpInCC;
    dVk =  (duk + alphak*dup - dek.*WakIn0CC - dep.*WapIn0CC ...
          +(dgk - dgp - Q).*mukIn0CC  )*dt;
    dVp =  (dup + alphap*duk - dep.*VapIn0CC - dek.*VakIn0CC ...
          -(dgk - dgp - Q).*mupIn0CC  )*dt;
    SkQ = (VkIn0CC - VkOutCC) + dVk;             
    SpQ = (VpIn0CC - VpOutCC) + dVp; 
        
end

end
