function lll = GetBKallocFloor2(ll,par,BKwp)
% Returns both players' transfers and realized expenditures when the
% parent has access to a means-tested consumption floor cMA (Medicaid).
% ProbMA returns the probability that this consumption floor is taken up 
% (convexification is governed by parameter par.kpa). Allows curvature
% of utility function to be different for the two agents. Wealth-pooling
% transfers are allowed (i.e. transfers to an unconstrained donor) if BKwp
% is set true, otherwise these are zero.
%
% Inputs:
% - ll: STRUCTURE with fields:
%     IncP:     (1,N,k,1 ,Nw,Nh) array. Parent's net income, where k is the
%               number of health dimensions handed over.
%     IncK:     (N,1,k,Nw,1 ,1 ) array. Child's net income. May have size 1
%               or k in health dimension.
%     epUnc:    (N,N,k,Nw,Nw,Nh) array. Parent's unconstrained expenditure.
%     ekUnc:    (N,N,k,Nw,Nw,Nh) array. Child's unconstrained expenditure.
%     thetak:   Scalar. Curvature of child's utility.
%     thetap:   (1,1,1,1, 1,Nh) array with curvature of parent's utility.
%     Ak:       Scalar. Marginal-utility shifter of kids,
%     Ap:       (1,1,1,1 ,1,Nh) array. Marginal-utility shifter of parents.
%               Does not depend on income state here since household size 
%               is always 1 for the disabled.
%     Vap:      (N,N,k,Nw,Nw,Nh) array. Parent's marginal value of assets.
%     Wak:      (N,N,k,Nw,Nw,Nh) array. Child's marginal value of assets.
%     nnk:      Scalar. Number of persons in child household.
%     hp:       (1,1,1,1 ,1 ,Nh) array. Parent's housing wealth.
%     valNoConsFeasible:
%               Scalar: consumption value that function assigns if no
%               positive consumption flows are feasible for the agents (when
%               both broke), e.g. sum of incomes is not enough to pay for
%               nursing home.
% par:          STRUCTURE with fields read out in beginning of function.
% BKwp:         Logical scalar. If wealth-pooling (WP) transfers are allowed.
%
% Output:
% The following fields are added to the input STRUCTURE ll:
%     gp:       (N,N,k,Nw,Nw,Nh) array. BK transfers from parent to child. 
%     gkNoMA:   (N,N,k,Nw,Nw,Nh) array. BK transfers from child to parent 
%               in the event that the parent does not take up MA.
%     epNoMA:   (N,N,k,Nw,Nw,Nh) array. Realized expenditure of parent after 
%               transfers outside MA.
%     ekNoMA:   (N,N,k,Nw,Nw,Nh) array. Realized expenditure of child after 
%               transfers in event that parent is outside MA.
%     ekMA:     (N,N,k,Nw,Nw,Nh) array with realized expenditure of child  
%               in event that parent inside MA.
%     ProbMA:   (N,1,1,Nw,Nw,Nh) array with probability of taking up the
%               floor .
%     gpsb:     (1,N,k,Nw,Nw,Nh) array. Parent's second-best gift (can be
%               negative, but parent cannot control kid's consumption).
%     gpstatfb: (1,1,k,Nw,Nw,Nh) array. Parent's first-best gift in static
%               altruism game when both are broke.
%     gksb:     (N,1,k,Nw,Nw,Nh) array. Kid's second-best gift (can be
%               negative, but kid cannot control parent's consumption).
%     gkstatfb: (1,1,k,Nw,Nw,Nh) array. Kid's first-best gift in static
%               altruism game when both are broke.
%     Ucp:      (N,N,k,Nw,Nw,Nh) array. Parent's flow felicity.
%     Uck:      (N,N,k,Nw,Nw,Nh) array. Kid's flow felicity.

epUnc= ll.epUnc;  ekUnc= ll.ekUnc;  IncP   = ll.IncP;    IncK   = ll.IncK; 
Ap   = ll.Ap;     Ak   = ll.Ak;     thetap = ll.thetap;  thetak = ll.thetak; 
Wak  = ll.Wak;   Vap   = ll.Vap;    hp     = ll.hp;   nnk    = ll.nnk;                              
valNoConsFeasible = ll.valNoConsFeasible;   % Read out inputs from ll.
pbar = par.pbar; taubar = par.taubar; Ns = par.Ns;

alphap = par.alphap;   alphak = par.alphak; % Read out altruism parameters,
Cma = par.Cma;      kpa = par.kpa;          % and Cma (the consumption floor)
h0  = par.h0;     h0 = par.h0;          % and grid parameters.
Nw = par.Nw;  N = par.N; hgrid = par.hgrid;                         
Nh = numel(hp);                             % Number of housing states we have
                                            % to consider: Is determined by
                                            % length of housing vector. For
                                            % formal care, only have to
                                            % consider renters and so it is
                                            % one.
                                            
[~,~,k,~,~,~] = size(epUnc);                % Read out k: how many health
                                            % dimensions were handed over.
uMA = utilde(Cma,h0,1,par);                 % Utility from consumption floor:
                                            % Don't get housing warm glow.

gpNoMA = zeros(N,N,Ns-1,Nw,Nw,Nh);   % Create zero arrays for parent's transfers.
gkNoMA = zeros(N,N,Ns-1,Nw,Nw,Nh);   % Kid's optimal gift if parent not in MA. 

if BKwp == true                    % If WP transfers are allowed:
    pBr = false(N,N,Ns-1,Nw,Nw,Nh);
    pBr(:,1,:,:,:,:) = 1;          % Indices where parent is broke.
    kBr = false(N,N,Ns-1,Nw,Nw,Nh);
    kBr(1,:,:,:,:,:) = 1;          % Indices where kid is broke.
    OneBroke = pBr|kBr;            % Indices where at least one is broke.
    NoneBroke= ~OneBroke;          % Indices where none is broke.
    kRich  = false(N,N,Ns-1,Nw,Nw,Nh);
    kRich(end,:,:,:,:,:)=1;        % Indices where kid has max wealth.
    pRich  = false(N,N,Ns-1,Nw,Nw,Nh);
    pRich(:,end,:,:,:,:)=1;        % Indices where parent has max wealth.
    OneRich = kRich|pRich;
    NoneRich= ~OneRich;            % Indices where none has max wealth
    wp      = NoneBroke & NoneRich;% Indices where neither broke nor wealthy:
                                   % This is where we allow WP gifts.
    % Call function QBounds with option BKwp=true in order to obtain gifts 
    % in region where no player is broke (note: in this case we have Qlb=Qub
    % for all quadrants, thus only return one output):
    [qWP,~] = QBounds(ll.tauk,ll.taup,ll.QbarL,ll.QbarU,pbar,taubar,BKwp);
    gpNoMA(wp)       = qWP(wp);
    gpNoMA(gpNoMA<0) = 0;          % Read out parent gifts which are positive qWPs.
    gkNoMA(wp)       = qWP(wp);
    gkNoMA(gkNoMA>0) = 0;          % Read out kid gifts which are negative qWPs.
    gkNoMA           = abs(gkNoMA);% Make them positive since we treat gifts as positive
end                                % numbers in budget constraints

% NOW, DO STANDARD REGIONS (WHERE ONE OF THE PLAYERS IS BROKE):

Vap1 =  Vap(1,:,:,:,:,:);               % Read out derivative.
% Get ratio rat=e'/e that parent desires when kid broke:  (1,N,1,Nw,Nw,Nh) array. 
ratp = (alphap*Ak).^(1/thetak) .* Ap.^(-1/thetap) .* Vap1.^(1/thetap - 1/thetak);
if any(Vap1(:)<0)                       % If any derivative is negative:
    ratp(Vap1<0) = 1;                   % In Consumption2.m, we had assigned a 
    ratp = real(ratp);                  % maximal consumption value for the
end                                     % parent. Say that the parent gives 
                                        % a gift that makes the kid abte to
                                        % consume that same amount. Have to
                                        % convert to real array since
                                        % Matlab has changed to complex
                                        % number in this case.

% Get ratio rat'=e/e' that kid desires when parent broke: (N,1,k,Nw,Nw,Nh) array.
Wak1 = Wak(:,1,:,:,:,:);                % Same treatment for kid.
ratk = (alphak*Ap).^(1/thetap) .* Ak.^(-1/thetak) .* Wak1.^(1/thetak - 1/thetap);
if any(Wak1(:)<0)
    ratk(Wak1<0) = 1;
    ratk = real(ratk);
end

% Get parent's first-best gift (may be negative, can control kid's 
% consumption): (1,N,k,Nw,Nw,Nh) array.
gpfb = ratp .* epUnc(1,:,:,:,:,:) - IncK(1,:,:,:,:,:);
% Get parent's gift when taking into account non-negativity constraint:
% This is the solution when parent is unconstrained. 
gpsb =  min( ekUnc(1,:,:,:,:,:) - IncK(1,:,:,:,:,:), gpfb );                                        
% Get parent's gift when taking into account non-negativity constraint:
% This is the solution when parent is unconstrained.
gpunc   = max( 0, gpsb );                   % Unconstrained transfers
                                            % parent to child (1,N,k,Nw,Nw,Nh)

% Get kid's first-best gift:   (N,1,k,Nw,Nw,Nh) array.                                           
gkfb =      ratk .* ekUnc(:,1,:,:,:,:) -  IncP(:,1,:,:,:,:);
% Get kid's second-best gift: (N,1,k,Nw,Nw,Nh) array.     
gksb = min( epUnc(:,1,:,:,:,:) -  IncP(:,1,:,:,:,:), gkfb);
% Get kid's actual gift, when kid is unconstrained
gkunc  = max( 0, gksb);                     % Unconstrained transfers
                                            % child to parent (N,1,k,Nw,Nw,Nh)
                                            
% Create variables where parent is broke: marker '0'.                                            
IncP0   =   IncP(:,1,:,:,:,:);              % (1,1,k,Nw,Nw,Nh) array.
% epUnc0  =  epUnc(:,1,:,:,:,:);              
Wak0    =    Wak(:,1,:,:,:,:);              % (N,1,k,Nw,Nw,Nh)

epThr = ( (1-thetap).*uMA./Ap ).^(1/(1-thetap)); % For renters, epThr=cMA.
                                            % Above this expenditure threshold 
                                            % parent stays out of MA:
                                            % (1,1,1,1,1 ,Nh) array. Only
                                            % depends on housing state.
gkThr = max(0, epThr-IncP0 );               % Minimal gift by child that 
                                            % makes parent stay out of MA:
                                            % (1,1,k,1,Nw,Nh) array.
                                            % Depends on both parent income
                                            % and parent health state.

% Now, get gifts that child gives outside MA: Either the minimal gift that
% keeps the parent outside MA or the unconditional child gift (if it is
% larger): 
gkNoMA(:,1,:,:,:,:)  = max( gkThr, gkunc ); % Matlab extends the parent 
                                        % threshold to correct dimension:
                                        % (N,1,k,Nw,Nw,Nh) array.
RHS =  alphak*utilde(IncP0+gkNoMA(:,1,:,:,:,:),hp,1,par) ...  
       - gkNoMA(:,1,:,:,:,:).*Wak0;     % Right-hand side of MA condition:
                                        % (N,1,k,Nw,Nw,Nh) array.
% From LTC paper: Convexify MA decision   
RHSneg = (RHS<0);                       % If RHS is negative, can get threshold
                                        % shock when moving to MA:
RHS(~RHSneg) = -1;                      % If positive: there is no shock 
                                        % to get kid prefer MA. Set RHS
                                        % negative to avoid complex
                                        % numbers in following step...:
curvRent = thetah(h0,par);              % Get curvature of utility function
                                        % for renters. Scalar.
K = log( RHS / (alphak*uMA) ) /kpa/(curvRent-1);
ProbMA = ( IncP0 < Cma ) .* normcdf( K );% Get prob. of sending to MA.
                                        % for negative RHS. 
                                        % (N,1,k,Nw,Nw,Nh) array.
ProbMA(~RHSneg) = 0;                    % Never send to MA if RHS is 
                                        % positive.
% Old: zero-one decision for MA:
% ProbMA = alphak*uMA > RHS;            % indifference condition.
                                            

ekMA   = ekUnc(:,1,:,:,:,:);            % Create arrays for equilibrium   
epNoMA = epUnc;                         % consumption under both contingencies:                                             
ekNoMA = ekUnc;                         % equals unconstrained as the default.
% gpNoMA = gpunc;                         % Set parent gifts outside MA to 
                                        % unconstrained gift by default.
                                        % Old's consumtion in MA is
                                        % always Cma, need not record.
epNoMA(2:end,1,:,:,:,:) = min( epUnc(2:end,1,:,:,:,:) , ...
                              gkNoMA(2:end,1,:,:,:,:) + IncP(:,1,:,:,:,:) );
% When kid broke and parent wealthy, parents give unconstrained transfer:                          
gpNoMA(1,:,:,:,:,:) = gpunc(1,:,:,:,:,:);                                            
% Kid eats what they want out of that:
ekNoMA(1,2:end,:,:,:,:) = min( ekUnc(1,2:end,:,:,:,:) , ...
                              gpNoMA(1,2:end,:,:,:,:) + IncK(1,:,:,:,:,:) );
                                            % Replace with BK values when
                                            % transfer recipient is broke.


% Now start case where both are broke:                                          
IncK00 = IncK(1,1,:,:,:,:);                 % Read out kid's income (1,1,1,Nw,1,1) array.
IncP00 = IncP(1,1,:,:,:,:);                 % and parent's (1,1,k,1,Nw,Nh) 
% ratk00 = ratk(1,1,:,:,:,:);                 % and the desired consumption
% ratp00 = ratp(1,1,:,:,:,:);                 % ratios: (1,1,k,Nw,Nw,Nh) arrays.

% % Get first-best static transfers: May be negative, both are broke. 
% % (1,1,k,Nw,Nw,Nh) arrays. THE FOLLOWING ARE ONLY CORRECT FOR THE CASE WITH
% % THE SAME CURVATURE (=RENTERS!)
% tolQ = 0.001;
% lb = (-IncP00 + valNoConsFeasible); %.*ones(1,1,k,Nw,Nw,Nh);
% ub = ( IncK00 - valNoConsFeasible);% .*ones(1,1,k,Nw,Nw,Nh);
% % val= (IncP00+IncK00>0);
% dx= (ub-lb)/2;
% xk= lb + dx;                % Initialize vector for kid...
% xp= xk     ;                % ... and parent.
% 
% %dx = 0.5*(ub(val) - lb(val));
% %x = lb(val) + dx;    % Find zero of the kid's surplus
% while any(abs(dx)>tolQ)         % by bisection method on [Tlb, Tub]
%     dx   = dx/2;                % to find the reservation transfer.
%     FOCk =  -Ak.*(IncK00-xk).^(-thetak) + alphak.*Ap.*(IncP00+xk).^(-thetap);
%                                             % Function for kid's static gift FOC
%     xk   = xk - sign(FOCk).*dx;     
%     FOCp =  -Ap.*(IncP00-xp).^(-thetap) + alphap.*Ak.*(IncK00+xp).^(-thetak);
%     xp   = xp - sign(FOCp).*dx;   % Update parent dictator gift.
% end 
% ??? TO DO: WRITE THIS INTO GPSTATFB ETC. AND USE IN LOOP BELOW. 
[gpstatfb,gkstatfb] = StatFBgift(IncP00,IncK00,Ap,Ak,thetap,thetak,alphap,alphak,...
                       valNoConsFeasible,0.001,[1,1,k,Nw,Nw,Nh]);
% This is for checking:                   
% [xp2,xk2] = StatFBgift(IncP00,IncK00,Ap,Ak,thetap,thetak,alphap,alphak,...
%                                  valNoConsFeasible,tolQ,[1,1,k,Nw,Nw,Nh]);
                   

% Transfers that would be given in static model (must be non-negative):
% (1,1,k,Nw,Nw,Nh) arrays.
gpstat = max( 0, gpstatfb );      
gkstat = max( 0, gkstatfb);                                               
                                            
% Note: Do a loop here and not the usual logical variables
% for cases where both are broke because:
% 1. If total dynasty income is negative there may by NaNs and infinities.
% 2. There are so many cases that some ifs help...
% 3. There are just few data points where both are constrained, so the
%    loop does not take that much time with Matlab's new engine.
%   BUT: THE CALLS TO fzero ARE REALLY COSTLY -- VECTORIZE THIS BEFORE?

for i=1:Nw                                  % Loop: all kid income levels.
    IncK_i = IncK00(1,1,1,i,1,1);           % Read out kid's income (does 
                                            % not depend on parent's
                                            % state).
    for j=1:Nh                              % Loop over housing levels.
        hp_j     =     hp(1,1,1,1,1,j);     % Read out housing and curvature
%        thetap_j = thetap(1,1,1,1,1,j);     % associated with it.
        
        for m=1:k                           % Loop over all health states.
            for n=1:Nw                      % Loop over parent income.
                % Read out at this point:
                IncP_mnj  =IncP00(1,1,m,1,n,j); % Parent's income, ...            
                gkThr_mnj = gkThr(1,1,m,1,n,j); % threshold transfer to go to MA
                                                % (does not depend on child's
                                                % income), ...
                ekUnc_minj= ekUnc(1,1,m,i,n,j); % and unconstrained 
                epUnc_minj= epUnc(1,1,m,i,n,j); % consumption for both.

                if IncK_i < gkThr_mnj       % If kid cannot pay transfer 
                                            % high enough that old can
                                            % afford nursing home (or that
                                            % she would prefer it to MA),
                    gpNoMA_minj = 0;        % set transfers to 
                    gkNoMA_minj = 0;        % zero (consumption floor is 
                                            % the only option).
                    ProbMA_minj = 1;        % MA happens for sure.
                    ekNoMA(1,1,m,i,n,j) = valNoConsFeasible; % Set both players' consumption
                                            % outside MA to very low level, but avoid
                                            % negative numbers --> won't be
                                            % chosen in equilibrium.
                    epNoMA(1,1,m,i,n,j) = valNoConsFeasible; % parent's consumption outside MA.
                      ekMA(1,1,m,i,n,j) = min(ekUnc_minj, IncK_i);

                else                        % Otherwise (usual case):
                    gpUncNoMA_minj= gpNoMA(1,1,m,i,n,j); % Read out unconstrained 
                    gkUncNoMA_minj= gkNoMA(1,1,m,i,n,j); % transfers,                  
                    gpStat_minj   = gpstat(1,1,m,i,n,j); % Checked: these give the same as fzero
                    gkStat_minj   = gkstat(1,1,m,i,n,j); % up to tolQ.
                    if isnan(gpStat_minj)||isnan(gkStat_minj)
                        break
                    end
                    if gpStat_minj>0        % If parent gave transfers in
                                            % static setting (this is an
                                            % upper bound on gifts): 
                        ParConstr_minj = (gpUncNoMA_minj + epUnc_minj > IncP_mnj);
                                            % Determine if parent is
                                            % constrained.
                        gpConstr_minj   = max( 0, min( gpStat_minj, ekUnc_minj-IncK_i) );           
                                            % Get parent's constrained
                                            % transfer.
                        gpNoMA_minj      = ParConstr_minj.*gpConstr_minj + (1-ParConstr_minj).*gpUncNoMA_minj;
                        gkNoMA_minj = 0;    % Assign correct transfer,
                        ProbMA_minj = 0;    % no MA here.

                    elseif gkStat_minj>0    % If child gave a static transfer:
                        gpNoMA_minj     = 0;    % Parent does not give transfer.

                        if gkUncNoMA_minj + ekUnc_minj <= IncK_i 
                                            % If unconstrained solution is feasible:
                            ProbMA_minj = ProbMA(1,1,m,i,n,j);  % Read out 
                            gkNoMA_minj = gkNoMA(1,1,m,i,n,j);  % unconstrained solution.
                        else                    % Otherwise look for constrained solution:
                            gkConstrNoMA_minj  = max( 0, min( gkStat_minj, epUnc_minj - IncP_mnj) );
                                                % Get kid's constrained
                                                % transfer if no MA.
                            gkNoMA_minj = max(gkThr_mnj, gkConstrNoMA_minj );
                                                % Impose the threshold.
                            RHS_minj = alphak*utilde(IncP_mnj+gkNoMA_minj,hp_j,1  ,par) ...
                                            + utilde(IncK_i  -gkNoMA_minj,h0,nnk,par) ...       
                                            + utilde(IncK_i              ,h0,nnk,par)    ;
                            K_minj = log( RHS_minj / (alphak*uMA) )/kpa/(curvRent-1);
                                                % Then, get probability of
                                                % sending to MA. kpa governs                                            
                            ProbMA_minj = ( IncP_mnj < Cma ) .* normcdf( K_minj );
                                                % strenght of smoothing here.                                            
                        end

                    else                        % In other cases (none of the 
                        gpNoMA_minj = 0;        % players gives static transfer),
                        gkNoMA_minj = 0;        % all transfers are zero.
                        ProbMA_minj = 1*(gkThr_mnj>0); % if threshold greater 0,
                    end                         % go to MA for sure, if not stay out.
                    
                                                % Get consumption for all usual
                                                % cases:
                    ekNoMA(1,1,m,i,n,j) = min(ekUnc_minj, IncK_i  +gpNoMA_minj-gkNoMA_minj);
                                                % Child's expenditure level
                    epNoMA(1,1,m,i,n,j) = min(epUnc_minj, IncP_mnj-gpNoMA_minj+gkNoMA_minj);  
                                                % Parent's consumption outside MA
                      ekMA(1,1,m,i,n,j) = min(ekUnc_minj, IncK_i);
                                                % Kid's expenditure in MA.
                end                                 

                ProbMA(1,1,m,i,n,j) = ProbMA_minj;   % Record MA choice.
                gkNoMA(1,1,m,i,n,j) = gkNoMA_minj;   % Fill in the equilibrium
                gpNoMA(1,1,m,i,n,j) = gpNoMA_minj;   % transfers.
            end
        end                                     % Close loops over four
    end                                        % states.
end

lll = ll;                                       % Copy input structure...
lll.ProbMA  = ProbMA;                           % and add the results as 
lll.gkNoMA = gkNoMA;  lll.gpNoMA = gpNoMA; % new fields.
% lll.gp      = gp;     % Took this out. It is understood that gifts must 
% be zero whenever MA is chosen.
lll.ekNoMA  = ekNoMA; lll.ekMA   = ekMA;    lll.epNoMA = epNoMA;
lll.gpsb     = gpsb;  lll.gksb   = gksb;   
lll.gpstatfb = gpstatfb;  lll.gkstatfb = gkstatfb;

lll.UckMA   = utilde(ekMA  , h0 , nnk,par);   % Flow felicity in MA.
lll.UcpMA   = utilde(Cma   , h0   ,   1,par);

lll.UckNoMA = utilde(ekNoMA, h0 , nnk,par);   % Flow felicity outside
lll.UcpNoMA = utilde(epNoMA, hgrid,   1,par);   % MA.
                                                                               