function ll = GetBKalloc2(ll,par,BKwp)
% Returns both players' transfers and realized stage-4 expenditures 
% given (net) income, unconstrained consumption levels. This is for
% the case where no Medicaid decision is to be made (e.g. parent is
% healthy, or decision against Medicaid has been taken). Allows curvature
% of utility function to be different for the two agents. Wealth-pooling
% transfers are allowed (i.e. transfers to an unconstrained donor) if BKwp
% is set true, otherwise these are zero.
%
% Inputs:
% STRUCTURE ll with fields:  
% IncP:     array with parent's net income (dimensions: N,N,1,Nw,Nw,Nh).
% IncK:     (N,N,1,Nw,Nw,Nh) array with child's net income.
% epUnc:    (N,N,1,Nw,Nw,Nh) array with parent's unconstrained expenditures 
%           for consumption plus rent. Equals cunc for home owners. 
% ekUnc:    (N,N,1,Nw,Nw,Nh) array with child's unconstrained consumption 
%           expenditures (kid never rents). 
% thetak:   Scalar. curvature of child's utility
% thetap:   (1,1,1,1,1,Nh) array with curvature of parent's utility.
% Ak:       Scalar. Marginal-utility shifter of kids,
% Ap:       (1,1,1,1,Nw,Nh) array. Marginal-utility shifter of parents.
% Vap:      (N,N,1,Nw,Nw,Nh) array. Parent's marginal value of savings, 
% Wak:      (N,N,1,Nw,Nw,Nh) array. Kid's marginal value of savings
% tauk:     Transfer motive.
% taup:     Parent transfer motive.  
% valNoConsFeasible:
%           Scalar: consumption value that function
%           assigns if no positive consumption flows are feasible for the
%           agents (when both broke), e.g. sum of incomes is not enough to
%           pay for nursing home.
% par:      STRUCTURE, containing the fields read out in the beginning of
%           this function.
% BKwp:     Logical scalar. If wealth-pooling (WP) transfers are allowed.
%
% Output:
% Input STRUCTURE ll with added/updated fields:
% gp:       (N,N,1,Nw,Nw,Nh) array. Transfer from parent to child.
% gk:       (N,N,1,Nw,Nw,Nh) array. Transfer from child to parent 
% ep:       (N,N,1,Nw,Nw,Nh) array. Realized stage-4 expenditures of
%           parent after transfers          
% ek:       (N,N,1,Nw,Nw,Nh) array. Realized consumption expenditures of 
%           child after transfers -- recall that kid never rents.
%           (Note: transfers are kept of full size with many zeros since 
%           convenient for addition in budget constraints etc. later).
% gpsb:     (1,N,1,Nw,Nw,Nh) array. Parent's second-best gift (can be
%           negative, but parent cannot control kid's consumption).
% gpstatfb: (1,1,1,Nw,Nw,Nh) array. Parent's first-best gift in static
%           altruism game when both are broke.
% gksb:     (N,1,1,Nw,Nw,Nh) array. Kid's second-best gift (can be
%           negative, but kid cannot control parent's consumption).
% gkstatfb: (1,1,1,Nw,Nw,Nh) array. Kid's first-best gift in static
%           altruism game when both are broke.
% Ucp:      (N,N,1,Nw,Nw,Nh) array. Parent's flow felicity.
% Uck:      (N,N,1,Nw,Nw,Nh) array. Kid's flow felicity.

epUnc = ll.epUnc; ekUnc = ll.ekUnc;  IncP   = ll.IncP;     IncK = ll.IncK;
Ap    = ll.Ap;    Ak    = ll.Ak;     thetap = ll.thetap; thetak = ll.thetak;
nnpArr= ll.nnpArr; nnk = ll.nnk;
valNoConsFeasible = ll.valNoConsFeasible;

alphap= par.alphap; alphak = par.alphak; % Read out altruism parameters
N = par.N;  Nw = par.Nw;  Nh = par.Nh;    % ...and grid sizes.                           
h0 = par.h0; pbar = par.pbar; taubar = par.taubar; hgrid = par.hgrid;
Cma = par.Cma;

gp = zeros(N,N,1,Nw,Nw,Nh);         % Set up zero arrays for transfers.
gk = zeros(N,N,1,Nw,Nw,Nh);       


if BKwp == true                     % If WP transfers are allowed:
    pBr = false(N,N,1,Nw,Nw,Nh);
    pBr(:,1,:,:,:,:) = 1;           % Mark indices where parent is broke.
    kBr = false(N,N,1,Nw,Nw,Nh);
    kBr(1,:,:,:,:,:) = 1;           % Mark Indices where kid is broke.
    OneBroke = pBr|kBr;             % Indices where at least one is broke.
    NoneBroke= ~OneBroke;           % Indices where none is broke.
    kRich  = false(N,N,1,Nw,Nw,Nh);
    kRich(end,:,:,:,:,:)=1;         % Indices where kid has max wealth.
    pRich  = false(N,N,1,Nw,Nw,Nh);
    pRich(:,end,:,:,:,:)=1;         % Indices where parent has max wealth.
    OneRich = kRich|pRich;          % Indides where on has max wealth.
    NoneRich= ~OneRich;             % Indices where none has max wealth
    wp      = NoneBroke & NoneRich; % Indices where none broke and none max wealth
    % Call function to get bounds in WP regions (note: Qlb=Qub):
    [qWP,~] = QBounds(ll.tauk,ll.taup,ll.QbarL,ll.QbarU,pbar,taubar,BKwp);
    gp(wp)  = qWP(wp);
    gp(gp<0)= 0; % read out parent gifts which are positive qWPs
    gk(wp)  = qWP(wp);
    gk(gk>0)= 0; % read out kid gifts which are negative qWPs
    gk      = abs(gk);  % make them positive since we treat gifts as positive
                           % numbers in budget constraints
end


% NOW, DO STANDARD REGIONS (WHERE ONE OF THE PLAYERS IS BROKE):

% Get ratio rat=e'/e that parent desires when kid broke:  (1,N,1,Nw,Nw,Nh) array. 
% Can simplify here now, but the old code was also correct:
ratp = (alphap*Ak./Ap).^(1/thetap);
% ratp = (alphap*Ak).^(1/thetak) .* Ap.^(-1/thetap) .* Vap(1,:,:,:,:,:).^(1/thetap - 1/thetak);
% Get ratio rat'=e/e' that kid desires when parent broke: (N,1,1,Nw,Nw,Nh) array.
ratk = (alphak*Ap./Ak).^(1/thetak);
% ratk = (alphak*Ap).^(1/thetap) .* Ak.^(-1/thetak) .* Wak(:,1,:,:,:,:).^(1/thetak - 1/thetap);
% Get parent's first-best gift (may be negative, can control recipient's 
% consumption): (1,N,1,Nw,Nw,Nh) array.
gpfb =      ratp .* epUnc(1,:,:,:,:,:) - IncK(1,:,:,:,:,:);
% Get parent's second-best gift (may be negative, but cannot control
% recipient's consumption):
gpsb = min( ekUnc(1,:,:,:,:,:) - IncK(1,:,:,:,:,:), gpfb);
% Get parent's gift when taking into account non-negativity constraint:
% This is the solution when parent is unconstrained.
gpunc  = max( 0, gpsb); 
                                            % Unconstrained transfers g_unc
                                            % parent to child (1,N,1,Nw,Nw,Nh)
% Get kid's first-best gift:   (N,1,1,Nw,Nw,Nh) array.                                           
gkfb =      ratk .* ekUnc(:,1,:,:,:,:) -  IncP(:,1,:,:,:,:);
% Get kid's second-best gift: (N,1,1,Nw,Nw,Nh) array.     
gksb = min( epUnc(:,1,:,:,:,:) -  IncP(:,1,:,:,:,:), gkfb);
% Get kid's actual gift, when kid is unconstrained
gkunc  = max( 0, gksb); 
                                               % Unconstrained transfers

gp(1,2:end,:,:,:,:) = gpunc(1,2:end,:,:,:,:); % Fill in unconstrained transfers 
gk(2:end,1,:,:,:,:) = gkunc(2:end,1,:,:,:,:); % when donor still has wealth

ep = epUnc;                                 % Create arrays for equilibrium
ek = ekUnc;                                 % expenditure: equals 
                                            % unconstrained as the default.
ep(2:end,1,:,:,:,:) = min( epUnc(2:end,1,:,:,:,:), ...
                            IncP( :   ,1,:,:,:,:)+gkunc(2:end,1,:,:,:,:) );
ek(1,2:end,:,:,:,:) = min( ekUnc(1,2:end,:,:,:,:), ...
                            IncK(1, :   ,:,:,:,:)+gpunc(1,2:end,:,:,:,:) );
                                            % Replace with BK values when
                                            % transfer recipient is broke.

% Now start case where both are broke: 
IncK00 = IncK(1,1,:,:,:,:);                 % Read out kid's income (1,1,1,Nw,1,1) array.
IncP00 = IncP(1,1,:,:,:,:);                 % and parent's (1,1,1,1,Nw,Nh) 
% ratk00 = ratk(1,1,:,:,:,:);                 % and the desired consumption
% ratp00 = ratp(1,1,:,:,:,:);                 % ratios: (1,1,1,Nw,Nw,Nh) arrays.

% Get first-best static transfers: May be negative, both are broke. 
% (1,1,1,Nw,Nw,Nh) arrays.
% gpstatfb = (IncP00+IncK00) .* ratp00./(1+ratp00) - IncK00;
% gkstatfb = (IncP00+IncK00) .* ratk00./(1+ratk00) - IncP00;

[gpstatfb,gkstatfb] = StatFBgift(IncP00,IncK00,Ap,Ak,thetap,thetak,alphap,alphak,...
                        valNoConsFeasible,0.001,[1,1,1,Nw,Nw,Nh]);
% Should get the same here for renters: works.                    
% squeeze( xp2(1,1,1,:,:,1) - gpstatfb(1,1,1,:,:,1) )
% squeeze( xk2(1,1,1,:,:,1) - gkstatfb(1,1,1,:,:,1) )                    
% Transfers that would be given in static model (must be non-negative):
% (1,1,1,Nw,Nw,Nh) arrays.
gpstat = max( 0, gpstatfb );      
gkstat = max( 0, gkstatfb );      
                                           

% No need to loop over disabled and healthy here: Only allow one input at a
% time!!
for i=1:Nw                                  % Loop over kid's income levels.
    Inck_i = IncK(1,1,1,i,1,1);             % Read out kid's income, which 
                                            % depends only on kid's prod.
    for k=1:Nw                              % Loop over all pension levels of old.
                                            % of parent.
        for j=1:Nh                          % Loop over all housing states.
            
            Incp_kj = IncP(1,1,1,1,k,j);    % Read out parent income, which 
                                            % depends only on parent's
                                            % state.
        
            if Incp_kj+Inck_i<Cma           % If both incomes together are less 
                                            % than the consumption floor:
                ep_ikj = valNoConsFeasible; % negative (i.e. not enough to 
                ek_ikj = valNoConsFeasible; % formal care, leave both with
                gp_ikj = 0;                 % very low consumption --> will
                gk_ikj = 0;                 % choose MA because u(0)~-inf.
                
            else                            % Otherwise:                                                         
                gpstat_ikj = gpstat(1,1,1,i,k,j); % Read out static 
                gkstat_ikj = gkstat(1,1,1,i,k,j); % transfers.                
                gpunc_ikj  =  gpunc(1,1,1,i,k,j); % Read out unconstrained 
                gkunc_ikj  =  gkunc(1,1,1,i,k,j); % transfers, ...
                
                eunck_ikj  =  ekUnc(1,1,1,i,k,j); % ... and unconstrained 
                euncp_ikj  =  epUnc(1,1,1,i,k,j); % consumption.
                if gpstat_ikj>0             % If parent gave transfers in
                                            % static setting (this is an
                                            % upper bound on g): 
                    ParConstr = (gpunc_ikj + euncp_ikj > Incp_kj);
                                            % Determine if parent is
                                            % constrained.
                    gpConstr   = max( 0, min( gpstat_ikj, eunck_ikj-Inck_i) );           
                    epConstr   = Incp_kj - gpConstr;
                    gp_ikj     = ParConstr.*gpConstr + (1-ParConstr).*gpunc_ikj;
                    ep_ikj     = ParConstr.*epConstr + (1-ParConstr).*euncp_ikj;
                    ek_ikj    = min(eunck_ikj,Inck_i+gp_ikj);
                    gk_ikj    = 0;
                elseif gkstat_ikj>0         % If child gave a static transfer:
                    KidConstr = (gkunc_ikj + eunck_ikj > Inck_i);
                    gkConstr  = max( 0, min( gkstat_ikj, euncp_ikj - Incp_kj) );
                    ekConstr  = Inck_i - gkConstr;
                    gk_ikj    = KidConstr.*gkConstr + (1-KidConstr).*gkunc_ikj;
                    ek_ikj    = KidConstr.*ekConstr + (1-KidConstr).*eunck_ikj;
                    ep_ikj    = min(euncp_ikj,Incp_kj+gk_ikj);
                    gp_ikj    = 0;
                else                        % In other cases:, no transfer:
                    gp_ikj    = 0;          % no transfers flow, consump-
                    gk_ikj    = 0;          % tion obtained in usual way
                    ep_ikj    = min(euncp_ikj,Incp_kj );
                    ek_ikj    = min(eunck_ikj,Inck_i);
                end

            end
            gp(1,1,1,i,k,j) = gp_ikj;       % Fill in the equilibrium 
            gk(1,1,1,i,k,j) = gk_ikj;       % transfer and consumption 
            ep(1,1,1,i,k,j) = ep_ikj;       % levels for this grid point.
            ek(1,1,1,i,k,j) = ek_ikj;
        end       
    end
end

% Add results as fields in output structure:
ll.ep       =       ep;  ll.ek       =       ek;
ll.gp       =       gp;  ll.gk       =       gk;
ll.gpsb     =     gpsb;  ll.gksb     =     gksb;
ll.gpstatfb = gpstatfb;  ll.gkstatfb = gkstatfb;

ll.Ucp = utilde( ep, hgrid, nnpArr, par);  % Get flow felicity of both.
ll.Uck = utilde( ek, h0 , nnk   , par);

