function out = ExtrapValFctCRRA(V0,c0,b,outa,g,A)
% Extrapolates a CRRA value function based on the assumption that
% consumption is a linear function. The utility function is
% u(c) = A*c(1-g)/(1-g)   , or log(c) for g=1
% A is a utility shifter, i.e. a constant (default: 1). The first-order
% condition for consumption is:
% u'(c) = A*c^(-g) = V'(a),
% where V(a) is the value at asset level a. We can use this to integrate:
%%
% $$
% V({a_1}) = V({a_0}) + \int_{a_0}^{a_1} V'(a) da  
%       = V_0    + \int_{a_0}^{a_1} A*c(a)^{-g} da 
%       = V_0    + A \int_{a_0}^{a_1} [c_0+b*(a-{a_0})]^{-g} da 
%       = V_0    + A*[ (c_0+b*outa)^{1-g} - c_0^{1-g} ] / (b*(1-g)) .    (CRRA)
% $$
%%
% where ${a_1} = {a_0} + outa$. For g=1 (log-utility), we obtain
% V({a_1}) = V0    + A*[ ln(c_0+b*outa) - ln(c_0) ] / b .                   (LOG)
%
%same size or scalar.
% V0:       value function value at outermost grid point {a_0}
% c0:       Consumption at outermost grid point. (Note: when using the 
%           backward difference we obtain consumption somewhere between the 
%           outermost grid point and the grid point before. Here we require
%           a better approximation to consumption at the outermost grid point.)
%           Approximate by c(N-1/2) + ( c(N-1/2) - c(N-3/2) )/2,
%           where c(N-1/2) is consumption as calculated from the slope
%           between grid points N and N-1.
% b:        slope of consumption function at {a_0}. Consumption is linearly
%           extrapolated as c({a_1}) = c_0 + b*({a_1}-{a_0}).
% outa:       distance of extrapolated wealth grid point a to {a_0}: 
%           outa = a_1 - {a_0}
% g:        risk aversion parameter in CRRA utility function.
% A:        utility shifter in CRRA utility: u(c) = A*c^(1-g)/(1-g).
%           Default is 1 (standard CRRA)
%
% Output:   array of same size as inputs. extrapolated value function at 
%           V({a_0}+outa)

if nargin<6                     % set utility shifter to default if not 
    A=1;                        % handed over.
end

if g==1                         % return result:
    out = V0 + A.* ( log(c0+b.*outa) - log(c0) ) ./ b;
else                            % see equation (LOG) above
    out = V0 + A.* ( (c0+b.*outa).^(1-g) - c0.^(1-g) )./b./(1-g); 
end                             % see equation (CRRA)

