function out  = ContinuationValue2(hazAkp,haz345,F,DeathVal,dt,par)
% Compute expected value of function 'F' tomorrow using Markov process on 
% state space with transition hazards in 'h', a value from dying 
% 'DeathValue' and a time increment 'dt', given model parameters 'par'.
%
% Inputs:
% hazAkp:       (N,N,Ns,Nw,Nw,,Nh,4) array. Hazard rates of moving up and 
%               down in assets due to savings and Brownian Motion. Created
%               by JumpHazards2.m. First 6 dimension refer to the usual
%               state space, dimension 7 tells us which hazard is meant:
%               dim. 7: index 1: akUp, hazard that kid's assets go up.
%                       index 2: akDn, kid's assets down.
%                       index 3: apUp, parent's assets up.
%                       index 4: apDn, parent's assets down.
% haz345:       Structure with fields containing hazard rates:
%                       Med:        medical shock to parent
%                       LtcBasic:   disabled, need basic care services
%                       LtcSkill:   disabled, need skilled care services
%                       Death:      death
%                       All hazards are (Ns,Nw,Nw) arrays. Dimensions:
%                       dim. 1: disability (healthy, LTC basic, LTC sk.)
%                       dim. 2: kid's productivity
%                       dim. 3: parent's productivity
% F:            (N+2,N+2,Ns,Nw,Nw,Nh) array. Value function tomorrow 
%               for the case that parent and kid are both alive, extended
%               by one grid point for the asset dimensions.
% DeathValue:   (N,N,Ns,Nw,Nw,Nh) array or scalar. Value function when old dies.
% dt:           scalar. The time increment.
% par:          STRUCTURE. Contains model parameters
%
% Output:
% Array of dim. (N,N,Ns,Nw,Nw,Nh) with expected value tomorrow (after time
% interval of dt) for each grid point.


N = par.N;   Nw = par.Nw;               % Read out parameters from 'par'.
Nh = par.Nh; Ns = par.Ns;
epsHaz1D = par.epsHaz1D;                % (Nw,Nw) matrix with hazards of 
                                        % productivity transitions for
                                        % child.
medProb  = par.medProb;                 % (N,N) sparse matrix with proba-
                                        % bilities of asset transitions 
                                        % conditional on medical shock off
                                        % the diagonal, and the negative of
                                        % the probability that the current
                                        % state is left on the diagonal.
                                        
FF         = F(2:end-1,2:end-1,:,:,:,:);% Read out the inner part of F, 
                                        % cutting out the enlargement in
                                        % dimensions 1 and 2.
                                        
if numel(DeathVal)<numel(FF)            % Extend DeathVal to (N,N,Ns,Nw,Nw,Nh)
    DeathVal = DeathVal.*ones(N,N,Ns,Nw,Nw,Nh);
end                                     % array if not of full size.
                                        
% Create "jump terms in savings" value of jumps to adjacent grid points in wealth.
jumpSavArr =  hazAkp(:,:,:,:,:,:,1) .* F(3:end  ,2:end-1,:,:,:,:) ... % kid up
            + hazAkp(:,:,:,:,:,:,2) .* F(1:end-2,2:end-1,:,:,:,:) ... % kid down
            + hazAkp(:,:,:,:,:,:,3) .* F(2:end-1,3:end  ,:,:,:,:) ... % parent up
            + hazAkp(:,:,:,:,:,:,4) .* F(2:end-1,1:end-2,:,:,:,:) ... % parent down
            - sum(hazAkp,7)         .* FF;           % hazard of leaving.                                       
                                 

% Now, go on to jumps in other dimensions. Do this by fast Kronecker matrix
% multiplaction: Takes advantage of the tensor structure, i.e. that we are
% applying linear maps (the sub-matrices of the Kronecker product)
% separately to different dimensions of the (N,N,Ns,Nw,Nw,Nh)-dimensional state
% space.
                                        
% Medical shocks:                                        
hazMatMed345 = Arr2DiagSparseMat(haz345.Med);   
                                        % Create square diagonal matrix 
                                        % (Ns*Nw*Nw rows) with hazard of
                                        % receiving a medical shock on
                                        % the diagonal.
jumpMed    = kronm( { N, medProb, hazMatMed345, Nh } , FF(:)        );
% Past line calculates the N*N*Ns*Nw*Nw*Nh square Kronecker matrix times FF(:):
% [ identity(Nh) kron hazMatMed345 kron medProb kron identity(N) ] * FF(:)
                                        % Medical spending shock: identity
                                        % matrix in first dim.,
                                        % probabilities of being knocked
                                        % down in dim. 2 (minus probality of
                                        % leaving current state on diagonal 
                                        % of that matrix), hazard of
                                        % medical shock in dim. 3-5, and
                                        % another identity matrix in
                                        % housing dimension since this
                                        % doesn't matter for this hazard.
% LTC shocks:      
hazMatLTCbasic345 = HazMatLTC345( haz345.LtcBasic, 2 ); % Basic care: jump to health index 2,
hazMatLTCskill345 = HazMatLTC345( haz345.LtcSkill, 3 ); % skilled care: jump to index 3.

% hazMatLTC345 = HazMatLTC345( haz345(:,:,:,2) );
                                        % Create transition matrix in
                                        % dimensions 3-5 due to LTC
                                        % shock by extra function.
                                        % Do the same as for medical shocks
jumpLTCbasic = kronm( { N^2       , hazMatLTCbasic345, Nh  } , FF(:)  );
jumpLTCskill = kronm( { N^2       , hazMatLTCskill345, Nh  } , FF(:)  );
% jumpLTC    = kronm( { N^2       , hazMatLTC345, Nh  } , FF(:)         );

% [ identity(Nh) kron hazMatLTC345 kron identity(N^2) ] * FF(:)
                                        % Kronecker-multiply to get jumps
                                        % due to these transitions:
% Kid's productivity shocks:
jumpEps    = kronm( { N^2 *Ns      , epsHaz1D, Nw*Nh } , FF(:)       );
% [ identity(Nw*Nh) kron epsHaz1D kron identity(Ns*N^2) ] * FF(:);
                                        
% Parent's death shock:
hazMatDth345 = Arr2DiagSparseMat(haz345.Death);
                                        % Create square diagonal matrix
                                        % (Ns*Nw*Nw rows) with death hazard
                                        % on diagonal.
jumpDthLoss = kronm( { N^2       , hazMatDth345, Nh  } , FF(:)       );
% [ identity(Nh) kron hazMatDth345 kron identity(N^2) ] * FF(:)
                                        % Then calculate loss from
                                        % dying,...
jumpDthGain = kronm( { N^2       , hazMatDth345, Nh  } , DeathVal(:) );
                                        % ...and the gain: death value.

                                        
% Add all jump terms to the current value (all in vectorized form):
out = FF(:) + dt*(jumpSavArr(:) + jumpMed + jumpLTCbasic + jumpLTCskill ...
                  + jumpEps - jumpDthLoss + jumpDthGain                    );     

out = reshape(out,[N,N,Ns,Nw,Nw,Nh]);   % Hand over result as 6D array. 

