function lll = Consumption2(ll,par,opt)
% Given the value function (ll.Vold, ll.Wold), get optimal (unconstrained) 
% consumption expenditures for both players (lll.epUnc and lll.ekUnc), the 
% derivatives of the value function (lll.Vap/Vak/Wap/Wak) and extrapolated
%  value-function arrays for updating (lll.VIold/WIold).
%
% Input:
% ll:  STRUCTURE WITH FIELDS:
% Vold, Wold:       Arrays (N,N,Ns,Nw,Nw,Nh) with old value function for 
%                   parent (Vold) and child (Wold).
% nnp:              Array (N,N,Ns,Nw,Nw,Nh) or scalar with number of persons 
%                   in parent household.
% nnk:              Scalar. number of persons in kid household.
% par: STRUCTURE that contains parameters read out in first code lines:
%
% Output: STRUCTURE lll with fields:
% epUnc,ekUnc:      (N,N,Ns,Nw,Nw,Nh) arrays with stage-4 consumption that
%                   parent/kid would choose if unconstrained.
% Vap,Vak,Wap,Wak:  (N,N,Ns,Nw,Nw,Nh) arrays. Derivatives of value
%                   functions.
% VIold, WIold:     (N+2,N+2,Ns,Nw,Nw) arrays. Extrapolated value-function
%                   arrays that will be used for value-function updating.

Vold = ll.Vold;    Wold = ll.Wold;      % Read out from ll: value functions
nnp  = ll.nnpArr;  nnk  = ll.nnk;       % and household sizes.
IncP = ll.NetIncPar;  IncK = ll.NetIncKidFC;

dak = par.dak;  % Read out grid sizes from par.
dap = par.dap; N = par.N; Nw = par.Nw; Nh= par.Nh; Ns = par.Ns;
hgrid = par.hgrid;  h0 = par.h0;    % Also house sizes.
ForwCrossDer = opt.ForwCrossDer;  BackwOwnDer = opt.BackwOwnDer;

if isscalar(nnp)                        % If parent household sizes handed
    nnp = nnp * ones( size(Vold) );     % over as scalar, expand it to
end                                     % full size.



Wak = GetCentDer1(Wold,dak); % derivatives.
Vap = GetCentDer2(Vold,dap);


ekUnc = Duinvtilde(Wak,h0,nnk,par);   % Consumption for kid:
if any(Wak(:)<0)                        % Make them consume a maximum
    ekMax = 5*IncK.*ones(size(Wak));    % amount if value of savings negative.
    ekUnc(Wak<0) = ekMax(Wak<0);
end

epUnc = Duinvtilde(Vap,hgrid,nnp,par);  % Same for parent consumption.
if any(Vap(:)<0)
    epMax = 5*IncP.*ones(size(Vap));
    epUnc(Vap<0) = epMax(Vap<0);
end

if BackwOwnDer                          % If backward derivative is asked
    Wak = GetBackwDer1(Wold,dak);       % for, call backward function for
    Vap = GetBackwDer2(Vold,dap);       % kid and parent.
end

if ForwCrossDer                        % If forward is asked for, call
    Wap = GetForwDer2(Wold,dap);        % forward derivative for parent
    Vak = GetForwDer1(Vold,dak);        % and child in other's assets.
else                                    % Otherwise, centered derivative.
    Wap = GetCentDer2(Wold,dap,approx3pt);
    Vak = GetCentDer1(Vold,dak,approx3pt);
end


% Extended value functions:
WIold                      = zeros(N+2,N+2,Ns,Nw,Nw,Nh);
WIold(2:N+1,2:N+1,:,:,:,:) = Wold;      % Now has N+2 rows and columns.
WIold(N+2  ,2:N+1,:,:,:,:) = Wold(end,:,:,:,:,:) + dak(end)*Wak(end,:,:,:,:,:);
WIold(2:N+1,N+2  ,:,:,:,:) = Wold(:,end,:,:,:,:) + dap(end)*Wap(:,end,:,:,:,:);

VIold   = zeros(N+2,N+2,Ns,Nw,Nw,Nh);   % Have 0's below zero wealth.
VIold(2:N+1,2:N+1,:,:,:,:) = Vold;      % Now has N+2 rows and columns.
VIold(N+2  ,2:N+1,:,:,:,:) = Vold(end,:,:,:,:,:) + dak(end)*Vak(end,:,:,:,:,:);
VIold(2:N+1,N+2  ,:,:,:,:) = Vold(:,end,:,:,:,:) + dap(end)*Vap(:,end,:,:,:,:);

% Finally, hand over results:
lll = ll;                               % Write output into structure lll:
lll.ekUnc = ekUnc;  lll.Wap = Wap;  lll.Wak = Wak;  lll.WIold = WIold;
lll.epUnc = epUnc;  lll.Vap = Vap;  lll.Vak = Vak;  lll.VIold = VIold;


end
