function [Z, hPolLin] = BuyHouseValue(Zhous,par,OLGiter)
% Returns the value for a 65-year old who just entered retirement without a
% house and the house-size policy function (i.e. which house size is bought
% conditional on buying). We let everybody buy a house here who can,
% renters will then immediately sell it off in the first time period. This
% is more parsimoniuous. The following approach worked best: We find the
% best house-size policy in the class of affine functions of wealth,
% interpolating the after-purchase values by 2D spline interpolation. 
% 
% Inputs:
% Zhous:   (N,1,Nw,Nh) array. Value of 65-year old parent (healthy) who
%          has chosen a house (for all sizes).
% par:     Structure. Contains model parameters.
% OLGiter: Scalar integer, from 0 to maxIterVF. 0 means that we just come
%          out of the value-function guess (i.e. alone loop), then count
%          OLG iterations 1,2... Is used to label figures.
%
% Output:
% Z:            (N,1,Nw,1) array. Value of entering retirement with given 
%               productivity and assets and buying the best house available
%               (and renting if buying is impossible).
% hPolLin:      (1,Nw) cell containing function handles. The optimal house
%               size bought (conditional on buying) in the class of affine
%               functions of wealth.


% Read out inputs from structure par:
N    = par.N;     Nw = par.Nw;  Nh   = par.Nh; 

hmin   = par.hvec(2  );                 % Minimal house size.
akgrid = par.avec(:);                   % Kid asset vector: N-by-1.

optInterp = 'spline';                   % Could change the interpolation
                                        % method here. Spline works best.

hval = log(par.hvec);                   % Log of houses: linearly spaced.
                                        % We interpolate in log of housing,
                                        % yields better results than to
                                        % interpolate in h itself.

hngrid = reshape(hval(2:end),[1,Nh-1]); % Grid vector for houses that can
                                        % be bought on grid:
                                        % 1-by-(Nh-1) vector. 
                                        
drawnow nocallbacks                     % Don't let user interfere with figure 
                                        % window while this function runs,
                                        %  show results right away.

%% Find best policy within the class of affine functions in assets:
Nalpha=100;                                 % Grid size for intercept and
Nbeta =100;                                 % slope parameters.
beta = linspace( 0.1,0.5,Nbeta) ;           % Slope of policy: 1-by-Nbeta.
% The ratio of maximal house size over maximal asset position,
% hvec(Nh)/aup, determines the maximal beta for which value functions are
% well-defined. If houses grow to large under a policy, interpolation
% cannot assign a value and we return NaN (extrapolation is instable!).
% Thus, going up to 0.5 here is reasonable.
alpha= linspace(-250,500,Nalpha)';          % Intercept of policy: Nalpha-by-1
% Create vector with asset grid points for which buying is possible:
afford = par.avec>hmin;           % 1-by-N logical vector if one
                                            % can buy (after paying penalty
                                            % in counterfactual).
Naff = sum(afford);                         % Number of grid points for which                    
                                            % one can buy house.
firstAff = N-Naff+1;                        % Index of first grid point 
                                            % at which one can afford a
                                            % house.
avec3 = reshape(par.avec(afford),[1,1,Naff]);   
                                            % (1,1,Naff) array with points
                                            % for which the asset levels we
                                            % want to get the housing
                                            % policy for.
rawHPol = alpha+beta.*avec3;                % Create (Nalpha,Nbeta,N) array
                                            % with "raw" housing policy:
                                            % Which house one would buy
                                            % under a given alpha-beta rule
                                            % for given assets avec3.
HPol = max(hmin,min(avec3,rawHPol));        % Impose that policy cannot be
                                            % more than all-in and not less
                                            % than minimum house size.
HPolStar = zeros(Naff,Nw);                  %  Array for optimal housing size under linear policy   
logHPol = log(HPol);                        % Get log of housing for
                                            % interpolation.                                         
AsalePol = avec3 - HPol;                    % (Nalpha,Nbeta,N) array with 
                                            % after-sales assets of buyer.
ASalePolStar = zeros(Naff,Nw);              % After-house-purchase financial wealth under linear policy
VPol = zeros(Nalpha,Nbeta,Naff,Nw);         % Set up array for the after-
                                            % purchase value of the
                                            % different policies, for all
                                            % Nw productivity levels on
                                            % dim. 4.
Zrent = Zhous(:,1,:,1);                     % Value function for renters N-by-1-by-Nw.
Z     = Zrent;                              % Set function output to value of
                                            % renting by default. Will
                                            % over-write this for buyers.
critPol = zeros(Nalpha,Nbeta,1,Nw);         % Set up array that gives us the
                                            % criterion (=avg.val.) for
                                            % each policy (for all prod.).
figure('Name',sprintf('Iter.-%1.0f house-buying policies: Avg. value across asset grid points',OLGiter));
alphaMat= alpha.*ones(1,Nbeta);             % Set up Nalpha-by-Nbeta matrices
betaMat = beta.*ones(Nalpha,1);             % to read out best policy later.
alphaStar = zeros(Nw,1);                    % Set up vectors for policies:
betaStar  = zeros(Nw,1);                    % One coefficient for each prod.
hPolLin  = cell(1,Nw);                      % Set up cell array for function
                                            % handles with the optimal
                                            % policy.

for i=1:Nw                          % Loop over all productivity levels.
    Viin = squeeze( Zhous(:,1,i,2:Nh) );    % Read out after-purchase values
                                            % for interpolation.
    VPol(:,:,:,i) = interp2(hngrid,akgrid,Viin,logHPol,AsalePol,optInterp,NaN);
                                            % As above, interpolate in 2D
                                            % to obtain the after-purchase
                                            % values for all possible
                                            % policies for prod. i.
    critPoli = sum(VPol(:,:,:,i),3);        % Obtain criterion: sum over all
                                            % values at all asset levels
                                            % (is proportional to average):
                                            % Nalpha-by-Nbeta matrix.    
    critPol(:,:,1,i) = critPoli;            % Record this as the criterium.
    [~,indi] = max(critPoli(:));            % Find (linear) index of best
                                            % policy on the alpha-beta
                                            % grid.
    alphaStar(i) = alphaMat(indi);          % Write optimal coeffisients 
    betaStar(i)  =  betaMat(indi);          % into star-vectors...
    hPolLin{i} = @(a) (a>=hmin).*min(a,max(hmin,alphaStar(i)+betaStar(i)*a));
                                            % ... and record the policy
                                            % function. Rent with assets
                                            % below hmin, then take affine
                                            % policy subject to feasibility
                                            % bounds for higher assets.
    % Obtain value of buying optimal house:
    hPolLini = hPolLin{i}(avec3(1,1,:));    % Obtain optimal policy: 
                                            % (1,1,Naff) array.
    HPolStar(1:Naff,i)= squeeze(hPolLini);  % Write into matrix.
    ASalePolStar(1:Naff,i) = akgrid(firstAff:end)-HPolStar(:,i);
                                            % Obtain after-purchase assets.
    Z(firstAff:end,1,i) = interp2(hngrid,akgrid,Viin,log(HPolStar(:,i)),ASalePolStar(:,i),optInterp,NaN);
                                            % Interpolate to get value of
                                            % buying and write into output
                                            % array (leave renting value
                                            % for those who can't afford
                                            % house).
    % Make contour plot of the criterion in alpha-beta space to check if 
    % we have a nice problem:
    subplot(1,Nw,i)                         
    contour(beta,alpha,critPol(:,:,1,i),'ShowText','on');
    hold on, plot(betaStar(i),alphaStar(i),'xk');   % Mark optimal choice.
    xlabel('slope (beta)'),     ylabel('intercept (alpha)')
    xlim(beta([1,Nbeta])),      ylim(alpha([1,Nalpha]))
    title(sprintf('Avg. value of policy for prod.%1.0f',i))
end


%% PLOT HOUSING POLICY FUNCTION:
figure('Name',sprintf('Iter.%1.0f house-buying rules',OLGiter))
hold on                     % Figure zooms in on housing choices by produdctivity
% pp = gca;
% COMPARED THE AFFINE-POLICY APPROACH WITH THE SEPARATE MAXIMIZATION
% APPROACH: RESULTS VERY SIMILAR IN OWNING REGIONS, IN RENTING REGIONS OR
% CLOSE TO INDIFFERENCE AFFINE POLICY LOOKS MUCH BETTER.
% pp.ColorOrderIndex = 1;     % Set color index so that each prod.level has
% for i=1:Nw                  % a fixed color. First, plot policies we got
%     ipol = hpol{i};         % by separately maximizing for each asset level.
%     plot(aipol,ipol(aipol),'*--')
% end
akPlot = linspace(hmin,par.aup,100);
% pp.ColorOrderIndex = 1;     % Second, plot in straight lines the best affine
for i=1:Nw                  % rules we found in the second approach.
    plot(akPlot,hPolLin{i}(akPlot),'-')
end
plot(akgrid,akgrid,'--k',akgrid,hmin*ones(size(akgrid)),'--k')
                            % Finally, mark the feasible cone: House at
                            % least of size hmin, but maximally can go
                            % all-in.
%xlim([hmin 800]), ylim([hmin 400])
legend('Prod 1','Prod 2','Prod 3','Prod 4','Feasibility bounds','Location','northwest')
xlabel('assets before housing choice')
ylabel('housing choice')    % Edit figure.
title('Housing policy function (conditional on buying)')


    

