% Solves illustrative ("toy") model with long-term care (LTC) from Paper
% Appendix B.2 ("Illustrative model with LTC"). This code adds LTC to
% ToyModelCommDevice.m.
%
% Runtime: ~1 second.
% Needed toolboxes: None (ie. only basic Matlab).
%
% TO CREATE FIGURE B.2 in paper appendix (title: "Payoffs from dt-allocations
% involving IC (a^p_t =1,illustrative model with LTC"):
% Run with default parameters and option LRplanner=false.
%
% TO CREATE FIGURE B.3 in paper appendix (title: "Planner’s solution in 
% illustrative model with LTC"):
% Set plotLRplanner=true and use default parameters.

close all; clear;                   % Create clean sheet.                 

%% Plotting options and preferred parameter configuration:
% Main option:                                    
LRplanner  = false;                 % If to plot long-run planner's allocation.
                                    % Default: false.    
% Model parameters:                                    
alpha =    0;                       % Parent altruism. Default: 0.
rho   = 0.04;                       % Discount rate (yearly). Default: 0.04.
delta =  1/3;                       % Death hazard: Default: 1/3, i.e.
                                    % Life exp. is 3 years.                        
yp    = 0.34;                       % Parent income. Default: 0.34. Chosen
                                    % so that parent lands in MA when broke:
                                    % yp - beta*yk = 0.34-0.25 = 0.09 < 0.1 = Cma.
yk    =   1;                        % Child income. Default: 1.
pnh   = 0.3;                        % Price of privately-paid care.
                                    % Need pnh = 0.3 > 0.25 = beta*yk.
                                    % Default: 0.3.
beta  =0.25;                        % Fraction of labor income child loses 
                                    % when giving IC. Default: 0.25.
Cma   = 0.1;                        % Consumption floor in Medicaid.
                                    % Default: 0.1.
ykFact=   1;                        % How child income changes after death.
                                    % Default: 1 (no change).
% Algorithm and plotting options:                                    
N = 81;                             % Number of grid points for plotting.
                                    % Default: 81.
plotDetail = false;                 % If to put details in plot. Default (to 
                                    % reproduce figure in paper appendix): false.                                   
LgtGrey = 0.9*[1,1,1];              % RGB value for light grey (used for iso-
                                    % apdot lines) and Pareto improvements
                                    % over renting allocation). Default:
                                    % 0.9*[1,1,1].
drkGrey = 0.6*[1,1,1];              % Dark grey to mark efficient allocations.
                                    % Default: 0.6*[1,1,1].                                   

%% Solve model:
u = @(c) log(c);                    % Log utility. CANNOT CHANGE THIS, SOME 
                                    % OF THE CLOSED FORMS USED BELOW ARE 
                                    % ONLY VALID FOR LOG-UTILITY.
ykIC   = (1-beta)*yk;               % Child income under IC: often needed, 
                                    % here 0.75.                                    
Dic = pnh - beta*yk;                % Efficiency advantage of IC over PPC.
                                    % Here: 0.05.                                    
B = ( u(ykFact*yk+rho) - u(ykFact*yk) )/rho;    
                                    % Get death value for child.
% We are only interested in equilibria in which MA is chosen here and in 
% IC dominates privately-paid care:
if Cma>yp-beta*yk                   % Check that MA is really chosen when 
    disp('Parent chooses MA when broke.')   % parent broke.
else                                % If not, give error message.
    disp('ERROR: PARENT WOULD CHOOSE IC WHEN BROKE.') 
end
if Dic>=0                           % Give error message if IC is dominated:
    disp('Equilibrium features IC.')
else
    disp('ERROR: IC IS DOMINATED. This code only finds IC equilibria.')
end
% Get values when parent alive and wealth is zero (scalars). Write into
% structure 'eqm' that contains equilibrium quantities:
eqm.Vk0 =                u(yk)  /(rho+delta);   
eqm.Vp0 = ( u(Cma)+alpha*u(yk) )/(rho+delta);   
                                    % Parent takes MA under our assumptions.

% Find equilibrium with IC and owning:
% First, find maximal transfer (i.e. lowest Q) that parent would accept
% without giving it back to the child; this gives us a reasonable lower bound
% for Q:
cpdict = (yp+ykIC)/(1+alpha);       % What parent would assign to herself of
                                    % flow income (given LOG-UTILITY!)
Qdict  = (yp - cpdict);             % Net transfer that parent would choose
                                    % if she could dictate.
                                    % Don't go all the way there to avoid
                                    % -Inf in case of zero altruism
                                    % (alpha=0).
Qmax   = 0.999*(yp-Cma);            % Maximal reasonable transfer is such 
                                    % the parent is still better off than
                                    % in Medicaid when broke. Don't go all
                                    % the way to avoid -Inf.
Qvec   = linspace(0.99*Qdict,0.9*Qmax,N); %#ok<NASGU>
                                    % Create vector with candidates for
                                    % Qeq (needed for plots): 1-by-N.                                    

Vp1 = @(Q) ( u(yp-Q) + alpha*u(ykIC+Q) + delta*alpha*B )/(rho+delta);
                                    % Function handle that obtains V^p_1 as
                                    % a function of Q in IC+own
                                    % equilibrium.
% plot(Qvec,Vp1(Qvec));             % Can check here: Vp1 is decreasing in 
%                                   % Q and maximized at Qdict.
Vk1 = @(Q) (                 u(ykIC+Q) + delta*      B )/(rho+delta);
                                    % Function handle that obtains V^k_1 as
                                    % a function of Q in IC+own
                                    % equilibrium.
% plot(Qvec,Vk1(Qvec));            % Vk1 is increasing in Q.

cpRent = @(Q) 1./(Vp1(Q)-eqm.Vp0);  % Function handle: parent consumption in
                                    % renting regime given V^p_1(Q).
% plot(Qvec,cpRent(Qvec))           % c^p_rent is increasing in Q.

% Hamiltonians under outside option:
Qout  = 0;                          % Guess equilibrium in which transfer
                                    % under outside option is zero.
Hpout = @(Q) u(cpRent(Q)) + alpha*u(yk+Qout) + (yp-cpRent(Q)-Qout-pnh).*(Vp1(Q)-eqm.Vp0);
Hkout = @(Q)                      u(yk+Qout) + (yp-cpRent(Q)-Qout-pnh).*(Vk1(Q)-eqm.Vk0);
% plot(Qvec,Hpout(Qvec))            % H^p_out is increasing in Q.
% plot(Qvec,Hkout(Qvec))            % H^k_out is decreasing in Q.

% Now, set up Hamiltonians under IC & own (drift is zero!):
HpICown = @(Q) u(yp   -Q) + alpha*u(ykIC+Q );
HkICown = @(Q)                    u(ykIC+Q );
% plot(Qvec,HpICown(Qvec))
% H^p_ill is decreasing in Q. 
% H^k_ill is increasing in Q.

% Now, get incentive-compatibility functions:
ICp = @(Q) HpICown(Q) - Hpout(Q);   % Difference between Hamiltonians: Positive
                                    % if parent strictly preferes owning 
                                    % regime under candidate Q, negative
                                    % if parent prefers outside option
                                    % (renting regime).
ICk = @(Q) HkICown(Q) - Hkout(Q);   % The same for the child: If this difference
                                    % is positive, the child prefers the
                                    % owning to renting regime.
% plot(Qvec,ICk(Qvec))
% IC^p is decreasing in Q.                                    
% IC^k is increasing in Q.

% Can make figure of IC-functions that shows if housing equilibrium exists:
% figure
% plot(Qvec,ICp(Qvec),'-r',Qvec,ICk(Qvec),'--b',Qvec,0*Qvec,'-k')
% xlabel('Q: candidate transfers')
% ylabel('IC^i(Q): incentive compatibility')
% title('parent owns')
% legend('IC^p: parent','IC^k: child')

% Now, find equilibrium transfer Q_eq (if possible):
if ICp(0.999*Qdict)>0 && ICp(Qmax)<0 % If IC^p cuts zero on relevant range:
    eqm.Q1 = fzero(ICp,[0.999*Qdict,Qmax]);     
                                    % Find equilibrium candidate: Q at which
                                    % parent is indifferent between renting
                                    % and housing regime.
    % eqm.Q1 = fzero(ICk,[Qdict,0]);  % Alternatively, can give all bargaining 
                                    % power to parent: Equilibrium looks
                                    % very similar.
    fprintf('Found unique housing equilibrium candidate with Q_eq =%1.3f.\n',eqm.Q1)
                                    % Have found housing equilbrium.
                                    % Will check IC^k below.
else                                % If IC^p does not cut zero:
    disp('ERROR: Could not find housing equilibrium candidate: IC^p does not cut zero.')
end

% Now, set up Hamiltonian functions: How do agents feel about different
% (c^p,Q) combinations over short dt, knowing that the equilibrium will be
% played after?
% First, find equilibrium objects and put into structure eqm (all scalars):
eqm.cp1   =    yp - eqm.Q1 ;        % Parent consumption in housing eq'm.
eqm.ck1   =  ykIC + eqm.Q1 ;        % Child consumption in IC+own eq'm.
eqm.Vp1   = Vp1(    eqm.Q1);        % Values at high wealth.  
eqm.Vk1   = Vk1(    eqm.Q1);
eqm.Hp1   = HpICown(eqm.Q1);        % Hamiltonians at high wealth.
eqm.Hk1   = HkICown(eqm.Q1);
eqm.cpRent= cpRent( eqm.Q1);        % Renting consumption.
eqm.HpOut = Hpout(  eqm.Q1);        % Hamiltonians under outside option
eqm.HkOut = Hkout(  eqm.Q1);        % in equilibrium.

% Now, find outcome in IC+rent subgame, given the continuation values
% from our guess for the IC+own equilibrium:
HpICrent = @(Q) u(eqm.cpRent) + alpha*u(ykIC+Q) + (yp-eqm.cpRent-Q).*(eqm.Vp1-eqm.Vp0);  
HkICrent = @(Q)                       u(ykIC+Q) + (yp-eqm.cpRent-Q).*(eqm.Vk1-eqm.Vk0);
% plot(Qvec,HpICrent(Qvec))         % H^p_rent is decreasing in Q.                                    
                                    % H^k_rent is increasing in Q.
% Obtain incentive compatibility functions for IC+rent equilibrium:                                    
ICpRent= @(Q) HpICrent(Q)-eqm.HpOut;
ICkRent= @(Q) HkICrent(Q)-eqm.HkOut;
% IC^p_rent is decreasing in Q 
% IC^k_rent is inccreasing in Q

% Now, find the transfer Q that the child would set under the
% allocation IC+rent:
if alpha==0                         % If the parent is not altruistic, the 
    eqm.ICrent.Q = pnh;             % child will just extract the entire
                                    % surplus. Put results in IC+rent
                                    % regime in substructure 'eqm.ICrent'.
else                                % If the parent is altruistic, the child
                                    % can extract even more:
    QQmax = 2*pnh;                  % Reasonable upper bound: Q_rent will 
                                    % be above p_pc!
    % First, plot to see what we are doing:                                        
    QQvec = linspace(Qdict,QQmax);          % Q_dict still a good lower bound
    figure, plot(QQvec,ICpRent(QQvec),'-r',QQvec,ICkRent(QQvec),'--b',QQvec,0*QQvec,'-k')
    xlabel('Q'), ylabel('incentive compatibility'), title('renting subgame')
    legend('IC^p_{rent}','IC^k_{rent}')
    % IC^p_rent decreasing in Q (for reasonable values)
    % IC^k_rent increasing in Q (for reasonable values)

    eqm.ICrent.Q = fzero(ICpRent,[0.999*Qdict,QQmax]);           
                                    % Solve for equilibrium in renting subgame:
                                    % Find the transfer just above p_pc
                                    % that makes parent indifferent to
                                    % outside option (Q_rent is above p_pc
                                    % since parent has gain from increasing
                                    % child's consumption when paying for
                                    % IC, but no such benefit when paying
end                                 % PP care).

fprintf('Found child''s optimal offer under IC+rent: Q_rent=%1.3f\n',eqm.ICrent.Q)
eqm.ICrent.Hp1 = HpICrent(eqm.ICrent.Q); % Evaluate Hamiltonians in IC+rent
eqm.ICrent.Hk1 = HkICrent(eqm.ICrent.Q); % regime: Scalars.


% Now, verify our guesses: Create logical scalars that tell us if our
% guesses worked out.
disp('Verifying guesses:')
guess1 = (1/yk >= eqm.Vk1-eqm.Vk0); % At Q=0, the child prefers to consume
                                    % a marginal dollar rather than giving
                                    % it to the parent.
guess2a= (eqm.Hk1 >= eqm.HkOut  );  % Child prefers IC+own equilibrium over
                                    % outside option,
guess2b= (eqm.Hk1 >= eqm.ICrent.Hk1); % and over the IC+rent option.

% 1. Message on Guess 1:
if guess1
    disp('1. Guess Q_out=0 verified.')
else
    disp('1. ERROR: GUESS Q_out=0 IS WRONG; CHILD WANTS TO SAVE THROUGH')
    disp('   PARENT UNDER OUTSIDE OPTION!')
end
% Message on Guess 2a:
if guess2a
    disp('2a. Checked: Child prefers IC+own over outside option.')
else    
    disp('2a. ERROR: CHILD PREFERS OUTSIDE OPTION OVER IC+OWNING!')
end
% Message on Guess 2b:
if guess2b
    disp('2b. Checked: Child prefers IC+own to IC+rent.')
else
    disp('ERROR: CHILD PREFERS OFFERING IC+RENT TO OFFERING IC+OWN!')
end
% If all guesses correct:
if guess1 && guess2a && guess2b
    disp('--> Found (unique) IC+own equilibrium.')
end

% Set up function handles for Hamiltonians in dt-game under IC when parent
% is wealthy:
HpIC = @(cp,Q) u(cp) + alpha*u(ykIC+Q) + (yp    -Q-cp)*(eqm.Vp1-eqm.Vp0);
HkIC = @(cp,Q)               u(ykIC+Q) + (yp    -Q-cp)*(eqm.Vk1-eqm.Vk0);
% Set up function handles for Hamiltonians in dt-game under FC (parent
% wealthy):
HpFC = @(cp,Q) u(cp) + alpha*u(yk  +Q) + (yp-pnh-Q-cp)*(eqm.Vp1-eqm.Vp0);
HkFC = @(cp,Q)               u(yk  +Q) + (yp-pnh-Q-cp)*(eqm.Vk1-eqm.Vk0);
% Hamiltonians under FC & own:
HpFCown = @(Q) u(yp-pnh-Q) + alpha*u(yk  +Q);
HkFCown = @(Q)                     u(yk  +Q);
% Now, find transfer that child would offer to parent in FHC regime (if we 
% reached that sugame... equilibrium is still IC+own!)
ICfhcOwn = @(Q) HpFCown(Q) - eqm.HpOut;
QfhcMin = min(yp-pnh,0);        % If formal care cost exceeds parent income,
                                % need to set minimal transfer such that 
                                % parent can afford this.
eqm.FHCown.Q1 = fzero( ICfhcOwn, [-0.99*yk,1.01*QfhcMin] );
                                % Get transfer in FHC+own regime.
eqm.FHCown.cp1= yp-pnh-eqm.FHCown.Q1;   
                                % Corresponding parent consumption.


%% No-housing world: only renting possible
% Collect solution to this game in structure 'noH'.
% Since parent always gets the value of the outside option in the high
% wealth state, can copy the outcome from above here (IC must happen since
% we assume Dic>0 throughout):
noH.Vp0 = eqm.Vp0;              % Values in low wealth state as in          
noH.Vk0 = eqm.Vk0;              % equilibrium before.
% High wealth state:
noH.Q1  = eqm.ICrent.Q;                     
noH.cp1 = eqm.cpRent  ;     noH.ck1 = ykIC-noH.Q1 ;
noH.dr1 = yp - noH.cp1 - noH.Q1;% Obtain drift.
noH.Vp1 = eqm.Vp1     ;         % Parent value must be as in IC+own eq'm.  
% Checked here: Vp1 is actually correct:
% noH.Vp1 = (          u(noH.cp1) - noH.dr1*noH.Vp0 )/(rho+delta-noH.dr1);
%(rho+delta-noH.dr1)*Vk1 = delta*B + u(noH.ck1) + noH.dr1*Vk0

% Then, solve for child's value in high wealth state from HJB:
%(rho+delta-noH.dr1)*Vk1 = delta*B + u(noH.ck1) - noH.dr1*Vk0  ==>
noH.Vk1 = ( delta*B + u(noH.ck1) - noH.dr1*noH.Vk0 )/(rho+delta-noH.dr1);


%% Make figure with preferences in dt-game under FC:
ypFC = yp-pnh;                      % Parent income under formal care.
% Obtain a small distance for plotting:
cpmax= max(eqm.FHCown.cp1,eqm.cpRent);
Qout = 0;                           % Transfer under outside option is zero
                                    % by our choice of B.                                    
ddc = (cpmax-ypFC)/20;              % Small distance for plotting.

% Now, set consumption bounds jointly with IC graph that comes later:
cpmin = min(ypFC, yp-Qmax);         % Bounds for c^p in plot.
cpmax = eqm.cpRent + 2*Dic; 
Qmin  = -(cpmax-cpmin);             % Minimal Q-value for plotting.
                                    % Find reasonable upper bound for plotting
                                    % parent consumption: eating the
                                    % maximal transfer, or highest
                                    % renting-regime consumption (under any
                                    % transfer).
cpvec = linspace(cpmin,cpmax,N)';   % N-by-1 vector for c^p.
Qvec  = linspace(Qmin,0,N);         % 1-by-N vector for Q (change Qvec here 
                                    % so that the grey points have similar
                                    % distance in both graphs).
% Calculate efficient allocations:                                            
etavec = linspace(0,1,N);           % 1-by-N vector with weights on parent.
etakvec= 1-etavec+alpha*etavec;     % Effective weight on child.
dVetavec  = etavec*(eqm.Vp1-eqm.Vp0) + (1-etavec)*(eqm.Vk1-eqm.Vk0);
                                    % Get V^eta_Delta
% First, get solutions with dissavings (adotp<=0 does not bind):
etaLR.cpUncVec = etavec./dVetavec;  % Turns out the c^p-allocation is the same
                                    % if Q constraint binds or not --
                                    % consistent with continuity of the
                                    % contract curve!
etaLR.QuncVec = min(0, etakvec./dVetavec - yk );
                                    % Impose non-positivity of Q.
% Second, get constrained solution (on housing line):
etaLR.QconVec  = (etakvec*yp - etavec*yk)./(etavec+etakvec);
etaLR.cpConVec = yp-etaLR.QconVec;
% Bring together constrained and unconstrained solution:
constr = (yp-etaLR.cpUncVec-etaLR.QuncVec)>0;
etaLR.Qvec = constr.* etaLR.QconVec + (~constr).* etaLR.QuncVec;
etaLR.cpVec= constr.*etaLR.cpConVec + (~constr).*etaLR.cpUncVec;
                                    % If unconstrained solution yields
                                    % positive drift, must take constrained
                                    % solution.                                    
                                                                      
% Get levels for which to plot parent indifference curves:
% Take some interesting allocations on Pareto frontier here with the following
% indeces for eta (note that ratio cp/ck for eta-planner must be
% eta/(1-eta+eta*alpha):
etaLo = 0.2;                        % For which Pareto weights to plot
etaMd = 0.38;                       
etaHi = 0.5;

indEtaLo   = 1+round(etaLo*(N-1));  % Get indeces on our vector.
indEtaMd   = 1+round(etaMd*(N-1));
indEtaHi   = 1+round(etaHi*(N-1));  % Same for higher weight on parent.

% Set vectors with (c^p/Q) allocations that we want indifference curves to
% run through: Take the eta's from above plus the equilibrium
% allocataion (which is on the same parent indifference curve as the renting
% subgame outcome; added eqm.FHCown.cp1-eqm.Q1 here in order to have nice plot when
% giving bargaining power to parent, Matlab seems to accept the same level
% twice):
cpPar = [etaLR.cpVec(indEtaHi), etaLR.cpVec(indEtaMd), eqm.cpRent, eqm.FHCown.cp1]; 
QPar  = [etaLR.Qvec( indEtaHi), etaLR.Qvec( indEtaMd),     Qout  , eqm.FHCown.Q1 ]; 
%                                   % Add equilibrium allocation here when
%                                   % giving bargaining power to parent.
levpar= HpFC(cpPar,QPar);           % Evaluate parent Hamiltonian H^p to 
                                    % obtain levels of parent indifference 
                                    % curves for plotting.

% In the same way, get levels for which to plot kid indifference curves:                           
cpKid = [etaLR.cpVec(indEtaHi), etaLR.cpVec(indEtaMd), eqm.cpRent, eqm.FHCown.cp1, etaLR.cpVec(indEtaLo)];
QKid  = [etaLR.Qvec( indEtaHi), etaLR.Qvec( indEtaMd),     Qout  , eqm.FHCown.Q1 , etaLR.Qvec( indEtaLo)]; 
levkid= HkFC(cpKid,QKid);

% Now, start plotting:  Put FC graph on left, IC will be right.  
figure('Name','Payoffs from dt-allocations involving IC ... (Fig.B1 appendix)');
subplot(1,2,1)
title('Formal care (FC)')
hold on

Hpmat = HpFC(cpvec,Qvec);           % Create large matrices with Hamiltonians
Hkmat = HkFC(cpvec,Qvec);           % evaluated on N-by-N Cartesian grid.
drmat = ypFC-cpvec-Qvec ;           % N-by-N matrix with drift (apdot).
drPos = (drmat>0);                  % N-by-N logical matrix: if drift positive.
Hpmat(drPos) = NaN;                 % Positive-drift allocations don't make
Hkmat(drPos) = NaN;                 % sense, remove these.

contour(Qvec,cpvec,Hpmat,levpar, '-r')
                                    % Parent ind. curves.
% contour(Qvec,cpvec,Hp(cpvec,Qvec), '-r') % This is with Matlab
% choosing the H^p-levels (do this if levpar does not work). 
                                   
plot([ypFC-eqm.cpRent,0],[eqm.cpRent,eqm.cpRent],'-r','LineWidth',2)
                                    % Parent best response in renting
                                    % regime.                                     
contour(Qvec,cpvec,Hkmat,levkid,'--b') 
                                    % Child ind. curves.
% contour(Qvec,cpvec,Hk(cpvec,Qvec),'--b')    % Again, can have Matlab 
                                    % choose the levels.
                                    
plot([Qout,Qout],[ypFC,cpmax],'--b','LineWidth',2)
                                    % Child best response in renting
                                    % regime.                                    
% Now, mark areas that are Pareto improvements over renting and eq'm alloc.:
Hpmat = HpFC(cpvec,Qvec);           % Create large matrices with Hamiltonians
Hkmat = HkFC(cpvec,Qvec);           % evaluated on N-by-N Cartesian grid.
% Create logical N-by-N matrices that tell us if a point on grid is a 
% Pareto improvement over...
ParImpRent =  (Hpmat>HpFC( eqm.cpRent    , Qout          ) ) ...
            & (Hkmat>HkFC( eqm.cpRent    , Qout          ) )  & ~drPos;
                                    % ... the renting subgame outcome, ...
ParImpEqm =   (Hpmat>HpFC( eqm.FHCown.cp1, eqm.FHCown.Q1 ) ) ...
            & (Hkmat>HkFC( eqm.FHCown.cp1, eqm.FHCown.Q1 ) )  & ~drPos; %#ok<NASGU>
                                    % ... or the equilibrium itself.
cpmat = cpvec.*ones(1,N);           % Set up N-by-N matrices for cp and Q
 Qmat =  Qvec.*ones(N,1);           % to index them:
% With logical indexing, only pick points on grid that are Pareto
% improvements and mark them with a dot in...
plot([Qdict,0],[ypFC-Qdict,ypFC],'-k')  % Mark zero-savings allocations (i.e. 
                                    % those possible housing regime).

if plotDetail                       % Only show this in detailed plot:
    plot(Qmat(ParImpRent(:)), cpmat(ParImpRent(:)),'.','Color',[0.9,0.9,0.9])
                                    % ... light grey for renting outcome,...
    plot(Qmat( ParImpEqm(:)), cpmat( ParImpEqm(:)),'.','Color',[0.6,0.6,0.6])
                                    % ... darker grey for eq'm outcome.
    % Plot iso-savings lines for the dynasty:
    isoLev = [-0.4,-0.2,0.2];
    [Cdr,hdr] = contour(Qvec,cpvec,drmat,isoLev,'-','Color',LgtGrey);
    clabel(Cdr,hdr,'Color',LgtGrey)
    plot(etaLR.Qvec,etaLR.cpVec,'-.','LineWidth',2,'Color',DrkGrey)  
                                    % Mark efficient allocations = Pareto
                                    % frontier.
else                                % If no detail wanted:
    plot(Qmat(ParImpRent(:)), cpmat(ParImpRent(:)),'.','Color',LgtGrey)
                                    % just grey for improvements over
end                                 % renting eq'm.

% Can mark parent bliss point here:
% dd = (cpmax-ypFC)/20;
% plot(etaLR.Qvec(end),etaLR.cpVec(end),'or','MarkerSize',5,'MarkerFaceColor','r')
%                                     % Mark parent's bliss point
% text(etaLR.Qvec(end),etaLR.cpVec(end)+dd,'parent bliss','Color','r',...
%      'HorizontalAlignment','left')

% Mark important allocations:
plot(eqm.FHCown.Q1,eqm.FHCown.cp1,'xk','MarkerSize',10,'LineWidth',2)        
text(eqm.FHCown.Q1-ddc/2,eqm.FHCown.cp1,{'FHC subgame outcome','($\dot{a}^p=0$)'},...
     'HorizontalAlignment','right', 'Interpreter','Latex')
                                    % Mark equilibrium.

plot(Qout,eqm.cpRent,'ok','MarkerSize',5,'LineWidth',2)    
text(Qout+ddc/2,eqm.cpRent,{'outside','option','($\dot{a}^p<0$)'},...
     'HorizontalAlignment','left', 'Interpreter','Latex'    )                                          
                                    % Mark renting allocation that would be
                                    % played if no bargain is reached.

plot(0,ypFC,'dk')                   % Endowment point.
text(ddc/2,ypFC,{'endowment','point'},'Interpreter','Latex', ...
     'HorizontalAlignment','left','VerticalAlignment','bottom' )

% Mark better-direction for parent:
arrQcnt = 0.4;                      % Where arrow is centered as fraction
                                    % of [Qmin,0].
plot( [(arrQcnt-0.1)*Qmin, (arrQcnt+0.1)*Qmin], [eqm.cpRent+0.5*ddc,eqm.cpRent+0.5*ddc],'-r') % arrow stem,
plot(  (arrQcnt+0.1)*Qmin, eqm.cpRent+0.5*ddc, '<r') % arrow head.
text( arrQcnt*Qmin, eqm.cpRent+ddc, 'better for parent',           ...
     'VerticalAlignment','bottom', 'HorizontalAlignment','center',...
     'Color','r',  'Interpreter','Latex'                             )

% Mark better-direction for child:
plot([Qout+0.5*ddc,Qout+0.5*ddc ],[0.4*ypFC+0.6*cpmax, 0.6*ypFC+0.4*cpmax],'--b')
plot( Qout+0.5*ddc, 0.6*ypFC+0.4*cpmax,'vb')   % again, arrow.
text( Qout+    ddc, 0.5*ypFC+0.5*cpmax, {'better','for','child'},...
      'HorizontalAlignment','left', 'Color','b', 'Interpreter','Latex')

if plotDetail  
    % Now, mark efficient allocations that were used to pick the indifference 
    % level curves:
    mrkSizEta = 5;
    plot(etaLR.Qvec(indEtaMd)      ,etaLR.cpVec(indEtaMd),'*','Color',DrkGrey,'MarkerSize',mrkSizEta);
    text(etaLR.Qvec(indEtaMd)+ddc/5,etaLR.cpVec(indEtaMd),...
         ['$\eta=', sprintf(etaFrm,etaMd), '$'],...
         'Color',DrkGrey,'VerticalAlignment','bottom','Interpreter','Latex')
                                    % Mark eta=0.5-planner's allocation.
    plot(etaLR.Qvec(indEtaHi  )      ,etaLR.cpVec(indEtaHi  ),'*','Color',DrkGrey,'MarkerSize',mrkSizEta)                                
    text(etaLR.Qvec(indEtaHi  )+ddc/5,etaLR.cpVec(indEtaHi  ), ...
         ['$\eta=', sprintf(etaFrm,etaHi), '$'],...
         'Color',DrkGrey,'VerticalAlignment','bottom','Interpreter','Latex')  % high eta: 2/3

    plot(etaLR.Qvec(indEtaLo  )      ,etaLR.cpVec(indEtaLo  ),'*','Color',DrkGrey,'MarkerSize',mrkSizEta)                                
    text(etaLR.Qvec(indEtaLo  )+ddc/5,etaLR.cpVec(indEtaLo  ), ...
         ['$\eta=', sprintf(etaFrm,etaLo), '$'],...
         'Color',DrkGrey,'VerticalAlignment','bottom','Interpreter','Latex')  % low eta: 1/4
end
% Finally, mark the area with positive drift that does not make sense for
% the family:
startX = -(cpmax-cpmin)*2/3;
startY = ypFC-startX-ddc/2;
endX   = startX-2*ddc;
endY   = startY-2*ddc;
plot( [startX,endX], [startY,endY],'-k')    % arrow stem,...
plot(         endX,          endY ,'^k')    % arrow head, and text:
text(endX, endY-ddc/2 ,                       ...
     {'$\dot{a}^p>0$:', 'dominated allocations', '(over-savings)'},...
     'Interpreter','Latex', 'HorizontalAlignment','center','VerticalAlignment','top'  ) 
 
xticks(   [-0.5,eqm.FHCown.Q1, 0] )          % Set ticks for x-axis.
yt = [ypFC, eqm.FHCown.cp1,  eqm.cpRent];    % 1-by-3 vector with ticks for y-axis.
                                 % Is potentially in wrong order, will sort
                                 % below.
% Set ticks on x-axis:
xticklabels({'-0.5',                                     ...
             ['$Q_{fhc,own}=', sprintf('%1.2f',eqm.FHCown.Q1    ), '$'],...
             '$0$'                                          } );
% Create 1-by-4 cell with tick labels for y: Sort this cell below
% to get in right order.
ytLbl = { [   '$y^p-p_{nh}=', sprintf('%1.2f',ypFC          ), '$'], ...                                         ...
          ['$c^p_{fhc,own}=', sprintf('%1.2f',eqm.FHCown.cp1), '$'], ...
          [   '$c^p_{rent}=', sprintf('%1.2f',eqm.cpRent    ), '$']      };


[yt,it] = sort(yt);             % Order y-ticks in yt.
yticks(yt)                      % Set ticks on y-axis.
yticklabels(ytLbl(it));         % Label (also need correct order).

set(gca,'TickLabelInterpreter','Latex','XTickLabelRotation',30)
xlim([Qmin ,ddc  ]);            % Set axis limits.                                
ylim([cpmin,cpmax])
% Label axes and figure:
xlabel('$Q$: transfer'            , 'Interpreter','Latex')
ylabel('$c^p$: parent consumption', 'Interpreter','Latex')

legShort = {'parent indifference curves',             ...
            'parent best response (renting)',          ...
            'child indifference curves',              ...
            'child best response (renting)',           ...
            'housing allocations ($\dot{a}^p=0$)',   ...
            'Pareto improvements over outside option'    };
                                % Cell with first 6 legend entries
                                % (always given).

if plotDetail                   % For detailed plot:
    title('Preferences (=Hamiltonians) in dt-game')
    legAddn  = {'Pareto improvements over equilibrium',     ...
                'iso-savings lines ($\dot{a}^p$=const)'    ,...
                'efficient allocations ($\eta$-dt-planner)'    };
                                % Cell with addition to legend: 1-by-3.
    legend([legShort,legAddn],'Location','southwest','Interpreter','Latex');
                                % Print detailed legend: Concatenate two
                                % cells with text to 1-by-9 cell.
else                            % For simple graph:
    legend(legShort,'Location','southwest','Interpreter','Latex');
end                             % Just put legend.
 

%% Now, make graph with preferences in dt-game under IC:
% Create grids for plotting:
Qmax = yp;                          % Consider transfers up to yp, and
Qmin = yp-eqm.cpRent-Dic;           % transfers higher than this.

ddcp = abs(eqm.cp1-eqm.cpRent);     % Small distance for setting apart texts.

cpgrid = linspace(cpmin,cpmax,N)';  % N-by-1 vector for c^p.
Qgrid  = linspace( Qmin, Qmax,N) ;  % 1-by-N vector for Q.

% Calculate efficient allocations: Short-run social planner. Collect
% results in structure 'etaSR'.
etavec = linspace(0,1,N);           % 1-by-N vector with weights on parent
etakvec= 1-etavec+alpha*etavec;     % Effective weight on child.
etaSR.dVvec  = etavec*(eqm.Vp1-eqm.Vp0) + (1-etavec)*(eqm.Vk1-eqm.Vk0);
                                    % Get Delta V^eta_SR: 1-by-N.
% First, get unconstrained solution:
etaSR.cpUncVec = etavec ./etaSR.dVvec;         
etaSR.QuncVec  = etakvec./etaSR.dVvec - ykIC ;
                                    % Can also choose positive Qs now.

% Second, get constrained solution (on housing line):
etaSR.QconVec  = (etakvec*yp - etavec*ykIC)./(etavec+etakvec);
etaSR.cpConVec = yp-etaSR.QconVec;
% Bring together constrained and unconstrained solution:
constr = (yp-etaSR.cpUncVec-etaSR.QuncVec)>0;
etaSR.QetaVec = constr.* etaSR.QconVec + (~constr).* etaSR.QuncVec;
etaSR.cpVec   = constr.*etaSR.cpConVec + (~constr).*etaSR.cpUncVec;
                                    % If unconstrained solution yields
                                    % positive drift: must use constrained
                                    % solution.

% To plot indifference curves, set up five allocations through which we want 
% parent and child's indifference curves to run: equilibrium,
% IC+rent offer child would choose if housing was ruled out for dt,
% outside option, and two points on Pareto frontier.
Qlev = zeros(5,1);  cplev = zeros(5,1);

etaLo = 0.18;                       % eta for first allocation on frontier,
etaHi = 0.4;                        % eta for second alloc. on frontier.
etaFrm= '%1.2f';                    % Format for labeling etas later. 
indEtaLo   = 1+round(etaLo*(N-1));  % Find indeces on eta vector with these
indEtaHi   = 1+round(etaHi*(N-1));  % weights.

% First point: eta=0.5 efficient allocation (which lies on the housing line):
Qlev(1) = etaSR.QetaVec(indEtaHi);     cplev(1) = etaSR.cpVec(indEtaHi);
% Second point: where kid outside option indifference curve intersects
% the housing line:
f2 = @(Q) HkICown(Q)-eqm.HkOut;
Qlev(2)  = fzero(f2,[Qmin,Qmax]);      cplev(2) = yp-Qlev(2);
% Third point: where kid's best renting indiff curve runs through housing
% line:
f3 = @(Q) HkICown(Q)-eqm.ICrent.Hk1;
Qlev(3) = fzero(f3,[Qmin,Qmax]);       cplev(3) = yp-Qlev(3);
% Fourth level: the (housing) equilibrium allocation.
Qlev(4) = eqm.Q1;                      cplev(4) = eqm.cp1;
% Fifth level: efficient allocation with eta=0.2 (which is interior).
Qlev(5) = etaSR.QetaVec(indEtaLo);     cplev(5) = etaSR.cpVec(indEtaLo);

levpar = HpIC(cplev,Qlev);          % Evaluate Hamiltonians to get 5-by-1
levkid = HkIC(cplev,Qlev);          % vectors with levels for both players' 
                                    % indifference curves.
levkid(3)=[];                       % Remove 3rd one for kid.. 
levpar(3)=[];                       % and parent (can comment out to see more).

Hpmat = HpIC(cpgrid,Qgrid);         % Create large matrices with Hamiltonians
Hkmat = HkIC(cpgrid,Qgrid);         % evaluated on N-by-N Cartesian grid.
drPos = (yp-cpgrid-Qgrid>0);        % N-by-N logical matrix: if grid positive.
Hpmat(drPos) = NaN;                 % Positive-drift allocations don't make
Hkmat(drPos) = NaN;                 % sense, set NaN so they are not plotted.

% First, mark areas that are Pareto improvements over renting and eq'm alloc.
% Do this first so that other lines go on top of it.
% Create logical N-by-N matrices that tell us if a point on grid is a 
% Pareto improvement over...
ParImpOut = (Hpmat>eqm.HpOut) & (Hkmat>eqm.HkOut) & ~drPos;
                                    % ... the outside option ...
ParImpEqm  = (Hpmat>eqm.Hp1   ) & (Hkmat>eqm.Hk1   ) & ~drPos;
                                    % ... or the equilibrium itself.
cpmat = cpgrid.*ones(1,N);          % Set up N-by-N matrices for cp and Q
 Qmat =  Qgrid.*ones(N,1);          % to index them:

% Now, start plotting: Make this the 2nd subplot.
subplot(1,2,2)
title('Informal care (IC)')
hold on
                                
if ~plotDetail                      % In less detailed plot: Just mark 
    ParetoDots = plot(Qmat(ParImpOut(:)), cpmat(ParImpOut(:)),'.','Color',LgtGrey);
end                                 % improvements over renting eq'm in grey.
                                 
% First, plot indifference curves:
% Plot parent's indifference curves (level lines of Hamiltonians):                                    
parCont = contour(Qgrid,cpgrid,Hpmat,levpar, '-r');
% contour(Qgrid,cpgrid,Hpmat, '-r') % This is with Matlab
% choosing the H^p-levels (do this if levpar does not work). 
cpLiqLine = plot([yp-eqm.cpRent,Qmax],[eqm.cpRent,eqm.cpRent],'-r','LineWidth',2);
                                    % Parent best response in renting
                                    % regime.  
% Plot child's indifference curves:                                    
kidCont = contour(Qgrid,cpgrid,Hkmat,levkid,'--b');
% contour(Qgrid,cpgrid,Hkmat,'--b') % This is with Matlab choosing the
                                    % levels.
IllLine = plot([Qdict,Qmax],[yp-Qdict,yp-Qmax],'-k');
                                    % Mark zero-savings allocations
                                    % (i.e. those possible in housing
                                    % regime).

% With logical indexing, only pick points on grid that are Pareto
% improvements and mark them with a dot in...
if plotDetail                       % (do this only for detailed plot)
    ParetoDots = plot(Qmat(ParImpOut(:)), cpmat(ParImpOut(:)),'.','Color',[0.9,0.9,0.9]);
                                    % ... light grey for outside option,...                               
    plot(Qmat( ParImpEqm(:)), cpmat( ParImpEqm(:)),'.','Color',[0.6,0.6,0.6])
                                    % ... darker grey for eq'm outcome.
    EffLine = plot(etaSR.QetaVec,etaSR.cpVec,'-.','LineWidth',2,'Color',drkGrey); 
end                                 % Mark efficient allocations.                                  

ddc = (cpmax-cpmin)/20;             % Small distance for plotting

% Mark important allocations:
plot(eqm.Q1,eqm.cp1,'xk','MarkerSize',10,'LineWidth',2)        
text(eqm.Q1-ddc/4,eqm.cp1,{'IC+own','equilibrium','($\dot{a}^p=0$)'},... %
     'HorizontalAlignment','right','VerticalAlignment','top',...
     'Interpreter','Latex')         % Mark equilibrium.

plot(eqm.ICrent.Q,eqm.cpRent,'ok','MarkerSize',5,'LineWidth',2) 
if plotDetail
    lblTxt = {'child''s best offer','under IC+rent',...
                         '=parent''s outside',...
                         ' option (as-if)',...
                         '($\dot{a}^p<0$)'};
else
    lblTxt = {'IC+rent','subgame','outcome',...
              '=','parent''s','outside','option','(as-if)'};                      
end
text(eqm.ICrent.Q+ddc/4,eqm.cpRent-ddc/4,lblTxt, ...
     'Interpreter','Latex',...
     'HorizontalAlignment','left','VerticalAlignment','top' );                                         
                                    % Mark allocation that would be
                                    % played if no housing possible (renting
                                    % wealth, but IC takes place).
                                   
plot(beta*yk,eqm.cpRent+Dic,'dk') 
text(beta*yk,eqm.cpRent+Dic+ddc/4,'child''s outside option (as-if)',...
     'HorizontalAlignment','right','VerticalAlignment','bottom',...
     'Interpreter','Latex')

% Mark better-direction for parent (took out, too much): 
% LeftWgt = 0.6;                                     % How much left (0 to 1)
%                                                    % to put arrow center.
% org = (LeftWgt-0.1)*Qmin+(1-LeftWgt+0.1)*Qmax;     % arrow tip
% tip = (LeftWgt+0.1)*Qmin+(1-LeftWgt-0.1)*Qmax;     % arrow origin.
% cnt = (tip+org)/2;                                  % center of arrow.
% plot( [tip,org], [cpmax-2*ddc,cpmax-2*ddc],'-r') % arrow stem,
% plot( tip, cpmax-2*ddc, '<r') % arrow tip.
% text( cnt, cpmax-1.75*ddc, 'better for parent',...
%      'VerticalAlignment','bottom','HorizontalAlignment','center',...
%      'Color','r','Interpreter','Latex'                               )

% Mark better-direction for child:
DownWgt = 0.8;                     % How far down (0 to 1) to put arrow center.
org = (DownWgt-0.1)*cpmin+(1-DownWgt+0.1)*cpmax;    % arrow tip
tip = (DownWgt+0.1)*cpmin+(1-DownWgt-0.1)*cpmax;    % arrow origin.
cnt = (tip+org)/2;                                  % center of arrow.
plot([Qmax-ddc,Qmax-ddc],[org, tip],'--b')
plot( Qmax-ddc, tip,'vb')                           % again, arrow.
text( Qmax-0.75*ddc, cnt, {'better','for','child'},...
      'HorizontalAlignment','left','Color','b','Interpreter','Latex')

if plotDetail

    % Now, mark efficient allocations that were used to pick the indifference 
    % level curves:
                                    % Mark eta=0.5-planner's allocation.
    plot(etaSR.QetaVec(indEtaHi  )      ,etaSR.cpVec(indEtaHi  ),'*k')                                
    text(etaSR.QetaVec(indEtaHi  )+ddc/4,etaSR.cpVec(indEtaHi  ),...
        ['$\eta=',sprintf(etaFrm,etaHi),'$'],'Color','k',...
         'Interpreter','Latex','VerticalAlignment','bottom')  % high eta: 1/2

    plot(etaSR.QetaVec(indEtaLo  )      ,etaSR.cpVec(indEtaLo  ),'*k')                                
    text(etaSR.QetaVec(indEtaLo  )+ddc/2,etaSR.cpVec(indEtaLo  ),...
        ['$\eta=',sprintf(etaFrm,etaLo),'$'],'Color','k',...
         'Interpreter','Latex','VerticalAlignment','bottom')  % low eta: 1/10

    % Mark how we add Dic to c^p_{rent} to get the child as-if allocation from
    % the outside option:
    plot([beta*yk,beta*yk],[eqm.cpRent,eqm.cpRent+Dic],'.-k')
    text( beta*yk         ,    eqm.cpRent+Dic/2    , ...
          ['$D_{ic}=',sprintf('%1.2f',Dic),'$']     ,...
          'HorizontalAlignment','right','Interpreter','Latex')
    % Mark how we add D_eff + eps to beta*yk to get to parent's as-if alloction
    % under outside option (and just D_eff if there is no altruism):
    plot([beta*yk,eqm.ICrent.Q],[eqm.cpRent-ddc/2,eqm.cpRent-ddc/2],'.-k')
    if alpha==0
        lbl = '$D_{ic}$';
    else
        lbl = '$D_{ic}+\epsilon$';
    end
    text( (beta*yk+eqm.ICrent.Q)/2,eqm.cpRent-ddc/2 , lbl, ...
          'VerticalAlignment','top','HorizontalAlignment','center',...
          'Interpreter','Latex')
else
    % Mark how we add Dic to c^p_{rent} to get the child as-if allocation from
    % the outside option:
    plot([beta*yk,beta*yk],[eqm.cpRent,eqm.cpRent+Dic],'.-k')
    text( beta*yk         ,    eqm.cpRent+Dic/2    , '$D_{ic}$'     ,...
          'HorizontalAlignment','right','Interpreter','Latex')
    plot([beta*yk,eqm.ICrent.Q],[eqm.cpRent,eqm.cpRent],'.-k') % plot without offset.
    if alpha==0
        lbl = '$D_{ic}$';
    else
        lbl = '$D_{ic}+\epsilon$';
    end
    text( (beta*yk+eqm.ICrent.Q)/2,eqm.cpRent-ddc/2 , lbl, ...
          'VerticalAlignment','top','HorizontalAlignment','center',...
          'Interpreter','Latex')  
end

if plotDetail  
    % Finally, mark the area with positive drift which does not make sense for
    % the family:
    plot( [0,-2*ddc]-ddc/4, [yp,yp-2*ddc],'-k')     % arrow stem,...
    plot(    -2*ddc -ddc/4,     yp-2*ddc ,'^k')     % arrow head, and text:
    text(-2*ddc -ddc/2, yp-2*ddc ,                       ...
         {'$\dot{a}^p>0$:', 'dominated allocations', '(over-savings)'},...
         'Interpreter','Latex', 'HorizontalAlignment','right'            )
end

% Some editing:
xlim([ Qmin, Qmax])                 % Axis limits.
ylim([cpmin,cpmax])
% Set axis ticks:
yticks([0, eqm.cp1, yp     , eqm.cpRent  ]) %          
xticks([0, eqm.Q1 , beta*yk, eqm.ICrent.Q])
yticklabels({  '$0$'                                             , ...
             [  '$c^p_{eq}=', sprintf('%1.2f',eqm.cp1     ), '$'], ...
             [       '$y^p=', sprintf('%1.2f',yp          ), '$'], ...
             ['$c^p_{rent}=', sprintf('%1.2f',eqm.cpRent  ), '$']      });
xticklabels({ '$0$'                                              , ...
             [   '$Q_{eq}=' , sprintf('%1.2f',eqm.Q1      ), '$'], ...
              '$\beta y^k$' ,... 
             [  '$Q_{rent}=', sprintf('%1.2f',eqm.ICrent.Q), '$']      });
% Change x-axis to top of graph:
set(gca,'TickLabelInterpreter','Latex','XTickLabelRotation',30)
if plotDetail
    set(gca, 'XAxisLocation', 'top')  
    % Switch title to bottom of graph here:
    title('Preferences in dt-game with LTC',...
      'Units', 'normalized', 'Position', [0.5, -0.1, 0]);
end           

xlabel('$Q$: transfer','Interpreter','Latex')
if plotDetail
    ylabel('$c^p$: parent consumption','Interpreter','Latex')
    % Give legend:
    legend({'Pareto improvements over outside option',       ...
            'Pareto improvements over equilibrium',          ... 
            'parent indifference curves: $H^p_{ic}(c^p,Q)$',...
            'parent best response (renting)',                 ...
            'child indifference curves: $H^k_{ic}(c^p,Q)$', ...
            'housing allocations: $\dot{a}^p=0$','efficient allocations'},...   
            'Location','southwest','Interpreter','Latex')
end

    
%% Long-run planner allocations
if LRplanner                            % Only if LR planner is asked for.
    par.yp=yp; par.yk=yk; par.ykIC=ykIC; par.Cma=Cma; par.delta=delta; %#ok<*UNRCH>
    par.rho=rho; par.alpha=alpha; par.B=B;
                                        % Pack structure with parameters.
    % Pack solution into structure etaLR:
    etaLR.cp1 = 0*etavec;               % Set up arrays for allocations the 
    etaLR.Q1  = 0*etavec;               % long-run eta-planner chooses at high
    etaLR.cp0 = 0*etavec;               % wealth, low wealth, ...
    etaLR.Q0  = 0*etavec;
    etaLR.MA  = false(size(etavec));    % ... and the Medicaid decision.
    etaLR.ck0 = 0*etavec;
    etaLR.dV  = 0*etavec;

    dVlo = 1/( 10*eqm.cpRent );         % Lower and upper bound for value 
    dVhi = 1/((eqm.cpRent/10));         % differential: Use renting consumption 
    dVvec = linspace( dVlo, dVhi,N);    % as guideline.
    par.yk=yk; par.yp=yp; par.rho=rho; par.delta=delta; par.alpha=alpha;
    par.B=B;   par.Qbd=0;               % Pack structure par to hand over below.
    for i=1:N                           % Loop over all etas.
        eta = etavec(i);                % Read out current eta.
        F = @(dV) ToyModelLtcLrPlannerCheckHJB(eta,dV,par);   
        etaLR.dV(i) = fzero(F,[dVlo,dVhi]);
                                        % Set up function handle with current 
                                        % eta and solve the HJB.
        [check,etaLR.cp1(i),etaLR.Q1(i),etaLR.cp0(i),etaLR.Q0(i),etaLR.MA(i),etaLR.ck0(i)] = ...
            ToyModelLtcLrPlannerCheckHJB(eta,etaLR.dV(i),par);
                                        % Call function again to get all
    end                                 % equilibrium policies at solution.

    % Get players' values at low wealth in the LR-planner allocations:
    etaLR.Vp0  = (u(etaLR.cp0) + alpha.*u(etaLR.ck0))./(rho+delta);
    etaLR.Vk0  = (                      u(etaLR.ck0))./(rho+delta);
    % Now, go to high wealth:
    etaLR.dr1  = yp - etaLR.cp1 - etaLR.Q1;     % drift.
    etaLR.ck1  = yk+etaLR.Q1;                                  
    % Get both players' values under the different etas at high wealth:
    etaLR.Vp1 = ( u(etaLR.cp1) + alpha.*u(etaLR.ck1) + delta.*alpha.*B - etaLR.dr1.*etaLR.Vp0 )./(rho+delta-etaLR.dr1);
    etaLR.Vk1 = (                       u(etaLR.ck1) + delta.*       B - etaLR.dr1.*etaLR.Vk0 )./(rho+delta-etaLR.dr1);
    
    % Now, by linear interpolation, find threshold etas for the interval
    % with Pareto improvements:
    FindEtaPar = @(eta) interp1(etavec,etaLR.Vp1,eta) - noH.Vp1;
    FindEtaKid = @(eta) interp1(etavec,etaLR.Vk1,eta) - noH.Vk1;
    etaPar = fzero(FindEtaPar,[etavec(2),etavec(end-1)]);
    etaKid = fzero(FindEtaKid,[etavec(2),etavec(end-1)]);
    
    % Set bounds for plotting:
    ybdC = [ 0 , max(ykIC+yp,etaLR.ck1(1))];   
                                    % Consumption plots.
    ybdQ = [min(-yk,etaLR.dr1(end)), max(yp,etaLR.Q1(1)) + 0.02 ];
                                    % Transfer pots.
    ybdV = [-8,2];                  % Set value bounds manually.

    % Logical variable: For which etas the LR planner achieves a Pareto
    % improvement over the equilibrium allocation:
    ParImp1EtaLR = (etaLR.Vp1>eqm.Vp1) & (etaLR.Vk1>eqm.Vk1);
    fprintf('Pareto improvements at high wealth for LR planner with eta=')
    for i=1:N                       % These are only a few, print them out.
        if ParImp1EtaLR(i)
            fprintf('%4.2f,',etavec(i))
        end
    end
    fprintf('\n')

    plot(etaLR.Q1,etaLR.cp1,'-g')   % Mark LR planner allocation in green.
    plot(etaLR.Q1(ParImp1EtaLR),etaLR.cp1(ParImp1EtaLR),'og')
                                    % Mark the ones that are Pareto
                                    % improvements with green circles.

    etaMd=0.4; indEtaMd=1+round(etaMd*(N-1));   % Mark eta=0.4 alloc.
    plot(etaLR.Q1(indEtaMd)      ,etaLR.cp1(indEtaMd),'*g','MarkerSize',5)
    text(etaLR.Q1(indEtaMd)-ddc/5,etaLR.cp1(indEtaMd),...
         ['$\eta=', sprintf(etaFrm,etaMd), '$ (LR)'],...
         'Color','g','VerticalAlignment','top','Interpreter','Latex',...
         'HorizontalAlignment','right')
                                    % Mark eta=0.2-planner's allocation.
    plot(etaLR.Q1(indEtaLo)      ,etaLR.cp1(indEtaLo),'*g','MarkerSize',5)

    text(etaLR.Q1(indEtaLo)-ddc/5,etaLR.cp1(indEtaLo),...
         { ['$\eta=', sprintf(etaFrm,etaLo), '$ (LR)'], '(o: Pareto improvements)' } ,...
         'Color','g','VerticalAlignment','top','Interpreter','Latex',...
         'HorizontalAlignment','right')                                

    % Figure that plots the LR planner's allocation:
    figure('Name','Planner’s solution in illustrative model with LTC (Fig.B3 appendix)');
    nRow = 3;  nCol = 2;            % How many subplots.
    % Treat low wealth in first column of plots:
    subplot(nRow,nCol,1), hold on
    MarkParImp(etaPar,etaKid,ybdC)  % Mark Pareto improvements with grey box
                                    % (use extra function below).
    etaMA = etavec(etaLR.MA);       % Get starting and 
    etaMAlo = etaMA(1  );           % ending point of MA 
    etaMAhi = etaMA(end);           % on y-axis.
    MarkMA(etaMAlo,etaMAhi,ybdC)    % Mark MA choice in darker grey (again, 
                                    % use function below).
    plot([0,1],[0,0],'-k')          % Mark zero line.
    % Use extra function below to plot and mark lines:
    PlotLine(etavec,etaLR.cp0,'--','r','$c^p_{\eta,0}$'     ,0.8,  'left','top'   ,ParImp1EtaLR) % parent, planner: mark commitment alloc.
    PlotLine(etavec,      Cma, '-','r','$c^p_{eq,0}=C_{ma}$',0.5,'center','bottom'             ) % parent, eq'm
    PlotLine(etavec,etaLR.ck0,'-.','b','$c^k_{\eta,0}$'     ,0.8,  'left','bottom',ParImp1EtaLR) % child, planner: mark commitm. alloc.
    PlotLine(etavec,      yk , '-','b','$c^k_{eq,0}=y^k$'   ,0.5,'center','bottom'             ) % child, eq'm

    ylim(ybdC)                      % Set limits for y-axis.
    title('low parent wealth: $a^p=0$','Interpreter','Latex')
    ylabel('consumption'              ,'Interpreter','Latex')

    
    % Now, plot transfers and drift at low wealth:
    subplot(nRow,nCol,3), hold on
    MarkParImp(etaPar,etaKid ,ybdQ) % Mark Pareto improvements with grey box.
    MarkMA(   etaMAlo,etaMAhi,ybdQ) % Mark MA choice in darker grey.
    plot([0,1],[0,0],'-k')          % Mark zero line.
    PlotLine(etavec,       0 ,'--','k','$\dot{a}^p_{\eta,0}=\dot{a}^p_{eq,0}=Q_{eq,0}$',0.5, 'left','bottom'             ) % 0 drift (by model ass'ns)
    PlotLine(etavec, etaLR.Q0, '-','k',                                  '$Q_{\eta,0}$',0.6,'right','top'   ,ParImp1EtaLR) % eq'm transfer
    ylim(ybdQ)                      % Some formatting.
    ylabel('transfer, drift','Interpreter','Latex')

    % Plot values at low wealth:
    subplot(nRow,nCol,5), hold on
    Vp0eq = ( u(Cma) + alpha*u(yk) )/(rho+delta);   % Value functions at low 
    Vk0eq = (                u(yk) )/(rho+delta);   % wealth in equilibrium.
    MarkParImp(etaPar,etaKid ,ybdV) % Mark Pareto improvements with grey box.
    MarkMA(   etaMAlo,etaMAhi,ybdV) % Mark MA choice in darker grey.

    PlotLine(etavec,Vp0eq    , '-','r','$V^p_{eq,0}$ (MA)',0.5,'center',   'top'             )
    PlotLine(etavec,etaLR.Vp0,'--','r','$V^p_{\eta,0}$'   ,0.7,  'left',   'top',ParImp1EtaLR)
    PlotLine(etavec,Vk0eq    , '-','b','$V^k_{eq,0}$ (MA)',0.5,'center','bottom'             )
    PlotLine(etavec,etaLR.Vk0,'-.','b','$V^k_{\eta,0}$'   ,0.8,  'left','bottom',ParImp1EtaLR)  
    ylim(ybdV)                      % Limits for y-axis.
    xlabel('$\eta$: planner weight','Interpreter','Latex'),
    ylabel('values'                ,'Interpreter','Latex') 

    % In second column of plots, describe allocations at high wealth:
    % First, consumption:
    subplot(nRow,nCol,2), hold on
    etaLR.ck1 = ykIC+etaLR.Q1;  
    MarkParImp(etaPar,etaKid,ybdC) % Mark Pareto improvements with grey box.
    plot([0,1],[0,0],'-k')         % Mark zero line.
    PlotLine(etavec,etaLR.cp1,'--','r','$c^p_{\eta,1}$',0.7,  'left',   'top',ParImp1EtaLR)   % parent, planner
    PlotLine(etavec,  eqm.cp1, '-','r',  '$c^p_{eq,1}$',0.5,'center','bottom'             )   % parent, eq'm
    PlotLine(etavec,etaLR.ck1,'-.','b','$c^k_{\eta,1}$',0.8,  'left','bottom',ParImp1EtaLR)   % child, planner
    PlotLine(etavec,  eqm.ck1, '-','b',  '$c^k_{eq,1}$',0.8,'center','bottom'             )   % child, eq'm
    ylim(ybdC)
    title('high parent wealth: $a^p=1$'                 ,'Interpreter','Latex')
    
    % Transfers and drift at high wealth:
    subplot(nRow,nCol,4), hold on
    MarkParImp(etaPar,etaKid,ybdQ) % Mark Pareto improvements with grey box.
    PlotLine(etavec,      0  , ':','k',  '$\dot{a}^p_{eq,1}$',0.8, 'left',   'top'             )  % eq'm'drift          
    PlotLine(etavec,  eqm.Q1 , '-','k',          '$Q_{eq,1}$',0.8, 'left','bottom'             )  % eq'm transfer
    PlotLine(etavec,etaLR.dr1, ':','k','$\dot{a}^p_{\eta,1}$',0.5,'right',   'top',ParImp1EtaLR)  % planner drift
    PlotLine(etavec,etaLR.Q1 , '-','k',        '$Q_{\eta,1}$',0.8,'right',   'top',ParImp1EtaLR)  % planner transfer
    ylim(ybdQ)
    
    % Values at high wealth:
    subplot(nRow,nCol,6), hold on                   % Plot value functions at high wealth:
    MarkParImp(etaPar,etaKid,ybdV)                  % Mark Pareto improvements with grey box.
     text(etaKid,0.95*ybdV(1)+0.05*ybdV(end),{'commitment','allocations'},... 
         'Color',0.6*ones(1,3),'HorizontalAlignment','left','VerticalAlignment','bottom')
    PlotLine(etavec,eqm.Vp1  , '-','r','$V^p_{eq,1}=V^p_{no-hous,1}$'  ,0.3,'left','top'                )
    PlotLine(etavec,etaLR.Vp1,'--','r','$V^p_{\eta,1}$',0.3,  'right','bottom'   ,ParImp1EtaLR)
    PlotLine(etavec,eqm.Vk1  , '-','b','$V^k_{eq,1}$'  ,0.5,'center','bottom'             )
    PlotLine(etavec,noH.Vk1  ,'--','b','$V^k_{no-hous,1}$',1,'right','top'             )
    PlotLine(etavec,etaLR.Vk1,'-.','b','$V^k_{\eta,1}$',0.93,  'right','top',ParImp1EtaLR) 
    ylim(ybdV)
    xlabel('$\eta$: planner weight'                     ,'Interpreter','Latex'),
end


%% Helper functions (called in this script):

function MarkParImp(etaPar,etaKid,ybd)
    % Marks Pareto-improvements in plot by light grey box.
    patch([etaPar,etaKid,etaKid,etaPar],...         % x-coordinates,
          [ybd(1),ybd(1),ybd(2),ybd(2)],...         % y-coordinates,
           0.9*ones(1,3),'LineStyle','none')        % light grey.
end


function MarkMA(etaMAlo,etaMAhi,ybd)
    % Marks Pareto-improvements in plot by light grey box.
    patch([etaMAlo,etaMAhi,etaMAhi,etaMAlo],...     % x-coordinates,
          [ybd(1),ybd(1),ybd(2),ybd(2)],...         % y-coordinates,
           0.8*ones(1,3),'LineStyle','none')        % darker grey.
    text(0.5*(etaMAlo+etaMAhi),0.5*sum(ybd),'MA','Interpreter','Latex',...
         'HorizontalAlignment','center')            % Mark with text.
end


function PlotLine(xData,yData,lineSpec,col,lbl,xRat,horAlign,vertAlign,ParImp) %#ok<*DEFNU>
    % Plots line (xData,yData) and places text (lbl) next to line. This 
    % is done at ratio (xRat) of the x-axis, 0 being all the way to the left
    % and 1 all the way to the right. horAlign and vertAlign specify
    % alignment of the text. Marks Pareto improvements (logical vector ParImp)
    % if handed over.
    if isscalar(yData)              % If y-data is only a scalar:
        yData = yData*ones(size(xData));
        plot(xData,yData,lineSpec,'Color',col,'LineWidth',1)
                                    % Expand to format of x-axis and make 
                                    % thin line.
    else                            % Otherwise, plot thick line:
        plot(xData,yData,lineSpec,'Color',col,'LineWidth',2);
    end                             
    if nargin>8                     % If Pareto-improvement indeces handed 
        plot(xData(ParImp),yData(ParImp),'.','Color',col,'LineWidth',2,'MarkerSize',10);
    end                             % over, mark them with dots.
    ind = round(xRat*numel(xData)); % Find index where to put text box.
    text(xData(ind),yData(ind),lbl,'Color',col,'Interpreter','Latex',...
         'HorizontalAlignment',horAlign,'VerticalAlignment',vertAlign)
                                    % Put text box next to line with the 
end                                 % given alignment.


function [hjb,cp1,Q,cp0,Q0,MA,ck0] = ToyModelLtcLrPlannerCheckHJB(eta,dV,par)
% For a given planner-weight eta (model WITH LTC) and a given value-differential dV
% (capturing the continuation value of the game at high wealth),
% this function first calculates the allocation that the eta-planner would
% choose at low wealth (with either MA, i.e. MA=true, or IC, i.e.
% MA=false). For high wealth, the function then calculates the allocation the
% planner would choose given marginal value dV over dt (cp and Q)
% evaluates the HJB: If output hjb=0, then the guess dV is correct.
%
% Inputs:
% eta:          Usually a scalar, in [0,1].
% dV:           Usually a scalar, but also vectors and arrays allowed
%               (have to be conformable with inputs in par).
% par:          Structure. Contains parameters of the model. Fields read out 
%               in lines 30-35 below.
%
% Outputs:      
% hjb:          Usually a scalar. Evaluates HJB at high wealth. If hjb=0,
%               the input dV constitutes an equilibrium at high wealth.
% par:          Structure. Fields that were inputted stay as they are, the
%               following results are added:
%   cp1:        Parent consumption under high wealth (given guessed dVp).
%    Q1:        Transfer under high wealth (given guessed dVp).
%   cp0:        Parent consutmpion under low wealth (in equilbrium)
%    Q0:        Transfer under zero wealth in equilibrium).
%    MA:        Logical variable. If MA is chosen at zero wealth.
%   ck0:        Child consumption under low wealth.

ykIC = par.ykIC;  yp = par.yp;      % Read out endowments, ...
yk   = par.yk  ;
Cma  = par.Cma;                     % Medicaid consumption floor, ...
rho= par.rho; delta=par.delta;      % discount and death rate, ...
alpha = par.alpha;                  % parent altruism, ...
B = par.B;                          % and B: kid's continuation value 
                                    % at death when receiving bequest (zero
                                    % bequest value normalized to zero).

u = @(c) log(c);                    % Utility must be log: formulae rely on this.
    
etak = 1-eta+alpha*eta;             % Effective weight on kid.
Beta = etak.*B;                     % Death value for eta-planner.

% Obtain constrained allocation (with apdot=0, i.e. either at ap=0 or when 
% no dissavings occur at ap=1) under IC:
 QCon = (etak*yp - eta*ykIC)./(eta+etak) ;
                                    % Get constrained transfer: The
                                    % allocation on the adot=0 line that
                                    % the eta planner would choose.
cpCon = yp - QCon;                  % Get constrained parent consumption.

% Now, use constrained allocation for zero wealth:
ucp0 = u(cpCon);                    % Parent felicity.
ucp0(cpCon==0) = 0;                 % If cpCon=0, replace -inf by 0 since  
                                    % this must be associated with eta=0.
ckCon = ykIC+QCon;                  % Child consumption.                                    
uck0 = u(ckCon);                    % Child felicity.
uck0(ckCon==0) = 0;                 % If child consumption zero, replace by 
                                    % zero since this must be associated
                                    % with eta=1 and alpha=0.
                                    
uCon =  eta.*ucp0 + etak.*uck0;     % Get planner's flow payoff under
                                    % constrained IC allocation.
uMa  = eta*u(Cma)+ etak*u(yk);      % Get planner's flow payoff when choosing 
                                    % Medicaid.
% Can check here, fine:
% plot(eta,uCon,'--r',eta,uMa,'-k'), legend('IC constrained','MA')                                    
MA   = (uMa>uCon);                  % Logical variable: Medicaid decision.
                                    % Planner chooses MA for very low
                                    % weight on parent (starve parent, give
                                    % everything to kid) and for high
                                    % weight (give more than MA to parent)
cp0 = MA.*Cma + ~MA.*cpCon;
Q0  =           ~MA.* QCon;
ck0 = MA.* yk + ~MA.*(ykIC+QCon);
% Can check: 
% plot(eta,cp0,'--r',eta,ck0,'-b')  % Plot consumption of both.
% legend('c^p_0','c^k_0')
                                    
V0   =  (MA.*uMa + ~MA.*uCon )./(rho+delta) ;
                                    % Get eta-planner's value at low... 
V1   = V0+dV;                       % ...and at high wealth. 

% Now, solve for allocation at high wealth for guessed dV:
cpUnc = eta ./dV;                   % Unconstrained solution for c^p.
 QUnc = etak./dV-ykIC;              % Unconstrained solution for Q. In case 
                                    % that Qbd=0, can just impose the zero
                                    % upper bound here. 
drUnc = yp-cpUnc-QUnc;              % Drift under unconstrained solution.                                  % 
drPos = (drUnc>0);                  % Logical variable: if drift implied by
                                    % c^p and Q is positive.
Q = drPos.* QCon + ~drPos.* QUnc;   % Take constrained solution if drift 
cp1= drPos.*cpCon + ~drPos.*cpUnc;  % possible, otherwise unconstrained is OK.

drift = yp-cp1-Q;                   % Get drift from global solution.
  
ucp = eta*u(cp1);
ucp(cp1==0)=0;                      % Replace by zero if weight on parent is zero.

ck1= ykIC+Q;
uck = etak*u(ck1);
uck(ck1==0)=0;                      % Replace by zero if effective weight on kid zero.

H = ucp + uck + drift.*dV;          % Get Hamiltonian implied by this.
                                    
hjb = H + delta.*Beta - (rho+delta).*V1;   
                                    % Evaluate HJB: Get hjb=0 if we have a
                                    % solution, i.e. if we have guessed the
                                    % right dV.
end                     