% Solves illustrative model (="toy model") in paper Section 3 ("An illustrative
% model: Housing as a commitment device"). 
% 
% Runtime: ~1 second.
% Needed toolboxes: None (ie. only basic Matlab).
%
% TO CREATE FIGURE 6 in paper (title: "Payoffs(=Hamiltonians) from
% dt-allocations at a^p_t=1 in basic illustrative model"): 
% Run this script with option plotLRplanner=false and default parameters.

% TO CREATE FIGURE B.1 (title: "Planner’s solutions in% illustrative model")
% AND TABLE B.1 ("Allocations in basic illustrative model") IN PAPER
% APPENDIX B:
% Set plotLRplanner=true and run (using default parameters).

% (To create figure with positive Qs, as in presentation slides for Richmond
% and Konstanz, use flipQ=true.)

close all; clear                    % Create clean sheet.

%% Parameter and option configuration:
% Main option:
plotLRplanner = false;              % If line with long-run planner's allocation 
                                    % should be shown. Default: false.
% Model parameters                                    
rho   =0.04;                        % Discount rate (yearly). Default: 0.04.
r     = rho;                        % By assumption: rent=interest rate=rho.
delta = 1/4;                        % Death hazard: Default: 1/4, i.e.
                                    % Life exp. is 4 years.                        
yp    = 0.25;                       % Parent income.s Default: 0.25.
yk    = yp;                         % Child income. Default: =0.25=yp.
ykFact=   1;                        % How child income changes after death.
                                    % Default: 1 (no change).
alpha =   0;                        % Parent altruism.
                                    % Default (model in paper): 0.                                    
% Algorithm and plotting options:                                  
N = 71;                             % Number of grid points for plotting,
                                    % matters for density of points in
                                    % Pareto lense. Default: 71.
Neta = 1001;                        % Number of grid points for planner weights.
                                    % Default: 1001.
KidPower      = true;               % If all bargaining power is given to kid.
                                    % Can change to false, then parent
                                    % makes take-it-or-leave-it offer. 
                                    % Default: true.
plotDetail    = false;              % If plot with many details is shown.
                                    % Default (to reproduce paper figure):
                                    % false.
flipQ         = false;              % If to flip the horizontal axis (Q). Then,
                                    % Q is interpreted as child-to-parent.
                                    % Default: false.
MakeLatexTable= false;              % If to make a Latex table with planner
                                    % outcomes (only done if
                                    % plotLRplanner=true). Default: false.
LgtGrey = 0.9*[1,1,1];              % RGB value for light grey (used for iso-
                                    % apdot lines) and Pareto improvements
                                    % over renting allocation). Default:
                                    % 0.9*[1,1,1].
DrkGrey = 0.6*[1,1,1];              % RGB value for dark grey (used for 
                                    % Pareto improvements over (housing)
                                    % equilibrium and dt-efficient
                                    % allocations. Default: 0.6*[1,1,1].

% Tell user which model we're solving:
fprintf('\n')
fprintf('Solving basic illustrative model (no LTC) with parameters:\n')
fprintf('rho=%1.3f, r=%1.3f, delta=%1.3f, yp=%1.3f, yk=%1.3f.\n',...
        [rho      , r      , delta      , yp      , yk      ]       );
if alpha==0 
    fprintf('No altruism, ')
else
    fprintf('PARENT IS ALTRUISTIC: alpha=%1.3f, ',alpha)
end
if KidPower
    fprintf('child has bargaining power.\n')
else
    fprintf('PARENT HAS BARGAINING POWER.\n')
end

%% Solve model:
u = @(c) log(c);                    % Log utility. CANNOT CHANGE THIS, SOME 
                                    % OF THE CLOSED FORMS USED BELOW ARE 
                                    % ONLY VALID FOR LOG-UTILITY.  
if r==rho                           % In special case rho=r:                                 
    B = ( u(ykFact*yk+rho) - u(ykFact*yk) )/rho;    
                                    % Get death value for child in closed
else                                % form. Otherwise:
    % First, solve model without normalization B_0=0:
    % HJB at low wealth:
    % rho*B_0 = u(ykFact*yk)
    B0 = u(ykFact*yk)/rho;          % Value once wealth is gone.
    % HJB at high wealth:
    % rho*B_1 = max u(ck) + (ykFact*yk + r - ck)*dB
    % FOC: dB = B_1-B_0 = 1/ck
    % Function handle to check HJB for given dB:
    HJBB = @(dB) -rho*(B0+dB) + u(1./dB) + (ykFact*yk + r).*dB - 1;
    ckLo = 0.9*(ykFact*yk + r);     % Reasonable bounds for consumption.
    ckHi = 10*ykFact*yk;
    % An equivalent way of solving this HJB:
  %  B1 = @(ck) -rho*(B0+1./ck) + u(ck) + (ykFact*yk + r - ck)./ck;
  %  ckvec = linspace( ckLo, ckHi, 100);
  %  plot(ckvec,B1(ckvec))
    % ckB = fzero(B1,[ckLo,ckHi]);
    dBvec = linspace(1./ckHi,1./ckLo,100);
    plot(dBvec,HJBB(dBvec))
    dB = fzero(HJBB,[1./ckHi,1./ckLo]);
    % Now, impose normalization and set B as difference to value when
    % broke:
    B = dB;
end
% Get values when parent alive but wealth is zero (scalars).  Will collect all
% equilibrium outcomes as fields of structure eqm:
eqm.ck0 = yk;                       % Child consumes endowment (guess Q=0).
eqm.cp0 = yp-r;                     % Parent consumes endowment minus rent.
eqm.Vk0 =                 u(yk)  /(rho+delta);   
eqm.Vp0 = ( u(yp-r)+alpha*u(yk) )/(rho+delta);

% Now, find maximal transfer (i.e. lowest Q) that parent would accept
% without giving it back, which gives us a reasonable lower bound for Q.
% Find all quantities as functions of guesses for Q.
cpDict = (yp-r+yk)/(1+alpha);       % What parent would assign to herself of
                                    % flow income (only valid for LOG-UTILITY!)
Qdict  = yp-r - cpDict;             % Net transfer that parent would choose
                                    % if she could dictate.
Qvec   = linspace(0.999*Qdict,0,N); % Create vector with candidates for eqm.Q.
                                    % Avoid full transfer (1*Qdict) since
                                    % this leads to NaNs for alpha=0:
                                    % 1-by-N vector.

% Values under high wealth in housing equilibrium given a guess for Q
% (function handles):
Vp1 = @(Q) ( u(yp-Q) + alpha*u(yk+Q) + delta*alpha*B )/(rho+delta);
% plot(Qvec,Vp1(Qvec));             % Vp1 is decreasing in Q and maximized at Qdict.
Vk1 = @(Q) (                 u(yk+Q) + delta*      B )/(rho+delta);
% Vk1 is increasing in Q.

cpRent = @(Q) 1./(Vp1(Q)-eqm.Vp0);  % Function handle: parent consumption in
                                    % renting subgame given V^p(Q).
% plot(Qvec,cpRent(Qvec))           % cpRent is increasing in Q.

% Now, get the child's best response in the renting game:
% (In the paper, we focus on the case where this Q~ is zero. However, code here
% also includes case in which child wants to save through parent.)
QrentNeg = @(Q) (1/yk - Vk1(Q) + eqm.Vk0) < 0 ;
                                    % Indicator function if child wants to
                                    % give a transfer Q~ < 0 under the
                                    % renting regime. This is the derivative
                                    % of H^k_rent in Q evaluated at zero.
QrentUnc = @(Q) 1./( Vk1(Q) - eqm.Vk0 ) - yk;  
                                    % Closed form for net transfer child
                                    % would choose (also positive).
Qrent    = @(Q) QrentNeg(Q).* QrentUnc(Q); % Optimal transfer: Zero if kid 
                                    % wants to  take away from parent, negative 
                                    % otherwise (if kid wants to save 
                                    % through parent.
% plot(Qvec,QrentUnc(Qvec))         % QrentUnc is decreasing in Q.
% plot(Qvec,Qrent(Qvec));                        

% Hamiltonians under outside option (renting): Function handles.                                    
HpOut = @(Q) u(cpRent(Q)) + alpha*u(yk+Qrent(Q)) + (yp-cpRent(Q)-Qrent(Q)).*(Vp1(Q)-eqm.Vp0);  
HkOut = @(Q)                      u(yk+Qrent(Q)) + (yp-cpRent(Q)-Qrent(Q)).*(Vk1(Q)-eqm.Vk0);                                      
% Note: Under guess in paper, Qrent(Q)=0 here.
% plot(Qvec,HpOut(Qvec))           % H^p_out is increasing in Q.                                    
% plot(Qvec,HkOut(Qvec))           % H^k_out is decreasing in Q.

% Now, set up Hamiltonians under inside option (housing; drift is zero):
HpIn  = @(Q) u(yp-Q) + alpha*u(yk+Q);
HkIn  = @(Q)                 u(yk+Q);
% plot(Qvec,HpIn(Qvec))            % H^p_in is decreasing in Q. 
% plot(Qvec,HkIn(Qvec))            % H^k_in is increasing in Q.

% Now, get "incentive-compatibility" (IC) functions:
ICp = @(Q) HpIn(Q) - HpOut(Q);      % Difference between Hamiltonians: Positive
% if parent strictly preferes housing regime under candidate Q, negative
% if parent prefers outside option (renting regime). 
                                    % IC^p is decreasing in Q.
ICk = @(Q) HkIn(Q) - HkOut(Q);      % The same for the child: If this difference
                                    % is positive, the child prefers the
                                    % housing to renting regime.
                                    % IC^k is increasing in Q.

% Can comment in the following lines to make figure of IC-functions that shows 
% us if housing  equilibrium exists: If there is an interval on which both 
% IC-functions are above zero, housing-trust equilibrium exists.
% figure
% plot(Qvec,ICp(Qvec),'-r',Qvec,ICk(Qvec),'--b',Qvec,0*Qvec,'-k')
% xlabel('Q: candidate transfers')
% ylabel('IC^i(Q): incentive compatibility')
% legend('IC^p: parent','IC^k: child')

% Now, find equilibrium transfer Q_eq (if possible).
if ICp(0.999*Qdict)>0 && ICp(0.001*Qdict)<0
                                    % If IC^p cuts zero on relevant range:
    if KidPower                     % Under default, kid has bargaining power:
        eqm.Q = fzero(ICp,[0.999*Qdict,0.001*Qdict]); 
                                    % Find equilibrium candidate: Q at which
                                    % parent is indifferent between renting
                                    % and housing regime.
        if ICk(eqm.Q)>0             % If child also prefers housing regime:
            fprintf('Found unique housing-trust equilibrium for Q_eq=%1.3f.\n',eqm.Q)
                                    % Have found housing equilbrium.
        else                        % Otherwise, no housing eq'm:
            fprintf('Parent indifferent at Q=%1.3f, but child prefers renting regime for this Q.\n',eqm.Q)
        end   
    else
        eqm.Q = fzero(ICk,[0.999*Qdict,0.001*Qdict]); %#ok<*UNRCH> % Alternatively, can give all bargaining 
                                    % power to parent: Equilibrium looks
    end                             % very similar.    
 
else                                % If IC^p does not cut zero:
    disp('Could not find housing equilibrium.')
end

% 
eqm.cp1  = yp-eqm.Q;                % Parent consumption in housing eq'm.
eqm.Vp1  = Vp1(eqm.Q);              % Get equilibrium values at high wealth.  
eqm.Vk1  = Vk1(eqm.Q);
eqm.RSG.cp = cpRent(eqm.Q);         % Parent consumption in renting subgame (RSG).
eqm.RSG.Q  = Qrent(eqm.Q);          % Child transfer in renting subgame.
% Set up function handles for Hamiltonians in dt-game:
Hp = @(cp,Q) u(cp) + alpha*u(yk+Q) + (yp-Q-cp)*(eqm.Vp1-eqm.Vp0);
Hk = @(cp,Q)               u(yk+Q) + (yp-Q-cp)*(eqm.Vk1-eqm.Vk0);


%% Make figure with preferences in dt-game:
% Obtain axis bounds and make grids:
cpmax= max( eqm.cp1, eqm.RSG.cp );  % Obtain highest (plotted) parent consumption.
ddc  = (cpmax-yp)/20;               % Small distance for plotting.
cpmax= cpmax + 2*ddc;               % Obtain axis bounds for cp.
cpmin= yp-r;                        % Lower bound: What parent would consume
                                    % in equilibrium at low wealth.
Qmin = max(-yk, yp-eqm.RSG.cp )-3*ddc;                                     
                                    % Find reasonable upper bound for plotting
                                    % parent consumption: eating the
                                    % maximal transfer, or highest
                                    % renting-regime consumption (under any
                                    % transfer).
cpvec = linspace(yp,cpmax,N)';      % N-by-1 vector for c^p.

% Calculate efficient allocations (short-run planner over dt at high wealth):                                            
etavec = linspace(0,1,Neta);        % 1-by-Neta vector with weights on parent.
etakvec= 1-etavec+alpha*etavec;     % Effective weight on child.
% Fill structure etaDt with outcomes for the short-run planner (sets
% allocations over dt in high wealth state, taking as given the equilibrium
% continuation values):
etaDt.dV  = etavec*(eqm.Vp1-eqm.Vp0) + (1-etavec)*(eqm.Vk1-eqm.Vk0);
                                    % Get V^eta_Delta.
% First, get solutions with dissavings (adotp<=0 does not bind):
etaDt.cpUnc = etavec./etaDt.dV;     % Turns out the c^p-allocation is the same
                                    % if Q constraint binds or not --
                                    % consistent with continuity of the
                                    % contract curve!
etaDt.Qunc = min(0, etakvec./etaDt.dV - yk );
                                    % Impose non-positivity of Q.
% Second, get constrained solution (on housing-trust line):
etaDt.Qcon  = (etakvec*yp - etavec*yk)./(etavec+etakvec);
etaDt.cpCon = yp-etaDt.Qcon;
% Bring together constrained and unconstrained solution:
constr = (yp-etaDt.cpUnc-etaDt.Qunc)>0;
etaDt.Q = constr.* etaDt.Qcon + (~constr).* etaDt.Qunc;
etaDt.cp= constr.*etaDt.cpCon + (~constr).*etaDt.cpUnc;
                                    % If unconstrained solution yields
                                    % positive drift, must take constrained
                                    % solution.                                    
                                                                      
% Get levels for which to plot parent's and child indifference curves:
QminIC = -0.9*yk;
wgt = linspace(0,1,3);
cpIC = [(1-wgt)*(yp-QminIC)+wgt*eqm.cp1, eqm.RSG.cp, yp-eqm.Q/2];
QIC  = [(1-wgt)*QminIC+wgt*eqm.Q,        eqm.RSG.Q,    eqm.Q/2];
ckIC = yk+QIC;
levpar = Hp(cpIC,QIC);              % Levels for parent...
levkid = Hk(cpIC,QIC);              % and for child.

% Take some interesting allocations on Pareto frontier here with the following
% indeces for eta: (note that ratio cp/ck for eta-planner must be
% eta/(1-eta+eta*alpha):
etaLo = 0.2;                        % For which Pareto weights to plot
etaMd = 0.38;                        
etaHi = 0.5;
etaFrm= '%1.2f';                    % In which format to label eta-allocations.

indEtaLo   = 1+round(etaLo*(Neta-1));  % Get indeces on the Neta-size vector.
indEtaMd   = 1+round(etaMd*(Neta-1));
indEtaHi   = 1+round(etaHi*(Neta-1));  % Same for higher weight on parent

% Now, start plotting:   
figure('Name','Payoffs (=Hamiltonians) in dt-game... (paper: Fig. 6)');

hold on

Hpmat = Hp(cpvec,Qvec);             % Create large matrices with Hamiltonians
Hkmat = Hk(cpvec,Qvec);             % evaluated on N-by-N Cartesian grid.
drmat = yp-cpvec-Qvec ;             % N-by-N matrix with drift (apdot)
drPos = (drmat>0);                  % N-by-N logical matrix: if grid positive.
Hpmat(drPos) = NaN;                 % Positive-drift allocations don't make
Hkmat(drPos) = NaN;                 % sense, remove these.

% Plot parent indifference curves::
contour(Qvec,cpvec,Hpmat,levpar, '-r')
% contour(Qvec,cpvec,Hp(cpvec,Qvec), '-r')      % This is with Matlab
% choosing the H^p-levels (do this if levpar does not work). 
                                    % Parent ind. curves
plot([max(-yk,yp-eqm.RSG.cp),0],[eqm.RSG.cp,eqm.RSG.cp],'-r','LineWidth',2)
                                    % Parent best response in renting
                                    % regime.                                     
contour(Qvec,cpvec,Hkmat,levkid,'--b')  % Then: child ind. curves.
% contour(Qvec,cpvec,Hk(cpvec,Qvec),'--b')    % Again, can have Matlab 
                                    % choose the levels.
plot([eqm.RSG.Q,eqm.RSG.Q],[yp,cpmax],'--b','LineWidth',2)
                                    % Child best response in renting
                                    % regime.

if flipQ                            % Flip Q-axis if told so.
    set(gca, 'XDir','reverse')                                    
end                                    
    
% Now, mark areas that are Pareto improvements over renting and eq'm alloc.:
Hpmat = Hp(cpvec,Qvec);             % Create large matrices with Hamiltonians
Hkmat = Hk(cpvec,Qvec);             % evaluated on N-by-N Cartesian grid.
% Create logical N-by-N matrices that tell us if a point on grid is a 
% Pareto improvement over...
ParImpRent = (Hpmat>Hp(eqm.RSG.cp,eqm.RSG.Q)) & (Hkmat>Hk(eqm.RSG.cp,eqm.RSG.Q) ) & ~drPos;
                                    % ... the renting subgame outcome, ...
ParImpEqm  = (Hpmat>Hp(eqm.cp1   ,eqm.Q    )) & (Hkmat>Hk(eqm.cp1   ,eqm.Q    ) ) & ~drPos;
                                    % ... or the equilibrium itself.
cpmat = cpvec.*ones(1,N);           % Set up N-by-N matrices for cp and Q
 Qmat =  Qvec.*ones(N,1);           % to index them:
% With logical indexing, only pick points on grid that are Pareto
% improvements and mark them with a dot in...
plot([Qdict,0],[yp-Qdict,yp],'-k')  % Mark zero-savings allocations (i.e. 
                                    % those possible in housing regime).

if plotDetail                       % Only show this in detailed plot:
    plot(Qmat(ParImpRent(:)), cpmat(ParImpRent(:)),'.','Color',[0.9,0.9,0.9])
                                    % ... light grey for renting outcome,...
    plot(Qmat( ParImpEqm(:)), cpmat( ParImpEqm(:)),'.','Color',[0.6,0.6,0.6])
                                    % ... darker grey for eq'm outcome.
    % Plot iso-savings lines for the dynasty:
    isoLev = [-0.4,-0.2,0.2];
    [Cdr,hdr] = contour(Qvec,cpvec,drmat,isoLev,'-','Color',LgtGrey);
    clabel(Cdr,hdr,'Color',LgtGrey) % Label curves.
    plot(etaDt.Q,etaDt.cp,'-.','LineWidth',2,'Color',DrkGrey)  
                                    % Mark efficient allocations = Pareto
                                    % frontier.
else                                % If no detail wanted:
    plot(Qmat(ParImpRent(:)), cpmat(ParImpRent(:)),'.','Color',LgtGrey)
                                    % just grey for improvements over
end                                 % renting eq'm.

plot([-yk,-yk],[yp+yk,cpmax],'-.k') % Plot vertical line with minimal transfer:
                                    % Child gives all income.
                                    

% Mark important allocations:
plot(eqm.Q,eqm.cp1,'xk','MarkerSize',10,'LineWidth',2)        
text(eqm.Q-ddc/2,eqm.cp1,{'housing-trust','equilibrium','($\dot{a}^p=0$)'},...
     'HorizontalAlignment','right','VerticalAlignment','top','Interpreter','Latex')
                                    % Mark equilibrium.

plot(eqm.RSG.Q,eqm.RSG.cp,'ok','MarkerSize',5,'LineWidth',2)    
text(eqm.RSG.Q+ddc/2,eqm.RSG.cp,{'renting','subgame','outcome','($\dot{a}^p<0$)'},...
     'HorizontalAlignment','left', 'Interpreter','Latex'    )                                          
                                    % Mark renting allocation that would be
                                    % played if no bargain is reached.

plot(0,yp,'dk')                     % Endowment point.
text(ddc/4,yp,{'endowment','point'},'Interpreter','Latex', ...
     'HorizontalAlignment','left','VerticalAlignment','bottom' )
 
plot(0,yp-r,'xk')                   % Equilibrium allocation under low wealth:
text(0+ddc/4,yp-r,{'eq''m allocation','at low wealth'},'VerticalAlignment','bottom','Interpreter','Latex'); %'HorizontalAlignment','left')


% Mark better-direction for parent:
arrQcnt = 0.55;                     % Where arrow is centered as fraction
                                    % of [Qmin,0].
plot( [(arrQcnt-0.1)*Qmin, (arrQcnt+0.1)*Qmin], [eqm.RSG.cp+0.5*ddc,eqm.RSG.cp+0.5*ddc],'-r') % arrow stem,
plot(  (arrQcnt+0.1)*Qmin, eqm.RSG.cp+0.5*ddc, '<r') % arrow head.
text( arrQcnt*Qmin, eqm.RSG.cp+ddc, 'better for parent',           ...
     'VerticalAlignment','bottom', 'HorizontalAlignment','center',...
     'Color','r',  'Interpreter','Latex'                             )

% Mark better-direction for child:
plot([eqm.RSG.Q+0.5*ddc,eqm.RSG.Q+0.5*ddc ],[0.4*yp+0.6*cpmax, 0.6*yp+0.4*cpmax],'--b')
plot( eqm.RSG.Q+0.5*ddc, 0.6*yp+0.4*cpmax,'vb')   % again, arrow.
text( eqm.RSG.Q+    ddc, 0.5*yp+0.5*cpmax, {'better','for','child'},...
      'HorizontalAlignment','left', 'Color','b', 'Interpreter','Latex')

if plotDetail                       % If detailed plot asked for:
    % Now, mark efficient allocations that were used to pick the indifference 
    % level curves:
    mrkSizEta = 5;
    plot(etaDt.Q(indEtaMd)      ,etaDt.cp(indEtaMd),'*','Color',DrkGrey,'MarkerSize',mrkSizEta);
    text(etaDt.Q(indEtaMd)+ddc/5,etaDt.cp(indEtaMd),...
         ['$\eta=', sprintf(etaFrm,etaMd), '$'],...
         'Color',DrkGrey,'VerticalAlignment','bottom','Interpreter','Latex')
                                    % Mark eta=0.5-planner's allocation.
    plot(etaDt.Q(indEtaHi  )      ,etaDt.cp(indEtaHi  ),'*','Color',DrkGrey,'MarkerSize',mrkSizEta)                                
    text(etaDt.Q(indEtaHi  )+ddc/5,etaDt.cp(indEtaHi  ), ...
         ['$\eta=', sprintf(etaFrm,etaHi), '$'],...
         'Color',DrkGrey,'VerticalAlignment','bottom','Interpreter','Latex')  % high eta: 2/3

    plot(etaDt.Q(indEtaLo  )      ,etaDt.cp(indEtaLo  ),'*','Color',DrkGrey,'MarkerSize',mrkSizEta)                                
    text(etaDt.Q(indEtaLo  )+ddc/5,etaDt.cp(indEtaLo  ), ...
         ['$\eta=', sprintf(etaFrm,etaLo), '$'],...
         'Color',DrkGrey,'VerticalAlignment','bottom','Interpreter','Latex')  % low eta: 1/4
end

% Finally, mark the area with positive drift that does not make sense for
% the family:
startX = -0.8*yk;
startY = yp-startX;
endX   = startX-2*ddc;
endY   = startY-2*ddc;
plot( [startX,endX], [startY,endY],'-k')     % arrow stem,...
plot(         endX,          endY ,'^k')     % arrow head, and text:
text(endX, endY-ddc/2 ,                       ...
     {'$\dot{a}^p>0$:', 'dominated allocations', '(over-savings)'},...
     'Interpreter','Latex', 'HorizontalAlignment','center','VerticalAlignment','top'  ) 

% Also, mark infeasible allocations (c_k<0!) 
startX = -yk;
startY = (yp+yk + eqm.RSG.cp)/2;               % Go half distance of vertical lign.
% startX = -(cpmax-cpmin)*2/3;
% startY = yp-startX-ddc/2;
endX   = startX-1.5*ddc;
endY   = startY      ;
plot( [startX,endX], [startY,endY],'-k')     % arrow stem,...
plot(         endX,          endY ,'<k')     % arrow head, and text:
text(endX, endY-ddc/2 ,                       ...
     {'$c_k<0$:', 'infeasible'},...
     'Interpreter','Latex', 'HorizontalAlignment','center','VerticalAlignment','top'  ) 


%% Solve for equilibrium without housing (noH)
% Guess equilibrium with Q=0:
cpLo = 1.01*yp;                     % Bounds for finding solution.
cpHi = 10*yp;

dVfct = @(cp) 1./cp;                % Get value differential dV=V1-V0 consistent
                                    % with cp from consumption FOC.
Vp1fct = @(cp) ( u(cp) + alpha*u(yk) + (yp-cp).*dVfct(cp) + delta*alpha*B )./(rho+delta);
                                    % Function that returns value V^p_1 
                                    % consistent with given cp.
cp1fct = @(vp1) 1./(vp1-eqm.Vp0);   % From FOC, get optimal consumption response
                                    % to some value vp1.
cp1Find = @(cp1) cp1fct(Vp1fct(cp1)) - cp1;
                                    % Construct function to find fixed
                                    % point of HJB.
% ccvec = linspace(cpLo,cpHi,100);  % Checked: nice and decreasing.
% figure
% plot(ccvec,cp1Find(ccvec))
% Pack structure 'noH' with equilibrium outcomes:
noH.cp1 = fzero(cp1Find,[cpLo,cpHi]);% Find equilibrium consumption.
noH.dr1 = yp-noH.cp1;           noH.cp0 = yp-r;
noH.Vp1 = Vp1fct(noH.cp1);      noH.Vp0 = eqm.Vp0;
% Child outcomes:
noH.ck1 = yk;                   noH.ck0 = yk;
noH.Vk1 = ( u(yk) - noH.dr1.*eqm.Vk0 + delta*B ) ./ ( rho+delta-noH.dr1 );
noH.Vk0 = eqm.Vk0;
disp('Found equilibrium without housing (assuming Q=0).')


%% Long-run planner allocations:
if plotLRplanner                    % Calculate these if asked for. Pack 
                                    % results in structure etaLR.
    etaLR.cp1= zeros(size(etavec)); % Set up arrays for allocations the 
    etaLR.Q1 = zeros(size(etavec)); % long-run eta-planner chooses.
    etaLR.cp0= zeros(size(etavec)); 
    etaLR.Q0 = zeros(size(etavec)); 
    dVlo = 1/( 100*eqm.RSG.cp );    % Lower and upper bound for value 
    dVhi = 1/((eqm.RSG.cp/100));    % differential: Use renting consumption 
    dVvec = linspace( dVlo, dVhi,N);% as guideline.
    par.yk=yk; par.yp=yp; par.rho=rho; par.delta=delta; par.alpha=alpha;
    par.r = r; par.B=B;   par.Qbd=Inf; 
                                    % Pack structure par to hand over parameters.

    for i=1:Neta                    % Loop over all etas.
        eta = etavec(i);            % Read out current eta.
        F = @(dV) ToyModelLRPlannerCheckHJB(eta,dV,par);   

                                    % Set up function handle that evaluates
                                    % HJB for current eta...
        dVstar = fzero(F,[dVlo,dVhi]);
                                    % ... and solve the HJB.
        [~,sol] = ToyModelLRPlannerCheckHJB(eta,dVstar,par);
                                    % Call function again and get back
                                    % equilibrium policies at high wealth.
        etaLR.cp0(i)= sol.cp0;  etaLR.cp1(i) = sol.cp1;
        etaLR.Q0(i) = sol.Q0 ;  etaLR.Q1(i)  = sol.Q1 ;
    end                             % Read equilibrium allocations into etaLR.
    
    etaLR.ck0 = yk+etaLR.Q0;        % Get child consumption at low...
    etaLR.ck1 = yk+etaLR.Q1;        % ...and high wealth.
    etaLR.dr1 = yp-etaLR.cp1-etaLR.Q1;     
                                    % Obtain drift of parent wealth.
    % Get players' values at low wealth in the LR-planner allocations:
    etaLR.Vp0  = (u(etaLR.cp0) + alpha.*u(etaLR.ck0))./(rho+delta);
    etaLR.Vk0  = (                     u(etaLR.ck0))./(rho+delta);
    % Get both players' values under the different etas at high wealth:
    etaLR.Vp1 = ( u(etaLR.cp1) + alpha.*u(etaLR.ck1) + delta.*alpha.*B - etaLR.dr1.*etaLR.Vp0 )./(rho+delta-etaLR.dr1);
    etaLR.Vk1 = (                      u(etaLR.ck1) + delta.*       B - etaLR.dr1.*etaLR.Vk0 )./(rho+delta-etaLR.dr1);
    % Now, by linear interpolation, find threshold etas for the interval
    % with Pareto improvements over the no-housing equilibrium:
    FindEtaPar = @(eta) interp1(etavec,etaLR.Vp1,eta) - noH.Vp1;
    FindEtaKid = @(eta) interp1(etavec,etaLR.Vk1,eta) - noH.Vk1;    
    % Before: had improvements over the housing-trust equilibrium:
%     FindEtaPar = @(eta) interp1(etavec,etaLR.Vp1,eta) - eqm.Vp1;
%     FindEtaKid = @(eta) interp1(etavec,etaLR.Vk1,eta) - eqm.Vk1;
    etaPar = fzero(FindEtaPar,[etavec(2),etavec(end-1)]);
    etaKid = fzero(FindEtaKid,[etavec(2),etavec(end-1)]);
    
    % Now, obtain (again by linear interpolation) the outcomes for this
    % eta:
    fdNames = {'ck0','ck1','cp0','cp1','dr1','Vp1','Vk1','Vp0','Vk0'};
                                    % 
    for i=1:numel(fdNames)          % Pack cell with variable names and
        fdi = fdNames{i};           % loop over them.
        range.(fdi) = interp1(etavec,etaLR.(fdi),[etaPar,etaKid]);
        % e.g.: range.ck0 = [interp1(etavec,etaLR.ck0,[etaPar,etaKid]);
    end                             % Interpolate linearly to get the two 
                                    % long-run planner allocations that
                                    % give the interval for Pareto
                                    % improvements.
    disp('Found long-run planner (=efficient) allocations.') 
    
    % Create logical vector: For which etas the LR planner achieves a Pareto
    % improvement over the no-housing allocation:
    ParImp1EtaLR = (etaLR.Vp1>noH.Vp1) & (etaLR.Vk1>noH.Vk1);
    % Used to have: improvements over housing-trust equilibrium:
    % ParImp1EtaLR = (etaLR.Vp1>eqm.Vp1) & (etaLR.Vk1>eqm.Vk1);
    % Mark line with Pareto efficient allocations at high wealth:
    plot(etaLR.Q1              ,etaLR.cp1              ,'-g',...
         etaLR.Q1(ParImp1EtaLR),etaLR.cp1(ParImp1EtaLR),'.g'    )
                                    % Mark those that are Pareto improvements
                                    % over the housing equilibrium with dots.
    plot(etaLR.Q0(ParImp1EtaLR),etaLR.cp0(ParImp1EtaLR),'.g')
                                    % Mark also allocation
    % Find first and last index with Pareto improvements:    
    ParImpIndFirst = find(ParImp1EtaLR,1);
    ParImpIndLast  = find(ParImp1EtaLR,1,'last');
    ParImpInd      = [ParImpIndFirst, ParImpIndLast];  % Set up vector with both.
    if ~isnan(ParImpIndFirst)       % Mark LR-planner allocations.
        text( etaLR.Q1(ParImpIndFirst), etaLR.cp1(ParImpIndFirst),...
              {'LR planner','high wealth','(Par.imp.)'},'Color','g','HorizontalAlignment','left')
        text( etaLR.Q0(ParImpIndFirst), etaLR.cp0(ParImpIndFirst),...
              {'LR planner', 'low wealth','(Par.imp.)'},'Color','g','HorizontalAlignment','right')
    end
       
    % Make table with outcomes (18 characters for consn entries, 10 for
    % hazard):
       disp('')
       disp('TABLE B.1 (paper appendix B):')
       disp('            | high wealth | low wealth  | drift/-hazard | value h.w. ')
       disp('---------------------------------------------------------------------')
    fprintf('c^p: eqm    |%12.3f |%12.3f |%14.3f |%12.2f\n',[eqm.cp1,yp-r,0,eqm.Vp1])
    fprintf('     rent-sg|%12.3f\n',eqm.RSG.cp)
    fprintf('     commit |[%5.3f,%5.3f]|[%5.3f,%5.3f]|[%6.3f,%6.3f]|[%5.2f,%5.2f]\n',...
                         [ range.cp1   , range.cp0   , range.dr1   , range.Vp1]);
    fprintf('     no-hous|%12.3f |%12.3f |%14.3f |%12.2f\n',[noH.cp1,noH.cp0,noH.dr1,noH.Vp1])
       disp('---------------------------------------------------------------------')
    fprintf('c^k: eqm    |%12.3f |%12.3f |%14.3f |%12.2f\n',[yk+eqm.Q,yk,0,eqm.Vk1])
    fprintf('     commit |[%5.3f,%5.3f]|[%5.3f,%5.3f]|[%6.3f,%6.3f]|[%5.2f,%5.2f]\n',...
                         [ range.ck1   , range.ck0   , range.dr1   , range.Vk1   ]);
    fprintf('     no-hous|%12.3f |%12.3f |%14.3f |%12.2f\n',[noH.ck1,noH.ck0,noH.dr1,noH.Vk1])
       disp('---------------------------------------------------------------------')
    fprintf('Note: "drift/-hazard": asset drift at high wealth                  \n')
    fprintf('      "value h.w.": respective agent''s value at high wealth       \n')
    fprintf('      "eqm": housing-trust equilibrium in basic model  \n')
    fprintf('      "rent-sg": renting subgame outcome in housing-trust equilibrium.\n')
    fprintf('      "commit": long-run planner allocations that Pareto-dominate  \n')
    fprintf('      the no-housing equilibrium outcome when starting game at high\n')
    fprintf('      wealth, here: eta in [%5.3f,%5.3f].\n',[etaPar,etaKid]);
    fprintf('      "no-hous": non-cooperative model without housing asset.      \n')
      disp('---------------------------------------------------------------------')
    
    if MakeLatexTable               % Make Latex table in file if asked to:
        fileID = fopen('PlannerTable.tex','w');     % Open tex-file
        fprintf(fileID,'\\begin{tabular}{ll|cccc}\n');
        fprintf(fileID,'  &  &  $c^i_1$  &  $c^i_0$  &  $\\dot{a}^p_1$ (drift)  & $V^i_1$ \\\\\n');
        fprintf(fileID,'\\hline \n');
        fprintf(fileID,'parent (i=p): &  eqm   &%12.3f &%12.3f &%14.3f &%12.2f\\\\\n',[eqm.cp1,yp-r,0,eqm.Vp1]);
        fprintf(fileID,'  &   LR-plan&[%5.3f,%5.3f]&[%5.3f,%5.3f]&[%6.3f,%6.3f]&[%5.2f,%5.2f]\\\\\n',...
                             [ range.cp1   , range.cp0   , range.dr1   , range.Vp1]);
        fprintf(fileID,'  &   no-hous&%12.3f &%12.3f &%14.3f &%12.2f\\\\\n',[noH.cp1,noH.cp0,noH.dr1,noH.Vp1]);
        fprintf(fileID,'\\hline \n');
        fprintf(fileID,'child (i=k): & eqm    &%12.3f &%12.3f &%14.3f &%12.2f\\\\\n',[yk+eqm.Q,yk,0,eqm.Vk1]);
        fprintf(fileID,'  &   LR-plan&[%5.3f,%5.3f]&[%5.3f,%5.3f]&[%6.3f,%6.3f]&[%5.2f,%5.2f]\\\\\n',...
                             [ range.ck1   , range.ck0   , range.dr1   , range.Vk1   ]);
        fprintf(fileID,'  &   no-hous&%12.3f &%12.3f &%14.3f &%12.2f\\\\\n',[noH.ck1,noH.ck0,noH.dr1,noH.Vk1]);
        fprintf(fileID,'\\end{tabular}\n');
        fclose(fileID);
        disp('Wrote output table into file PlannerTable.tex.')
    end
end


%% Some final figure editing
% Set ticks on axes: 
xticks(   [-yk, eqm.Q, 0] )         % Set ticks for x-axis. 
yt = [yp-r, yp, eqm.cp1, yp+yk, eqm.RSG.cp]; 
                                    % 1-by-4 vector with ticks for y-axis. 
                                    % Is potentially in wrong order,
                                    % will sort below.
% Set ticks on x-axis:
xticklabels({[ '$-y^k='  , sprintf('%1.2f',-yk       ), '$'],...
             [ '$Q_{eq}=', sprintf('%1.2f',eqm.Q     ), '$'],...
             '$0$'                                          } );
% Create 1-by-4 cell with tick labels for y: Sort this cell below
% to get in right order.
ytLbl = { [     '$y^p-r=', sprintf('%1.2f',yp-r      ), '$'], ...  
          [       '$y^p=', sprintf('%1.2f',yp        ), '$'], ...                                         ...
          [  '$c^p_{eq}=', sprintf('%1.2f',eqm.cp1   ), '$'], ...
          [   '$y^p+y^k=', sprintf('%1.2f',yp+yk     ), '$'], ...
          ['$c^p_{rent}=', sprintf('%1.2f',eqm.RSG.cp), '$']      };

if abs(r)<0.00001                   % If r is very close to zero, remove 
    yt = yt(2:end);  ytLbl = ytLbl{2:end};  % first entry.
end

[yt,it] = sort(yt);                 % Order y-ticks in yt.
yticks(yt)                          % Set ticks on y-axis.
yticklabels(ytLbl(it));             % Label (also need correct order).

set(gca,'TickLabelInterpreter','Latex')
xlim([Qmin,2*ddc]);                 % Set axes limits.                               
ylim([cpmin,cpmax    ])
% Label axes and figure:
xlabel('$Q$: transfer'            , 'Interpreter','Latex')
ylabel('$c^p$: parent consumption', 'Interpreter','Latex')

legShort = {'parent indifference curves',                ...
            'parent best response (renting)',            ...
            'child indifference curves',                 ...
            'child best response (renting)',             ...
            'housing-trust allocations ($\dot{a}^p=0$)', ...
            'Pareto improvements over renting outcome'      };
                                    % Cell with first 6 legend entries.

if plotDetail                       % For detailed plot:
    title('Preferences (=Hamiltonians) in dt-game')
    legAddn  = {'Pareto improvements over equilibrium',     ...
                'iso-savings lines ($\dot{a}^p$=const)'    ,...
                'efficient allocations ($\eta$-dt-planner)'    };
                                    % Cell with addition to legend:
                                    % 1-by-3.
    legend([legShort,legAddn],'Location','southwest','Interpreter','Latex');
                                    % Print detailed legend: Concatenate
                                    % two cells with text to 1-by-9 cell.
else                                % For simple graph: just short legend.
    legend(legShort,'Location','southwest','Interpreter','Latex');
end 

%% Finally, make figure with planner's allocations.
if plotLRplanner
    figure('Name',' Planner’s solutions in illustrative model (Fig.B1 in paper appendix)');
    nRow=3;                         % Make 6 supblots.
    nCol=2; 
    ybdC = [ 0 , max(yk+yp,etaLR.ck1(1))];
    ybdQ = [min(-yk,etaLR.dr1(end)), max(yp,etaLR.Q1(1)) ];                       
                                    % Transfers.
    
    eqm.Vp0eq = ( u(yp-r) + alpha*u(yk) )/(rho+delta); % Value functions at low 
    eqm.Vk0eq = (                 u(yk) )/(rho+delta); % wealth in equilibrium.
    maxV = max([etaLR.Vp1(:); etaLR.Vk1(:)]);
    minV = min(eqm.Vp0eq,eqm.Vk0eq);
    ybdV = [minV - 0.4*(maxV-minV), maxV ];
    % Plot consumption at low wealth first:
    subplot(nRow,nCol,1)            % Upper left subplot:
    hold on   
    ybdQc = [-0.5,1];               % Set bounds for y-axis in consumption-
                                    % transfer plots.
    % Use helper function below to mark Pareto improvements in light grey
    % box:
    MarkParImp(etaPar,etaKid,ybdQc); 
    % Use another helper function (below) to plot lines, label them.
    % First, plot helper lines with equilibrium outcomes:
    PlotLine(etavec,  yp-r   ,'-' ,'r','$c^p_{eq,0}=y^p-r$',1  ,'right','top'   )
    PlotLine(etavec,  yk     ,'-' ,'b','$c^k_{eq,0}=y^k$'  ,1  ,'right','bottom')
    % Then thicker lines with planner solution:
    PlotLine(etavec,etaLR.cp0,'--','r','$c^p_{\eta,0}$'    ,0.9,'right','bottom',ParImp1EtaLR)
    PlotLine(etavec,etaLR.ck0,'-.','b','$c^k_{\eta,0}$'    ,0.1,'left' ,'bottom',ParImp1EtaLR)
    % Some editing:
    ylim(ybdC)                      % Limits for y-axis.
    title('low parent wealth: $a^p=0$','Interpreter','Latex')       
                                    % Both plots in column are for low wealth.
    ylabel('consumption','Interpreter','Latex')         
                                    % Both plots in first row are for cons/transfers.

    % Plot transfer and drift at low wealth:
    subplot(nRow,nCol,nCol+1), hold on % Subplot middle row, left column.
    Q0eq = zeros(size(etavec));     % Equilibrium drift is zero.
    plot([0,1],[0,0],'-k');         % Mark zero line.
    MarkParImp(etaPar,etaKid,ybdQ); % Mark Pareto improvements.
    PlotLine(etavec,etaLR.Q0,'-','k','$Q_{\eta,0}$'                         ,0.2,'left' ,'bottom',ParImp1EtaLR)
    PlotLine(etavec,Q0eq    ,':','k','$\dot{a}^p_{\eta,0}=\dot{a}^p_{eq,0}$',1  ,'right','bottom',ParImp1EtaLR)
    ylim(ybdQ)
    ylabel('transfer, drift','Interpreter','Latex')         
                                    % Both plots in first row are for cons/transfers.
     
    % Plot values at low wealth:
    subplot(nRow,nCol,2*nCol+1), hold on % Lower left subplot.
    MarkParImp(etaPar,etaKid,ybdV);      
    % Thin helper lines with equilibrium outcomes:
    PlotLine(etavec,  eqm.Vp0eq  ,'-' ,'r','$V^p_{eq,0}$'  ,1  ,'right','top'   )
    PlotLine(etavec,  eqm.Vk0eq  ,'-' ,'b','$V^k_{eq,0}$'  ,1  ,'right','bottom')
    % Thick lines with planner solutions:
    PlotLine(etavec,etaLR.Vp0,'--','r','$V^p_{\eta,0}$',0.9,'right','bottom',ParImp1EtaLR)
    PlotLine(etavec,etaLR.Vk0,'-.','b','$V^k_{\eta,0}$',0.1,'left' ,'bottom',ParImp1EtaLR)
    xlabel('$\eta$: weight on parent','Interpreter','Latex')  % Mark x-axis only on bottom plots.
    ylabel('values','Interpreter','Latex')                    % Both plots in lower row are values.
    ylim(ybdV);                   

    % Consumption at high wealth:
    subplot(nRow,nCol,2)            % Upper right subplot.
    hold on
    MarkParImp(etaPar,etaKid,ybdC); % Mark Pareto improvements in grey.
    PlotLine(etavec, eqm.cp1 ,'-' ,'r','$c^p_{eq,1}$'  ,1  ,'right','bottom')
    PlotLine(etavec, yk+eqm.Q,'-' ,'b','$c^k_{eq,1}$'  ,1  ,'right','top'   )
    PlotLine(etavec,etaLR.cp1,'--','r','$c^p_{\eta,1}$',0.7,'left' ,'top'   ,ParImp1EtaLR)
    PlotLine(etavec,etaLR.ck1,'-.','b','$c^k_{\eta,1}$',0.2,'left' ,'bottom',ParImp1EtaLR)
    ylim(ybdC)                      % Use common bounds for cp-Q plots.                          
    title('high parent wealth: $a^p=1$','Interpreter','Latex')
    
    % Plot transfers and drift at high wealth:
    subplot(nRow,nCol,nCol+2), hold on  % Subplot middle row, right column.
    % Mark zero line (don't use function, want this thin):
    plot([0,1],[0,0],'-k');         % Mark zero line.
    PlotLine(etavec,        0,':' ,'k','$\dot{a}^p_{eq,1}$'  ,1  ,'right','bottom'             )
    PlotLine(etavec,      eqm.Q,'-' ,'k','$Q_{eq,1}$'          ,0.2,'left','top'                 )
    MarkParImp(etaPar,etaKid,ybdQ); % Rest as before...
    PlotLine(etavec,etaLR.Q1 ,'-' ,'k','$Q_{\eta,1}$'        ,0.2 ,'left','bottom',ParImp1EtaLR)
    PlotLine(etavec,etaLR.dr1,':' ,'k','$\dot{a}^p_{\eta,1}$',0.8,'right','top'   ,ParImp1EtaLR)
    ylim(ybdQ)
    
    % Plot value functions at high wealth: 
    subplot(nRow,nCol,2*nCol+2), hold on % Lower right subplot.
    MarkParImp(etaPar,etaKid,ybdV)  % As above.
    text(etaKid,ybdV(end),{'commitment','allocations'},... 
         'Color',0.6*ones(1,3),'HorizontalAlignment','left','VerticalAlignment','top')
    PlotLine(etavec,  eqm.Vp1  ,'-' ,'r','$V^p_{eq,1}$'  ,1   ,'right','top'   )
    PlotLine(etavec,  eqm.Vk1  ,'-' ,'b','$V^k_{eq,1}$'  ,1   ,'right','bottom')
    PlotLine(etavec,etaLR.Vp1,'--','r','$V^p_{\eta,1}$',0.3,'right','bottom',ParImp1EtaLR)
    PlotLine(etavec,etaLR.Vk1,'-.','b','$V^k_{\eta,1}$',0.2,'right','top'   ,ParImp1EtaLR)
    xlabel('$\eta$: weight on parent','Interpreter','Latex'), 
    ylim(ybdV);                         
end


%% Helper functions (called in this script):

function PlotLine(xData,yData,lineSpec,col,lbl,xRat,horAlign,vertAlign,ParImp) %#ok<*DEFNU>
    % Plots line (xData,yData) and places text (lbl) next to line. Text 
    % is placed at ratio (xRat) of the x-axis, 0 being all the way left and
    % 1 all the way right, horAlign and vertAlign are alignment specifications.
    % Marks Pareto improvements (logical vector ParImp) if input handed over.
    if isscalar(yData)              % If y-data is only a scalar:
        yData = yData*ones(size(xData));
        plot(xData,yData,lineSpec,'Color',col,'LineWidth',1)
                                    % Expand to format of x-axis and make 
                                    % thin line.
    else                            % Otherwise, plot thick line:
        plot(xData,yData,lineSpec,'Color',col,'LineWidth',2);
    end                             
    if nargin>8                     % If Pareto-improvement indeces handed 
        plot(xData(ParImp),yData(ParImp),'.','Color',col,'LineWidth',2,'MarkerSize',10);
    end                             % over, mark them with dots.
    ind = round(xRat*numel(xData)); % Find index where to put text box.
    text(xData(ind),yData(ind),lbl,'Color',col,'Interpreter','Latex',...
         'HorizontalAlignment',horAlign,'VerticalAlignment',vertAlign)
                                    % Put text box next to line with the 
end                                 % given alignment.


function MarkParImp(etaPar,etaKid,ybd)
    % Marks Pareto-improvements in plot by light grey box.
    patch([etaPar,etaKid,etaKid,etaPar],...         % x-coordinates,
          [ybd(1),ybd(1),ybd(2),ybd(2)],...         % y-coordinates,
           0.9*ones(1,3),'LineStyle','none')        % light grey.
end


function [hjb,par] = ToyModelLRPlannerCheckHJB(eta,dV,par)
% For a given planner-weight eta and a given value-differential dV
% (capturing the continuation value of the game at high wealth),
% this program calculates the allocation that the long-run eta-planner would
% choose over dt (cp and Q) and evaluates the HJB: If output is hjb=0, then the
% guess dV is correct. 
%
% Inputs:
% eta:          Usually a scalar, in [0,1].
% dV:           Usually a scalar, but also vectors and arrays allowed
%               (have to be conformable with inputs in par).
% par:          Structure. Contains parameters of the model read out in
%               lines 25 to 31.
%
% Outputs:      
% hjb:          Usually a scalar. Evaluates HJB at high wealth. If hjb=0,
%               the input dV constitutes an equilibrium at high wealth.
% par:          Structure. Fields that were inputted stay as they are, the
%               following results are added in new fields:
%   cp1:        Parent consumption under high wealth (given guessed dVp).
%    Q1:        Transfer under high wealth (given guessed dVp).
%   cp0:        Parent consutmpion under low wealth (in equilbrium)
%    Q0:        Transfer under zero wealth in equilibrium).

yk = par.yk;  yp = par.yp;          % Read out endowments, ...
rho= par.rho; delta=par.delta;      % discount and death rate, ...
alpha = par.alpha; r=par.r;         % parent altruism, ...
B = par.B;                          % and B: kid's continuation value 
                                    % at death when receiving bequest (zero
                                    % bequest value normalized to zero).
Qbd=par.Qbd;                        % Bound on Q: either zero or infinite, 
                                    % depending on model.

u = @(c) log(c);                    % Utility must be log; formulae rely on this.
etak = 1-eta+alpha*eta;             % Effective weight on kid.
Beta = etak.*B;                     % Death value for eta-planner.

% Get consumption the planner assigns to parent at zero wealth using extra
% (nested) function below:
[cp0,ck0,Q0] = StaticAlloc(yp-r,yk);

% Now, use constrained allocation for zero wealth:
par.cp0 = cp0;                      % Write parent consumption and 
par.Q0  =  Q0;                      % transfer into structure.
V0 = FlowUeta(cp0,ck0) / (rho+delta);                                    
                                    % Using another nested function (below),
                                    % get eta-planner's value at low
                                    % wealth.

% Now, go to high wealth (backward induction):  
% Obtain constrained allocation (with apdot=0, i.e. either at ap=0 or when 
% no dissavings occur at ap=1):
[cpCon,~,QCon] =  StaticAlloc(yp,yk);
V1   = V0+dV;                       % Add guessed dV to get planner's value
                                    % at high wealth.                     
cpUnc = eta./dV;                    % Unconstrained solution for c^p.
 QUnc = min(Qbd,etak./dV-yk);       % Unconstrained solution for Q. In case 
                                    % that Qbd=0, can just impose the zero
                                    % upper bound here. 

drUnc = yp-cpUnc-QUnc;              % Drift under unconstrained solution.                                  % 
drPos = (drUnc>0);                  % Logical variable: if drift implied by
                                    % c^p and Q is positive.
 Q1= drPos.* QCon + ~drPos.* QUnc;  % Take constrained solution if drift 
cp1= drPos.*cpCon + ~drPos.*cpUnc;  % positive, otherwise unconstrained is OK.
ck1= yk+Q1;
par.Q1  =  Q1;                      % Write these into output structure.
par.cp1 = cp1;
par.ck1 = ck1;
drift = yp-cp1-Q1;                  % Get drift from global solution.                                   
H = FlowUeta(cp1,ck1) + drift.*dV;  % Get Hamiltonian implied by this.                              
                                    
hjb = H + delta.*Beta - (rho+delta).*V1;   
                                    % Evaluate HJB: Get hjb=0 if we have a
                                    % solution, i.e. if we have guessed the
                                    % right dV.
  
% Extra (nested) function that solves static transfer problem:                                    
    function [cp,ck,Q] = StaticAlloc(incp,inck)
        Inc = incp+inck;            % Get aggregate endowment
        cpNoBd = Inc.*eta./(eta+etak);
                                    % What planner would assign to parent
                                    % if there was no bound on transfer Q.
        QNoBd  = incp - cpNoBd;     % Transfer that planner would choose if 
                                    % there was no bound.
        Q = min( Qbd, QNoBd );      % Now, impose the upper bound on Q.
        cp = incp - Q;              % Get parent consumption.
        ck = inck + Q;              % and child consumption.
    end

% Extra (nested) function that obtains flow utility for the planner given
% consumption rates.
    function U = FlowUeta(cp,ck)
       up = u(cp);                  % Get parent felicity.
       up(cp==0) = 0;               % If cp=0, replace -inf by 0 since  
                                    % this must be associated withm eta=0
                                    % to avoid infinities here.
        uk = u(ck);                 % Get child felicity.
        uk(ck==0) = 0;              % Replace -inf by 0 since this must be 
                                    % associated with eta=1.
        U =  eta*up + etak*uk;      % Aggregate to planner utility.
    end

end
          

                   