# This supplementary material is hosted by Eurosurveillance as supporting information alongside the \textit{Epidemiological and clinical insights from SARS-CoV-2 RT-PCR crossing threshold values} on behalf of the authors who remain responsible for the accuracy and appropriateness of the content. The same standards for ethics, copyright, attributions and permissions as for the article apply. Supplements are not edited by Eurosurveillance and the journal is not responsible for the maintenance of any links or email addresses provided therein.


### R initialisation 
rm(list=ls(all=TRUE)) # clears variables1 
def.par<-par(no.readonly=TRUE) # default graphical parameters

#############################################
##  loading packages


library("dplyr")
library("tidyr")

library("ggplot2")
library("ggpubr")
library("ggsci")       # for nice color palettes

library("texreg")
library("Hmisc")
library("summarytools")

library("raster") #for graphical plots
library("ggmap")
library("sf")
library("maps")
library("mapproj")

library("nlme")

library("tidyquant")



setwd("/home/samuel/Documents/Hote-parasite/COVID-2019/Ct/national/")

Sys.setlocale("LC_TIME", "en_IN") #use English for dates


################################################################
##  loading data
################################################################


Df<-read.csv2(file="DATA_cleaned_all.csv",header=T)
Df<-dplyr::select(Df,-1)
Df<-as_tibble(Df)



# filter values
Df <- Df %>% dplyr::filter(
  (replicat==1)  #one value per patient 
  &(Ct>10) # remove 0 Cts
  &(Ct<45) # remove Cts too large
  &(age>0)&(age<90)
)

#replacing NAs by "unknown" for the plots
Df$symptom_details = replace(Df$symptom_details,which(is.na(Df$symptom_details)),"unknown")
Df$symptoms = replace(Df$symptoms,which(is.na(Df$symptoms)),"unknown")
Df$lieu_prelevement = replace(Df$lieu_prelevement,which(is.na(Df$lieu_prelevement)),"unknown")
Df$nature_prelevement = replace(Df$nature_prelevement,which(is.na(Df$nature_prelevement)),"unknown")


Df$date_prelevement<-as.Date(Df$date_prelevement)
Df$resultat_qualitatif<-as.factor(Df$resultat_qualitatif)
Df$resultat_qualitatif <- factor(Df$resultat_qualitatif, levels=c("0", "3", "1"), ordered=TRUE) 
Df$departement<-as.factor(Df$departement)
Df$infos_date_symptomes<-as.factor(Df$infos_date_symptomes)
Df$lieu_prelevement<-as.factor(Df$lieu_prelevement)
Df$symptoms<-as.factor(Df$symptoms)
Df$symptom_details<-as.factor(Df$symptom_details)
Df$partenaire_etude<-as.factor(Df$partenaire_etude)
Df$laboratoire_analyse<-as.factor(Df$laboratoire_analyse)
Df$sexe<-as.factor(Df$sexe)

names(Df)

################################################################
#### Plotting the tests done (Figure S1)
################################################################


tests_done <-  Df %>% group_by(id_patient,date_prelevement)

n_groups(tests_done)

format(min(as.Date(tests_done$date_prelevement)),"%d %b %Y")
format(max(as.Date(tests_done$date_prelevement)),"%d %b %Y")



Df_tests_done <- ungroup(Df) %>% group_by(date_prelevement,id_patient) %>% summarise(first(departement))
Df_tests_done <- Df_tests_done %>% rename(departement=`first(departement)`)



plot_testing<-ggplot(Df_tests_done)+
  geom_histogram(aes(x=as.Date(date_prelevement,format="%B %Y")),bins=100,alpha=0.9)+
  labs(x="sampling date",y="number of tests",fill="")+
  theme_light()+
  theme(legend.position = "none") 

# ggsave(
#   filename="all_tests_labs.pdf",
#   plot = plot_testing,
#   width = 8,
#   height = 5,
#   units = c("cm"),
#   dpi = 300,
# )



### plot data per French department (Figure S1)

formes <- getData(name="GADM", country="FRA", level=2) #map at the department level

formes$CC_2 <- replace(formes$CC_2,which(formes$CC_2=="2A"),"20")
formes$CC_2 <- replace(formes$CC_2,which(formes$CC_2=="2B"),"20")

the_table<-table(as.numeric(as.character(Df_tests_done$departement)))

idx <- match(as.numeric(as.character(formes$CC_2)),names(the_table))  #build index
formes$CC_2[which(is.na(idx))]

concordance <- the_table[idx]


formes$values <- log10(as.numeric(concordance))
formes$values <- replace(formes$values,which(is.na(formes$values)),0)


couleurs <- colorRampPalette(c('yellow', 'black'))
plot1<-raster::spplot(formes, "values",col.regions=couleurs(20),
                      main=list(label="number of tests analysed (in log10)"),
                      par.settings = list(fontsize = list(text = 10)),
                      margins=F,frame.plot=F,lwd=0.2)


ggarrange(plot_testing,plot1,
          labels = c("A", "B"),
          ncol = 2, nrow = 1,
          font.label = list(size = 10)
)


# ggsave(
#   filename="testing_panels.pdf",
#   plot = last_plot(),
#   width = 18,
#   height = 6,
#   units = c("cm"),
#   dpi = 300,
# )


################################################################
#### Plotting Ct data per asay
################################################################


old_Df<-Df  #store database before modifications

Df$technique_PCR<-as.character(Df$technique_PCR)
Df$technique_PCR = replace(Df$technique_PCR,which(Df$technique_PCR == "Technique Drosten"),"Technique Charité")
Df$technique_PCR = replace(Df$technique_PCR,which(Df$technique_PCR == "Perkin"),"PerkinElmer")
Df$technique_PCR<-as.factor(Df$technique_PCR)
Df$technique_PCR<-relevel(Df$technique_PCR, ref = "PerkinElmer")


Df<-Df%>%dplyr::filter((n_tech>0)&(!is.na(technique_PCR))&((technique_PCR!="other")))

table(Df$technique_PCR)




#removing NA values
data_plot_kits<-drop_na(Df,Rt_noshift,date_prelevement,age,
                        laboratoire_analyse,resultat_qualitatif)

#rescaling quantitative factors
data_plot_kits$age<-as.numeric(scale(data_plot_kits$age))
data_plot_kits$Rt_noshift<-as.numeric(scale(data_plot_kits$Rt_noshift))
data_plot_kits$date_prelevement<-as.numeric(scale(as.Date(data_plot_kits$date_prelevement)))



#performing a linear model to predict values and correct potential biases
modele_kits = lm(Ct ~  
                   Rt_noshift*date_prelevement + age+sexe+   
                 gene_cible+technique_PCR
                 + laboratoire_analyse   + symptoms + 
                   + nature_prelevement + lieu_prelevement  +
                   + resultat_qualitatif 
                 ,data = data_plot_kits)

summary(modele_kits)


n_fun <- function(x){
  return(data.frame(y = median(x), label = paste0("n=",length(x))))
}


#data_plot_kits$residuals_kits<-predict(modele_kits,data_plot_kits)
data_plot_kits$residuals_kits<-fitted.values(modele_kits)



plot_kits<-ggplot(data_plot_kits)+
  geom_violin(aes(x=gene_cible,fill=gene_cible,y=residuals_kits))+
  labs(x="target gene",y="estimated Ct")+
  facet_wrap( ~ technique_PCR, ncol=3) +
  geom_boxplot(aes(x=gene_cible,y=residuals_kits),width=0.1,outlier.shape = NA)+
  theme_light()+
  stat_summary(aes(x=as.factor(gene_cible),y=max(residuals_kits)),
    fun.data = n_fun,
    geom = "text",
    hjust = 0.5,
    vjust = 0.9,
    size=2
  )+
  theme(legend.position = "none")+
  scale_color_aaas()+
  scale_fill_aaas()+
  scale_x_discrete(limits=c("O","S","E","N","N_O","ci"),labels=c("ORF1","S","E","N","N&O","IC"))
                   


# ggsave(
#   filename="Ct_kit_gene_new.pdf",
#   plot = plot_kits,
#   width = 18,
#   height = 14,
#   units = c("cm"),
#   dpi = 300,
# )




################################################################
# Main paper figure with 6 panels
################################################################


Df<-old_Df



Df_model<-drop_na(Df,Rt_noshift,date_prelevement,age,gene_cible,technique_PCR,
                  laboratoire_analyse,resultat_qualitatif)



Df_model<-dplyr::filter(Df_model,(gene_cible!="ci")&(age>0)&(age<90)) #removing internal control genes

#rescaling quantitative factors
Df_model$age<-as.numeric(scale(Df_model$age))
Df_model$Rt_noshift<-as.numeric(scale(Df_model$Rt_noshift))
Df_model$date_prelevement<-as.numeric(scale(as.Date(Df_model$date_prelevement)))



modele_result = lm(Ct ~  
                     Rt_noshift*date_prelevement + age+sexe+
                      gene_cible+technique_PCR
                    + laboratoire_analyse   + symptoms + 
                      + nature_prelevement + lieu_prelevement  +
                      + resultat_qualitatif  ,
                    data = Df_model)



summary(modele_result)

#Df_model$residuals_result<-predict(modele_result,Df_model)

Df_model$residuals_result<-fitted.values(modele_result)

plot_Ct_qual_res<-ggplot(Df_model)+
  geom_violin(aes(x=as.factor(resultat_qualitatif),fill=as.factor(resultat_qualitatif),
                  y=as.numeric(residuals_result)),size=0.2)+
  labs(x="",y="estimated Ct")+
  geom_boxplot(aes(x=as.factor(resultat_qualitatif),y=residuals_result),width=0.1,outlier.shape = NA)+
  theme_light()+
  theme(legend.position = "none")+
  scale_x_discrete(limits=c("0","3","1"),labels=c("negative","weak positive","positive"))+
  scale_color_aaas()+
  scale_fill_aaas()


# ggsave(
#   filename="Ct_qualitative_result.pdf",
#   plot = plot_Ct_qual_res,
#   width = 7,
#   height = 4,
#   units = c("cm"),
#   dpi = 300,
# )




plot_Ct_target_gene<-ggplot(Df_model)+
  geom_violin(aes(x=as.factor(gene_cible),fill=as.factor(gene_cible),
                  y=as.numeric(residuals_result)),size=0.2)+
  geom_boxplot(aes(x=as.factor(gene_cible),y=residuals_result),width=0.1,outlier.shape = NA)+
  # stat_summary(aes(x=as.factor(gene_cible),y=max(residuals_target)),
  #   fun.data = n_fun, 
  #   geom = "text", 
  #   hjust = 0.5,
  #   vjust = 0.9
  # )+
    labs(x="target gene",y="estimated Ct")+
  theme_light()+
  theme(legend.position = "none")+
  scale_x_discrete(limits=c("O","E","N"),labels=c("ORF1","E","N")
                   )

# ggsave(
#   filename="Ct_target_gene.pdf",
#   plot = plot_Ct_target_gene,
#   width = 7,
#   height = 4,
#   units = c("cm"),
#   dpi = 300,
# )



plot_Ct_symptomatic<-ggplot(Df_model)+
  geom_violin(aes(x=as.factor(symptom_details),fill=as.factor(symptom_details),
                  y=as.numeric(residuals_result)),size=0.2)+
  labs(x="",y="estimated Ct")+
  geom_boxplot(aes(x=as.factor(symptom_details),y=residuals_result),width=0.1,outlier.shape = NA)+
  theme_light()+
  theme(legend.position = "none")+
   scale_x_discrete(limits=c("unknown","symptomatic","asymptomatic"),
                    labels=c("unknown","symptomatic","asymptomatic"))+
  scale_color_aaas()+
  scale_fill_aaas()

# 
# ggsave(
#   filename="Ct_symptoms.pdf",
#   plot = plot_Ct_symptomatic,
#   width = 8,
#   height = 4,
#   units = c("cm"),
#   dpi = 300,
# )




plot_Ct_days_post_symptoms<-ggplot(Df_model)+
  geom_violin(aes(x=as.factor(symptoms),fill=as.factor(symptoms),
                  y=as.numeric(residuals_result)),size=0.2)+
  geom_boxplot(aes(x=as.factor(symptoms),y=residuals_result),width=0.1,outlier.shape = NA)+
  labs(x="days post symptoms",y="estimated Ct")+
  theme_light()+
  theme(legend.position = "none") +
  scale_x_discrete(limits=c("unknown","moins_4_jours","entre_4_7_jours","entre_8_14_jours","plus_de_14_jours")
                   ,labels=c("unknown","< 4","4 to 7","8 to 14","> 14" ))+
  scale_color_nejm()+
  scale_fill_nejm()


# ggsave(
#   filename="Ct_days_post_symptoms.pdf",
#   plot = plot_Ct_days_post_symptoms,
#   width = 8,
#   height = 4,
#   units = c("cm"),
#   dpi = 300,
# )





plot_Ct_sample_type<-ggplot(Df_model)+
  geom_violin(aes(x=as.factor(nature_prelevement),fill=as.factor(nature_prelevement),
                  y=as.numeric(residuals_result)),size=0.2)+
  labs(x="sample type",y="estimated Ct")+
  geom_boxplot(aes(x=as.factor(nature_prelevement),y=residuals_result),width=0.1,outlier.shape = NA)+
  theme_light()+
  theme(legend.position = "none") +
  scale_x_discrete(limits=c("unknown","OP","NP"),labels=c("unknown","other","NP"))+
  scale_color_aaas()+
  scale_fill_aaas()

# ggsave(
#   filename="Ct_sample_type.pdf",
#   plot = plot_Ct_sample_type,
#   width = 8,
#   height = 4,
#   units = c("cm"),
#   dpi = 300,
# )




plot_Ct_sampling_location<-ggplot(Df_model)+
  geom_violin(aes(x=as.factor(lieu_prelevement),fill=as.factor(lieu_prelevement),
                  y=as.numeric(residuals_result)),size=0.2)+
  labs(x="sampling facility",y="estimated Ct")+
  geom_boxplot(aes(x=as.factor(lieu_prelevement),y=residuals_result),width=0.1,outlier.shape = NA)+
  theme_light()+
  theme(legend.position = "none") +
  scale_x_discrete(limits=c("screening","hospital","ehpad","prison","unknown"),labels=c("city","hospital","ACF","prison","unknown"))+
  scale_color_aaas()+
  scale_fill_aaas()

# ggsave(
#   filename="Ct_sampling_location.pdf",
#   plot = plot_Ct_sampling_location,
#   width = 8,
#   height = 4,
#   units = c("cm"),
#   dpi = 300,
# )



### same analysis without rescaling for age, date, and Rt

#### Ct vs Rt 

Df<-old_Df

Df_model<-drop_na(Df,Rt_noshift,date_prelevement,age,gene_cible,technique_PCR,
                  laboratoire_analyse,resultat_qualitatif)



Df_model<-dplyr::filter(Df_model,(gene_cible!="ci")&(age>0)&(age<90)) #removing internal control genes




modele_result = lm(Ct ~  
                     Rt_noshift*date_prelevement + age+sexe+
                     gene_cible+technique_PCR
                   + laboratoire_analyse   + symptoms + 
                     + nature_prelevement + lieu_prelevement  +
                     + resultat_qualitatif  ,
                   data = Df_model)



summary(modele_result)

#Df_model$residuals_result<-predict(modele_result,Df_model)

Df_model$residuals_result<-fitted.values(modele_result)





plot_Ct_Rt<-ggplot(dplyr::filter(Df_model,Rt_noshift<=1.5),
                   aes(y=residuals_result,x=Rt_noshift)) +
  geom_smooth(size=0.3)+
  theme_light()+
  labs(x="reproduction number",y="estimated Ct")+
  scale_color_aaas()+
  scale_fill_aaas()


# ggsave(
#   filename="Ct_Rt.pdf",
#   plot = plot_Ct_Rt,
#   width = 8,
#   height = 6,
#   units = c("cm"),
#   dpi = 300,
# )



######## Ct vs date



plot_Ct_date<-ggplot(Df_model,
                     aes(y=residuals_result,x=as.Date(date_prelevement))) +
  geom_smooth(size=0.3)+
  theme_light()+
  labs(x="sampling date",y="estimated Ct")+
  scale_color_aaas()+
  scale_fill_aaas()



# 
# ggsave(
#   filename="Ct_date.pdf",
#   plot = plot_Ct_Rt,
#   width = 8,
#   height = 6,
#   units = c("cm"),
#   dpi = 300,
# )




#### Ct vs age and sex


plot_Ct_age_sex<-ggplot(Df_model,
                        aes(x=as.numeric(age),y=as.numeric(residuals_result)))+
  theme_light()+
  geom_smooth(aes(color=sexe,fill=sexe),size=0.3)+
  labs(x="age",y="estimated Ct",fill="sex")+
  guides(color = FALSE)+
  scale_color_aaas()+
  scale_fill_aaas()



# ggsave(
#   filename="Ct_age_sex.pdf",
#   plot = plot_Ct_age_sex,
#   width = 8,
#   height = 6,
#   units = c("cm"),
#   dpi = 300,
# )




ggarrange(#plot_residuals,
          plot_Ct_qual_res,plot_Ct_days_post_symptoms,
#         plot_Ct_sampling_location,plot_Ct_sample_type,
          plot_Ct_age_sex,plot_Ct_target_gene,
          plot_Ct_date,plot_Ct_Rt,
          labels = c("A", "B", "C", "D", "E", "F"#,"G", "H"
                     ),
          ncol = 2, nrow = 3,
          font.label = list(size = 10)
          )



# ggsave(
#   filename="all_panels_new.pdf",
#   plot = last_plot(),
#   width = 18,
#   height = 12,
#   units = c("cm"),
#   dpi = 300,
# )





################################################################
# Plotting residuals of the linear model (supplementary figure)
################################################################


Df<-old_Df

# scaling numerical variables 
Df$date_scaled<-scale(as.Date(Df$date_prelevement))
Df$date_scaled<-as.numeric(Df$date_scaled)
Df$age_scaled<-scale(as.numeric(Df$age))
Df$age_scaled<-as.numeric(Df$age_scaled)
Df$Rt_scaled<-scale(as.numeric(Df$Rt_noshift))
Df$Rt_scaled<-as.numeric(Df$Rt_scaled)


#recalculating the linear model
modele_general = lm(Ct ~  
                      Rt_scaled*date_scaled + age_scaled+sexe +
                      gene_cible+technique_PCR
                    + laboratoire_analyse   + symptoms + 
                      + nature_prelevement + lieu_prelevement  +
                      + resultat_qualitatif  +variable_controle_id,
                    data = dplyr::filter(Df,(gene_cible!="ci")))



plot_residuals<-ggplot(as.data.frame(residuals(modele_general)))+
  geom_density(aes(residuals(modele_general)),size=0.3)+
  theme_light()+
  labs(x="linear model Ct residual",y="density")


# ggsave(
#   filename="residuals_lm.pdf",
#   plot = plot_residuals,
#   width = 6,
#   height = 4,
#   units = c("cm"),
#   dpi = 300,
# )


Du <- old_Df %>% dplyr::filter(
  (resultat_qualitatif == 1) # test is positive
  &(Ct>10) # remove Cts smaller than 10
  &(Ct<45) # remove Cts larger than 45
  #  & (date_prelevement>"2020-05-11") #post lock-down
  & (date_prelevement>="2020-07-01") #post summer 2020
  & (gene_cible!="ci") #remove internal control gene
  & (lieu_prelevement=="screening") # from screening
  & (age>5) # not too young
  & (age<81) # not too old
)



# scaling numerical variables 
Du$date_scaled<-scale(as.Date(Du$date_prelevement))
Du$age<-scale(as.numeric(Du$age))
Du$Rt_scaled<-scale(as.numeric(Du$Rt_noshift))



## cleaning variables to drop removed factors
Du$laboratoire_analyse = drop.levels(as.factor(Du$laboratoire_analyse))
Du$technique_PCR = drop.levels(as.factor(Du$technique_PCR))
Du$gene_cible = drop.levels(as.factor(Du$gene_cible))
Du$nature_prelevement = drop.levels(as.factor(Du$nature_prelevement))
Du$lieu_prelevement = drop.levels(as.factor(Du$lieu_prelevement))



Du<-drop_na(Du,laboratoire_analyse,technique_PCR,gene_cible,age)
Du_saved<-Du



# mixed effect model
modele_noR = lme(Ct ~ 1+technique_PCR+age,
                 random=(~1|laboratoire_analyse),data = Du)


plot_residuals_glmm<-ggplot(as.data.frame(residuals(modele_noR)))+
  geom_density(aes(residuals(modele_noR)))+
  theme_light()+
  labs(x="linear mixed model Ct residual",y="density")


# ggsave(
#   filename="residuals_glmm.pdf",
#   plot = plot_residuals_glmm,
#   width = 6,
#   height = 4,
#   units = c("cm"),
#   dpi = 300,
# )



ggarrange(plot_residuals,plot_residuals_glmm,
  labels = c("A", "B"#, "C", "D", "E", "F","G", "H"
  ),
  ncol = 2, nrow = 1,
  font.label = list(size = 10)
)

# ggsave(
#   filename="residuals.pdf",
#   plot = last_plot(),
#   width = 18,
#   height = 5,
#   units = c("cm"),
#   dpi = 300,
# )


################################################################
# Calcute correlation between Rt and Ct residuals times series 
################################################################



Du$Ct_residuals_raw<-residuals(modele_noR)


# compute daily summary statistics
Du_daily <- Du %>% group_by(date_prelevement) %>% summarise(median_Ct_raw=median(Ct),
                                                            median_Ct_residuals=median(Ct_residuals_raw),
                                                            median_Rt=median(Rt_noshift),
                                                            skew_Ct_raw=skewness(Ct),
                                                            skew_Ct_residuals=skewness(Ct_residuals_raw)
)

Du_daily$Ct_residuals<-zoo::rollmean(Du_daily$median_Ct_residuals,7,align="right",fill=NA)
Du_daily$Ct_residuals_skew<-zoo::rollmean(Du_daily$skew_Ct_residuals,7,align="right",fill=NA)


# calculating cross-correlation functions
ccf_corr<-ccf(drop_na(Du_daily)$Ct_residuals,drop_na(Du_daily)$median_Rt,lag.max=21,main="",xlab="lag (days)", type = c("correlation"))[,1][[1]]
ccf_time<-ccf(drop_na(Du_daily)$Ct_residuals,drop_na(Du_daily)$median_Rt,lag.max=21,main="",xlab="lag (days)", type = c("correlation"))[,1][[4]]

clim0 <- qnorm((1 + 0.95)/2)/sqrt(ccf(drop_na(Du_daily)$Ct_residuals,drop_na(Du_daily)$median_Rt,lag.max=21,main="",xlab="lag (days)")[,1][[3]])



max_time<-ccf_time[which(abs(ccf_corr)==max(abs(ccf_corr)))]


plot_ccf<-ggplot(as.data.frame(cbind(ccf_time,ccf_corr)))+
  #  geom_hline(yintercept=c(clim0,-clim0),col="blue",linetype=2,size=0.5)+
  geom_rect(aes(xmin=min(ccf_time),ymin=-clim0,xmax=max(ccf_time),ymax=clim0),alpha=0.01,fill="blue")+
  geom_hline(yintercept=c(0),size=0.5)+
  geom_vline(xintercept=max_time,size=0.3,col="red",linetype=2)+
  geom_point(aes(x=ccf_time,y=ccf_corr),size=0.7)+
  theme_light()+
  xlim(c(min(ccf_time),max(ccf_time)))+
  ylim(c(-max(abs(ccf_corr)),max(abs(ccf_corr))))+
  labs(y="correlation",x="lag (days)")


# ggsave(
#   filename="ccf.pdf",
#   plot = plot_ccf,
#   width = 10,
#   height = 6,
#   units = c("cm"),
#   dpi = 300,
# )


# calculating cross-correlation functions
ccf_corr_skew<-ccf(drop_na(Du_daily)$Ct_residuals_skew,drop_na(Du_daily)$median_Rt,lag.max=21,main="",xlab="lag (days)", type = c("correlation"))[,1][[1]]
ccf_time_skew<-ccf(drop_na(Du_daily)$Ct_residuals_skew,drop_na(Du_daily)$median_Rt,lag.max=21,main="",xlab="lag (days)", type = c("correlation"))[,1][[4]]

clim0_skew <- qnorm((1 + 0.95)/2)/sqrt(ccf(drop_na(Du_daily)$Ct_residuals_skew,drop_na(Du_daily)$median_Rt,lag.max=21,main="",xlab="lag (days)")[,1][[3]])

max_time_skew<-ccf_time[which(abs(ccf_corr_skew)==max(abs(ccf_corr_skew)))]


plot_ccf_skew<-ggplot(as.data.frame(cbind(ccf_time_skew,ccf_corr_skew)))+
  #  geom_hline(yintercept=c(clim0,-clim0),col="blue",linetype=2,size=0.5)+
  geom_rect(aes(xmin=min(ccf_time_skew),ymin=-clim0_skew,xmax=max(ccf_time_skew),ymax=clim0_skew),alpha=0.01,fill="blue")+
  geom_hline(yintercept=c(0),size=0.5)+
  geom_vline(xintercept=max_time_skew,size=0.3,col="red",linetype=2)+
  geom_point(aes(x=ccf_time_skew,y=ccf_corr_skew),size=0.7)+
  theme_light()+
  xlim(c(min(ccf_time_skew),max(ccf_time_skew)))+
  ylim(c(-max(abs(ccf_corr_skew)),max(abs(ccf_corr_skew))))+
  labs(y="correlation",x="lag (days)")



ggarrange(plot_ccf,plot_ccf_skew,
          labels = c("A", "B"#, "C", "D", "E", "F","G", "H"
          ),
          ncol = 2, nrow = 1,
          font.label = list(size = 10)
)

# ggsave(
#   filename="ccfs.pdf",
#   plot = last_plot(),
#   width = 18,
#   height = 6,
#   units = c("cm"),
#   dpi = 300,
# )



