# Data and Code for "Portfolio Optimization via Quantum Zeno Dynamics on a Quantum Processor"

## Setup

### Python Environment

Required Python packages are specified in requirements.txt. Run the following command to install all dependencies.
```bash
conda create --name qzd python=3.8
conda activate qzd
pip install -r requirements.txt
```

Note that we tested our code using Python 3.8. Although other Python versions may also work, it is not guaranteed to be so.

### Directory for Output

A directory named `figures` must be created before executing the notebooks generating the figures.
<!-- language: lang-bash-->
    mkdir figures

## Portfolio Optimization Test Cases

As mentioned in the paper, we tested quantum optimization techniques on a set of generated portfolio optimization problems.
All problems assume an objective function according to the mean-variance portfolio optimization portfolio (see text in paper for details).

The first set of problems can be retrieved through

```python
from problem_instances import get_instance
num_assets = 4  # number of assets to choose from
inst = get_instance("single", num_assets)
mu = inst["mu"]  # expected return for each asset in the pool
sigma = inst["sigma"]  # covariance matrix of returns for the assets in the pool
budget = inst["budget"]  # maximum number of assets allowed in the portfolio
```

These problems have a single inequality constraint on the portfolio size, defined by `budget`:

```python
sum(x) <= budget
```

where `x` is the sequence the values of the binary variables representing the positions on the assets.

The second set of problems can be retrieved through

```python
from problem_instances import get_instance
num_assets = 4  # number of assets to choose from
inst = get_instance("multi", num_assets)
mu = inst["mu"]  # expected return for each asset in the pool
sigma = inst["sigma"]  # covariance matrix of returns for the assets in the pool
budget = inst["budget"]  # maximum number of assets allowed in the portfolio
min_return = inst["min_return"]  # minimum total expected return of the portfolio
```

These problems have two inequality constraints, one on the portfolio size, defined by `budget` in the same way as the single-constraint problems, the other constraint is an inequality constraint on the minimum total expected return, i.e.

```python
sum(mu@x) >= min_return
```

where `x` is the sequence the values of the binary variables representing the positions on the assets, and `mu@x` denotes the vector inner product of `x` and `mu`.

For both sets of problems, `num_assets` ranges from 4 to 10.

## Data Files

A description of each data file in `data/` (data type, column name description for DataFrames, etc.)

### **1. simulation_results.pickle**

 A dictionary containing the numerical simulation results from applying various quantum optimization techniques to constrained portfolio optimization problems as specified in [Portfolio Optimization Test Cases](#portfolio-optimization-test-cases).
The layout of the dictionary is as follows

```python
{
    "single":  # single-constraint problems
    {
        "penalty": <pandas.DataFrame>,  # results from QAOA with penalty terms
        "zeno": <pandas.DataFrame>,  # results from QAOA with quantum Zeno dynamics
        "lvqe": <pandas.DataFrame>,  # results from L-VQE with quantum Zeno dynamics
    },
    "multi":  # multi-constraint problems
    {
        "zeno": <pandas.DataFrame>,  # results from QAOA with quantum Zeno dynamics
        "lvqe": <pandas.DataFrame>,  # results from L-VQE with quantum Zeno dynamics
    }
}
```

Columns in the `pandas.DataFrame`s:

* `num_assets`: number of assets to choose from in the portfolio optimization problem
* `mixer` (`"penalty"` and `"zeno"` only): mixer used in QAOA circuit
* `nt` (`"penalty"` and `"zeno"` only): number of QAOA layers (i.e. $p$ in paper)
* `initial_state`: initial state feeding into the variational circuit:
  * `uniform` = uniform superposition of all computational basis states
  * `uniform feasible` = uniform superposition of all in-constraint computational basis states
* `betas_opt` (`"penalty"` and `"zeno"` only): optimized QAOA parameters for the mixer layers
* `gammas_opt` (`"penalty"` and `"zeno"` only): optimized QAOA parameters for the phase operator layers
* `thetas_opt` (`"lvqe"` only): optimized variational parameters for the L-VQE circuit
* `cost` (`"penalty"` only): expectation value of the objective function with the penalty terms when evaluated on the optimal solution
* `cost_projected`: expectation value of the original objective function when evaluated on the optimal solution projected into the in-constraint subspace (i.e. dropping all out-of-constraint computational basis states)
* `p_success`: probability of the measured state from the optimized circuit being in constraint
* `cost_scale` (`"penalty"` and `"zeno"` only): a scaling factor on the objective function (objective function with penalty in the `"penalty"` case), see [Objective function rescaling](#objective-function-rescaling) for details
* `penalty_factor` (`"penalty"` only): penalty factor scaled by the range of the original objective function, see [Scaling of the penalty factor](#scaling-of-the-penalty-factor) for details
* `cost_min`: minimum value of the objective function (objective function with penalty in the `"penalty"` case) among all computational basis states
* `cost_max`: maximum value of the objective function (objective function with penalty in the `"penalty"` case) among all computational basis states
* `cost_projected_min`: minimum value of the original objective function among all in-constraint computational basis states
* `cost_projected_max`: maximum value of the original objective function among all in-constraint computational basis states
* `ar` (`"penalty"` only): approximation ratio with the penalty terms: `(cost_max - cost) / (cost_max - cost_min)`
* `ar_projected`: approximation ratio with the original objective function in the in-constraint subspace: `(cost_projected_max - cost_projected) / (cost_projected_max - cost_projected_min)`
* `method`: quantum optimization technique used:
  * `penalty`: QAOA with constraints enforced by penalty terms
  * `zeno`: QAOA with constraints enforced by quantum Zeno dynamics
  * `lvqe`: L-VQE with constraints enforced by quantum Zeno dynamics
* `eta` (`"zeno"` only): hyperparameter in the quantum Zeno dynamics approach to determine the number of measurements used, see paper for details
* `total_num_meas` (`"zeno"` and `"lvqe"` only): total number of projective measurements used in the circuit for quantum Zeno dynamics with the optimal parameters specified in `betas_opt` and `gammas_opt` (`"zeno"`) or `thetas_opt` (`"lvqe"`)

### **2. penalty_parameter_transfer.pickle**

A `pandas.DataFrame` containing results from applying optimized parameters (`betas_opt`, `gammas_opt`) with a certain penalty factor (`penalty_factor_opt`) to circuits with different penalty factors (`penalty_factor`).
Specifically, in the row with `penalty_factor = penalty_factor_opt`, we run optimization with a penalty factor of `penalty_factor_opt`, and get the optimal QAOA parameters `betas_opt` and `gammas_opt`.
Then we apply these `betas_opt` and `gammas_opt` to circuits with other `penalty_factor` values, and evaluate the circuit without optimization, which generates the entries for the rows with `penalty_factor != penalty_factor_opt`.
All problems considered are from the single-constraint problem set.

Columns in the `DataFrame`:

* `instance_type`: string indicating which set of problems (as specified in [Portfolio Optimization Test Cases](#portfolio-optimization-test-cases)) was used in producing the result
    * `"single"`: single-constraint problems with budget constraint
    * `"multi"`: multi-constraint problems with budget and minimum total expected return constraints
* `num_assets`: number of assets to choose from in the portfolio optimization problem
* `mixer`: mixer used in QAOA circuit
* `nt`: number of QAOA layers (i.e. $p$ in paper)
* `penalty_factor`: penalty factor scaled by the range of the original objective function, see [Scaling of the penalty factor](#scaling-of-the-penalty-factor) for details
* `penalty_factor_opt`: penalty factor (scaled by the range of the original objective function) used to get the optimized QAOA parameters `betas_opt` and `gammas_opt`
* `cost`: expectation value of the objective function with the penalty terms when evaluated on the optimal solution
* `cost_projected`: expectation value of the original objective function when evaluated on the optimal solution projected into the in-constraint subspace (i.e. dropping all out-of-constraint computational basis states)
* `p_success`: probability of the measured state from the optimized circuit being in constraint
* `ar`: approximation ratio with the penalty terms: `(cost_max - cost) / (cost_max - cost_min)`
* `ar_projected`: approximation ratio with the original objective function in the in-constraint subspace: `(cost_projected_max - cost_projected) / (cost_projected_max - cost_projected_min)`

### **3. penalty_factor_sweep_1d.pickle**

A `pandas.DataFrame` containing results from running QAOA with penalty terms applied on the single-constraint problems with varying penalty factors, i.e. we are running a penalty factor sweeping.

Columns in the `DataFrame`:

* `num_assets`: number of assets to choose from in the portfolio optimization problem
* `mixer`: mixer used in QAOA circuit
* `nt`: number of QAOA layers (i.e. $p$ in paper)
* `initial_state`: initial state feeding into the variational circuit:
  * `uniform` = uniform superposition of all computational basis states
  * `uniform feasible` = uniform superposition of all in-constraint computational basis states
* `betas_opt`: optimized QAOA parameters for the mixer layers
* `gammas_opt`: optimized QAOA parameters for the phase operator layers
* `cost`: expectation value of the objective function with the penalty terms when evaluated on the optimal solution
* `cost_projected`: expectation value of the original objective function when evaluated on the optimal solution projected into the in-constraint subspace (i.e. dropping all out-of-constraint computational basis states)
* `p_success`: probability of the measured state from the optimized circuit being in constraint
* `cost_scale`: a scaling factor on the objective function (with penalty), see [Objective function rescaling](#objective-function-rescaling) for details
* `penalty_factor`: penalty factor scaled by the range of the original objective function, see [Scaling of the penalty factor](#scaling-of-the-penalty-factor) for details
* `cost_min`: minimum value of the objective function (with penalty) among all computational basis states
* `cost_max`: maximum value of the objective function (with penalty) among all computational basis states
* `cost_projected_min`: minimum value of the original objective function among all in-constraint computational basis states
* `cost_projected_max`: maximum value of the original objective function among all in-constraint computational basis states
* `ar`: approximation ratio with the penalty terms: `(cost_max - cost) / (cost_max - cost_min)`
* `ar_projected`: approximation ratio with the original objective function in the in-constraint subspace: `(cost_projected_max - cost_projected) / (cost_projected_max - cost_projected_min)`

### **4. penalty_factor_sweep_2d.pickle**

A `pandas.DataFrame` containing results from running QAOA with penalty terms applied on the multi-constraint problems with varying penalty factors, i.e. we are running a penalty factor sweeping.

Columns in the `DataFrame`:

* `num_assets`: number of assets to choose from in the portfolio optimization problem
* `mixer`: mixer used in QAOA circuit
* `nt`: number of QAOA layers (i.e. $p$ in paper)
* `initial_state`: initial state feeding into the variational circuit:
  * `uniform` = uniform superposition of all computational basis states
  * `uniform feasible` = uniform superposition of all in-constraint computational basis states
* `betas_opt`: optimized QAOA parameters for the mixer layers
* `gammas_opt`: optimized QAOA parameters for the phase operator layers
* `cost`: expectation value of the objective function with the penalty terms when evaluated on the optimal solution
* `cost_projected`: expectation value of the original objective function when evaluated on the optimal solution projected into the in-constraint subspace (i.e. dropping all out-of-constraint computational basis states)
* `p_success`: probability of the measured state from the optimized circuit being in constraint
* `cost_scale`: a scaling factor on the objective function (with penalty), see [Objective function rescaling](#objective-function-rescaling) for details
* `penalty_factor_1`: penalty factor (scaled by the range of the original objective function) applied on the penalty term for the budget constraint
* `penalty_factor_2`: penalty factor (scaled by the range of the original objective function) applied on the penalty term for the minimum total expected return constraint
* `cost_min`: minimum value of the objective function (with penalty) among all computational basis states
* `cost_max`: maximum value of the objective function (with penalty) among all computational basis states
* `cost_projected_min`: minimum value of the original objective function among all in-constraint computational basis states
* `cost_projected_max`: maximum value of the original objective function among all in-constraint computational basis states
* `ar`: approximation ratio with the penalty terms: `(cost_max - cost) / (cost_max - cost_min)`
* `ar_projected`: approximation ratio with the original objective function in the in-constraint subspace: `(cost_projected_max - cost_projected) / (cost_projected_max - cost_projected_min)`

### **5. zeno_eta_sweep.pickle**

A `pandas.DataFrame` containing results from running QAOA with quantum Zeno dynamics applied on the single- and multi-constraint problems with varying constraint enforcing strengths controlled by $\eta$ (see paper for details).
In all simulations, the initial state is always a uniform superposition of all in-constraint computational basis states.

Columns in the `DataFrame`:

* `instance_type`: string indicating which set of problems (as specified in [Portfolio Optimization Test Cases](#portfolio-optimization-test-cases)) was used in producing the result
    * `"single"`: single-constraint problems with budget constraint
    * `"multi"`: multi-constraint problems with budget and minimum total expected return constraints
* `num_assets`: number of assets to choose from in the portfolio optimization problem
* `mixer`: mixer used in QAOA circuit
* `nt`: number of QAOA layers (i.e. $p$ in paper)
* `cost_scale`: a scaling factor on the objective function (with penalty), see [Objective function rescaling](#objective-function-rescaling) for details
* `eta`: hyperparameter in the quantum Zeno dynamics approach to determine the number of measurements used, see paper for details
* `eta_opt`: the $\eta$ value used to obtain the optimal QAOA parameters ($\beta$ and $\gamma$)
    * when `eta` and `eta_opt` are equal, the data in the row shows the result from optimizing the QAOA circuit with the specified `eta` value
    * when `eta` and `eta_opt` are different, the data in the row shows the result from optimizing the QAOA circuit with $\eta =$ `eta_opt` and then apply the optimal $\beta$ and $\gamma$ onto a QAOA circuit with  $\eta =$ `eta`
* `ar_projected`: approximation ratio with the original objective function in the in-constraint subspace: `(cost_projected_max - cost_projected) / (cost_projected_max - cost_projected_min)`
* `p_success`: probability of the measured state from the optimized circuit being in constraint

### **6. h1_2_experiments.pickle**

Contains raw counts and qaoa params from the execution of circuits on the Quantinuum H1-2. Used by `experiment_plots.ipynb`. It is a pickled dictionary with the following structure:
```python
    { 
        "[constraint type]": { 
                                "constraint" : "[string presenting constraint]", 
                                "num_shots" : <int>, 
                                "beta": <float>, 
                                "gamma": <float>,
                                "[circuit type]" : [ {
                                                        "raw_counts" : <dict>, 
                                                        "n_x" : <int> # number of segments
                                                     }, ...]},
        ...
    }
```
`raw_counts` contains the results from measuring the auxilary qubits. The keys in the counts dictionary of are the form `[problem_variables(big-endian)] [auxliary_bits(big-endian))]`.

## Notes on Simulation Procedures

### Objective function rescaling

In the QAOA simulations, we rescale the objective function (objective function with penalty when using QAOA with penalty terms) by a multiplicative factor `cost_scale`.
Effectively, this is equivalent to multiplying the phase operator parameters $\gamma$ by `cost_scale` before plugging them into the variational circuit.
The purpose of this rescaling is to ensure that the gradients of the expected cost w.r.t. $\gamma$ and $\beta$ are roughly the same, so to ease the gradient descent by the classical optimizer.
We choose the value of `cost_scale` by visually examining the expected cost profile as a function of $\gamma$ and $\beta$ at $p=1$ (i.e. single-layer QAOA).

### Scaling of the penalty factor

For ease of comparison, we display all penalty factors (here and in the paper) scaled by the range of the objective function as defined by the difference between the maximum and minimum values of the objective function across all Boolean inputs (i.e. computational basis states).

## Quantum circuits

* `equality_qcl_N_*.qasm` contains the circuits for equality constrained problem with the semiclassical QFT (i.e. QCL)
* `equality_qft_N_*.qasm` contains the circuits for equality constrained problem with without QCL (i.e. regular QFT)
* `inequality_qcl_N_*.qasm` contains the circuits for inequality constrained problem

## Notebooks

1. **measurement_scaling_plots.ipynb** (*Figure 1*): Generates a plot showing how the number of measurements required grows with the mixer parameter $\beta$ in order to maintain a given minimum in-constraint probability $\delta_{\min}$, for `x` and `plus` mixers.

2. **simulation_plots.ipynb** (*Figures 2 & 3*): Generates the following plots visualizing the simulations results for portfolio optimization using QAOA with penalty terms and quantum Zeno dynamics (QZD):
    * (*Figure 2, a-d*) Approximation ratio and in-constraint probability using QAOA with penalty terms and QAOA with QZD applied to single-constraint portfolio optimization problems with 4 - 10 assets. Both `x` and `plus` mixers were considered.
    * (*Figure 2, e-f*) Approximation ratio and in-constraint probability using QAOA with QZD applied to multi-constraint portfolio optimization problems with 4 - 10 assets. Both `x` and `plus` mixers were considered.
    * (*Figure 3*) Approximation ratio including the penalty term using QAOA with penalty terms applied to single-constraint portfolio optimization problems with 4 - 10 assets. Both `x` and `plus` mixers were considered.

3. **zeno_eta_sweep_plots.ipynb** (*Figure 4*): Generates plots of the in-constraint probability and approximation ratio of optimal solutions obtained using QAOA with quantum Zeno dynamics with varying hyperparameter $\eta$.
A QAOA with $p=1$ layer and $p=5$ layers are used with `x` mixer applied to the single-constraint portfolio optimization problem with 4 assets in the pool.
Also shows results from transferring the optimized QAOA parameters ($\beta$ and $\gamma$) with a certain $\eta$ to circuits with different $\eta$'s.

4. **penalty_sweep_plots.ipynb** (*Figures 5-7*): Generates plots of the performance of the QAOA with penalty terms approach with varying penalty factors.
   * (*Figure 5*) Approximation ratio with penalty ($r_\text{penalty}$), approximation ratio ($r$), and in-constraint probability ($\delta$) of the solution obtained by QAOA with constraints enforced through penalties with parameters transferred from a fixed value of penalty factor $\lambda = 0.1$ (source marked with a star). Figure shows results for `plus` mixer and $p = 1$ QAOA layer applied on the 4-asset multi-constraint portfolio optimization problem.
   * (*Figure 6*) Approximation ratio with penalty ($r_\text{penalty}$), approximation ratio ($r$), and in-constraint probability ($\delta$) of the solution obtained by QAOA with a single constraint enforced through a penalty term with varying penalty factors $\lambda$. Figure shows results for `plus` mixer and $p = 1$ QAOA layer applied on the 4-asset single-constraint portfolio optimization problem.
   * (*Figure 7*) In-constraint probability of optimized solution using the penalty approach with varied penalty factors on the maximum budget and minimum return constraints. Figure shows results for `plus` mixer and $p = 3$ QAOA layers applied on the 4-asset multi-constraint portfolio optimization problem.

5. **experiment_plots.ipynb** (*Figures 8 & 9*): Generates plots of results from hardware demonstrations.
    * (*Figure 8 a-b*) Plot of in-constraint probability of $p=1$ QAOA `x` mixer applied to a single equality constraint portfolio optimization problem (a) and a single inequality constraint portfolio optimization problem (b). For the equality constraint the oracle was implement utilizing both the semiclassical QFT (labeled `QCL`) and the coherent QFT (labeled `w/o QCL`).
    * (*Figure 9 a-b*) Plot of meausurement results from p=1 QAOA `x` mixer applied to a single equality constraint portfolio optimization problem (a) and a single inequality constraint portfolio optimization problem (b).

6. **experiment_circuits.ipynb** (*Figures 8 & 9*): Loads QASM files containing circuits executed on hardware and runs them in simulation.
