import scipy.io
import scipy.sparse.linalg
from experiment_4.assemble_exp4 import FE_assemble
from common.euler_method import euler_method_sparse
import numpy as np
import os
import pickle
import dill # supports pickling of functions

### We consider the ordinary differential equation: Find u(t) in R^n such that
### Mu_t(t) + Au(t) = f and u(0) = u_0.
### In this file we compute and store the high fidelity FE solution and FE matrices.


# 2D domain with nx/ny elements in x/y direction:
domain = [[0,2.2],[0,0.6]]
nx = 220
ny = 60

# Time interval and number of time steps:
T_start = 0
T_finish = 10
nt = 500

# Coefficient:
SPE10_coeff_matrix = scipy.io.loadmat('perm_40.mat')['perm']

def alpha_t_1(t): return 10**3* ((t>=3)*(t<=7.5)+(t>=8)*(t<=10))
def alpha_x_1(x): return (x>=0.5)*(x<=0.6)
def alpha_y_1(y): return (y>=0.2)*(y<=0.6)
def alpha_t_2(t): return 10**3* ((t>=3)*(t<=7.5)+(t>=8)*(t<=10))
def alpha_x_2(x): return (x>=1)*(x<=1.1)
def alpha_y_2(y): return (y>=0.2)*(y<=0.6)
def alpha_t_3(t): return 10**3* ((t>=8)*(t<=10))
def alpha_x_3(x): return (x>=1.6)*(x<=1.7)
def alpha_y_3(y): return (y>=0.2)*(y<=0.6)
alpha_t = [alpha_t_1,alpha_t_2,alpha_t_3]
alpha_x = [alpha_x_1,alpha_x_2,alpha_x_3]
alpha_y = [alpha_y_1,alpha_y_2,alpha_y_3]

# Initial condition:
def u0(x,y): return np.ones_like(x)*(x>=0.5)*(x<=0.7)*(y>=0.3)*(y<=0.4)


# Assemble FE matrices etc:
grid_x, grid_y, grid_t, mass, stiff, stiffs_coeff_in_time, alpha_matrix = FE_assemble(domain, nx, ny, T_start, T_finish,nt, alpha_t, alpha_x, alpha_y, SPE10_coeff_matrix)

# Rhs vector/matrix with f=0, g_D=0 bottom, g_N=0 left and right, g has values 1 or 5 on top at [0.4,1.8] for two time intervals:
rhs_vector = np.zeros((nx+1)*ny)
rhs_vector[(nx+1)*(ny-1)+40:(nx+1)*(ny-1)+181] = np.hstack((1/200,1/100*np.ones(139),1/200))
rhs_matrix = np.zeros(((nx+1)*ny, nt+1))
def rhs_t(t): return (t>=1)*(t<=5.5) + 5*(t>=8)*(t<=9)
for i in range(len(grid_t)):
    rhs_matrix[:,i] += rhs_t(grid_t[i]) * rhs_vector

# Assemble discrete initial condition:
X,Y = np.meshgrid(grid_x,grid_y[1:])
u_0 = u0(X,Y).flatten()

# Compute high fidelity FE solution:
lhs_solves = []
ht = grid_t[1]-grid_t[0]
for t in range(*(1,nt+1)):
    stiffness_temp = stiffs_coeff_in_time[t]
    matrix_left_temp = mass + ht * stiffness_temp
    lhs_solve_temp = scipy.sparse.linalg.factorized(matrix_left_temp.tocsc())
    lhs_solves.append(lhs_solve_temp)

solution = euler_method_sparse(u_0, T_start, T_finish, nt, mass, lhs_solves, rhs_matrix)

# Store all results:
os.makedirs('results', exist_ok=True)
results = {}
results['FE_solution'] = solution
results['grid_t'] = grid_t
results['grid_x'] = grid_x
results['grid_y'] = grid_y
results['mass'] = mass
results['stiff'] = stiff
results['stiffs_coeff_in_time'] = stiffs_coeff_in_time
results['rhs_matrix'] = rhs_matrix
results['alpha_matrix'] = alpha_matrix
results['u0_discrete'] = u_0
pickle.dump(results, open('results/FE_data_problem=SPE10.pickle', 'wb'))
dill.dump(rhs_t, open('results/rhs_t_problem=SPE10.dill', 'wb'))